/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

/* 
   TODO:

   - handle keyboard remap on the fly. 

 */

#include "structs.h"
#include "keys.h"

#ifndef NO_KBD

static Bool
keys_keysym_needs_shift(Wm *w, KeySym keysym)
{
  int col;     /* XXX double check this is correct / easiest way  */
  int keycode;
  KeySym k;
  int min_kc, max_kc;

  XDisplayKeycodes(w->dpy, &min_kc, &max_kc);

  for (keycode = min_kc; keycode <= max_kc; keycode++) {
    for (col = 0; (k = XKeycodeToKeysym (w->dpy, keycode, col))
           != NoSymbol; col++)
      if (k == keysym && col == 1) {
        return True;
      }
  }  
  return False;
}

Bool
keys_add_entry(Wm *w, char *keystr, int action, int idata, char *sdata)
{
  char *p = keystr;
  char *q;
  int i = 0, mask = 0;
  char *keydef = NULL;
  KeySym ks;
  MBConfigKbdEntry *entry;

  MBConfigKbd *kb =  w->config->kb;

  struct {

    char *def;
    int mask;

  } lookup[] = {
    { "ctrl", ControlMask },
    { "alt", kb->AltMask },
    { "meta", kb->MetaMask },
    { "super", kb->SuperMask },
    { "hyper", kb->HyperMask },
    { "mod1", Mod1Mask },
    { "mod2", Mod2Mask },
    { "mod3", Mod3Mask },
    { "mod4", Mod4Mask },
    { "mod5", Mod5Mask },
    { NULL, 0 }
  };

  while (*p != '\0')
    {
      if (*p == '<') 		/* XXX check for esacped < */
	{
	  q = ++p;
	  while (*q != '\0' && *q != '>') q++;
	  if (*q == '\0') return False; /* Parse error */
	  
	  i = 0;
	  while (lookup[i].def != NULL)
	    {
	      if (!strncasecmp(p, lookup[i].def, q-p))
		{
		  mask |= lookup[i].mask;
		}
	      i++;
	    }
	  p = q;
	  /* XXX check something is actually found 
	     XXX check for things like <tab>
	  */
	} 
      else if (!isspace(*p)) {
	  /* set the keysym */
	keydef = p;
	break;
      }
      p++;
    }

  if (!keydef) return False;

  dbg("%s() keydefinition is %s\n", __func__, keydef);

  if ((ks = XStringToKeysym(keydef)) == (KeySym)NULL)
    {
      dbg("%s() failed to look up %s\n", __func__, keydef);
      return False;
    }

  if (keys_keysym_needs_shift(w, ks))
    mask |= ShiftMask;

  if (w->config->kb->entrys == NULL)
    {
      w->config->kb->entrys = malloc(sizeof(MBConfigKbdEntry));
      entry = w->config->kb->entrys;
    }
  else
    {
      entry = w->config->kb->entrys;
      while (entry->next_entry != NULL) 
	entry = entry->next_entry; 
    
      entry->next_entry = malloc(sizeof(MBConfigKbdEntry));
      entry = entry->next_entry;
    }

  entry->next_entry   = NULL;
  entry->action       = action;
  entry->ModifierMask = mask;
  entry->key          = ks;
  entry->idata        = idata;
  entry->sdata        = (( sdata == NULL ) ? NULL : strdup(sdata));

  dbg("added new key entry mask: %i\n", entry->ModifierMask);

  return True;
}
 

Bool
keys_load_config(Wm *w)
{
  FILE *fp;
  char data[256];
  char *key = NULL, *val = NULL, *str = NULL;
  int i = 0;
  struct stat stat_info;
  char *conf_path = NULL;

  struct {

    char *str;
    int act;

  } lookup[] = {
    { "next", KEY_ACTN_NEXT_CLIENT },
    { "prev", KEY_ACTN_PREV_CLIENT },
    { "close", KEY_ACTN_CLOSE_CLIENT },
    { "desktop", KEY_ACTN_TOGGLE_DESKTOP },
    { "taskmenu", KEY_ACTN_TASK_MENU_ACTIVATE },
    { NULL, 0 }
  };

  if (getenv("HOME"))
    {
      conf_path = malloc(sizeof(char)*(strlen(getenv("HOME"))+15));
      sprintf(conf_path, "%s/.mbkbdconfig", getenv("HOME"));
      if (stat(conf_path, &stat_info) == -1)
	{
	  free(conf_path);
	  conf_path = NULL;
	}
    }

  if (conf_path == NULL) conf_path = strdup(PREFIX "/kbdconfig");

  if (!(fp = fopen(conf_path, "r"))) 
    {
      free(conf_path);
      return False;
    }

  dbg("%s() parsing keyboard config: %s\n", __func__, conf_path);

  free(conf_path);

  while(fgets(data,256,fp) != NULL)
    {
      if (data[0] == '#') 
	continue;
      str = strdup(data);
      if ( (val = strchr(str, '=')) != NULL)
	{
	  *val++ = '\0'; key = str;
	  if (*val != '\0')
	    {
	      int action = 0;
	      if (val[strlen(val)-1] == '\n') val[strlen(val)-1] = '\0';

	      if (*val == '!')
		{		/* It some kind of exec */
		  val++;
#ifdef USE_LIBSN

		  if (*val == '!')
		    {
		      action = KEY_ACTN_EXEC_SN;
		      val++;
		    }
		  else if (*val == '$')
		    {
		      action = KEY_ACTN_EXEC_SINGLE;
		      val++;
		    }
		  else 
#else
		    if ((*(val+1) == '!') || (*(val+1) == '$'))
		      val++;
#endif
		    action = KEY_ACTN_EXEC;
		}
	      else 
		{
		  i = 0;
		  while (lookup[i].str != NULL)
		    {
		      if (!strncasecmp(lookup[i].str, val, 
				       strlen(lookup[i].str)))
			{
			  action = lookup[i].act;
			  break;
			}
		      i++;
		    }
		}
	      if (!(action && keys_add_entry(w, key, action, 0, val)))
		fprintf(stderr, "matchbox: failed to parse %s\n", data);

	    }
	}
      free(str);
    }
  fclose(fp);

  return 1;
}

void
keys_get_modifiers(Wm *w)
{
  int mod_idx, mod_key, col, kpm;
  XModifierKeymap *mod_map = XGetModifierMapping(w->dpy);

  MBConfigKbd *kbd =  w->config->kb;

  kbd->MetaMask = 0;
  kbd->HyperMask = 0;
  kbd->SuperMask = 0;
  kbd->AltMask = 0;
  kbd->ModeMask = 0; 
  kbd->NumLockMask = 0;
  kbd->ScrollLockMask = 0;

  kpm = mod_map->max_keypermod;
  for (mod_idx = 0; mod_idx < 8; mod_idx++)
    for (mod_key = 0; mod_key < kpm; mod_key++) 
      {
	KeySym last_sym = 0;
	for (col = 0; col < 4; col += 2) 
	  {
	    KeyCode code = mod_map->modifiermap[mod_idx * kpm + mod_key];
	    KeySym sym = (code ? XKeycodeToKeysym(w->dpy, code, col) : 0);

	    if (sym == last_sym) continue;
	    last_sym = sym;

	    switch (sym) 
	      {
	      case XK_Mode_switch:
		/* XXX store_modifier("Mode_switch", mode_bit); */
		break;
	      case XK_Meta_L:
	      case XK_Meta_R:
		kbd->MetaMask |= (1 << mod_idx); 
		break;
	      case XK_Super_L:
	      case XK_Super_R:
		kbd->SuperMask |= (1 << mod_idx);
		break;
	      case XK_Hyper_L:
	      case XK_Hyper_R:
		kbd->HyperMask |= (1 << mod_idx);
		break;
	      case XK_Alt_L:
	      case XK_Alt_R:
		kbd->AltMask |= (1 << mod_idx);
		break;
	      case XK_Num_Lock:
		kbd->NumLockMask |= (1 << mod_idx);
		break;
	      case XK_Scroll_Lock:
		kbd->ScrollLockMask |= (1 << mod_idx);
		break;
	      }
	  }
      }

  if (mod_map) XFreeModifiermap(mod_map);

}

void
keys_grab(Wm *w)
{
  MBConfigKbdEntry *entry =  w->config->kb->entrys;

  while (entry != NULL)
    {
      dbg("keys, grabbing %i , %i\n", 
	  XKeysymToKeycode(w->dpy, entry->key), entry->ModifierMask);
      XGrabKey(w->dpy, XKeysymToKeycode(w->dpy, entry->key), 
	       entry->ModifierMask,
	       w->root, True, GrabModeAsync, GrabModeAsync);
      entry = entry->next_entry;
    }
}

void
keys_init(Wm *w)
{
  w->config->kb = malloc(sizeof(MBConfigKbd));

  w->config->kb->entrys = NULL;
  
  keys_get_modifiers(w);

  if (!keys_load_config(w))
    {
      fprintf(stderr, "matchbox: failed to load keyboard config\n");
      return;
    }

  keys_grab(w);

}

#endif 
