/*
   mbmenu - a small application launcher

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

/*
  TODO
  o fix crash on menu's bigger than display.
  o add logout / lock buttons on root menu.
     - add use_gpe defines for this !
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <signal.h>
#include <sys/wait.h>

#ifdef USE_DNOTIFY
#include <linux/fcntl.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>
#include <X11/Xresource.h>
#include <X11/extensions/shape.h>

#include "mbtray.h"
#include "mbmenu.h"
#include "mbutil.h"
#include "mbdotdesktop.h"
#include "mbpixbuf.h"
#include "config.h"

#ifdef USE_XFT
#include <X11/Xft/Xft.h>
#endif

#ifdef USE_LIBSN
#define SN_API_NOT_YET_FROZEN 1
#include <libsn/sn.h>
#endif

#ifndef USE_PNG
#include <setjmp.h>
#endif

#ifdef USE_XSETTINGS
#include <xsettings-client.h>
#endif 

#define SINGLE_INSTANCE 1
#define STARTUP_NOTIFY  2

#define UP   0
#define DOWN 1

#define MAX_DISPLAYS 10


typedef struct _app
{
  Display *dpy;
  Window root;
  int screen;
   
#ifdef USE_XFT
  XftFont *xftfont;
  XftColor fg_xftcol;
  XftColor bg_xftcol;
  XftColor hl_xftcol;

#else
  XFontStruct* font;
#endif
  XColor   fg_xcol;
  XColor   bg_xcol;
  XColor   hl_xcol;
  GC gc;
  GC mask_gc;
   
  MBMenu* mbmenu;

  Pixmap pxm_icon; /* Dock Icon */
  Pixmap pxm_icon_down;

  MBPixbuf* pb;
  MBPixbufImage *img_tray, *img_backing;
  XColor xcol_bg; 
   
  char icon_fn[64]; 
  int width;
  int height;
  Window win_panel;

  Bool no_dock;
  int gx,gy;

  char *display_names[MAX_DISPLAYS];
  int display_name_cnt;

  char *theme_name;

#ifdef USE_LIBSN
  SnDisplay  *sn_display;
#endif

#ifdef USE_XSETTINGS
  XSettingsClient *xsettings_client;
#endif 

  Atom mbtheme_atom;
  Atom mbcommand_atom;
   
} App;

App    *app;     /* Global for sig callback - way round this ? */
Menu   *root;
Bool    Update_Pending = False;
jmp_buf Jbuf;
Menu   *active[10];
int     menu_is_active = False;

static void reap_children(int signum);
static void fork_exec(char *cmd);
#ifdef USE_DNOTIFY
static void reload_menu(int sig, siginfo_t *si, void *data);
#endif
static void usage();
static int parse_menu_file(char *data);
static void build_menu(void);
static void catch_sigsegv(int sig);
static void paint_dock(int state);
static void menu_set_theme_from_root_prop();
#ifdef USE_LIBSN
static void sn_exec(char* name, char* bin_name, char *desc);
#endif
#ifdef USE_XSETTINGS
static void 
menu_xsettings_notify_cb (const char       *name,
			  XSettingsAction   action,
			  XSettingsSetting *setting,
			  void             *data);
#endif

#define MENUDIR "/usr/lib/menu"

#define MAX(x,y) ((x>y)?(x):(y))
#define NEW(OBJ) ((OBJ *)(malloc(sizeof(OBJ))))
#define IS_WHITESPACE(c) ((c) == ' ' || (c) == '\\' \
                          || (c) == '\t' || (c)=='\n' || (c)=='?')

#ifdef DEBUG
#define DBG(txt, args... ) fprintf(stderr, "DEBUG: " txt , ##args )
#else
#define DBG(txt, args... ) /* nothing */
#endif



/* ---------------------------- App --------------------------------- */

static void usage()
{
  printf("usage: mbmenu [options ....]\n"
	  "Where options are\n"
	  "  -display <display> Display to connect to\n"
          "\n Mbmenu Version %s"
	  "\n", VERSION

	  );
   exit(1);
}

static App *
new_app(int argc, char **argv)
{
  int i;
   XGCValues gv;

   char fgdef[32] = "black";
   char bgdef[32] = "grey";
   char hldef[32] = "white";
   char bddef[32] = "black";

#ifdef USE_PNG
   char *app_undef_img_filename = DATADIR "/matchbox/mbmenu/app.png";
   char *dir_undef_img_filename = DATADIR "/pixmaps/mbfolder.png";
#else
   char *app_undef_img_filename = DATADIR "/matchbox/mbmenu/app.xpm";
   char *dir_undef_img_filename = DATADIR "/matchbox/mbmenu/folder.xpm";
#endif

   MBMenuOptions mbmenu_ops;
   int           mbmenu_mask = MBMENU_FG_COL|MBMENU_BG_COL|MBMENU_HL_COL
                               |MBMENU_FONT|MBMENU_ICON_SZ|MBMENU_ICON_FN
                               |MBMENU_ICON_FOLDER_FN|MBMENU_BD_COL;

   int border_width;
   
   char *display_name = NULL;
#ifdef USE_XFT	  
   char *fontname = "verdana-7";
#else
   char *fontname = "fixed";
#endif

   App *app = NEW(App);

   mbmenu_ops.fontname = fontname;
   mbmenu_ops.foreground_col_spec = fgdef;
   mbmenu_ops.background_col_spec = bgdef;
   mbmenu_ops.highlight_col_spec  = hldef;
   mbmenu_ops.border_col_spec     = bddef;
   mbmenu_ops.border_size         = 1;
   mbmenu_ops.icon_dimention      = 16;
   mbmenu_ops.default_icon_filename = app_undef_img_filename;
   mbmenu_ops.default_folder_icon_filename = dir_undef_img_filename;
   mbmenu_ops.bevel_size          = 1;

   border_width = 0;
   app->icon_fn[0] = (char)NULL;
   //icon_ = 16;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      display_name = argv[i];
      continue;
    }
    usage(argv[0]);
  }

   if ((app->dpy = XOpenDisplay(display_name)) == NULL)
   {
      printf("mbmenu: Cant open display\n"); exit(1);
   }

   app->screen = DefaultScreen(app->dpy);
   app->root   = RootWindow(app->dpy, app->screen);

   app->mbtheme_atom = XInternAtom(app->dpy, "_MB_THEME", False);
   app->mbcommand_atom = XInternAtom(app->dpy, "_MB_COMMAND", False);

   app->theme_name = NULL;

#ifdef USE_XSETTINGS

  app->xsettings_client = xsettings_client_new(app->dpy, app->screen,
					      menu_xsettings_notify_cb,
					      NULL, (void *)app );
#endif

   gv.graphics_exposures = False;
   gv.function   = GXcopy;
   gv.foreground = (app->fg_xcol).pixel;
   app->gc = XCreateGC(app->dpy, app->root,
			GCGraphicsExposures|GCFunction|GCForeground, &gv);

   app->mbmenu = mb_menu_new(app->dpy, app->screen, mbmenu_mask, &mbmenu_ops);

   app->pb = mb_pixbuf_new(app->dpy, app->screen);

#ifdef USE_PNG
#define TRAY_IMG "mbmenu.png"
#else
#define TRAY_IMG "mbmenu.xpm"
#endif

   if (!(app->img_tray = mb_pixbuf_img_new_from_file(app->pb, DATADIR "/pixmaps/" TRAY_IMG )))
   {
     fprintf(stderr, "mbmenu: failed to get dock image\n");
     exit(1);
   }
       
   app->width  = mb_pixbuf_img_get_width(app->img_tray) + 4; 
   app->height = mb_pixbuf_img_get_height(app->img_tray) + 2;
   
   app->win_panel = XCreateSimpleWindow(app->dpy,
					app->root,
					0, 0,
					app->width, app->height, 0,
					BlackPixel(app->dpy, app->screen),
					WhitePixel(app->dpy, app->screen));

   app->pxm_icon = XCreatePixmap(app->dpy, app->win_panel, 
				 app->width, app->height,
				 app->pb->depth);

   app->pxm_icon_down = XCreatePixmap(app->dpy, app->win_panel, 
				      app->width, app->height,
				      app->pb->depth);

   XStoreName(app->dpy, app->win_panel,"App Launcher");

   XSelectInput(app->dpy, app->win_panel, 
		StructureNotifyMask|ExposureMask|
		ButtonPressMask|ButtonReleaseMask ); 



   return app;
}


/* ------------------ signals etc ------------------------------------ */

static void catch_sigsegv(int sig)
{
   DBG("ouch\n");
   signal(SIGSEGV, SIG_DFL);
   longjmp(Jbuf, 1);

}

static void reap_children(int signum)
{
    pid_t pid;

    do {
	pid = waitpid(-1, NULL, WNOHANG);
    } while (pid > 0);
}

#ifdef USE_LIBSN

static void 
sn_exec(char* name, char* bin_name, char *desc)
{
  SnLauncherContext *context;
  pid_t child_pid = 0;

  context = sn_launcher_context_new (app->sn_display, app->screen);
  
  if (name)     sn_launcher_context_set_name (context, name);
  if (desc)     sn_launcher_context_set_description (context, desc);
  if (bin_name) sn_launcher_context_set_binary_name (context, bin_name);
  
  sn_launcher_context_initiate (context, "mbmenu launch", bin_name,
				CurrentTime);

  switch ((child_pid = fork ()))
    {
    case -1:
      fprintf (stderr, "Fork failed\n" );
      break;
    case 0:
      sn_launcher_context_setup_child_process (context);
      /* execlp(bin_name, bin_name, NULL); */
      mb_exec(bin_name);
      fprintf (stderr, "mbmenu: Failed to exec %s \n", bin_name);
      _exit (1);
      break;
    }
}

#endif

static void fork_exec(char *cmd)
{
    pid_t pid;
    
    pid = fork();
    
    switch (pid) 
      {
      case 0:
	mb_exec(cmd);
	/* execlp("/bin/sh", "sh", "-c", cmd, NULL); */
	fprintf(stderr, "mbmenu: exec failed, cleaning up child\n");
	exit(1);
      case -1:
	fprintf(stderr, "mbmenu: can't fork\n"); break;
      }
}

#ifdef USE_LIBSN
static
void menu_clicked_sn_cb(MenuItem *item)
{
  sn_exec(item->title, item->info, item->info);
}

static
void menu_clicked_si_cb(MenuItem *item)
{
  App *app = (App *)item->cb_data;

  Window win_found;

  win_found = mb_single_instance_get_window(app->dpy, item->info);

  if (win_found != None)
    {
      mb_util_window_activate(app->dpy, win_found);
    }
  else menu_clicked_sn_cb(item);
}
#endif

static
void menu_clicked_cb(MenuItem *item)
{
  fork_exec(item->info);
}

static int
parse_menu_file(char *data)
{
   char *p, *key, *val;

   char *tmp_title;
   char *tmp_section;
   char *tmp_cmd;
   char *tmp_icon;
   char *tmp_needs;

   signal(SIGSEGV, catch_sigsegv);
   if (setjmp(Jbuf)) return 0;  /* catch possible parse segfualt  */

   tmp_title = NULL;
   tmp_section = NULL;
   tmp_cmd = NULL;
   tmp_icon = NULL;
   tmp_needs = NULL;
   p = data;
   
   if (*p != '?') return 0;   /* check is an actual menu entry  */
   
   while(*(++p) != ':'); *p = ' '; /* skip to first entry */
   
   while(*p != '\0')
   {
      if ((!IS_WHITESPACE(*p))
	  || (IS_WHITESPACE(*p) && *(p+1) != '\0'
	      && (!IS_WHITESPACE(*(p+1)))))
      {
	 /* process key=pair */
         char *lc = " \t\n\\";
	 char *sc = "\"";
	 char *tc = lc; 
         if (IS_WHITESPACE(*p)) p++;
	 key = p;
	 while(*p != '=') p++; *p = '\0';
	 DBG("\tkey %s ", key);
         if (*(++p) == '"') { p++; tc = sc; } /* skip "'s */
	 val = p;
	 while(index(tc,*p) == NULL)
	 {
	    if (*p == '\\' && *(p+1) == '"') p++;  /* skip \" */
	    p++;
	 } *p = '\0';
	 DBG("value %s \n", val);

	 if(!strcmp(key,"title"))        { tmp_title = val;  }
	 else if(!strcmp(key,"section")) { tmp_section = val; }
	 else if(!strcmp(key,"command")) { tmp_cmd = val; }
	 else if(!strcmp(key,"icon"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon16"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon32"))    { tmp_icon = val; }
	 else if(!strcmp(key,"icon48"))    { tmp_icon = val; }
	 else if(!strcmp(key,"needs"))    { tmp_needs = val; }

      }
      p++;

      if (tmp_section && (*p == '?' || *p == '\0'))
      {
	 if ( (!strcmp(tmp_needs,"x11"))
	      || (!strcmp(tmp_needs,"X11"))
	      || (!strcmp(tmp_needs,"text"))
	      || tmp_needs == NULL )
	 {
	   Menu *m = NULL;

	   char *tmpstr = 
	     (char *)malloc(sizeof(char)*(strlen(tmp_section)+
					  strlen(tmp_title)+12));
	   sprintf(tmpstr, "Other/%s", tmp_section);

	   m = mb_menu_add_path(app->mbmenu, tmpstr, NULL, 0);
	   mb_menu_add_item_to_menu(app->mbmenu, m, tmp_title, 
				       tmp_icon, tmp_cmd, 
				    menu_clicked_cb, NULL, 0); 
	 }
      }
   }	 
	 /* reset all */
      tmp_title = NULL;
      tmp_section = NULL;
      tmp_cmd = NULL;
      tmp_icon = NULL;
      
      /* new menu entry */
      
      if ( *p == '?')
	 {
	   DBG("new entry, igonoring package(foo) data\n");
	   while(*(++p) != ':');
	   *p = ' ';
	 }
      
   
   signal(SIGSEGV, SIG_DFL);
   return 1;
}


#ifdef USE_XSETTINGS

static void 
menu_xsettings_notify_cb (const char       *name,
			  XSettingsAction   action,
			  XSettingsSetting *setting,
			  void             *data)
{
  App *app = (App *)data;

  DBG("%s called\n", __func__);

  if (!strcmp(name, "Net/ThemeName")
      && setting != NULL 
      && setting->type == XSETTINGS_TYPE_STRING )
    {
      switch (action)
	{
	case XSETTINGS_ACTION_NEW:
	case XSETTINGS_ACTION_CHANGED:
	  if (app->theme_name != NULL) free (app->theme_name);
	  app->theme_name = strdup(setting->data.v_string);
	  if (app->mbmenu != NULL)
	    {
	      DBG("Net/ThemeName XSetting caught\n");
	      mb_menu_free(app->mbmenu);
	      build_menu();
	    }
	  break;
	case XSETTINGS_ACTION_DELETED:
	  break;
	}
    }
}

#endif

#ifdef USE_DNOTIFY
static void reload_menu(int sig, siginfo_t *si, void *data)
{
  DBG("Reload menu callback\n");

  mb_menu_free(app->mbmenu);
  build_menu();

}
#endif

static void
build_menu(void)
{
  int i = 0;
  DIR *dp;
  struct dirent *dir_entry;
  struct stat stat_info;
  char orig_wd[256];

  char *dd_dir = DATADIR "/applications";

  char dirs[2][256] = { MENUDIR };
  FILE *fp;
  char *buf;
  int len;
  Menu *menu_panel;
  char *tmp_path = NULL, *tmp_path2 = NULL ;

  struct _folderlookup { 	/* XXX this hardcodedness is silly ... */

    char *key;
    Menu *menu;

  } folders[] = {
    { "Utility", mb_menu_add_path(app->mbmenu, "Utilities" , NULL, MBMENU_NO_SORT ) },
    { "Game",    mb_menu_add_path(app->mbmenu, "Games" , NULL, MBMENU_NO_SORT)    },
    { "Setting", mb_menu_add_path(app->mbmenu, "Settings" , NULL, MBMENU_NO_SORT) },
    { "Other",    mb_menu_add_path(app->mbmenu, "Other" , NULL, MBMENU_NO_SORT)    },
    { NULL, NULL }
  };

  menu_panel = mb_menu_add_path(app->mbmenu, "Utilities/Panel" , NULL, MBMENU_NO_SORT );

  mb_menu_add_seperator_to_menu(app->mbmenu, app->mbmenu->rootmenu, 
				MBMENU_PREPEND);
  mb_menu_add_seperator_to_menu(app->mbmenu, app->mbmenu->rootmenu, 
				MBMENU_NO_SORT);

  tmp_path = mb_dot_desktop_icon_get_full_path (app->theme_name, 
						16, 
						"mbfolder.png" );

  tmp_path2 = mb_dot_desktop_icon_get_full_path (app->theme_name, 
						16, 
						"mbnoapp.png" );

  mb_menu_set_default_icons(app->mbmenu, tmp_path, tmp_path2);

  if (tmp_path) free(tmp_path);
  if (tmp_path2) free(tmp_path2);

  /* XXX - this should only happen ifdef USE_GPE  */

  tmp_path = mb_dot_desktop_icon_get_full_path (app->theme_name, 
						16, 
						"mbdesktop.png" );

  mb_menu_add_item_to_menu(app->mbmenu, app->mbmenu->rootmenu, 
			   "Desktop",
			   tmp_path,
			   "mbcontrol -desktop",
			   menu_clicked_cb, 
			   NULL, MBMENU_NO_SORT); 

  if (tmp_path) free(tmp_path);


  tmp_path = mb_dot_desktop_icon_get_full_path (app->theme_name, 
						16, 
						"mbexit.png" );


  mb_menu_add_item_to_menu(app->mbmenu, app->mbmenu->rootmenu, 
			   "Log Out",
			   tmp_path,
			   "gpe-logout",
			   menu_clicked_cb, 
			   NULL, MBMENU_NO_SORT); 

  if (tmp_path) free(tmp_path);

  tmp_path = mb_dot_desktop_icon_get_full_path (app->theme_name, 
						16, 
						"mblock.png" );


  mb_menu_add_item_to_menu(app->mbmenu, app->mbmenu->rootmenu, 
			   "Lock Display",
			   tmp_path,
			   "gpe-lock-display",
			   menu_clicked_cb, 
			   NULL, MBMENU_NO_SORT); 

  if (tmp_path) free(tmp_path);

  /* XXXX */

  if (getcwd(orig_wd, 255) == (char *)NULL)
    {
      fprintf(stderr, "mbmenu: cant get current directory\n");
      exit(0);
    }
  
  if ((dp = opendir(dd_dir)) == NULL)
    {
      fprintf(stderr, "mbmenu: failed to open %s\n", dd_dir);
      exit(0);
    }
  
  chdir(dd_dir);
  while((dir_entry = readdir(dp)) != NULL)
    {
      lstat(dir_entry->d_name, &stat_info);
      if (!(S_ISDIR(stat_info.st_mode)))
	{
	  MBDotDesktop *dd;
	  int flags = 0;
	  dd = mb_dotdesktop_new_from_file(dir_entry->d_name);
	  if (dd)
	    {
	      char *png_path = NULL;
	      MBMenuActivateCB activate_callback = NULL;

	      if (mb_dotdesktop_get(dd, "Icon") 
		  && mb_dotdesktop_get(dd, "Name")
		  && mb_dotdesktop_get(dd, "Exec"))
		{
		  Menu *m = NULL;
		  /*
		  unsigned char *icon_str = mb_dotdesktop_get(dd, "Icon");


		  if (icon_str[0] == '/')
		    strcpy(png_path, icon_str);
		  else
		    sprintf(png_path, 
			    "%s/%s", 
			    pixmaps_dir, mb_dotdesktop_get(dd, "Icon"));
		  */
		  
		  png_path = mb_dot_desktop_icon_get_full_path(app->theme_name, 16, mb_dotdesktop_get(dd, "Icon"));
		  
		  if (mb_dotdesktop_get(dd, "Categories") == NULL)
		    {
		      m = folders[3].menu; /* Other */
		    }
		  else if (strstr(mb_dotdesktop_get(dd, "Categories"), "PIM"))
		    {
		      flags = MBMENU_PREPEND | MBMENU_NO_SORT; 
		      m = app->mbmenu->rootmenu;
		    }
		  else 
		    {
		      Bool found = False;
		      i = 0;
		      while( folders[i].key != NULL )
			{			  
			  if (strstr(mb_dotdesktop_get(dd, "Categories"), 
				     folders[i].key))
			    {
			      m = folders[i].menu;
			      found = True;
			      break;
			    }
			  i++;
			}
		      if (!found) m = folders[3].menu; /* Other */
		    }

		  activate_callback = menu_clicked_cb;

#ifdef USE_LIBSN

		  if (mb_dotdesktop_get(dd, "SingleInstance")
		      && !strcasecmp(mb_dotdesktop_get(dd, "SingleInstance"),
				     "true"))
		    {
		      activate_callback = menu_clicked_si_cb;
		    }
		  else if (mb_dotdesktop_get(dd, "StartupNotify")
		      && !strcasecmp(mb_dotdesktop_get(dd, "StartupNotify"),
				     "true"))
		    {
		      activate_callback = menu_clicked_sn_cb;
		    }
#endif
		  if (mb_dotdesktop_get(dd, "Type") 
		      && !strcmp(mb_dotdesktop_get(dd, "Type"), "PanelApp"))
		    {
		      m = menu_panel;
		    }

		  if (png_path)
		    {
		      mb_menu_add_item_to_menu(app->mbmenu, m, 
					       mb_dotdesktop_get(dd, "Name"),
					       png_path,
					       mb_dotdesktop_get(dd, "Exec"),
					       activate_callback, 
					       (void *)app, flags); 
		      /* mb_menu_add_seperator_to_menu(app->mbmenu, m); */

		      free(png_path);
		    }
		}
	      else fprintf(stderr, 
			   "mbmenu: %s has no icon, png or name\n", 
			   dir_entry->d_name);
	      
	      mb_dotdesktop_free(dd);
	    }
	  else
	    fprintf(stderr, "mbmenu: failed to parse %s :( \n", dir_entry->d_name);
	}
    }
  
  closedir(dp);

  /* Now parse old Debian / Familiar Menu entrys */
  chdir(orig_wd);


   strcpy(dirs[1], (char *)getenv("HOME"));
   strcat(dirs[1], "/.menu");

   for(i=0; i<2; i++)
   {
      if ((dp = opendir(dirs[i])) == NULL)
      {
	 fprintf(stderr, "mbmenu: failed to open %s\n", dirs[i]);
	 continue;
      }

      chdir(dirs[i]);
      while((dir_entry = readdir(dp)) != NULL)
      {
	 lstat(dir_entry->d_name, &stat_info);
	 if (!(S_ISDIR(stat_info.st_mode)))
	 {
	    DBG("file %s \n", dir_entry->d_name);
	    
	    fp = fopen(dir_entry->d_name, "r");
	    buf = malloc(sizeof(char) * (stat_info.st_size + 1));
	    len = fread(buf, 1, stat_info.st_size, fp);
	    if (len >= 0) buf[len] = '\0';
	    if (!(parse_menu_file(buf)))
	       fprintf(stderr, "mbmenu: had problems parsing %s. Ignoring. \n",
		       dir_entry->d_name);
	    DBG("done\n\n");
	    fclose(fp);
	    free(buf);
	 }
      }

      closedir(dp);

   }
   chdir(orig_wd);

}


static void
paint_dock(int state)
{

  if (state == UP)
    XSetWindowBackgroundPixmap(app->dpy, app->win_panel, app->pxm_icon);
  else
    XSetWindowBackgroundPixmap(app->dpy, app->win_panel, app->pxm_icon_down);

  XClearWindow(app->dpy, app->win_panel);

}

static void
set_backing( void *user_data )
{
  App *app = (App *)user_data;
  MBPixbufImage *img_backing_clone = NULL;

  app->img_backing = mb_tray_get_bg_img(app->pb, app->win_panel);

  img_backing_clone = mb_pixbuf_img_clone(app->pb, app->img_backing);

  /* Inactive button */
  mb_pixbuf_img_composite(app->pb, app->img_backing, app->img_tray , 2, 1);

  mb_pixbuf_img_render_to_drawable(app->pb, app->img_backing, app->pxm_icon, 
				   0, 0);

  /* Active Button */
  mb_pixbuf_img_composite(app->pb, img_backing_clone, app->img_tray , 2, 2 );

  mb_pixbuf_img_render_to_drawable(app->pb, img_backing_clone, 
				   app->pxm_icon_down, 0, 0);

  mb_pixbuf_img_free(app->pb, img_backing_clone);
  mb_pixbuf_img_free(app->pb, app->img_backing);
  
  paint_dock(UP);
}


void
menu_set_theme_from_root_prop(void)
{
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  char * value;
  struct stat stat_info;
  char app_cfg[256];

  status = XGetWindowProperty(app->dpy, app->root,
			      app->mbtheme_atom,
			      0L, 512L, False,
			      AnyPropertyType, &realType,
			      &format, &n, &extra,
			      (unsigned char **) &value);
	    
  if (status != Success || value == 0
      || *value == 0 || n == 0)
    {
      printf("mbmenu: no _MB_THEME set on root window\n");
    } else {
      strcpy(app_cfg, value);
      strcat(app_cfg, "/theme.desktop");
      if (stat(app_cfg, &stat_info) != -1)
	{
	  MBDotDesktop *theme  = NULL;
	  theme = mb_dotdesktop_new_from_file(app_cfg);
	  if (theme)
	    {
	      if (mb_dotdesktop_get(theme, "MenuBgColor"))
		{
		  mb_menu_set_col(app->mbmenu, BG_COL, 
				  mb_dotdesktop_get(theme, "MenuBgColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuFont"))
		{
		  mb_menu_set_font (app->mbmenu, 
				    mb_dotdesktop_get(theme, "MenuFont"));
		}
	      if (mb_dotdesktop_get(theme, "MenuFgColor"))
		{
		  mb_menu_set_col(app->mbmenu, FG_COL, 
				  mb_dotdesktop_get(theme, "MenuFgColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuHlColor"))
		{
		  mb_menu_set_col(app->mbmenu, HL_COL, 
				  mb_dotdesktop_get(theme, "MenuHlColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuBdColor"))
		{
		  mb_menu_set_col(app->mbmenu, BD_COL, 
				  mb_dotdesktop_get(theme, "MenuBdColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuTransparency"))
		{
		  mb_menu_set_trans(app->mbmenu,  
				    atoi(mb_dotdesktop_get(theme, 
						      "MenuTransparency")));
		}

	      mb_dotdesktop_free(theme);
	    }
	}
    }
  return;
}


static void
handle_events(App *app)
{
      XEvent an_event;
      int done = False;
      int dock_x = 0, dock_y = 0, state = UP;
      Bool was_active = False, next_closes = False;

#ifdef USE_DNOTIFY
      sigset_t block_sigset;
      sigemptyset(&block_sigset);
      sigaddset(&block_sigset, SIGRTMIN);
#endif

      XSelectInput (app->dpy, app->root, 
		    PropertyChangeMask|SubstructureNotifyMask);

      while (!done)
      {
#ifdef USE_DNOTIFY
	if (!mb_menu_is_active(app->mbmenu))
	  sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
#endif
	XNextEvent(app->dpy, &an_event);
	
	was_active = mb_menu_is_active(app->mbmenu);
	mb_menu_handle_xevent(app->mbmenu, &an_event);
	
	switch (an_event.type)
	  {
	  case ReparentNotify:
	    if (an_event.xreparent.window == app->win_panel)
	      {
		set_backing( (void *)app );
		break;
	      }

	  case ButtonRelease:
	    if (an_event.xbutton.window == app->win_panel)
	      {
		if (state == DOWN && !next_closes) {
		  if (dock_y < 20) 
		    dock_y = 20;
		  if (!was_active) {
#ifdef USE_DNOTIFY
		    sigprocmask(SIG_BLOCK, &block_sigset, NULL); 
#endif
		    mb_menu_activate(app->mbmenu,dock_x,dock_y);
		  }
		} else {
		  mb_menu_deactivate(app->mbmenu);
#ifdef USE_DNOTIFY
		  sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
#endif
		}
		state = UP;
		next_closes = False;
		paint_dock(state);
	      }
	    break;
	  case ButtonPress:
	    if (an_event.xbutton.window == app->win_panel)
	      {
		if (was_active) 
		  next_closes = True;
		state = DOWN;
		paint_dock(state);
		XSync(app->dpy, False);
	      }
	    break;
	  case Expose:
	    if (an_event.xexpose.count != 0 )
	      break;
	    paint_dock(state);
	    break;
	  case ConfigureNotify:
	    if (an_event.xconfigure.window == app->win_panel)
	      {
		set_backing( (void *)app );
		dock_x = an_event.xconfigure.x;
		dock_y = an_event.xconfigure.y;
	      }
	    break;
	   
	  case ClientMessage:

#define MB_CMD_SHOW_EXT_MENU 6

	       if (an_event.xclient.message_type == app->mbcommand_atom
		   && an_event.xclient.data.l[0] == MB_CMD_SHOW_EXT_MENU )
		 {
		   if (dock_y < 20) 
		     dock_y = 20;

		   if (!mb_menu_is_active(app->mbmenu)) {
#ifdef USE_DNOTIFY
		     sigprocmask(SIG_BLOCK, &block_sigset, NULL); 
#endif
		     mb_menu_activate(app->mbmenu,dock_x,dock_y);
		   } else {
		     mb_menu_deactivate(app->mbmenu);
#ifdef USE_DNOTIFY
		     sigprocmask(SIG_UNBLOCK, &block_sigset, NULL); 
#endif
		   }
		 }
	       break;
	  case PropertyNotify:
	    if (an_event.xproperty.atom == app->mbtheme_atom)
	      {
		menu_set_theme_from_root_prop();
	      }
	    break;
	  }
	mb_tray_handle_event(app->dpy, app->win_panel, &an_event);
#ifdef USE_XSETTINGS
	if (app->xsettings_client != NULL)
	  xsettings_client_process_event(app->xsettings_client, &an_event);
#endif

      }

}

int
main( int argc, char *argv[])
{
   char **argv_copy;  /* make a copy of argc as XResources drinks it */
   int argc_copy;
   int i;

  struct sigaction act;
#ifdef USE_DNOTIFY
  int fd;
#endif

   argc_copy = argc;
   argv_copy = (char **)malloc(sizeof(char*)*argc);
   for(i=0; i<argc; i++)
   {
      argv_copy[i] = (char *)malloc(sizeof(char)*(strlen(argv[i])+1));
      strcpy(argv_copy[i], argv[i]);
   }
   
   app  = new_app(argc, argv);

   build_menu();

   mb_tray_init_session_info(app->dpy, app->win_panel, argv_copy, argc_copy);
   mb_tray_init(app->dpy, app->win_panel);

   mb_tray_window_icon_set(app->dpy, app->win_panel, app->img_tray);

   //set_backing( (void *)app );
   mb_tray_bg_change_cb_set(set_backing, (void *)app);

   menu_set_theme_from_root_prop();

#ifdef USE_LIBSN

   /*XXX error callbacks*/
   app->sn_display = sn_display_new (app->dpy, NULL, NULL); 

#endif

   act.sa_flags = 0;
   sigemptyset(&act.sa_mask);
   act.sa_handler = reap_children;
   sigaction(SIGCHLD, &act, NULL);

#ifdef USE_DNOTIFY

#define DD_DIR DATADIR "/applications"

   act.sa_sigaction = reload_menu;
   sigemptyset(&act.sa_mask);
   act.sa_flags = SA_SIGINFO;
   sigaction(SIGRTMIN, &act, NULL);
   
   fd = open(DD_DIR, O_RDONLY);
   fcntl(fd, F_SETSIG, SIGRTMIN);
   fcntl(fd, F_NOTIFY, DN_MODIFY|DN_CREATE|DN_DELETE|DN_MULTISHOT);
   
#endif 
   
   handle_events(app);

   return 1;
}


