/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "misc.h"

#ifndef HAVE_STRSEP
char *
strsep(char **stringp, char *delim)
{
  char *start = *stringp;
  char *cp;
  char ch;

  if (start == NULL)
    return NULL;

  for (cp = start; ch = *cp; cp++) {
    if (strchr(delim, ch)) {
      *cp++ = 0;
      *stringp = cp;
      return start;
    }
  }
  *stringp = NULL;
  return start;
}
#endif

void err(const char *fmt, ...)
{
#ifdef DEBUG
    va_list argp;
    fprintf(stderr, "X-error: ");
    va_start(argp, fmt);
    vfprintf(stderr, fmt, argp);
    va_end(argp);
    fprintf(stderr, "\n");
#endif
}

void fork_exec(char *cmd)
{
    pid_t pid = fork();

    switch (pid) {
        case 0:
            execlp("/bin/sh", "sh", "-c", cmd, NULL);
            err("exec failed, cleaning up child");
            exit(1);
        case -1:
            err("can't fork");
    }
}

void sig_handler(int signal)
{
    switch (signal) {
        case SIGINT:
        case SIGTERM:
        case SIGHUP:
	  exit(1); break;
        case SIGCHLD:
          wait(NULL); break;
    }
}

int handle_xerror(Display *dpy, XErrorEvent *e)
{
    if (e->error_code == BadAccess &&
	e->resourceid == RootWindow(dpy, DefaultScreen(dpy)) ) {
        fprintf(stderr, "root window unavailible (maybe another wm is running?)\n");
        exit(1);
    } else {
        char msg[255];
        XGetErrorText(dpy, e->error_code, msg, sizeof msg);
        err("X error (%#lx): %s (opcode: %i)",
	    e->resourceid, msg, e->request_code);
    }
    return 0;
}

int ignore_xerror(Display *dpy, XErrorEvent *e)
{
   return 0;
}

int  /* check for ageing mwm hints  */
mwm_get_decoration_flags(Wm *w, Window win)
{

#define PROP_MOTIF_WM_HINTS_ELEMENTS    5
#define MWM_HINTS_DECORATIONS   (1L << 1)

#define MWM_DECOR_BORDER              (1L << 1)
#define MWM_DECOR_RESIZEH             (1L << 2)
#define MWM_DECOR_TITLE               (1L << 3)
#define MWM_DECOR_MENU                (1L << 4)
#define MWM_DECOR_MINIMIZE            (1L << 5)
#define MWM_DECOR_MAXIMIZE            (1L << 6)

  typedef struct
  {
    unsigned long       flags;
    unsigned long       functions;
    unsigned long       decorations;
    long                inputMode;
    unsigned long       status;
  } PropMotifWmHints;

  Atom type;
  int format;
  long bytes_after;
  PropMotifWmHints *hints;
  long n_items;

  if (XGetWindowProperty (w->dpy, win, w->atoms[_MOTIF_WM_HINTS],
                          0, PROP_MOTIF_WM_HINTS_ELEMENTS,
                          False, AnyPropertyType, &type, &format, &n_items,
                          &bytes_after, (unsigned char **)&hints) != Success ||
      type == None)
    {
      dbg("MWM xgetwinprop failed\n");
      return 0;
    }
  
  if (type == None || n_items != PROP_MOTIF_WM_HINTS_ELEMENTS)
    {
      dbg("MWM xgetwinprop failed - %li items != 5\n", n_items);
      return 0;
    }

  dbg("MWM xgetwinprop success \n");

  if (hints->flags & MWM_HINTS_DECORATIONS)
  {
    if (hints->decorations == 0)
      return CLIENT_TITLE_HIDDEN_FLAG; /* XXX should really change flag to 
					  NO_DECOR */

    if (hints->decorations == MWM_DECOR_BORDER)
      return CLIENT_BORDERS_ONLY_FLAG;
  }

  return 0;

}
