/* 

Minitime - A mini dockable clock.

Copyright (c) 2003 Matthew Allum

This software is provided 'as-is', without any express or implied
warranty. In no event will the authors be held liable for any damages
arising from the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute it
freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you
       must not claim that you wrote the original software. If you use
       this software in a product, an acknowledgment in the product
       documentation would be appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and
       must not be misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
       distribution.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>

#include "mbtray.h"
#include "mbpixbuf.h"
#include "mbdotdesktop.h"

#ifdef USE_XFT
#include <X11/Xft/Xft.h>
#endif


/* 
   Globals :( , just to keep things simple.

*/

static Display* dpy;
static Window win_panel, win_root;
static Pixmap pxm_icon;
static int screen;	
static GC gc;
static int win_width, win_height;
static MBPixbuf *pb;
static MBPixbufImage *img_icon, *img_backing = NULL;
static Atom atom_mb_theme;

#ifdef USE_XFT
static XftFont *    xftfont;
static XftColor     xftcol;
static XftDraw     *xftdraw = NULL;
static XRenderColor  colortmp;
static XGlyphInfo    extents;
#else
static XFontStruct* font;
#endif

static void paint( void *user_data );
static void set_backing( void *user_data );
static void theme_change(void);


/* Parse a color spec like #aabbcc and set the global xcol */
static void
set_fg_col(char* spec)
{
  XColor col;
  if (!XParseColor(dpy, DefaultColormap(dpy, screen), spec, &col))
    {
      fprintf(stderr, "minitime: failed to parse color %s\n", spec);
      return;
    } else {
      XAllocColor(dpy, DefaultColormap(dpy, screen), &col);
    }

  XSetForeground(dpy, gc, col.pixel);

#ifdef USE_XFT   
  colortmp.red   = col.red;
  colortmp.green = col.green;
  colortmp.blue  = col.blue;
  colortmp.alpha = 0xffff; 
  XftColorAllocValue(dpy,DefaultVisual(dpy, screen), 
		     DefaultColormap(dpy, screen),
		     &colortmp, &xftcol);
#endif
}

/* Called when root window theme property changes */
static void
theme_change(void)
{
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  char *value = NULL;
  struct stat stat_info;
  char dock_cfg[256];

  /* Attemp to read the _MB_THEME property */
  status = XGetWindowProperty(dpy, win_root,
			      atom_mb_theme, 
			      0L, 512L, False,
			      AnyPropertyType, &realType,
			      &format, &n, &extra,
			      (unsigned char **) &value);
	    
  if (status != Success || value == 0
      || *value == 0 || n == 0)
    {
      ; 			/* failed to get it */
    } else {
      /* Open the theme.desktop file in property specified directory */
      strncpy(dock_cfg, value, 230); 
      strcat(dock_cfg, "/theme.desktop");
      if (stat(dock_cfg, &stat_info) != -1)
	{
	  /* Use libMB dotdesktop parser to get local based hash */
	  MBDotDesktop *theme  = NULL;
	  theme = mb_dotdesktop_new_from_file(dock_cfg);
	  if (theme)
	    {
	      /* Get the PanelFgColor key value if exists */
	      if (mb_dotdesktop_get(theme, "PanelFgColor"))
		{
		  /* Set out font foreground color and repaint */
		  set_fg_col(mb_dotdesktop_get(theme, "PanelFgColor"));
		}
	      mb_dotdesktop_free(theme);
	    }
	}
    }

  if (value) XFree(value);
  return;
}

static void
set_backing( void *user_data )
{
  /* 
     Use libMB to get a mbpixbuf representation of the panel background
     behind us. We can then use this to alpha blend to it
  */
  if (img_backing) mb_pixbuf_img_free(pb, img_backing);

  img_backing = mb_tray_get_bg_img(pb, win_panel);

  if (img_backing == NULL)
    {
      fprintf(stderr, "minitime: cannot get dock backing !!!");
      return;
    }

  paint(NULL);
}


static void 
paint( void *user_data ) 	/* XXX usedata not needed ? */
{
  struct timeval  tv;
  struct timezone tz;
  struct tm *localTime = NULL; 
  time_t actualTime;
  char timestr[5] = "";

  /* Figure out  the actual time */
  gettimeofday(&tv, &tz);
  actualTime = tv.tv_sec;
  localTime = localtime(&actualTime);

  if (!img_backing) {
    set_backing( NULL ); return; /* set_backing will recall this */
  }

  /* Free any X resources, were going to 'reuse'  */
#ifdef USE_XFT
  if (xftdraw) XftDrawDestroy (xftdraw);
#endif
  if (pxm_icon) XFreePixmap(dpy, pxm_icon);

  /* Create a pixmap for the window background */
  pxm_icon = XCreatePixmap(dpy, win_panel, win_width, win_height,
			   DefaultDepth(dpy, screen));

  /* Paint the current mbpixbuf background ( from set_backing ) there */
  mb_pixbuf_img_render_to_drawable(pb, img_backing, pxm_icon, 0, 0);

  /* Write the font with the current time onto the pixmap */
#ifdef USE_XFT
  xftdraw = XftDrawCreate(dpy, (Drawable)pxm_icon,
			  DefaultVisual(dpy, screen),
			  DefaultColormap(dpy, screen));
#endif

  sprintf(timestr, "%.2d:%.2d", localTime->tm_hour, localTime->tm_min);

#ifdef USE_XFT	 	 
  XftDrawString8(xftdraw, &xftcol, xftfont, 2, xftfont->ascent+2,
		 (unsigned char *) timestr, strlen(timestr));
#else
  XDrawString(dpy, pxm_icon, gc, 2, font->ascent+2, timestr, strlen(timestr) );
#endif

  /* Set the panel app window background to be the pixmap.
     This lets X deal with exposes and we dont have to worry about it :-)
  */
  XSetWindowBackgroundPixmap(dpy, win_panel, pxm_icon);
  XClearWindow(dpy, win_panel);

  XFlush(dpy);
}

static Bool
get_xevent_timed(Display* dpy, XEvent* event_return, struct timeval *tv)
{
  if (tv == NULL) 
    {
      XNextEvent(dpy, event_return);
      return True;
    }

  XFlush(dpy);

  if (XPending(dpy) == 0) 
    {
      int fd = ConnectionNumber(dpy);
      fd_set readset;
      FD_ZERO(&readset);
      FD_SET(fd, &readset);
      if (select(fd+1, &readset, NULL, NULL, tv) == 0) 
	{
	  return False;
	} else {
	  XNextEvent(dpy, event_return);
	  return True;
	}
    } else {
      XNextEvent(dpy, event_return);
      return True;
    }
}

void 
usage(char *progname)	
{
  printf("usage: %s [options ....]\n"
	 "Where options are\n"
	 "  -display    <display>    Display to connect to\n"
	 "  --font, -fn <font spec>  Font name to use.    \n"
	 "  --col, -c   <col spec>   Color to use ( defualt #ffffff )\n"
	 "  -geometry   <+x>         Specify initial panel position\n"
	 "\n", progname
	  );
  exit(1);
}

int 
main(int argc, char **argv)
{
  XEvent xevent;
  XSizeHints size_hints;

  /* Defaults  */
  char *dpy_name     = NULL;
  char *font_name    = "verdana-8";
  char *geometry_str = NULL;
  char *col_spec     = NULL;
  int panel_offset   = 0;

  int i;
  int dummy_a, dummy_b, dummy_c;

  /* Pass command line params */
  for (i = 1; i < argc; i++) 
    {
      if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
	if (++i>=argc) usage (argv[0]);
	dpy_name = argv[i];
	continue;
      }
      
      if (!strcmp ("--font", argv[i]) || !strcmp ("-fn", argv[i])) {
	if (++i>=argc) usage (argv[0]);
	font_name = argv[i];
	continue;
      }

      if (!strcmp ("--col", argv[i]) || !strcmp ("-c", argv[i])) {
	if (++i>=argc) usage (argv[0]);
	col_spec = argv[i];
	continue;
      }

      if (!strcmp ("-geometry", argv[i]) || !strcmp ("-g", argv[i])) {
	if (++i>=argc) usage (argv[0]);
	geometry_str = argv[i];
	continue;
      }

      usage(argv[0]);
    }

  /* Open X Display Connection, etc */
  if ((dpy = XOpenDisplay(dpy_name)) == NULL)
    {
      fprintf(stderr, "Cannot connect to X server on display %s.",
	      dpy_name);
      exit(1);
    }

  screen   = DefaultScreen(dpy);
  win_root = DefaultRootWindow(dpy);
  pb       = mb_pixbuf_new(dpy, screen);


  /* Get atom for figuring out where the current theme is */
  atom_mb_theme = XInternAtom(dpy, "_MB_THEME", False);

  /* Do we want an specified initial position on the panel ? */
  if (!(geometry_str && XParseGeometry(geometry_str, &panel_offset, 
				     &dummy_a, &dummy_b, &dummy_c)))
    panel_offset = 0;

  /* Load X Fonts */
#ifdef USE_XFT   
  if ((xftfont = XftFontOpenName(dpy, screen, font_name)) == NULL)
#else
  if (!(font = XLoadQueryFont(dpy, font_name)))
#endif
    { printf("%s: Cant open font %s\n", argv[0], font_name); exit(0); }

  gc = XCreateGC(dpy, win_root, 0, NULL);

  /* Parse color spec */
  if (col_spec) 
    set_fg_col(col_spec);
  else
    set_fg_col("#ffffff");

  /* Use font metrics to figure out panel size. */

#ifdef USE_XFT

  XftTextExtents8(dpy, xftfont, (unsigned char *) "99:99", 5, &extents);

  win_width  = extents.width + 4;
  win_height = xftfont->ascent + xftfont->descent + 4; 

#else

  win_width = XTextWidth(font, "99:99", 5);
  win_height = font->ascent + font->descent + 4; 

  XSetFont(dpy, gc, font->fid);

#endif 

  if (!(img_icon = mb_pixbuf_img_new_from_file(pb, PIXMAPSDIR "/minitime.png")))
    {
      fprintf(stderr, "%s: failed to load image %s.\n", argv[0], 
	      PIXMAPSDIR "/minitime.png" );
      exit(1);
    }

  /* Create the panel window */

  win_panel = XCreateSimpleWindow(dpy, win_root, panel_offset, 0,
				  win_width, win_height, 0,
				  BlackPixel(dpy, screen),
				  WhitePixel(dpy, screen));

  /* Set up standard hints */

  size_hints.flags = PPosition | PSize | PMinSize | PMaxSize;
  size_hints.x = panel_offset;
  size_hints.y = 0;
  size_hints.max_width = size_hints.min_width = size_hints.width =  win_width;
  size_hints.max_height = size_hints.min_height = size_hints.height =  win_height;

  XSetStandardProperties(dpy, win_panel, "Clock", 
			 NULL, 0, argv, argc, &size_hints);




  /* Load in an image for our rgba window icon */


  /* libMB calls */

  /* This sets various props on the window needed for panel sessioning */
  mb_tray_init_session_info(dpy, win_panel, argv, argc);

  /* Set atoms, attempt to find the tray and initially message it */
  mb_tray_init(dpy, win_panel);

  /* Call utility function to set EWMH rgba icon for panel window
      - this is used mainly by the panel remove menu. 
  */
  mb_tray_window_icon_set(dpy, win_panel, img_icon); 

  /* Define a callback, to get notified when the panel background changes */
  mb_tray_bg_change_cb_set(set_backing, NULL);

  if (col_spec == NULL) theme_change(); /* Read data from theme prop */

  /* Select to get notified when root properties change 
      - Used for the _MB_THEME prop.


  XSelectInput (dpy, win_root, PropertyChangeMask|SubstructureNotifyMask);
  */

  /* Select for interested events on panel window */
  XSelectInput(dpy, win_panel, StructureNotifyMask|ExposureMask
	       |ButtonPressMask|ButtonReleaseMask );


  paint(NULL);

  XFlush(dpy);


  /* Main loop */
  while (1)
    {
      Bool have_painted = False;
      time_t now;
      struct timeval tvt;
      struct tm *lt;
      now = time (0);
      lt = gmtime (&now);
      tvt.tv_usec = 0;
      tvt.tv_sec = 10;  

      if (get_xevent_timed(dpy, &xevent, &tvt))
      {
	  switch (xevent.type) 
	    {
	    case ReparentNotify:
	      /* We've been reparented ( ie docked ) so set/paint background */
	      set_backing( NULL );
	      have_painted = True;
	      break;
	    case ConfigureNotify:

	      set_backing( NULL ); /* XXX needed ? */
	      have_painted = True;
	      break;
	    case Expose:
	      break; 
	    case ButtonPress:
	      /* Ask the tray for a 'message bubble' showing current time  */
	      now = time (0);
	      mb_tray_send_message(dpy, win_panel, 
				   ctime(&now), 
				   5000);
	      break;
	    case PropertyNotify:
	      /* Has the root window theme prop changed ? */
	      if (xevent.xproperty.atom == atom_mb_theme)
		{
		  theme_change();
		  paint(NULL);
		}
	      break;
	    }
	  /* Let mbtray.c handle its events */
	  mb_tray_handle_event(dpy, win_panel, &xevent);
      }
      else
	{
	  paint(NULL);
	}
    }      
  return 0;
}

