/*
 *  miniwave - Tiny 820.11 wireless 
 *
 *  originally based on wmwave
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "miniwave.xpm"
#include "mbtray.h"

#define SYSTEM_TRAY_REQUEST_DOCK    0
#define SYSTEM_TRAY_BEGIN_MESSAGE   1
#define SYSTEM_TRAY_CANCEL_MESSAGE  2

struct {

   char iface[5];
   char status[3];
   float link;
   float level;
   float noise;
   int nwid;
   int crypt;
   int misc;
   int mode;    // default: no card detected

} Mwd;

int
xcol_from_str(Display *display, XColor *col, const char *defstr)
{
  char *str;
  const char delim[] = ",:";
  char *token;
  XColor exact;

  if ((strchr(defstr, delim[0]) != NULL)
      || (strchr(defstr, delim[1]) != NULL) )
  {
     str = strdup(defstr);

     token = strsep (&str, delim); 
     col->red = ( atoi(token) * 65535 ) / 255; 
     token = strsep (&str, delim); 
     col->green = ( atoi(token) * 65535 ) / 255;
     token = strsep (&str, delim); 
     col->blue = ( atoi(token) * 65535 ) / 255;

     return XAllocColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			col);
  } else {
     return XAllocNamedColor(display,
			     DefaultColormap(display, DefaultScreen(display)),
			     defstr, col, &exact);
  }
}


void update_wireless(void) {
  FILE *wireless;   // File handle for /proc/net/wireless
  int count;					      
  char line[255];

  Mwd.link = 0;
  Mwd.level = 0;
  Mwd.noise = 0;
  Mwd.nwid = 0;
  Mwd.crypt = 0;
  Mwd.misc = 0;

  if ((wireless = fopen ("/proc/net/wireless", "r")) != NULL)
  {
     fgets(line,sizeof(line),wireless);
     fgets(line,sizeof(line),wireless);
     if (fgets(line,sizeof(line),wireless) == NULL) {
	Mwd.mode = 0;
	Mwd.iface[0]=0;
     } else {
	sscanf(line,"%s %s %f %f %f %d %d %d",
	       Mwd.iface, Mwd.status, &Mwd.link, &Mwd.level,
	       &Mwd.noise, &Mwd.nwid, &Mwd.crypt, &Mwd.misc);
	for(count=0;(count<strlen(line)) && (line[count]==0x20);count++);
        strncpy(Mwd.iface,&line[count],5);
        Mwd.iface[4]=0;
	Mwd.mode = 1;
     }
     fclose(wireless);
  }
  else
  {
     printf ("Wirless device /proc/net/wireless not found\nEnable radio networking and recompile your kernel\n");
     exit (0);
  }

}


void usage(void)
{
   printf("Usage: miniwave [options...]\n");
   printf("Options:\n");
   printf("  -display  <display>\n");
   printf("  -c  <color> color for monitor background\n");
   printf("  -r  align to right of dock ( matchbox only )\n");
   printf("  -h  this help\n\n");
   exit(1);
}

int
main(int argc, char *argv[])
{
   Window win;
   Display *dpy;
   GC gc;
   
   int i;
   Atom window_type_atom;
   Atom window_type_dock_atom;

   Atom atom_SYSTEM_TRAY;
   
   Pixmap icon, mask;
   XpmAttributes xpm_attr;	

   Atom mb_align_atom;
   Atom mb_align_east_atom;
   int align_dock_right = False;
   
   XGCValues gv;
   XEvent an_event;
   XColor xcol;

   char *col_name = "red";
   int done = 0;
   char *display_name = NULL; 

   float last_link = 0.0;
   int update_req = 0;

   char tray_msg[256];
   
   for (i=1; argv[i]; i++) {
      char *arg = argv[i];
      if (*arg=='-') {
	 switch (arg[1]) {
	    case 'd' :
	       display_name = argv[i+1];
	       i++;
	       break;
	    case 'r' :
	       align_dock_right = True;
	       break;
	    case 'c' :
	       col_name = argv[i+1];
	       i++;
	       break;
	    default:
	       usage();
	       exit(0);
	       break;
	 }
      }	
   }

   update_wireless();

   if ((dpy = XOpenDisplay(display_name)) == NULL)
   {
      printf("Cant open display\n"); exit(1);
   }

   xcol_from_str(dpy, &xcol, col_name);
   
   window_type_atom = XInternAtom(dpy, "_NET_WM_WINDOW_TYPE" , False);
   window_type_dock_atom = XInternAtom(dpy,
				       "_NET_WM_WINDOW_TYPE_DOCK",False);
   mb_align_atom = XInternAtom(dpy, "_MB_DOCK_ALIGN", False);
   mb_align_east_atom = XInternAtom(dpy, "_MB_DOCK_ALIGN_EAST",False);

   atom_SYSTEM_TRAY = XInternAtom(dpy, "_NET_SYSTEM_TRAY_S0", False);
   
   win = XCreateSimpleWindow(dpy,
			     RootWindow(dpy, DefaultScreen(dpy)),
			     0, 0,
			     16, 16,
			     0, BlackPixel(dpy, DefaultScreen(dpy)),
			     WhitePixel(dpy, DefaultScreen(dpy)));
   
   gv.graphics_exposures = False;
   gv.function           = GXcopy;
   gv.foreground         = xcol.pixel;
   gc = XCreateGC(dpy, win,
		  GCGraphicsExposures|GCFunction|GCForeground, &gv);

   xpm_attr.valuemask = XpmColormap | XpmCloseness;
   xpm_attr.closeness = 40000;
   xpm_attr.colormap = DefaultColormap(dpy, DefaultScreen(dpy));

   if (XpmCreatePixmapFromData( dpy, win, miniwave_xpm,
			    &icon, &mask, &xpm_attr) != XpmSuccess )
   {
      fprintf(stderr, "failed to get icon image\n");
      exit(1);
   }
       
   XCopyArea(dpy, icon, win, gc, 0, 0,
	     xpm_attr.width, xpm_attr.height, 0, 0);
   
   XShapeCombineMask (dpy, win, ShapeBounding, 0, 0, mask, ShapeSet);

   if (align_dock_right)
      XChangeProperty(dpy, win, mb_align_atom, XA_ATOM, 32, 
		      PropModeReplace,
		      (unsigned char *) &mb_align_east_atom, 1);

   XStoreName(dpy, win, "Wireless Monitor");

   mb_tray_init_session_info(dpy, win, argv, argc);
   mb_tray_init(dpy, win);
   //mb_tray_unmap_window (dpy, win);
   
   XSelectInput(dpy, win, ExposureMask|ButtonPressMask); 
      
   while (!done)
   {
      update_wireless();
      //mb_tray_map_window (dpy, win);
      if (last_link != Mwd.link)
      {
	 update_req = 1;
      }
      last_link = Mwd.link;
      while (XPending(dpy)) {
	 XNextEvent(dpy, &an_event);
	 switch (an_event.type) {
	    case Expose:
	       update_req = 1;
	       break;
	    case DestroyNotify:	      
	       done = 1;
	       break;
	 case ButtonPress:
	   if (Mwd.mode)
	     sprintf(tray_msg, "%s: Link %.1f, Level %.1f, Noise %.1f\n",
		     Mwd.iface, Mwd.link, Mwd.level, Mwd.noise );
	   else
	     sprintf(tray_msg, "No wireless cards detected\n");
	   mb_tray_send_message(dpy, win, tray_msg, 5000);
	    default:
	       break;
	 }
	 mb_tray_handle_event(dpy, win, &an_event);
      }
      if (update_req)
      {
	 XCopyArea(dpy, icon, win, gc, 0, 0,
		   xpm_attr.width, xpm_attr.height, 0, 0);
	 
	 if (Mwd.mode)
	 {
	    //printf("%s, %f\n", Mwd.iface, Mwd.link);
	    XFillRectangle(dpy, win, gc, 0, 0, (Mwd.link/100.00)*16, 16);
	 }

	 update_req = 0;
      } 
      usleep(300000L);
   }

   return 0;
}
