/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "select_client.h"


client*
select_client_new(Wm *w)
{
   Window win;
   XSetWindowAttributes attr;
   int button_x = 0;

   int width, height;
   Client *c;

   if (!theme_frame_menu_get_dimentions(w->mbtheme, &width, &height))
     return NULL;

   if (width == 0 || height == 0) return NULL;

   height += theme_frame_defined_height_get(w->mbtheme, FRAME_MENU);
   width  += theme_frame_defined_width_get(w->mbtheme, FRAME_MENU);

   button_x = theme_frame_button_get_x_pos(w->mbtheme, FRAME_MAIN, 
					   BUTTON_ACTION_MENU,
					   w->main_client->width);

   if (XGrabPointer(w->dpy, w->root, True,
		    (ButtonPressMask|ButtonReleaseMask),
		    GrabModeAsync,
		    GrabModeAsync, None, w->curs, CurrentTime)
       != GrabSuccess)
      return NULL;

   XGrabKeyboard(w->dpy, w->root, True, GrabModeAsync, 
		 GrabModeAsync, CurrentTime);
   
   attr.override_redirect = True;
   attr.background_pixel = BlackPixel(w->dpy, w->screen);
   attr.event_mask = ButtonPressMask|ExposureMask|
     EnterWindowMask|LeaveWindowMask|PointerMotionMask;

   win = XCreateWindow(w->dpy, w->root,
		       button_x + wm_get_offsets_size(w, WEST, NULL, True),
		       wm_get_offsets_size(w, NORTH, NULL, True) 
		       + main_client_title_height(w->main_client),
		       width, height, 0,
		       CopyFromParent, CopyFromParent, CopyFromParent,
		       CWOverrideRedirect|CWBackPixel|CWEventMask,
		       &attr);

   c = base_client_new(w, win);
   c->type = menu;
   c->title_frame = c->frame = c->window;
   
   c->destroy      = &select_client_destroy;
   c->button_press = &select_client_button_press;
   c->redraw       = &select_client_redraw;

   w->flags ^= MENU_FLAG;

   c->redraw(c, False);
   c->mapped = True;
   client_set_state(c, NormalState);

   c->height = height;
   c->width  = width;
   
   XMapWindow(c->wm->dpy, c->frame);

   return c;
}

static void
select_client_highlight_entry(Client *c, 
			      MBClientButton *button, 
			      Drawable drawable)
{
  MBTheme *t = c->wm->mbtheme;

  dbg("%s() painting +%i+%i %ix%i\n", __func__, 
      button->x, button->y, button->w, button->h);

  XDrawRectangle(t->wm->dpy, c->frame, t->band_gc, 
		 button->x + 2, button->y, button->w - 4, button->h);

}

void 
select_client_event_loop( Client *c, struct list_item *button_item_cur )
{
  client *p;
  XEvent ev;
  Bool in_window = True;
  MBClientButton *button;
  struct list_item *button_item_new = NULL;
  KeySym key;

  if (button_item_cur == NULL)
    {
      button_item_cur = c->buttons;
      in_window = False;
    }
  
  button = (MBClientButton *)button_item_cur->data;

  select_client_highlight_entry( c, button, c->frame );

  for (;;) 
    {
      XMaskEvent(c->wm->dpy, 
		 EnterWindowMask|LeaveWindowMask|KeyReleaseMask
		 |ExposureMask|ButtonReleaseMask|PointerMotionMask, &ev);
      switch (ev.type)
	{
	case KeyRelease:

	  switch (key = XKeycodeToKeysym (c->wm->dpy, ev.xkey.keycode, 0))
	    {
	    case XK_Up:
	      button_item_new = c->buttons;
	      if (button_item_cur == c->buttons)
		{
		  while(button_item_new->next != NULL)
		    button_item_new = button_item_new->next;
		}
	      else
		{
		  while(button_item_new->next != button_item_cur)
		    button_item_new = button_item_new->next;
		}

	      if (button_item_cur)
		select_client_highlight_entry( c, button, c->frame );
	      button_item_cur = button_item_new;
	      button = (MBClientButton *)button_item_cur->data;
	      select_client_highlight_entry( c, button, c->frame );
	      break;
	      
	    case XK_Down:

	      button_item_new = button_item_cur->next;
	      if (button_item_new == NULL)
		button_item_new = c->buttons; 

	      if (button_item_cur)
		select_client_highlight_entry( c, button, c->frame );
	      button_item_cur = button_item_new;
	      button = (MBClientButton *)button_item_cur->data;
	      select_client_highlight_entry( c, button, c->frame );
	      break;

	    case XK_Return:
	    case XK_KP_Enter:
	    }
	  break;
	case ButtonRelease:
	  if (button && in_window)
	    {
	      p = (Client *)button->data;
	      select_client_highlight_entry( c, button, c->frame );
	      if (p->type == desktop)
		wm_toggle_desktop(c->wm);
	      else
		{
		  base_client_hide_transients(c->wm->main_client);
		  wm_activate_client(p);
		}
	    }
	  return;
	case EnterNotify:
	  printf("enter\n");
	  in_window = True;
	  break;
	case MotionNotify:
	  /* XXX get list_item back, then can get next / prev for keys */
	
	  button_item_new 
	    = client_get_button_list_item_from_event(c, &ev.xbutton);
	  if (button_item_new != NULL && in_window)
	    {
	      if (button_item_new != button_item_cur)
		{
		  dbg("%s() got a button again\n", __func__ );
		  if (button_item_cur)
		    select_client_highlight_entry( c, button, c->frame );
		  button_item_cur = button_item_new;
		   button = (MBClientButton *)button_item_cur->data;
		   select_client_highlight_entry( c, button, c->frame );
		 }
	     }
	   break;
	 case LeaveNotify:
	   printf("leave\n");
	   in_window = False;
	   button_item_cur = NULL;
	   /* Clear button highlight */
	   select_client_highlight_entry( c, button, c->frame );
	   break;
	 }
     }
}

void
select_client_button_press( Client *c, XButtonEvent *e)
{

  struct list_item *button_item = NULL;

  if ((button_item = client_get_button_list_item_from_event(c, e)) == NULL)
    {
      dbg("%s() failed to get a button\n", __func__ );
      select_client_destroy(c);
      return;
    }

  select_client_event_loop( c, button_item );

  select_client_destroy(c);

}

void
select_client_redraw(Client *c, Bool use_cache)
{
   MBTheme *theme = c->wm->mbtheme;
   Bool is_shaped = False;

   if (use_cache && c->backing)
     {
       XSetWindowBackgroundPixmap(c->wm->dpy, c->frame, c->backing);
       XClearWindow(c->wm->dpy, c->frame);
       return;
     }

   is_shaped = theme_frame_wants_shaped_window( theme, FRAME_MENU);

   if (c->backing == (Pixmap)NULL)
     client_init_backing(c, c->width, c->height);

   if (is_shaped) client_init_backing_mask(c, c->width, c->height);

   theme_frame_menu_paint( theme, c);
  
   if (is_shaped)
     XShapeCombineMask( c->wm->dpy, c->frame, ShapeBounding, 0, 0, 
			c->backing_mask, ShapeSet);

   XSetWindowBackgroundPixmap(c->wm->dpy, c->frame, c->backing);
   XClearWindow(c->wm->dpy, c->frame);

}

void
select_client_destroy(Client *c)
{
   XUngrabPointer(c->wm->dpy, CurrentTime);
   XUngrabKeyboard(c->wm->dpy, CurrentTime);

   c->wm->flags ^= MENU_FLAG;

   XUnmapWindow(c->wm->dpy, c->frame);

   base_client_destroy(c);
}


