/* miniapm - A tiny battery monitor

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/
/* 
   TODO
   o Add define + install scripts for apm.png ( and rename )

 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "config.h"

#ifdef USE_PNG
#define MINIAPM_IMG     "/miniapm.png"
#define MINIAPM_PWR_IMG "/miniapm-power.png"
#else
#define MINIAPM_IMG     "/miniapm.xpm"
#define MINIAPM_PWR_IMG "/miniapm-power.xpm"
#endif

#ifdef HAVE_APM_H 		/* Linux */
#include <apm.h>
#endif

#ifdef HAVE_APMVAR_H		/* *BSD  */
#include <err.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <machine/apmvar.h>

#define APMDEV "/dev/apm"

enum apm_state {
    NORMAL,
    SUSPENDING,
    STANDING_BY
};

struct apm_reply {
    int vno;
    enum apm_state newstate;
    struct apm_power_info batterystate;
};

#endif

#define TIME_LEFT  0
#define PERCENTAGE 1
#define AC_POWER   2

#define WIN_WIDTH  8
#define WIN_HEIGHT 14

#define CLOCK_DISP 1
#define BATT_DISP  0

#ifndef AC_LINE_STATUS_ON
#define AC_LINE_STATUS_ON 1
#endif

#include "mbtray.h"
#include "mbpixbuf.h"

static Display* dpy;	
static Window win_panel, win_root;
static int screen;
static GC gc;
static int apm_vals[3];
static MBPixbuf *pb;
static MBPixbufImage *img_icon = NULL, *img_icon_pwr = NULL, *img_backing = NULL; 
static Pixmap pxm_backing = None;

static int time_left_alerts[] = { 0, 2, 5, 10, 20 }; /* times to alert on */
static int time_left_idx = 4;

static Bool ac_power = False;

#ifdef HAVE_APM_H

static int 
read_apm(int *values)
{
  /* add stat function here ? */
 
  apm_info info;
  apm_read(&info);

  values[TIME_LEFT] = info.battery_time;
  values[PERCENTAGE] = info.battery_percentage;
  values[AC_POWER] = info.ac_line_status;

  return 1;
}

#else  /* BSD */

static int 
read_apm(int *values)
{
  int fd;
  struct apm_reply reply;
  struct apm_power_info *api = &reply.batterystate;

  /* open the device directly and get status */
  fd = open(APMDEV, O_RDONLY);
  if (fd == -1) 
	return 0;
  memset(&reply, 0, sizeof(reply));
  if (ioctl(fd, APM_IOC_GETPOWER, &reply.batterystate) == -1) {
        close (fd);
  return 0;
  }
  close (fd);

  values[TIME_LEFT]  = api->minutes_left;
  values[PERCENTAGE] = api->battery_life;
  values[AC_POWER]   = api->ac_state;
 
  return 1;
}

#endif


static void 
paint( void *user_data )
{
  int power_pixels = 0;
  Bool old_power = ac_power;
  unsigned char r = 0, g = 0, b = 0;
  int x, y;

  while (!read_apm(apm_vals))
    usleep(50000L);

  if (apm_vals[PERCENTAGE] <= 0 || apm_vals[PERCENTAGE] > 99)
    { 
      r = 0x66; g = 0xff; b = 0x33; ac_power = True; 
      apm_vals[PERCENTAGE] = -1;
    }
  else if (apm_vals[PERCENTAGE] <= 30)
    { r = 0xff; g = 0; b = 0; }
  else if (apm_vals[PERCENTAGE] < 75)
    { r = 0xff; g = 0x99; b = 0x33; }
  else if (apm_vals[PERCENTAGE] <= 100)
    { r = 0x66; g = 0xff; b = 0x33; }

  if (apm_vals[AC_POWER] == AC_LINE_STATUS_ON) 
    ac_power = True;
  else 
    ac_power = False;

  if (old_power != ac_power)
    {
      /*
      mb_pixbuf_img_fill(pb, img_backing, 
			 xcol_bg.red, xcol_bg.green, xcol_bg.blue, 0);
      */
      if (ac_power)
	mb_pixbuf_img_composite(pb, img_backing, img_icon_pwr , 0, 0);
      else
	mb_pixbuf_img_composite(pb, img_backing, img_icon , 0, 0);
    }

  /* Clear out bar first */
  for ( y = 13; y < 15; y++) 
    for ( x = 3; x < 14; x++)
      if (ac_power)
	{
	  mb_pixbuf_img_set_pixel(img_backing, x, y, 0xff, 0xff, 0);
	}
      else
	{
	  mb_pixbuf_img_set_pixel(img_backing, x, y, 0, 0, 0);
	}

  if (apm_vals[PERCENTAGE] > 0)
    { 
      power_pixels = apm_vals[PERCENTAGE]/10;

      for ( y = 13; y < 15; y++) 
	for ( x = 3; x < 4 + power_pixels; x++)
	    mb_pixbuf_img_set_pixel(img_backing, x, y, r, g, b);
    }

  /* Bubble alerts  */
  if ((time_left_idx > 0) 
      && !ac_power
      && apm_vals[PERCENTAGE] > 0
      && apm_vals[TIME_LEFT]  > 0
      && (apm_vals[TIME_LEFT] < time_left_alerts[time_left_idx]))
    {
      char tray_msg[256];
      sprintf(tray_msg, 
	      "Battery power very low !\n\nTime Left: %.2i minutes", 
	      time_left_alerts[time_left_idx]);
      mb_tray_send_message(dpy, win_panel, tray_msg, 0);
      time_left_idx--;
    }
  else if (time_left_idx < 4 
	   && apm_vals[TIME_LEFT] > time_left_alerts[time_left_idx+1])
    {
      time_left_idx++;
    }

  if (pxm_backing) XFreePixmap(dpy, pxm_backing);

  pxm_backing = XCreatePixmap(dpy, win_panel, 
			      mb_pixbuf_img_get_width(img_icon), 
			      mb_pixbuf_img_get_height(img_icon),
			      pb->depth);

  mb_pixbuf_img_render_to_drawable(pb, img_backing, pxm_backing, 0, 0);

  XSetWindowBackgroundPixmap(dpy, win_panel, pxm_backing);
  XClearWindow(dpy, win_panel);
  XFlush(dpy);
}

static void
set_backing( void *user_data )
{
  /*
  mb_tray_get_bg_col(dpy, &xcol_bg);
  mb_pixbuf_img_fill(pb, img_backing, 
		     xcol_bg.red, xcol_bg.green, xcol_bg.blue, 0);
  */
  if (img_backing) mb_pixbuf_img_free(pb, img_backing);

  img_backing = mb_tray_get_bg_img(pb, win_panel);

  if (ac_power)
    mb_pixbuf_img_composite(pb, img_backing, img_icon_pwr , 0, 0);
  else
    mb_pixbuf_img_composite(pb, img_backing, img_icon , 0, 0);


  paint(NULL);
}

void 
usage(char *progname)
{
  ;
}

int 
main(int argc, char **argv)
{
  int i;
  XSizeHints size_hints;
  XEvent xevent;
  char *dpy_name = NULL;
  int xfd;
  fd_set fd;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) usage (argv[0]);
      dpy_name = argv[i];
      continue;
    }
    usage(argv[0]);
  }

  if ((dpy = XOpenDisplay(dpy_name)) == NULL)
    {
      fprintf(stderr, "Cannot connect to X server on display %s.",
	      dpy_name);
      exit(1);
    }

  screen   = DefaultScreen(dpy);
  win_root = DefaultRootWindow(dpy);
  pb       = mb_pixbuf_new(dpy, screen);

  if (!(img_icon = mb_pixbuf_img_new_from_file(pb, PIXMAPSDIR MINIAPM_IMG)))
    {
      fprintf(stderr, "%s: failed to load image XXX \n", argv[0] );
      exit(1);
    }

  if (!(img_icon_pwr = mb_pixbuf_img_new_from_file(pb, PIXMAPSDIR MINIAPM_PWR_IMG)))
    {
      fprintf(stderr, "%s: failed to load image XXX \n", argv[0] );
      exit(1);
    }
  /*
  img_backing = mb_pixbuf_img_new(pb, 
				  mb_pixbuf_img_get_width(img_icon), 
				  mb_pixbuf_img_get_height(img_icon));
  */


  win_panel = XCreateSimpleWindow(dpy, win_root, 0, 0,
				  mb_pixbuf_img_get_width(img_icon), 
				  mb_pixbuf_img_get_height(img_icon), 
				  0,
				  BlackPixel(dpy, screen),
				  WhitePixel(dpy, screen));

  size_hints.flags = PPosition | PSize | PMinSize | PMaxSize;
  size_hints.x = 0;
  size_hints.y = 0;
  size_hints.width      =  mb_pixbuf_img_get_width(img_icon);
  size_hints.height     =  mb_pixbuf_img_get_height(img_icon);
  size_hints.min_width  =  mb_pixbuf_img_get_width(img_icon);
  size_hints.min_height =  mb_pixbuf_img_get_height(img_icon);
  size_hints.max_width  =  mb_pixbuf_img_get_width(img_icon);
  size_hints.max_height =  mb_pixbuf_img_get_height(img_icon);

    
  XSetStandardProperties(dpy, win_panel, "Battery Monitor", 
			 NULL, 0, argv, argc, &size_hints);

  gc = XCreateGC(dpy, win_root, 0, NULL);

  // XStoreName(dpy, win_panel, "Battery Monitor");

  XSelectInput(dpy, win_panel, StructureNotifyMask|ExposureMask
	       |ButtonPressMask|ButtonReleaseMask); 
  
  mb_tray_init_session_info(dpy, win_panel, argv, argc);
  mb_tray_init(dpy, win_panel);
  mb_tray_window_icon_set(dpy, win_panel, img_icon);

  //set_backing(NULL);
  mb_tray_bg_change_cb_set(set_backing, NULL);

  XFlush(dpy);

  xfd = ConnectionNumber (dpy);

  while (1)
    {
      char tray_msg[256];

      struct timeval tvt;
      tvt.tv_usec = 0;
      tvt.tv_sec = 2; 		/* check batt stats every 2 seconds  */
      

      XNextEvent(dpy, &xevent);
      switch (xevent.type) 
	{
	case ReparentNotify:
	  set_backing(NULL);
	  break;
	case ConfigureNotify:
	  set_backing(NULL);
	  break;
	case Expose:
	      break;
	case ButtonPress:
	  if (apm_vals[AC_POWER] == AC_LINE_STATUS_ON)
	    {
	      if (apm_vals[PERCENTAGE] > 0 
		  && apm_vals[PERCENTAGE] < 100 )
		sprintf(tray_msg, "AC Connected\nCharging: %.2i %%\n"
			, apm_vals[PERCENTAGE]);
	      else
		sprintf(tray_msg, "AC Connected\nFully charged.\n");
	    } else {
	      if (apm_vals[PERCENTAGE] > 0 
		  && apm_vals[PERCENTAGE] < 100 
		  && apm_vals[TIME_LEFT] > 0)
		{
		  sprintf(tray_msg, 
			  "Battery Power\nJuice %.2i %%\nTime left: %.2i mins\n", apm_vals[PERCENTAGE], apm_vals[TIME_LEFT]);
		    }
	      else sprintf(tray_msg, "Battery Power\n Device read error.\n");
	    }
	  mb_tray_send_message(dpy, win_panel, tray_msg, 5000);
	  break;
	}
      mb_tray_handle_event(dpy, win_panel, &xevent);
      
      FD_ZERO (&fd);
      FD_SET (xfd, &fd);
      
      XFlush(dpy);
      
      if (select (xfd+1, &fd, NULL, NULL, &tvt) == 0)
	{
	  paint(NULL);
	}
    }

}
