/****************************************************************************/
/* Copyright 1999 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/

/****************************************************************************
Module Name:
    itsyeth.c

PNP ethernet/USB driver for the Itsy Pocket Computer

Environment:
    Kernel Mode - 

Revision History
    - JCB 8/14/97 Example Driver Created
	- PD  23/7/99 Created from sample driver in Win2000 DDK

*****************************************************************************/

#include <ndis.h>
#include <efilter.h>

#include "equates.h"
#include "itsyethdbg.h"
#include "itsyethsw.h"
#include "itsyethpr.h"

NDIS_STATUS 
ItsyEthTransferData(
       OUT PNDIS_PACKET Packet,
       OUT PUINT BytesTransferred,
       IN NDIS_HANDLE MiniportAdapterContext,
       IN NDIS_HANDLE MiniportReceiveContext,
       IN UINT ByteOffset,
       IN UINT BytesToTransfer
       )
{
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthTransferData\n"));
	return NDIS_STATUS_SUCCESS;
}


//-----------------------------------------------------------------------------
// Procedure:   DriverEntry
//
// Description: This is the primary initialization routine for the Itsy
//              driver. It has two tasks;
//              1. initialise the driver object with entry points 
//                 for plug and play IRPs
//              2. Intializing the wrapper and registering the 
//                 adapter driver.  The routine gets called once per driver.
//                 operation therein is slightly different from the normal
//                 NDIS because operations are started from PnP entry points.
//
// Arguments:
//      DriverObject - Pointer to driver object created by the system.
//      RegistryPath - The registry path of this driver
//
// Returns:
//  The status of the operation, normally this will be NDIS_STATUS_SUCCESS
//-----------------------------------------------------------------------------
NTSTATUS
DriverEntry(PDRIVER_OBJECT DriverObject,
            PUNICODE_STRING RegistryPath)
{

    NDIS_STATUS         Status = STATUS_SUCCESS;
    NDIS_HANDLE         NdisWrapperHandle;
    PNDIS_MINIPORT_CHARACTERISTICS ItsyEthChar;
	NDIS_HANDLE DriverHandle;

    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthDriverEntry %p\n", DriverObject));

    // Now we must initialize the wrapper, and then register the Miniport
    NdisMInitializeWrapper( &NdisWrapperHandle,
        DriverObject,
        RegistryPath,
        NULL
        );

	NdisAllocateMemory(&ItsyEthChar, sizeof(*ItsyEthChar), 0, HighestAcceptableMax);
    NdisZeroMemory(ItsyEthChar, sizeof(*ItsyEthChar));

    // Initialize the Miniport characteristics for the call to
    // NdisMRegisterMiniport.
    ItsyEthChar->MajorNdisVersion       = ITSYETH_NDIS_MAJOR_VERSION;
    ItsyEthChar->MinorNdisVersion       = ITSYETH_NDIS_MINOR_VERSION;
    ItsyEthChar->CheckForHangHandler    = NULL;                     // up to ndis for now
	// This guy doesn't generate interrupts directly so these
	// guys aren't required.
    ItsyEthChar->DisableInterruptHandler= NULL;
    ItsyEthChar->EnableInterruptHandler = NULL;
    ItsyEthChar->HandleInterruptHandler = NULL;
	ItsyEthChar->ISRHandler             = NULL;
    ItsyEthChar->ReconfigureHandler     = NULL;
    ItsyEthChar->SendHandler            = NULL;

	ItsyEthChar->HaltHandler            = ItsyEthHalt;
    ItsyEthChar->InitializeHandler      = ItsyEthInitialize;
    ItsyEthChar->QueryInformationHandler= ItsyEthQueryInformation;
    ItsyEthChar->ResetHandler           = ItsyEthReset;
    ItsyEthChar->SetInformationHandler  = ItsyEthSetInformation;
    ItsyEthChar->SendPacketsHandler     = ItsyEthMultipleSend;
    ItsyEthChar->ReturnPacketHandler    = ItsyEthReturnPackets;
    ItsyEthChar->TransferDataHandler    = ItsyEthTransferData;
    ItsyEthChar->AllocateCompleteHandler = ItsyEthAllocateComplete;

    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("DriverEntry: About to call NdisMRegisterMiniport\n"));

    // Register this driver with the NDIS wrapper
    //   This will cause ItsyEthInitialize to be called before returning
    Status = NdisMRegisterMiniport(NdisWrapperHandle,
		                           ItsyEthChar,
                                   sizeof(NDIS_MINIPORT_CHARACTERISTICS));

    ItsyEth_KdPrintCond(DBGLVL_DEFAULT, 
		                (Status != STATUS_SUCCESS), 
						("NdisMRegisterMiniport failed (Status = 0x%x)\n", Status));

	// now initialise the WDM stuff
//	ItsyEth_DriverInit(DriverObject);

    return Status;
}



//-----------------------------------------------------------------------------
// Procedure:   ItsyEthHalt
//
// Description: Removes an adapter instance that was previously initialized.
//              To "halt" or "remove" an adapter, we disable its interrupt,
//              abort its receive unit (otherwise it would continue to DMA in
//              data), and release all of the resources (memory, i/o space,
//              etc.) that the adapter instance was using.
//              This routine is only called when the adapter is "stopped"
//              or unloaded with a "net stop e100b". To see what is called
//              at machine shutdown see D100ShutdownHandler.
//
// Arguments:
//      MiniportAdapterContext - pointer to the adapter object data area.
//
// Returns:     (none)
//-----------------------------------------------------------------------------

VOID
ItsyEthHalt(NDIS_HANDLE MiniportAdapterContext)
{
	PItsyEth_ADAPTER Adapter;

    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthHalt\n"));

	Adapter = (PItsyEth_ADAPTER) MiniportAdapterContext;
#if 1
    //NdisMDeregisterAdapterShutdownHandler(Adapter->D100AdapterHandle);
//	NdisMDeregisterDevice(Adapter->ItsyEthAdapterHandle);

    NdisFreeSpinLock(&Adapter->Lock);

	ItsyEth_RemoveDevice(Adapter->WdmHandle);
    // Free the entire adapter object, including the shared memory structures.
    NdisFreeMemory(Adapter, sizeof(ItsyEth_ADAPTER), 0);
#endif
	return;
}


//-----------------------------------------------------------------------------
// Procedure:   ItsyEthInitialize/ItsyEthAddAdapter
//
// Description: This routine is called once per each supported adapter card in
//              the system.  This routine is responsible for initializing each
//              adapter.  This includes parsing all of the necessary parameters
//              from the registry, allocating and initializing shared memory
//              structures, and whatever it needs for USB....
//
// Arguments:
//      OpenErrorStatus (mini) - Returns more info about any failure
//      SelectedMediumIndex (mini) - Returns the index in MediumArray of the
//                                   medium that the miniport is using
//      MediumArraySize (mini) - An array of medium types that the driver
//                               supports
//      MiniportAdapterHandle (mini) - pointer to the adapter object data area.
//
//      WrapperConfigurationContext (both) - A value that we will pass to
//                                           NdisOpenConfiguration.
//
//
// Returns:
//      NDIS_STATUS_SUCCESS - If the adapter was initialized successfully.
//      <not NDIS_STATUS_SUCCESS> - If for some reason the adapter didn't
//                                  initialize
//-----------------------------------------------------------------------------

NDIS_STATUS
ItsyEthInitialize(PNDIS_STATUS OpenErrorStatus,
			   PUINT SelectedMediumIndex,
               PNDIS_MEDIUM MediumArray,
               UINT MediumArraySize,
               NDIS_HANDLE MiniportAdapterHandle,
               NDIS_HANDLE WrapperConfigurationContext)
{
	PItsyEth_ADAPTER Adapter;
	NDIS_STATUS Status;
	UINT i;
	UCHAR tmpNodeAddress[ETH_LENGTH_OF_ADDRESS] = {0x40, 0x00, 0x00, 0x00, 0xff, 0x01};
	UCHAR tmpDevAddress[ETH_LENGTH_OF_ADDRESS] =  {0x40, 0x00, 0x00, 0x00, 0x00, 0x01}; 
	NDIS_HANDLE         ConfigHandle;
	PDEVICE_OBJECT PhysicalDeviceObject;
    PDEVICE_OBJECT FunctionalDeviceObject;
    PDEVICE_OBJECT NextDeviceObject;
	NTSTATUS ntStatus;
	NDIS_STATUS returnStatus = NDIS_STATUS_SUCCESS;
	
    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthInitialize %p\n", WrapperConfigurationContext));
	
    // if medium type 802.3 not found in list, exit with error)
    for (i = 0; i < MediumArraySize; i++)
    {
        if (MediumArray[i] == NdisMedium802_3) 
			break;
    }
    if (i == MediumArraySize)
    {
        ItsyEth_KdPrint(DBGLVL_DEFAULT, ("802.3 Media type not found.\n"));
        return NDIS_STATUS_UNSUPPORTED_MEDIA;
    }

    // Select ethernet
    *SelectedMediumIndex = i;

    // Allocate the Adapter Object, exit if error occurs
    Status = NdisAllocateMemory(&Adapter, sizeof(ItsyEth_ADAPTER), 0, HighestAcceptableMax);
    if (Status != NDIS_STATUS_SUCCESS)
    {
        ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ADAPTER Allocate Memory failed (Status = 0x%x)\n", 
			Status));
        return Status;
    }

    //Zero out the adapter object space
    NdisZeroMemory(Adapter, sizeof(ItsyEth_ADAPTER));

    Adapter->ItsyEthAdapterHandle = MiniportAdapterHandle;

    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Adapter structure pointer is %08x\n", Adapter));

    // XXX Open Registry, exit if error occurs.
	// for now I will forget about the registry

	// create a local MAC address based on the adapter context.
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Itsy MAC address %x %x %x %x %x %x\n",
		                             tmpNodeAddress[0],
									 tmpNodeAddress[1],
									 tmpNodeAddress[2],
									 tmpNodeAddress[3],
									 tmpNodeAddress[4],
									 tmpNodeAddress[5]));

	NdisMoveMemory(Adapter->PermanentNodeAddress, tmpNodeAddress, 3);
	NdisMoveMemory(Adapter->NodeAddress, tmpNodeAddress, ETH_LENGTH_OF_ADDRESS);
	NdisMoveMemory(Adapter->DevAddress, tmpDevAddress, ETH_LENGTH_OF_ADDRESS);

	// our line speed is always 12Mbps, well that's what i am going to report
	Adapter->LineSpeedCur = 12;
	Adapter->LinkIsActive = NdisMediaStateConnected;

    // call NdisMSetAttributesEx in order to let NDIS know
    // what kind of driver and features we support
    NdisMSetAttributesEx(
        Adapter->ItsyEthAdapterHandle,
        (NDIS_HANDLE) Adapter,
        0,
        (ULONG)
        NDIS_ATTRIBUTE_DESERIALIZE |
        NDIS_ATTRIBUTE_INTERMEDIATE_DRIVER |
		NDIS_ATTRIBUTE_IGNORE_PACKET_TIMEOUT |
		NDIS_ATTRIBUTE_IGNORE_REQUEST_TIMEOUT,
        NdisInterfaceInternal
        );

	// now can start up the USB specific stuff....
	// 1. config the device
	// 2. start an async receive.

    // allocate a spin lock for locking at all our entry points
    NdisAllocateSpinLock(&Adapter->Lock);

	// get the PDo and FDO and configure the itsy for use
    NdisMGetDeviceProperty(Adapter->ItsyEthAdapterHandle,
	                      &PhysicalDeviceObject,
                          &FunctionalDeviceObject,
                          &NextDeviceObject,
						  NULL,
						  NULL);

	// allocate a buffer pool for receive packets
	ItsyEthInitialiseRecvQueue(Adapter);

	ntStatus = ItsyEth_StartDevice(PhysicalDeviceObject, NextDeviceObject, &Adapter->WdmHandle);
	if (NT_SUCCESS(ntStatus))
	{
		ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthInitialize - starting\n"));
		ItsyEthStartReceive(Adapter);
	}
	else
	{
		returnStatus = NDIS_STATUS_FAILURE;
	}
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("Leaving ItsyEthInitialize %x\n", returnStatus));

    return returnStatus;
}

//-----------------------------------------------------------------------------
// Procedure:   ItsyEthReset
//
// Description: Instructs the Miniport to issue a hardware reset to the
//              network adapter.  The driver also resets its software state.
//              this function also resets the transmit queues.
//
// Arguments:
//      AddressingReset - TRUE if the wrapper needs to call
//                        MiniportSetInformation to restore the addressing
//                        information to the current values
//      MiniportAdapterContext - pointer to the adapter object data area.
//
// Returns:
//      NDIS_STATUS_PENDING - This function sets a timer to complete, so
//                            pending is always returned
//-----------------------------------------------------------------------------

NDIS_STATUS
ItsyEthReset(PBOOLEAN AddressingReset,
          NDIS_HANDLE MiniportAdapterContext)
{
    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthReset\n"));

    return(NDIS_STATUS_SUCCESS);
}

//-----------------------------------------------------------------------------
// ItsyEthResetComplete
//
// PARAMETERS: NDIS_HANDLE MiniportAdapterContext
//
// DESCRIPTION: This function is called by a timer indicating our
//              reset is done (by way of .5 seconds expiring)
//
// RETURNS: nothing, but sets NdisMResetComplete, enables ints
//          and starts the receive unit
//
//-----------------------------------------------------------------------------

VOID
ItsyEthResetComplete(PVOID sysspiff1,
                  NDIS_HANDLE MiniportAdapterContext,
                  PVOID sysspiff2, PVOID sysspiff3)
{
    ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthResetComplete\n"));
	
#if 0
    INITSTR(("\n"));
    Adapter = PD100_ADAPTER_FROM_CONTEXT_HANDLE(MiniportAdapterContext);

    DEBUGCHAR(Adapter,'@');

    NdisMResetComplete(Adapter->D100AdapterHandle,
        (NDIS_STATUS) NDIS_STATUS_SUCCESS,
        FALSE);

    Adapter->ResetInProgress = FALSE;


    DEBUGSTR(("D100Reset: Calling StartReceiveUnit\n"));
    // Start the receive unit and indicate any pending receives that we
    // had left in our queue.

    if (StartReceiveUnit(Adapter))
    {
        TRACE2(Adapter, ("Indicating Receive complete\n"));
        DEBUGCHAR(Adapter,'^');
        NdisMEthIndicateReceiveComplete(Adapter->D100AdapterHandle);
    }

    D100EnableInterrupt(Adapter);
#endif

    return;
}
//-----------------------------------------------------------------------------
// Procedure: ItsyEthAllocateComplete
//
// Description: This function handles initialization of new receive memory
//              when the os returns some shared memory to us because we
//              called NdisMAllocateSharedMemoryAsync when we ran low on
//              receive buffers.
//
// Arguments: MiniportAdapterContext - a pointer to our adapter structure
//            VirtualAddress - The virtual address of the new memory
//            PhysicalAddress - _pointer to_ The physical address of the
//                              new memory
//            Length - The length of the new memory
//            Context - The offset into our MemoryDescriptor array to
//                      initialize (zero-based)
//
// Returns: Nothing
//
//-----------------------------------------------------------------------------

VOID
ItsyEthAllocateComplete(NDIS_HANDLE MiniportAdapterContext,
                     IN PVOID VirtualAddress,
                     IN PNDIS_PHYSICAL_ADDRESS PhysicalAddress,
                     IN ULONG Length,
                     IN PVOID Context)
{
	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEthAllocateComplete\n"));

#if 0
    current = &Adapter->ReceiveMemoryDescArray[PtrToUint(Context)];
    INITSTR(("\n"));

    // check if any of the three variables that indicate failure are zero,
    // indicating that a part of the allocation failed
    if ((VirtualAddress == 0)
        || (NdisGetPhysicalAddressLow(*PhysicalAddress) == 0)
        || (Length == 0))
    {
        // ndismfreesharedmemory?
        Adapter->AsynchronousAllocationPending = FALSE;
        return;
    }


    // catch the case where we have waaaay too many receive buffers.
    if((Adapter->NumRfd + packet_count[PtrToUint(Context)]) >= MAX_RECEIVE_DESCRIPTORS)
    {
        NdisMFreeSharedMemory(Adapter->D100AdapterHandle,
            Length,
            FALSE,
            (PVOID) VirtualAddress,
            *PhysicalAddress);
        Adapter->AsynchronousAllocationPending = FALSE;
        return;
    }

    NdisAcquireSpinLock(&Adapter->Lock);

    // allocate and setup some cached memory
    // also initialize the uncached areas of the RMD
    // (receive memory descriptor)
    Status = AllocateRMD(current,
        packet_count[PtrToUint(Context)],
        (ULONG_PTR) VirtualAddress,
        *PhysicalAddress,
        Length);

    if (Status != NDIS_STATUS_SUCCESS)
    {
        D100LogError(Adapter, EVENT_20, NDIS_ERROR_CODE_OUT_OF_RESOURCES, 0);

        INITSTR(("Could not allocate %d bytes for ExtraRecvCached mem\n", current->CachedMem.Size));

        NdisReleaseSpinLock(&Adapter->Lock);

        NdisMFreeSharedMemory(Adapter->D100AdapterHandle,
            Length,
            FALSE,
            (PVOID) VirtualAddress,
            *PhysicalAddress);

        Adapter->AsynchronousAllocationPending = FALSE;
        return;
    }

    // increment our RMD counter
    ++(Adapter->Last_RMD_used);

    // update the number of Rfds we have
    Adapter->NumRfd += packet_count[PtrToUint(Context)];

    //    SwRfdNext = (D100SwRfd *)current->CachedMem.VirtualAddress;// cached RFDs logical
    //    HwRfdNext = (RFD_STRUC *)xtraRecvUnCached;   // uncached RFDs logical


    for (RfdCount = 0;
    RfdCount < (Adapter->NumRfd - OldNumRfds);
    ++RfdCount)
    {
        SwRfdPtr = BuildSwRfd(Adapter,current, RfdCount);

        // if something goes wrong with packet pool (should be impossible)
        if (SwRfdPtr == NULL)
            break;

        // call init and chain packet to put the packet on our available list
        InitAndChainPacket(Adapter,SwRfdPtr);

    }


    Adapter->AsynchronousAllocationPending = FALSE;

    NdisReleaseSpinLock(&Adapter->Lock);
#endif
    return;
}

