/****************************************************************************/
/* Copyright 1999 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/

/*++

Copyright (c) 1997-1998  Microsoft Corporation

Module Name:

    ItsyEthPnp.c 

Abstract:

    Plug and Play module for the Itsy Ethernet/USB driver.
	This module does not handle PnP IRPs because NDIS handles them for us,
	instead this module takes care of the starting and stopping the 
	device as the NDIS miniport wrapper tells us to.

Environment:

    kernel mode only

Notes:

  THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
  KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
  PURPOSE.

  Copyright (c) 1997-1998 Microsoft Corporation.  All Rights Reserved.


Revision History:

    2/7/98: created

--*/

#include "wdm.h"
#include "stdarg.h"
#include "stdio.h"

#include "usbdi.h"
#include "usbdlib.h"

#include "itsyethdbg.h"
#include "itsyethwdm.h"
#include "itsyethcom.h"

/*++

Routine Description:

    Called from the MiniportInitialize entry point.

    Initializes a given instance of the device on the USB.
    USB client drivers such as us set up URBs (USB Request Packets) to send requests
    to the host controller driver (HCD). The URB structure defines a format for all
    possible commands that can be sent to a USB device.
    Here, we request the device descriptor and store it, and configure the device.

Arguments:

    DeviceObject     - pointer to the PDO (PhysicalDevice Object)
	NextDeviceObject - pointer to the FDO (FunctionalDevice Object)
	WdmHandle        - pointer to a device extension structure for storing 
	                   WDM context, if NULL, one is created.

Return Value:

    NT status code

--*/
NTSTATUS
ItsyEth_StartDevice(
	IN  PDEVICE_OBJECT DeviceObject,
	IN  PDEVICE_OBJECT NextDeviceObject,
	OUT PITSYETH_WDM_HANDLE *WdmHandle
    )
{

    NTSTATUS ntStatus;
    PUSB_DEVICE_DESCRIPTOR deviceDescriptor = NULL;
	PITSYETH_DEVICE_EXTENSION deviceExtension;
    PURB urb;
    ULONG siz;
	ULONG idx;
	PUSBD_PIPE_INFORMATION PipeInfo;

    ItsyEth_KdPrint( DBGLVL_DEFAULT,("enter ItsyEth_StartDevice\n"));

	// if the device extention has not already been created create it.
	if (*WdmHandle == NULL)
	{
		*WdmHandle = ExAllocatePool(NonPagedPool, sizeof(struct _ITSYETH_DEVICE_EXTENSION));
		ItsyEth_KdPrintCond( DBGLVL_DEFAULT,!WdmHandle, 
			("ItsyEth_StartDevice() FAILED ExAllocatePool() for wdm_extension\n"));
	}
	deviceExtension = (PITSYETH_DEVICE_EXTENSION) *WdmHandle;
	deviceExtension->TopOfStackDeviceObject = NextDeviceObject;
	deviceExtension->PhysicalDeviceObject = DeviceObject;

	ItsyEth_InitFrameList(deviceExtension);

    urb = ExAllocatePool(NonPagedPool, sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST));

    ItsyEth_KdPrintCond( DBGLVL_DEFAULT,!urb, ("ItsyEth_StartDevice() FAILED ItsyEth_ExAllocatePool() for URB\n"));

    if (urb) 
	{
        siz = sizeof(USB_DEVICE_DESCRIPTOR);
        deviceDescriptor = ExAllocatePool(NonPagedPool, siz);

        ItsyEth_KdPrintCond( DBGLVL_DEFAULT, !deviceDescriptor, ("ItsyEth_StartDevice() FAILED ItsyEth_ExAllocatePool() for deviceDescriptor\n"));
        if (deviceDescriptor) 
		{
            UsbBuildGetDescriptorRequest(urb,
                                         (USHORT) sizeof (struct _URB_CONTROL_DESCRIPTOR_REQUEST),
                                         USB_DEVICE_DESCRIPTOR_TYPE,
                                         0,
                                         0,
                                         deviceDescriptor,
                                         NULL,
                                         siz,
                                         NULL);

            ntStatus = ItsyEth_CallUSBD(DeviceObject, NextDeviceObject, urb);

            ItsyEth_KdPrintCond( DBGLVL_DEFAULT, !NT_SUCCESS(ntStatus), ("ItsyEth_StartDevice() FAILED ItsyEth_CallUSBD(DeviceObject, urb)\n"));

            if (NT_SUCCESS(ntStatus)) 
			{
                ItsyEth_KdPrint( DBGLVL_DEFAULT,("Device Descriptor = %x, len %x\n",
                                deviceDescriptor,
                                urb->UrbControlDescriptorRequest.TransferBufferLength));

                ItsyEth_KdPrint( DBGLVL_MEDIUM,("I82930 Device Descriptor:\n"));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("-------------------------\n"));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bLength %d\n", deviceDescriptor->bLength));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bDescriptorType 0x%x\n", deviceDescriptor->bDescriptorType));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bcdUSB 0x%x\n", deviceDescriptor->bcdUSB));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bDeviceClass 0x%x\n", deviceDescriptor->bDeviceClass));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bDeviceSubClass 0x%x\n", deviceDescriptor->bDeviceSubClass));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bDeviceProtocol 0x%x\n", deviceDescriptor->bDeviceProtocol));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bMaxPacketSize0 0x%x\n", deviceDescriptor->bMaxPacketSize0));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("idVendor 0x%x\n", deviceDescriptor->idVendor));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("idProduct 0x%x\n", deviceDescriptor->idProduct));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bcdDevice 0x%x\n", deviceDescriptor->bcdDevice));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("iManufacturer 0x%x\n", deviceDescriptor->iManufacturer));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("iProduct 0x%x\n", deviceDescriptor->iProduct));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("iSerialNumber 0x%x\n", deviceDescriptor->iSerialNumber));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("bNumConfigurations 0x%x\n", deviceDescriptor->bNumConfigurations));
            }
        } 
		else 
		{
			// if we got here we failed to allocate deviceDescriptor
            ntStatus = STATUS_INSUFFICIENT_RESOURCES;
        }

        if (NT_SUCCESS(ntStatus)) 
		{
            deviceExtension->UsbDeviceDescriptor = deviceDescriptor;
        } 
		else if (deviceDescriptor) 
		{
            ExFreePool(deviceDescriptor);
        }
        ExFreePool(urb);
    } 
	else 
	{
		// if we got here we failed to allocate the urb
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }

    if (NT_SUCCESS(ntStatus)) 
	{
        ntStatus = ItsyEth_ConfigureDevice(deviceExtension);
        ItsyEth_KdPrintCond( DBGLVL_DEFAULT,!NT_SUCCESS(ntStatus),("ItsyEth_StartDevice FAILURE (%x)\n", ntStatus));
    }

    if (NT_SUCCESS(ntStatus)) 
	{
		ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_StartDevice ready to go..(%x)\n", ntStatus));
        deviceExtension->DeviceStarted = TRUE;
		deviceExtension->DeviceStalled = FALSE;
		deviceExtension->StopDeviceRequested = FALSE;
		deviceExtension->OutPipeState = OUTPIPE_RUNNING;
		deviceExtension->ResetWork = NULL;
		deviceExtension->PendingOutIoCount = 0;
		deviceExtension->FastCompleteSpinlock = ExAllocatePool(NonPagedPool, sizeof(KSPIN_LOCK));
		RtlZeroMemory(deviceExtension->FastCompleteSpinlock, sizeof(KSPIN_LOCK));
		KeInitializeSpinLock(deviceExtension->FastCompleteSpinlock);
		ItsyEth_KdPrint( DBGLVL_DEFAULT, ("ItsyEth_StartDevice spinlock %p %p\n", 
			deviceExtension, deviceExtension->FastCompleteSpinlock));
	
		// now initialise the pipes that we are going to use.
		// find the appropriate pipe
		for (idx=0;idx<deviceExtension->UsbInterface->NumberOfPipes;idx++)
		{
			PipeInfo = &deviceExtension->UsbInterface->Pipes[idx];
			if ((PipeInfo->EndpointAddress & 0x3) == 0x1)
			{
				deviceExtension->OutPipe = PipeInfo;
			}
			if ((PipeInfo->EndpointAddress & 0x3) == 0x2)
			{
				deviceExtension->InPipe = PipeInfo;
			}
		}

		GetAddress(deviceExtension);
	}
	ItsyEth_KdPrint( DBGLVL_DEFAULT, ("exit ItsyEth_StartDevice (%x)\n", ntStatus));
    return ntStatus;
}

/*++

Routine Description:

	Called from MiniportHalt entrypoint to
    clean up our device instance's allocated buffers; 

Arguments:

    DeviceObject - pointer to the FDO

Return Value:

    NT status code from free symbolic link operation

--*/
NTSTATUS
ItsyEth_RemoveDevice(
	IN  PITSYETH_WDM_HANDLE *WdmHandle
    )
{
    PITSYETH_DEVICE_EXTENSION deviceExtension;
    NTSTATUS ntStatus = STATUS_SUCCESS;

	ItsyEth_KdPrint(DBGLVL_DEFAULT, ("ItsyEth_RemoveDevice\n"));

    deviceExtension = (PITSYETH_DEVICE_EXTENSION) WdmHandle;

	deviceExtension->DeviceStarted = FALSE;
	deviceExtension->StopDeviceRequested = TRUE;

	// If any pipes are still open, call USBD with URB_FUNCTION_ABORT_PIPE
	// This call will also close the pipes; if any user close calls get through,
	// they will be noops
	ItsyEth_AbortPipes(deviceExtension);

    //
    // Free device descriptor structure
    //
    if (deviceExtension->UsbDeviceDescriptor) 
	{
        ExFreePool(deviceExtension->UsbDeviceDescriptor);
    }

    //
    // Free up the UsbInterface structure
    //
    if (deviceExtension->UsbInterface) 
	{
        ExFreePool(deviceExtension->UsbInterface);
    }

	// XXX free up the frames also...

	ExFreePool(deviceExtension);
    ItsyEth_KdPrint( DBGLVL_DEFAULT,("exit ItsyEth_RemoveDevice() status = 0x%x\n", ntStatus ));

    return ntStatus;
}


/*++

Routine Description:

    Initializes a given instance of the device on the USB and
	selects and saves the configuration.

Arguments:

    DeviceObject - pointer to the physical device object for this instance of the 82930
                    device.


Return Value:

    NT status code

--*/
NTSTATUS
ItsyEth_ConfigureDevice(IN PITSYETH_DEVICE_EXTENSION deviceExtension)
{
    NTSTATUS ntStatus;
    PURB urb;
    ULONG siz;
	PUSB_DEVICE_DESCRIPTOR UsbDeviceDescriptor;
	PUSB_CONFIGURATION_DESCRIPTOR UsbConfigurationDescriptor;


    ItsyEth_KdPrint( DBGLVL_DEFAULT,("enter ItsyEthUsb_ConfigureDevice\n"));

    urb = ExAllocatePool(NonPagedPool,
                         sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST));
	if ( !urb )
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	// When USB_CONFIGURATION_DESCRIPTOR_TYPE is specified for DescriptorType
	// in a call to UsbBuildGetDescriptorRequest(),
	// all interface, endpoint, class-specific, and vendor-specific descriptors 
	// for the configuration also are retrieved. 
	// The caller must allocate a buffer large enough to hold all of this 
	// information or the data is truncated without error.
	// Therefore the 'siz' set below is just a 'good guess', and we may have to retry

    siz = sizeof(USB_CONFIGURATION_DESCRIPTOR) + 512;  
	UsbConfigurationDescriptor = ExAllocatePool(NonPagedPool, siz);

	if ( !UsbConfigurationDescriptor ) 
	{
	    ExFreePool(urb);
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	UsbBuildGetDescriptorRequest(urb,
								 (USHORT) sizeof (struct _URB_CONTROL_DESCRIPTOR_REQUEST),
								 USB_CONFIGURATION_DESCRIPTOR_TYPE,
								 0,
								 0,
								 UsbConfigurationDescriptor,
								 NULL,
								 siz,
								 NULL);

	ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
		                        deviceExtension->TopOfStackDeviceObject,
								urb);

	ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_CallUSBD() Configuration Descriptor = %x, len %x\n",
					UsbConfigurationDescriptor,
					urb->UrbControlDescriptorRequest.TransferBufferLength));
	
    ExFreePool(urb);
	ITSYETH_ASSERT( UsbConfigurationDescriptor );

    //
    // We have the configuration descriptor for the configuration we want.
    // Now we issue the select configuration command to get
    // the  pipes associated with this configuration.
    //
    ntStatus = ItsyEth_SelectInterface(deviceExtension, UsbConfigurationDescriptor);
	
	ExFreePool(UsbConfigurationDescriptor);

	// now flag that the device is ok
    ItsyEth_KdPrint( DBGLVL_DEFAULT,("exit ItsyEth_ConfigureDevice (%x)\n", ntStatus));
    return ntStatus;
} 

/*++

Routine Description:

    Initializes Itsy interface.
	This driver only supports one interface (with multiple endpoints).

Arguments:

    DeviceObject - pointer to the device object for this instance of the 82930
                    device.

    ConfigurationDescriptor - pointer to the USB configuration
                    descriptor containing the interface and endpoint
                    descriptors.

Return Value:

    NT status code

--*/
NTSTATUS
ItsyEth_SelectInterface(IN PITSYETH_DEVICE_EXTENSION deviceExtension,
                        IN PUSB_CONFIGURATION_DESCRIPTOR ConfigurationDescriptor)
{
    NTSTATUS ntStatus;
    PURB urb = NULL;
    ULONG i;
    PUSB_INTERFACE_DESCRIPTOR interfaceDescriptor = NULL;
	PUSBD_INTERFACE_INFORMATION Interface = NULL;
    USHORT siz;

    ItsyEth_KdPrint( DBGLVL_DEFAULT,("enter ItsyEth_SelectInterface\n"));

    //
    // BulkUsb driver only supports one interface, we must parse
    // the configuration descriptor for the interface 
    // and remember the pipes.
    //
    urb = USBD_CreateConfigurationRequest(ConfigurationDescriptor, &siz);
    if (urb) 
	{
		//
		// USBD_ParseConfigurationDescriptorEx searches a given configuration
		// descriptor and returns a pointer to an interface that matches the 
		//  given search criteria. We only support one interface on this device
		//
        interfaceDescriptor =
            USBD_ParseConfigurationDescriptorEx(ConfigurationDescriptor,
								  ConfigurationDescriptor, //search from start of config  descriptro
								  -1,	// interface number not a criteria; we only support one interface
								  -1,   // not interested in alternate setting here either
								  -1,   // interface class not a criteria
								  -1,   // interface subclass not a criteria
								  -1    // interface protocol not a criteria
								  );

		if ( !interfaceDescriptor ) 
		{
			ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_SelectInterface() ParseConfigurationDescriptorEx() failed\n  returning STATUS_INSUFFICIENT_RESOURCES\n"));
			ExFreePool(urb);
			return STATUS_INSUFFICIENT_RESOURCES;
		}

        Interface = &urb->UrbSelectConfiguration.Interface;
        for (i=0; i< Interface->NumberOfPipes; i++) 
		{
            //
            // perform any pipe initialization here
            //
            Interface->Pipes[i].MaximumTransferSize = 4096;
            Interface->Pipes[i].PipeFlags = 0;
        }

        UsbBuildSelectConfigurationRequest(urb,
                                          (USHORT) siz,
                                          ConfigurationDescriptor);


        ntStatus = ItsyEth_CallUSBD(deviceExtension->PhysicalDeviceObject,
			                        deviceExtension->TopOfStackDeviceObject,
									urb);

    } 
	else 
	{
        ItsyEth_KdPrint( DBGLVL_DEFAULT,("ItsyEth_SelectInterface() USBD_CreateConfigurationRequest() failed\n  returning STATUS_INSUFFICIENT_RESOURCES\n"));
        ntStatus = STATUS_INSUFFICIENT_RESOURCES;
    }


    if (NT_SUCCESS(ntStatus)) 
	{
        deviceExtension->UsbInterface = ExAllocatePool(NonPagedPool,
                                                       Interface->Length);

        if (deviceExtension->UsbInterface) 
		{
            ULONG j;
            //
            // save a copy of the interface information returned
            //
            RtlCopyMemory(deviceExtension->UsbInterface, Interface, Interface->Length);

            //
            // Dump the interface to the debugger
            //
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("---------\n"));
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("NumberOfPipes 0x%x\n", deviceExtension->UsbInterface->NumberOfPipes));
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("Length 0x%x\n", deviceExtension->UsbInterface->Length));
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("Alt Setting 0x%x\n", deviceExtension->UsbInterface->AlternateSetting));
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("Interface Number 0x%x\n", deviceExtension->UsbInterface->InterfaceNumber));
            ItsyEth_KdPrint( DBGLVL_DEFAULT,("Class, subclass, protocol 0x%x 0x%x 0x%x\n",
                deviceExtension->UsbInterface->Class,
                deviceExtension->UsbInterface->SubClass,
                deviceExtension->UsbInterface->Protocol));

            // Dump the pipe info

            for (j=0; j<Interface->NumberOfPipes; j++) 
			{
                PUSBD_PIPE_INFORMATION pipeInformation;

                pipeInformation = &deviceExtension->UsbInterface->Pipes[j];

                ItsyEth_KdPrint( DBGLVL_MEDIUM,("---------\n"));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("PipeType 0x%x\n", pipeInformation->PipeType));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("EndpointAddress 0x%x\n", pipeInformation->EndpointAddress));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("MaxPacketSize 0x%x\n", pipeInformation->MaximumPacketSize));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("Interval 0x%x\n", pipeInformation->Interval));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("Handle 0x%x\n", pipeInformation->PipeHandle));
                ItsyEth_KdPrint( DBGLVL_MEDIUM,("MaximumTransferSize 0x%x\n", pipeInformation->MaximumTransferSize));
            }

            ItsyEth_KdPrint( DBGLVL_MEDIUM,("---------\n"));
        }
    }

    if (urb) 
	{
		// don't call the ItsyEthExFreePool since the buffer was 
		//  alloced by USBD_CreateConfigurationRequest, not ItsyEthExAllocatePool()
        ExFreePool(urb);
    }
    ItsyEth_KdPrint( DBGLVL_DEFAULT,("exit ItsyEth_SelectInterface (%x)\n", ntStatus));

    return ntStatus; 
}
