#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <regex.h>

#include <libmb/mb.h>
#include <mbdesktop_module.h>

/* 
   XXX Todo . 
   - some how check we have jpg/png support *done*
   - figure out some way to get rid of folder paths
     - use item->comment ? *done*
     - Add to simple filebrowser
   - Simple progress dialog when making thumbnails *done*
     - imp thumbnail spec
   - Slideshow mode ?
   - Add icon to distro, all it to be theme
 */

enum {
  WINDOW_STATE = 0,
  WINDOW_STATE_FULLSCREEN,
  WINDOW_STATE_MODAL,
  WINDOW_TYPE_DIALOG,
  WINDOW_TYPE,    
  _NET_WM_NAME,
  UTF8_STRING,
  ATOM_CNT
};

typedef struct BrowserData { 
  char* BrowserCurrentPath;
  char* BrowserFolderName;
  char* BrowserPath;
  
  Window win;

  Atom   atoms[ATOM_CNT];
  int    win_w;
  int    win_h;
  Bool   have_backing;

  MBPixbuf      *pixbuf;
  MBPixbufImage *img;
  MBDesktopItem *item_viewed;

} BrowserData;

#define MOD_VIS_NAME "Pictures"


static int trapped_error_code = 0;
static int (*old_error_handler) (Display *d, XErrorEvent *e);

void
imgbrowser_open_cb (void *data1, void *data2);

static Bool
imgbrowser_load_image_for_item (MBDesktop *mb, MBDesktopItem *item);

static int
error_handler(Display     *display,
              XErrorEvent *error)
{
  trapped_error_code = error->error_code;
  return 0;
}

static void
trap_errors(void)
{
  trapped_error_code = 0;
  old_error_handler = XSetErrorHandler(error_handler);
}
 
static int
untrap_errors(void)
{
  XSetErrorHandler(old_error_handler);
  return trapped_error_code;
}


static void
imgbrowser_win_handle_resize (MBDesktop   *mb, 
			      BrowserData *data,
			      int          w,
			      int          h)
{
  if (data->win_w != w || data->win_h != h)
    {
      data->win_w  = w;
      data->win_h  = h;
      data->have_backing = False;
    }
}

static void
imgbrowser_win_handle_expose (MBDesktop   *mb, 
			      BrowserData *data)
{
  Pixmap backing;
  MBPixbufImage *img_scaled = NULL, *img_checked = NULL;

  int scaled_w, scaled_h, x, y, img_h, img_w;
  double x_zoom, y_zoom, zoom;

  if (data->have_backing || !data->win_w || !data->win_h ) return;

  backing = XCreatePixmap(mbdesktop_xdisplay(mb),
			  mbdesktop_xrootwin(mb),
			  data->win_w, data->win_h,
			  data->pixbuf->depth ); 

  /* Zoom to Fit */

  scaled_w = img_w = mb_pixbuf_img_get_width(data->img);   
  scaled_h = img_h = mb_pixbuf_img_get_height(data->img);   

  x_zoom = (double) data->win_w / img_w;
  y_zoom = (double) data->win_h / img_h;

  zoom = (x_zoom < y_zoom) ? x_zoom : y_zoom;

  scaled_w = zoom * img_w; 
  scaled_h = zoom * img_h;

  y = (data->win_h - scaled_h)/2;
  x = (data->win_w - scaled_w)/2;

  printf("%s win_w: %i, win_h: %i, img_w: %i img_h: %i, scaled_w: %i, scaled_h: %i\n",
	 __func__, data->win_w, data->win_h, img_w, img_h, scaled_w, scaled_h);

  img_scaled = mb_pixbuf_img_scale(data->pixbuf, data->img, 
				   scaled_w, scaled_h);

#define CHECKED_SIZE 20

  /* Paint checkboard background */
  if (img_scaled->has_alpha 
      || scaled_w != data->win_w || scaled_h != data->win_h )
    {
      int cx, cy, r = 0x66, g = 0x66, b = 0x66;
      Bool darken = False;
      img_checked = mb_pixbuf_img_rgb_new(data->pixbuf, data->win_w, data->win_h);
      for (cy=0; cy < data->win_h; cy++)
	{
	  if ( cy % CHECKED_SIZE == 0)
	    {
	      if (darken)
		{ r = 0x66, g = 0x66, b = 0x66; darken = False; }
	      else
		{ r = 0x33, g = 0x33, b = 0x33; darken = True; }
	    }

	  for (cx=0; cx < data->win_w; cx++)
	    {
	      if ( cx % CHECKED_SIZE == 0)
		{
		  if (darken)
		    { r = 0x66, g = 0x66, b = 0x66; darken = False; }
		  else
		    { r = 0x33, g = 0x33, b = 0x33; darken = True; }
		}

	      mb_pixbuf_img_plot_pixel(data->pixbuf, img_checked, 
				       cx, cy, r, g, b);
	    }
	}

      mb_pixbuf_img_copy_composite (data->pixbuf, img_checked, img_scaled,
				    0, 0, scaled_w, scaled_h, x, y);

      mb_pixbuf_img_free(data->pixbuf, img_scaled);
      img_scaled = img_checked; 
      x=0; y=0;
    }

  mb_pixbuf_img_render_to_drawable(data->pixbuf, img_scaled, 
				   (Drawable)backing,
				   x, y);

  XSetWindowBackgroundPixmap(mbdesktop_xdisplay(mb), data->win, backing);
  XClearWindow(mbdesktop_xdisplay(mb), data->win);

  XFreePixmap(mbdesktop_xdisplay(mb), backing);
  mb_pixbuf_img_free(data->pixbuf, img_scaled);
  data->have_backing = True;
}


static void
imgbrowser_win_open (MBDesktop *mb, BrowserData *data)
{

  data->win
    = XCreateSimpleWindow(mbdesktop_xdisplay(mb),
			  mbdesktop_xrootwin(mb),
			  10, 10, 200, 200, 0,
			  BlackPixel(mbdesktop_xdisplay(mb), 
				     mbdesktop_xscreen(mb)),
			  WhitePixel(mbdesktop_xdisplay(mb), 
				     mbdesktop_xscreen(mb)));

  XChangeProperty(mbdesktop_xdisplay(mb), data->win, 
		  data->atoms[WINDOW_STATE], XA_ATOM, 32, 
		  PropModeReplace, 
		  (unsigned char *) &data->atoms[WINDOW_STATE_FULLSCREEN], 1);

  XSelectInput(mbdesktop_xdisplay(mb), data->win, 
	       ExposureMask | ButtonPressMask | ButtonReleaseMask |
	       KeyPress | KeyRelease | StructureNotifyMask );



  imgbrowser_win_handle_expose (mb, data); 

  XFlush(mbdesktop_xdisplay(mb));

  XMapWindow(mbdesktop_xdisplay(mb), data->win);

}

static void
imgbrowser_win_close (MBDesktop *mb, BrowserData *data)
{
  XDestroyWindow(mbdesktop_xdisplay(mb), data->win);
  mb_pixbuf_img_free(data->pixbuf, data->img);
  data->img = NULL;
}

static MBDesktopItem *
imgbrowser_get_prev_img_item(MBDesktop     *mb, 
			     BrowserData   *data, 
			     MBDesktopItem *item_sibling)
{	
  MBDesktopItem *item = NULL;

  item = mbdesktop_item_get_prev_sibling(item_sibling);

  while (item != NULL && mbdesktop_item_is_folder(mb,item))
    {
      item = mbdesktop_item_get_prev_sibling(item);
    }
  
  if (item == NULL)
    {
      return imgbrowser_get_prev_img_item(mb, data, mbdesktop_item_get_last_sibling(item_sibling));
    }

  return item;
}


static MBDesktopItem *
imgbrowser_get_next_img_item(MBDesktop     *mb, 
			     BrowserData   *data, 
			     MBDesktopItem *item_sibling)
{	      
  MBDesktopItem *item = NULL;  

  item = mbdesktop_item_get_next_sibling(item_sibling);

  while (item != NULL && mbdesktop_item_is_folder(mb,item))
    item = mbdesktop_item_get_next_sibling(item);
  
  if (item == NULL)
    {
      return imgbrowser_get_next_img_item(mb, data, mbdesktop_item_get_first_sibling(item_sibling));
    }
  
  return item;
}

static void
imgbrowser_win_event_loop (MBDesktop *mb, BrowserData *data)
{
  Bool done = False;
  KeySym key;
  XEvent xevent;
  MBDesktopItem *item = NULL;

  while(!done)
    {
      XNextEvent(mbdesktop_xdisplay(mb), &xevent);

      switch (xevent.type) 
	{
	case Expose:
	  printf("expose\n");
	  if (xevent.xexpose.window == data->win)
	    imgbrowser_win_handle_expose (mb, data); 
	  break;
	case ConfigureNotify:
	  printf("config\n");
	  if (xevent.xconfigure.window == data->win)
	    imgbrowser_win_handle_resize (mb, data, 
					  xevent.xconfigure.width,
					  xevent.xconfigure.height
					  ); 
	  printf("winspew log: %li, ConfigureNotify event\n", 
		 xevent.xconfigure.window);
	  /* XXX compress configure notifys ? */
	  break;

	case KeyPress:
	  switch (key = XKeycodeToKeysym (mbdesktop_xdisplay(mb), 
					  xevent.xkey.keycode, 0))
	    {
	    case XK_Left:
	      item = imgbrowser_get_prev_img_item(mb, data, data->item_viewed);
	      imgbrowser_load_image_for_item (mb, item);
	      imgbrowser_win_handle_expose (mb, data); 
	      break;
	    case XK_Right:
	    case XK_space:
	      item = imgbrowser_get_next_img_item(mb, data, data->item_viewed);
	      imgbrowser_load_image_for_item (mb, item);
	      imgbrowser_win_handle_expose (mb, data); 
	      break;
	    case XK_q:
	    case XK_Escape:
	    case XK_Return:
	    case XK_KP_Enter:
	      done = True;
	    }
	  break;
	}
    }


}

static Bool
imgbrowser_load_image_for_item (MBDesktop *mb, MBDesktopItem *item)
{
  BrowserData *mod_data;
  char *current_path = NULL;
  char *img_path;

  mod_data = (BrowserData*)mbdesktop_item_get_user_data (mb, item);

  /* Items contain full path to there image  */
  current_path = mbdesktop_item_get_extended_name(mb, item);

  img_path = alloca(strlen(current_path)+strlen(item->name)+3);
  sprintf(img_path, "%s/%s", current_path, item->name);

  if (mod_data->img != NULL)  
    mb_pixbuf_img_free(mod_data->pixbuf, mod_data->img);
  mod_data->img = NULL;

  mod_data->img = mb_pixbuf_img_new_from_file(mod_data->pixbuf, img_path);

  mod_data->have_backing = False;

  if (mod_data->img == NULL) return False;

  mod_data->item_viewed = item;

  return True;
}

static void
imgbrowser_file_activate_cb (void *data1, void *data2)
{
  BrowserData *mod_data;
  MBDesktop *mb = (MBDesktop *)data1; 
  MBDesktopItem *item = (MBDesktopItem *)data2; 

  mod_data = (BrowserData*)mbdesktop_item_get_user_data (mb, item);

  trap_errors(); 		/* XXX need to figure out why we get
				   unexplained x errors
				*/
  if (imgbrowser_load_image_for_item (mb, item))
    {
      imgbrowser_win_open (mb, mod_data);
      imgbrowser_win_event_loop (mb, mod_data);
      imgbrowser_win_close (mb, mod_data);
    }

  untrap_errors();

  mod_data->item_viewed = NULL;
}

int
imgbrowser_init (MBDesktop             *mb, 
                 MBDesktopFolderModule *folder_module, 
		 char                  *arg_str)
{
  DIR           *dp;
  struct dirent *dir_entry;
  struct stat    stat_info;
  MBDesktopItem *folder;
  BrowserData   *data = NULL;

  /* XXX args can be location of user def config folder */
  if (arg_str == NULL)
    {
      fprintf(stderr, "imgbrowser: missing directoy path arg.\n" );
      return -1;
    }

  if ((dp = opendir(arg_str)) == NULL)
    {
      fprintf(stderr, "imgbrowser: failed to open %s\n", arg_str);
      return -1;
    }

  /* Clip trailing '/' from path */
  if (arg_str[strlen(arg_str)-1] == '/')
    arg_str[strlen(arg_str)-1] = '\0';

  data = malloc(sizeof(BrowserData));
  memset(data, 0, sizeof(BrowserData));

  data->BrowserPath = strdup(arg_str);
  data->BrowserFolderName = MOD_VIS_NAME;
  data->have_backing = False;
  data->win_w  = 0;
  data->win_h  = 0;
  data->pixbuf = mb_pixbuf_new(mbdesktop_xdisplay(mb), 
			       mbdesktop_xscreen(mb));

  folder = mbdesktop_module_folder_create (mb, MOD_VIS_NAME, "mbpicturefolder.png");

  mbdesktop_item_set_user_data (mb, folder, (void *)data);
  mbdesktop_item_set_extended_name(mb, folder, MOD_VIS_NAME);

  mbdesktop_items_append_to_top_level (mb, folder);
  
  mbdesktop_item_set_activate_callback (mb, folder, imgbrowser_open_cb);

  data->atoms[WINDOW_STATE]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_STATE",False);
  data->atoms[WINDOW_STATE_FULLSCREEN]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_STATE_FULLSCREEN",False);
  data->atoms[WINDOW_STATE_MODAL]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_STATE_MODAL",False);
  data->atoms[WINDOW_TYPE_DIALOG]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_WINDOW_TYPE_DIALOG",False);
  data->atoms[WINDOW_TYPE]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_WINDOW_TYPE",False);
  data->atoms[_NET_WM_NAME]
    = XInternAtom(mbdesktop_xdisplay(mb), "_NET_WM_NAME",False);
  data->atoms[UTF8_STRING]
    = XInternAtom(mbdesktop_xdisplay(mb), "UTF8_STRING",False);


  return 1;
}

void 				/* Called on a folder open */
imgbrowser_open_cb (void *data1, void *data2)
{
  DIR *dp;
  BrowserData   *mod_data;
  struct dirent **namelist;
  char orig_wd[512] = { 0 };
  regex_t re;
  int n, i = 0;
  char *current_path = NULL;
  char *current_path_stripped = NULL;

  MBDesktopItem *subfolder = NULL;
  MBDesktopItem *item_new  = NULL; 

  MBDesktop     *mb          = (MBDesktop *)data1; 
  MBDesktopItem *item_folder = (MBDesktopItem *)data2; 

  Bool is_subfolder   = False;
  Bool showing_dialog = False;

  mod_data = (BrowserData*)mbdesktop_item_get_user_data (mb, item_folder);

  if (!strcmp(mbdesktop_item_get_name(mb, item_folder), MOD_VIS_NAME))
    {				/* Is top level */
      current_path = strdup(mod_data->BrowserPath);
      current_path_stripped = strdup("");
    } 
  else
    {				/* Is sub folder  */
      /* Figure out the 'real' directory path from name etc */

      char *base_path = mbdesktop_item_get_extended_name(mb, item_folder) 
	+ (strlen(MOD_VIS_NAME) +1) ;

      int len = strlen(mod_data->BrowserPath) 
	+ strlen(mbdesktop_item_get_extended_name(mb, item_folder))
	+ strlen(MOD_VIS_NAME);
     
      current_path = malloc(sizeof(char)*len);
      current_path_stripped = malloc(sizeof(char)*(strlen(base_path)+3));

      snprintf(current_path, len, "%s/%s", mod_data->BrowserPath, base_path);
      sprintf(current_path_stripped, "%s/", base_path);
      
      is_subfolder = True;
    }

  if (mbdesktop_item_folder_has_contents(mb, item_folder))
      mbdesktop_item_folder_contents_free(mb, item_folder); 

  if (regcomp(&re, ".*(jpg|jpeg|png|xpm)$", 
	      REG_EXTENDED|REG_ICASE|REG_NOSUB) != 0) 
    {
      fprintf(stderr, "mbdesktop-filebrowser: failed to compile regexp\n");
      return;
    }

  if ((dp = opendir(current_path)) == NULL)
    {
      fprintf(stderr, "mbdesktop-filebrowser: failed to open %s\n", 
	      mod_data->BrowserPath);
      return;
    }
  
  if (getcwd(orig_wd, 255) == (char *)NULL)
    {
      fprintf(stderr, "mbdesktop-filebrowser: cant get current directory\n");
      return;
    }



  chdir(current_path);

  printf("ok starting loop\n");

  n = scandir(".", &namelist, 0, alphasort);
  while (i < n && n > 0)
    {
      struct stat stat_buf;

      if (namelist[i]->d_name[0] ==  '.')
	goto end;

      printf("loop\n");

      /* Check for directory */
      if (stat(namelist[i]->d_name, &stat_buf) == 0 
	  && S_ISDIR(stat_buf.st_mode))
	{
	  char *subfolderlongname = NULL;


	  int l = strlen(MOD_VIS_NAME) 
	    + strlen(current_path) + strlen(namelist[i]->d_name);
	  
	  subfolderlongname = malloc(sizeof(char)*l);

	  sprintf(subfolderlongname, MOD_VIS_NAME "/%s%s", 
		  current_path_stripped, namelist[i]->d_name);

	  subfolder = mbdesktop_module_folder_create (mb,
						      namelist[i]->d_name,
						      "mbpicturefolder.png");

	  mbdesktop_item_set_extended_name (mb, subfolder, subfolderlongname);
	  mbdesktop_item_set_user_data (mb, subfolder, (void *)mod_data);

	  mbdesktop_items_append_to_folder (mb, item_folder, subfolder);

	  mbdesktop_item_set_activate_callback (mb, subfolder, 
						imgbrowser_open_cb);

	  free(subfolderlongname);
	  goto end;
	}

      if (regexec(&re, namelist[i]->d_name, 0, NULL, REG_NOTBOL) == 0)
	{
	  struct stat stat_info;
	  char *full_path = NULL;
	  MBPixbufImage *img = NULL, *img_backing = NULL;
	  int img_w, img_h;

	  full_path = malloc(sizeof(char)*(strlen(current_path)+strlen(namelist[i]->d_name)+2));
	  sprintf(full_path, "%s/%s", current_path, namelist[i]->d_name); 

	  if (stat(full_path, &stat_info) == -1)
	    {
	      free(full_path); goto end;
	    }

	  item_new = mbdesktop_item_from_cache (mb, full_path, 
						stat_info.st_mtime);
	  
	  if (item_new != NULL)
	    {
	      printf("Item from cache success - %s\n", item_new->name);

	    } else {

	      /* Only show progress dialog if we hit non cached items */
	      if (!showing_dialog) 
		{
		  mbdesktop_progress_dialog_open(mb); 
		  showing_dialog = True;
		}

	      img = mb_pixbuf_img_new_from_file(mod_data->pixbuf, full_path);
	      if (img == NULL)
		{
		  fprintf(stderr, "imgbrowser: Unable to load '%s'. No support for image format ?\n", full_path);
		  free(full_path);
		  goto end;
		}

	      item_new = mbdesktop_item_new_with_params( mb,
							 namelist[i]->d_name,
							 NULL,
							 NULL,
							 ITEM_TYPE_MODULE_ITEM
							 );
	      img_w = mb_pixbuf_img_get_width(img);   
	      img_h = mb_pixbuf_img_get_height(img);   
	      
	      /* Scaled image if needed  */
	      if (img_w > mbdesktop_icon_size(mb) 
		  || img_h > mbdesktop_icon_size(mb))
		{
		  int scaled_w, scaled_h;
		  MBPixbufImage *img_scaled = NULL;
		  
		  if ( img_w > img_h )
		    {
		      scaled_w = mbdesktop_icon_size(mb);
		      scaled_h = ( mbdesktop_icon_size(mb) * img_h )/ img_w;
		    }
		  else
		    {
		      scaled_h = mbdesktop_icon_size(mb);
		      scaled_w = ( mbdesktop_icon_size(mb) * img_w )/ img_h;
		    }
		  
		  img_scaled = mb_pixbuf_img_scale(mod_data->pixbuf, img, 
						   scaled_w, scaled_h);
		  
		  mb_pixbuf_img_free(mod_data->pixbuf, img);
		  img   = img_scaled;
		  img_w = scaled_w;
		  img_h = scaled_h;
		  
		}
	      
	      img_backing = mb_pixbuf_img_rgba_new(mod_data->pixbuf, 
						   mbdesktop_icon_size(mb),
						   mbdesktop_icon_size(mb));
	      
	      mb_pixbuf_img_copy_composite (mod_data->pixbuf, img_backing, img,
					    0, 0, img_w, img_h,
					    ( mbdesktop_icon_size(mb) - img_w)/2,
					    ( mbdesktop_icon_size(mb) - img_h)/2);
	      
	      mbdesktop_item_set_icon_data (mb, item_new, img_backing);

	      mb_pixbuf_img_free(mod_data->pixbuf, img);
	      mb_pixbuf_img_free(mod_data->pixbuf, img_backing);

	      mbdesktop_item_set_extended_name (mb, item_new, current_path);
	      
	      mbdesktop_item_cache(mb, item_new, full_path);	      
	    }
	  
	  mbdesktop_item_set_user_data (mb, item_new, (void *)mod_data);
      
	  mbdesktop_item_set_activate_callback (mb, item_new, 
						imgbrowser_file_activate_cb); 
	  
	  mbdesktop_items_append_to_folder( mb, item_folder, item_new );

	  free(full_path);
	}


    end:
      free(namelist[i]);
      if (showing_dialog)
	mbdesktop_progress_dialog_set_percentage (mb, (i * 100) / n ) ;
      ++i;
    }

  regfree(&re);

  free(namelist);

  closedir(dp);

  chdir(orig_wd);

  free(current_path);
  free(current_path_stripped);

  if (showing_dialog)
    mbdesktop_progress_dialog_close(mb); 

  mbdesktop_item_folder_activate_cb(data1, data2);
}


#define MODULE_NAME         "Simple Image Browser"
#define MODULE_DESC         "Simple Image Browser"
#define MODULE_AUTHOR       "Matthew Allum"
#define MODULE_MAJOR_VER    0
#define MODULE_MINOR_VER    0
#define MODULE_MICRO_VER    2
#define MODULE_API_VERSION  0
 
MBDesktopModuleInfo imgbrowser_info = 
  {
    MODULE_NAME         ,
    MODULE_DESC         ,
    MODULE_AUTHOR       ,
    MODULE_MAJOR_VER    ,
    MODULE_MINOR_VER    ,
    MODULE_MICRO_VER    ,
    MODULE_API_VERSION
  };

MBDesktopFolderModule folder_module =
  {
    &imgbrowser_info,
    imgbrowser_init,
    NULL,
    NULL
  };
