#!/bin/sh
#set -x

argv=$*

bindir=`echo $0 | sed 's![^/]*$!!'`
old=`pwd`
cd $bindir
bindir=`pwd`
cd $old || fatal "Could not return to orignal dir >$old<"

. $bindir/reflash-utils.sh
. $bindir/reflash-defs.h

PATH="$bindir:$PATH"

just_show="-P-"

progname=`basename $0`
bindir=`dirname $0`
exe=`getexe`
wrdiropt=
f_opt=
y_opt=
wrdir=

b_off=bootldr.base
b_len=bootldr.size
b_args="-O0 -o \$b_off -u \$b_len -e \$b_len -i \$bfile -wl"
b_prompt_rz='-m "" -m "Please send bootdlr file" -m ""'
b_prompt=

p_off=params.base
p_len=params.size
p_args="-o \$p_off -e \$p_len -i \$pfile -wl"
p_prompt_rz='-m "" -m "Please send params/parts file" -m ""'
p_prompt=

k_off=kernel.base
k_len=kernel.size
k_args="-o \$k_off -e \$k_len -i \$kfile -wl"
k_prompt_rz='-m "" -m "Please send kernel file" -m ""'
k_prompt=

j_off=0x140000
j_len="\$REST_OF_FLASH"
j_args="-o \$j_off -e \$j_len -i \$jfile -wl"
j_prompt_rz='-m "" -m "Please send jffs file" -m ""'
j_prompt=

set_defaults()
{
    b_off_def=$1
    b_len_def=$2
    p_off_def=$3
    p_len_def=$4
    k_off_def=$5
    k_len_def=$6
}

#
# set up defaults using one call to save time
#
defaults=`get_partition_value $exe $M_opt $b_off $b_len $p_off $p_len $k_off $k_len`
set_defaults $defaults

usage_msg=`cat <<EOF
$progname [-b <file>] [-k <file>] [-j <file>] [-other-options]

Upgrade the specified components.
You must separate the option character from its argument with a space.

The options are:

Option        Region       reflash commands
--------- ---------------  ----------------
-b <file> --> bootldr -->  $b_args
-p <file> --> params  -->  $p_args
-k <file> --> kernel  -->  $k_args
-j <file> --> jff2 fs -->  $j_args

<file> is the name of the file you wish to program into
the indicated region.  If file == "@rz" then the file
will be downloaded with zmodem.  Use your terminal
program's zmodem command to transfer the file.

Other options:
-bo --> change bootldr offset
-bl --> change bootldr len
-po --> change params offset
-pl --> change params len
-ko --> change kernel offset
-kl --> change kernel len
-jo --> change jffs2 fs offset
-jl --> change jffs2 fs len
-y  --> autoanswer yes to all prompts
-R  --> do not reboot after successful reflashing
-K  --> do not kill processes
-M  --> set file name for params data (e.g. /dev/mtdblockXX)
-B  --> set len of params data. Use 0 to prevent reading of params

The lengths and offsets can be specified as values from the existing
bootdlr partition table or as numbers.
Values from the ptable are given as the name of the partition 
followed by .base for base addr and .size for size.
E.g. params.size is the size of the bootldr partition named params
Number can be given as <number>[suffix] where suffix is:
K --> * 1024
M --> * 1048576
s --> * sector size (0x40000 by default).

Defaults:
(symbolic names refer to partition table entries.  The values shown
are current partition table entries or built-in defaults)
b_off == $b_off($b_off_def); b_len == $b_len($b_len_def)
p_off == $p_off($p_off_def); p_len == $p_len($p_len_def)
k_off == $k_off($k_off_def); k_len == $k_len($k_len_def)
j_off == $j_off; j_len == $j_len

Note: rz is smart enough to understand 
both Y and Z modem protocols.

EOF
`

[ "$#" = "0" ] && usage
reflash_args="-E0"
b_add=
p_add=
k_add=
j_add=

b_done=
p_done=
k_done=
j_done=
meg32=
j_len_set=
y_opt=
rz_opt=
R_opt=

# check for any rz files in argses...
if echo "$*" | egrep '@rz' >/dev/null
then
    rz_opt="-rz"
    . $bindir/wrdir-def.h
fi

# echo "args>$*<"
while [ -n "$*" ]
do
    # echo "\$1>$1<"
    case $1 in
    -b)
	[ -z "$b_done" ] || fatal "$1 option already specified."
	[ -z "$2" ] && fatal "Argument required for $1 option"
	bfile=$2
	b_add="$b_args"
	b_done=y
	case "$bfile" in
	*@rz*)
	    b_add="$b_prompt_rz $b_add"
	    ;;
	esac
	shift
	;;
    -bo|-ob)
	b_off=$2
	shift
	;;
    -bl|-lb)
	b_len=$2
	shift
	;;
    -p)
	[ -z "$p_done" ] || fatal "$1 option already specified."
	[ -z "$2" ] && fatal "Argument required for $1 option"
	pfile=$2
	p_add="$p_args"
	p_done=y
	case "$pfile" in
	*@rz*)
	    p_add="$p_prompt_rz $p_add"
	    ;;
	esac
	shift
	;;
    -po|-op)
	p_off=$2
	shift
	;;
    -pl|-lp)
	p_len=$2
	shift
	;;
    -k)
	[ -z "$k_done" ] || fatal "$1 option already specified."
	[ "$2" = "" ] && fatal "Argument required for $1 option"
	kfile=$2
	k_add="$k_args"
	k_done=y
	case "$kfile" in
	*@rz*)
	    k_add="$k_prompt_rz $k_add"
	    ;;
	esac
	shift
	;;
    -ko|-ok)
	k_off=$2
	shift
	;;
    -kl|-lk)
	k_len=$2
	shift
	;;
    -j)
	[ -z "$j_done" ] || fatal "$1 option already specified."
    	[ "$2" = "" ] && fatal "Argument required for $1 option"
	jfile=$2
	j_add="$j_args"
	j_done=y
	case "$jfile" in
	*@rz*)
	    j_add="$j_prompt_rz $j_add"
	    ;;
	esac
	shift
	;;
    -jo|-oj)
	j_off=$2
	shift
	;;
    -jl|-lj)
	j_len=$2
	j_len_set=y
	shift
	;;
    -n)
	just_show="-x"
	;;

    -y)
	y_opt='-y'
	;;

    -doit)
	just_show="-P-"
	;;
    -wrdir)
	wrdir=$2
	wrdiropt="-wrdir $2"
	f_opt="-F $2"
	shift
	;;
    -h|-help|--help)
	usage
	;;
    -R)
	R_opt='-R'
	;;
    -K)
	no_kill_I="-k"
	;;
    -B)
	B_opt="-B $2"		# params sector size. 0 --> do not read
	shift;;
    -M)
	M_opt="-M $2"		# params file (e.g. /dev/mtdblockXX
	shift;;
    -kmod)
	kmod_opt="-kmod $2"
	shift
	;;
    *)
	fatal "Unrecognized argument >$1<"
	;;
    esac
    shift
done

[ -n "$wrdir" ] && PATH="$wrdir:$PATH"

reflash_args="$reflash_args $y_opt $p_add $k_add $j_add $b_add $B_opt $M_opt"
#echo "reflash_args, before>$reflash_args<"

#
# get the values of our vars
#
eval j_len=\"$j_len\"
eval 'reflash_args="$reflash_args"'
#echo "reflash_args>$reflash_args<"

arch=`uname -m`
case $arch in
    arm*)
	prep $rz_opt $wrdiropt $no_kill_I $kmod_opt
	;;
esac
eval echo exec $exe $f_opt $R_opt $reflash_args -q $just_show
eval exec $exe $f_opt $R_opt $reflash_args -q $just_show
