/* rec_file.c -- functions to deal with loading a recognition file

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"

#define MAX_LINE_LENGTH 512

static char *strip_whitespace(char *str);

/* Return a pointer the the given string with initial and trailing
 * whitespace removed. Warning: This function modifies the original
 * string. */
static char *strip_whitespace(char *str) {
  if (str && *str) {
    int n;
    while (*str && isspace (*str)) str++;

    n = strlen (str) - 1;
    while (n > 0 && isspace (str [n])) n--;
    str [n+1] = '\0';
  }
  return str;
}

struct rec_node *create_rec_tree_from_file(char *filename) {
  FILE *file;
  int line_num = 0;
  int err;
  char buf[MAX_LINE_LENGTH];
  char *key, *seq;
  struct rec_node *root;

  file = fopen(filename, "r");
  if (!file) {
    /*
    fprintf(stderr, "%s:%d: create_rec_tree_from_file: failed to open file %s\n", __FILE__, __LINE__, filename);
    perror("create_rec_tree_from_file");
    */
    return NULL;
  }

  root = new_rec_tree();
  while (fgets(buf, MAX_LINE_LENGTH, file)) {
    char *p;
    line_num++;

    p = buf;
    while (isspace(p[0]))
      p++;
    if (p[0] == '\0' || p[0] == '#')
      goto NEXT_LINE;

    key = strtok(buf, "=");
    seq = strtok(NULL, "\n");

    key = strip_whitespace(key);
    seq = strip_whitespace(seq);

    if (strspn(seq,"123456789?[]*") != strlen(seq)) {
      fprintf(stderr,"%s:%d: Unrecognized character %c in digit sequence %s\n", filename, line_num, seq[strspn(seq,"123456789?")], seq);
      goto NEXT_LINE;
    }

    /* TODO: It probably makes sense to do string -> keysym -> keycode
       mapping now, up front and simply add the keycode to the
       database rather than the whole string. */
    err = add_sequence(root, seq, key, strlen(key) + 1);
    if (err) {
      fprintf(stderr, "%s:%d: Error adding sequence %s to recognition tree\n", filename, line_num, seq);
    }

    NEXT_LINE:
  }

  return root;
}

