/* rec_baseline.c -- correct stroke coordinates according to inferred baseline

   Copyright 2001 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "rec_baseline.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

static void compute_correction_matrix(rec_baseline_t *baseline);

int rec_baseline_init(rec_baseline_t *baseline)
{
    int i;

    baseline->orientation = 0.0;

    for (i=0; i < REC_BASELINE_HISTORY_MAX; i++) {
	baseline->history[i] = 0.0;
    }

    compute_correction_matrix(baseline);

    return 0;
}

void rec_baseline_deinit(rec_baseline_t *baseline)
{
    /* nothing to see here, move along... */
}

void rec_baseline_correct(rec_baseline_t *baseline, int *x, int *y)
{
    int new_x, new_y;

    new_x = baseline->correction_matrix[0][0] * *x 
	  + baseline->correction_matrix[0][1] * *y;
    new_y = baseline->correction_matrix[1][0] * *x 
	  + baseline->correction_matrix[1][1] * *y;

    *x = fixed_to_i_round(new_x, REC_BASELINE_PRECISION_BITS);
    *y = fixed_to_i_round(new_y, REC_BASELINE_PRECISION_BITS);
}

void rec_baseline_nudge(rec_baseline_t *baseline, double correction)
{
    int i;

    fprintf(stderr, "%s: baseline %5.2f nudged by %5.2f to",
	    __FUNCTION__,
	    (180 / M_PI) * baseline->orientation,
	    (180 / M_PI) * correction);
    for (i=0; i < REC_BASELINE_HISTORY_MAX - 1; i++) {
	baseline->history[i] = baseline->history[i+1];
    }
    baseline->history[i] = baseline->orientation + correction;

    baseline->orientation = 0.0;
    for (i=0; i < REC_BASELINE_HISTORY_MAX; i++) {
	baseline->orientation += baseline->history[i];
    }
    baseline->orientation /= (double) REC_BASELINE_HISTORY_MAX;

    fprintf(stderr, "%5.2f\n",
	    (180 / M_PI) * baseline->orientation);
    compute_correction_matrix(baseline);
}

static void compute_correction_matrix(rec_baseline_t *baseline)
{
    /* correct orientation by rotating the opposite direction */
    double theta = -baseline->orientation;
    int max_val = (1L << REC_BASELINE_PRECISION_BITS) - 1;

    int cos_theta = max_val * cos(theta);
    int sin_theta = max_val * sin(theta);

    baseline->correction_matrix[0][0] = cos_theta;
    baseline->correction_matrix[0][1] = sin_theta;

    baseline->correction_matrix[1][0] = -sin_theta;
    baseline->correction_matrix[1][1] = cos_theta;
}
