/* 
   xstroke - X based gesture recognition program based on libstroke.

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>

#include "stroke.h"
#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"

char *progname;

/* This parameter only affects the smoothness of the displayed stroke,
 * not the quality of the recognition. */
#define MAX_STROKE_POINTS 100
/* When MAX_STOKE_POINTS is exceeded, we throw out every Nth
 * point. This is N. */
#define STROKE_DECIMATION 4

/* Default values. */
/* How wide should the line be that is used to draw the stroke. */
#define DEFAULT_STROKE_WIDTH 3
#define DEFAULT_STROKE_FILE ".xstrokerc"
#define DEFAULT_STROKE_DIR  "."
#define DEFAULT_FONT        "fixed"
#define DEFAULT_NAME        "xstroke"
#define DEFAULT_CLASS       "Xstroke"
#define DEFAULT_GEOMETRY    "100x15"
#define DEFAULT_FG_COLOR    "black"
#define DEFAULT_BG_COLOR    "white"

/* State values for the recognizer */
#define NULL_STATE         0
#define RECOGNITION_STATE  1
#define WINDOW_CLICK_STATE 2

struct stroke_lines {
  XPoint pts[MAX_STROKE_POINTS];
  int num_pts;
  int last_decimated;
};

static XrmOptionDescRec op_table[] = {
  {"-foreground", "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-background", "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-fg",         "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-bg",         "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-display",    ".display",     XrmoptionSepArg, (XPointer) NULL},
  {"-strokeWidth",".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-width",      ".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-fn",         "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-font",       "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-geometry",   "*geometry",    XrmoptionSepArg, (XPointer) NULL},  
  {"-f",          ".strokeFile",  XrmoptionSepArg, (XPointer) NULL},
  {"-strokeFile", ".strokeFile",  XrmoptionSepArg, (XPointer) NULL}
};

/* Global data (could probably use some cleanup -- maybe organize it
   into appropriate structs) */
/* General X goop */
static Display *display;
static Window root_window;
static Pixmap pixmap;
static int pixmap_width;
static int pixmap_height;
static Window window;
static GC window_gc;
static GC root_gc;
static XFontStruct *font;
static XrmDatabase db;
static int window_x;
static int window_y;
static int window_width;
static int window_height;

static KeyCode shift_keycode;
static XSizeHints wm_size_hints;

/* One pointer for libstroke */
static struct rec_node *rec_tree_root;

/* Data for drawing */
static struct stroke_lines lines;
static char sequence[MAX_SEQUENCE+1];
#define MAX_STATUS_LEN 2*MAX_SEQUENCE
static XTextItem status_text;

/* State information for the program's main logic */
static int state = NULL_STATE;
static int prev_state = NULL_STATE;

/* Prototypes for private functions */
static void initialize(int *argc, char *argv[]);
static void initializeX(int *argc, char *argv[]);
static void initialize_keymap(void);
static void initialize_stroke_rec(int *argc, char *argv[]);
static void main_event_loop(void);
static void cleanup_and_exit(int exit_code);
static void decimate_stroke(struct stroke_lines *stroke, int mod);
static void erase_stroke(struct stroke_lines *stroke);
static void draw_stroke(struct stroke_lines *stroke);

static int  find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret);
static void send_key_press_release(char *string);
static void change_state_to(int new_state);
static void toggle_recognition_state(void);
static int event_in_window(XButtonEvent *xev);

static void client_message_handler(XEvent *xev);
static void expose_event_handler(XEvent *xev);
static void button_press_handler(XEvent *xev);
static void button_release_handler(XEvent *xev);
static void motion_notify_handler(XEvent *xev);
static void configure_notify_handler(XEvent *xev);
static void key_press_handler(XEvent *xev);

int main(int argc, char *argv[])
{
  progname = argv[0];
  initialize(&argc, argv);

  /*
    print_rec_tree(rec_tree_root);
  */
  /*
    print_node_stats();
  */

  main_event_loop();

  /* This never happens... */
  return 1;
}

static void main_event_loop() {
  XSelectInput(display, window, 
	       ExposureMask
	       | ButtonPressMask
	       | ButtonReleaseMask
	       | ButtonMotionMask
	       | KeyPressMask
               | StructureNotifyMask
	       );
  while (1)  {
    XEvent xev;

    XNextEvent(display, &xev);
    switch  (xev.type) {
    case ClientMessage:
      client_message_handler(&xev);
      break;
    case Expose:   
      expose_event_handler(&xev);
      break;
    case ButtonPress:
      button_press_handler(&xev);
      break;
    case ButtonRelease:
      button_release_handler(&xev);
      break;
    case MotionNotify:
      motion_notify_handler(&xev);
      break;
    case KeyPress:
      key_press_handler(&xev);
      break;
    case ConfigureNotify:
      configure_notify_handler(&xev);
      break;
    }
  }
}

static void initialize(int *argc, char *argv[]) {
  initializeX(argc, argv);
  initialize_stroke_rec(argc, argv);
  initialize_keymap();

  lines.num_pts = 0;
  lines.last_decimated = 0;
}

/* Ick! This function is getting rather hairy.
   Maybe Xt will come to the rescue soon! :) 
*/
static void initializeX(int *argc, char *argv[]) {
  int screen;
  char *displayname = NULL;
  char *fontname = DEFAULT_FONT;
  char *fg_name = DEFAULT_FG_COLOR;
  char *bg_name = DEFAULT_BG_COLOR;
  int stroke_width = DEFAULT_STROKE_WIDTH;
  Colormap colormap;
  XColor fg_color;
  XColor bg_color;
  XrmDatabase cmd_db = NULL;
  char *res_type;
  XrmValue res_value;
  int x = 0, y = 0, width = 100, height = 15, gravity;
  unsigned long gcm;
  XGCValues gcv;

  XrmInitialize();
  XrmParseCommand(&cmd_db,
		  op_table, sizeof(op_table) / sizeof(op_table[0]),
		  DEFAULT_NAME,
		  argc,
		  argv);
  if (XrmGetResource(cmd_db, "xstroke.display", "Xstroke.Display",
		     &res_type, &res_value) && (strcmp(res_type,"String") ==0)) {
    displayname = res_value.addr;
  }

  display = XOpenDisplay(displayname);
  if (display) {
    Atom wm_protocols[] = { XInternAtom(display, "WM_DELETE_WINDOW", False)};
    XClassHint class_hint;
    XWMHints wm_hints;

    screen = DefaultScreen(display);

    /* TODO: Figure out the right way to merge default resources with
       those on the command line. I've looked over the Xlib
       documentation and I'm guessing that this code is correct, but
       XrmGetDatabase is always returning NULL for some reason. */
    db = XrmGetDatabase(display);
    XrmMergeDatabases(cmd_db, &db);
    XrmGetResource(db, "xstroke.geometry", "Xstroke.Geometry",
		   &res_type, &res_value);
    /* XWMGeometry(display, screen, NULL, DEFAULT_GEOMETRY, 0, &wm_size_hints, &x, &y, &width, &height, &gravity);*/
    wm_size_hints.flags |= PWinGravity;
    wm_size_hints.win_gravity = gravity;

    colormap = DefaultColormap(display, screen);
    if (XrmGetResource(db, "xstroke.foreground", "Xstroke.Foreground",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, fg_name, &fg_color)) {
      fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to %s.\n", progname, fg_name, DEFAULT_FG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_FG_COLOR, &fg_color)) {
	fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to BlackPixel.\n", progname, DEFAULT_FG_COLOR);
	fg_color.pixel = BlackPixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &fg_color);

    if (XrmGetResource(db, "xstroke.background", "Xstroke.Background",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      bg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, bg_name, &bg_color)) {
      fprintf(stderr,"%s: Failed to parse background color %s. Falling back to %s.\n", progname, bg_name, DEFAULT_BG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_BG_COLOR, &bg_color)) {
	fprintf(stderr,"%s: Failed to parse background color %s. Falling back to WhitePixel.\n", progname, DEFAULT_BG_COLOR);
	bg_color.pixel = WhitePixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &bg_color);

    if (XrmGetResource(db, "xstroke.strokeWidth", "Xstroke.StrokeWidth",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      stroke_width = atoi(res_value.addr);
    }
    if (stroke_width <= 0) {
      fprintf(stderr, "%s: Invalid stroke_width %s. Using %d\n", progname, res_value.addr, DEFAULT_STROKE_WIDTH);
      stroke_width = DEFAULT_STROKE_WIDTH;
    }

    root_window = DefaultRootWindow(display);
    pixmap_width = DisplayWidth(display, screen);
    pixmap_height = DisplayHeight(display, screen);
    pixmap = XCreatePixmap(display, root_window,
                           pixmap_width,
                           pixmap_height,
                           DefaultDepth(display, screen));
                           
    window = XCreateSimpleWindow(display, root_window,
				 x, y, width, height,
				 1,
				 fg_color.pixel, bg_color.pixel);
    window_width = width;
    window_height = height;

    gcm = 0;
    gcm |= GCSubwindowMode; gcv.subwindow_mode = IncludeInferiors;
    gcm |= GCFunction;      gcv.function = GXxor;
    gcm |= GCForeground;    gcv.foreground = 0xffffffff;
    gcm |= GCLineWidth;     gcv.line_width = stroke_width;
    gcm |= GCCapStyle;      gcv.cap_style = CapRound;
    gcm |= GCJoinStyle;     gcv.join_style = JoinRound;
    root_gc = XCreateGC(display, root_window, gcm, &gcv);

    gcm = 0;
    gcm |= GCSubwindowMode; gcv.subwindow_mode = IncludeInferiors;
    gcm |= GCForeground;    gcv.foreground = fg_color.pixel;
    gcm |= GCBackground;    gcv.background = bg_color.pixel;
    window_gc = XCreateGC(display, root_window, gcm, &gcv);

    if (XrmGetResource(db, "xstroke.font", "Xstroke.Font",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fontname = res_value.addr;
    }
    font = XLoadQueryFont(display, fontname);
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Falling back to %s.\n", progname, fontname, DEFAULT_FONT);
      font = XLoadQueryFont(display, DEFAULT_FONT);
    }
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Continuing without a font!\n", progname, DEFAULT_FONT);
    } else {
      XSetFont(display, window_gc, font->fid);
    }
    status_text.chars = xmalloc(MAX_STATUS_LEN);
    status_text.delta = 0;
    status_text.font = None;

    wm_hints.flags = InputHint;
    wm_hints.input = False;
    class_hint.res_name = DEFAULT_NAME;
    class_hint.res_class = DEFAULT_CLASS;
    XmbSetWMProperties(display, window, DEFAULT_NAME, DEFAULT_NAME, argv, *argc,
		       &wm_size_hints,
		       &wm_hints, &class_hint);

    XSetWMProtocols(display, window, wm_protocols, sizeof(wm_protocols) / sizeof(Atom));
    XStoreName(display, window, DEFAULT_NAME);
    XMapWindow(display, window);
  } else {
    fprintf(stderr, "%s: Could not open display %s\n", progname, displayname ? displayname : getenv("DISPLAY"));
    exit(1);
  }
}

static void initialize_stroke_rec(int *argc, char *argv[]) {
  char *stroke_file = DEFAULT_STROKE_FILE;
  char *stroke_dir = getenv("HOME");
  char *filename;
  char *res_type;
  XrmValue res_value;

  stroke_init();

  if (stroke_dir == NULL)
    stroke_dir = DEFAULT_STROKE_DIR;

  if (XrmGetResource(db, "xstroke.strokeFile", "Xstroke.StrokeFile",
		     &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
    filename = res_value.addr;
    rec_tree_root = create_rec_tree_from_file(filename);
  } else {
    filename = xmalloc(strlen(stroke_dir) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", stroke_dir, stroke_file);
    rec_tree_root = create_rec_tree_from_file(filename);
    free(filename);
  }

  if (rec_tree_root == NULL && strcmp(stroke_dir, DEFAULT_STROKE_DIR)) {
    /* Try again with DEFAULT_STROKE_DIR */
    filename = xmalloc(strlen(DEFAULT_STROKE_DIR) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", DEFAULT_STROKE_DIR, stroke_file);
    rec_tree_root = create_rec_tree_from_file(filename);
    free(filename);
  }

  if (rec_tree_root == NULL) {
    fprintf(stderr, "%s: Could not load stroke definition file.\n", progname);
    fprintf(stderr, "%s: Tried %s/%s", progname, stroke_dir, stroke_file);
    if (strcmp(stroke_dir,DEFAULT_STROKE_DIR))
      fprintf(stderr, " and %s/%s", DEFAULT_STROKE_DIR, stroke_file);
    fprintf(stderr, "\n%s: Exiting.\n\n", progname);
    exit(1);
  }
}

/* Is the given event inside the main window */
static int event_in_window(XButtonEvent *xev) {
  if (state == RECOGNITION_STATE)
    return ((xev->x > 0)
	    && (xev->y > 0)
	    && (xev->x < window_width)
	    && (xev->y < window_height));
  else
    return ((xev->x > window_x)
	    && (xev->y > window_y)
	    && (xev->x < window_x + window_width)
	    && (xev->y < window_y + window_height));
}

static void initialize_keymap(void) {
  find_key(display, XK_Shift_L, &shift_keycode, 0);
}

static void redraw() {
  XClearWindow(display, window);
  if (font)
    XDrawText(display, window, window_gc, 1, 1 + font->ascent, &status_text, 1);
}

static void expose_event_handler(XEvent *xev) {
  if (font)
    XDrawText(display, window, window_gc, 1, 1 + font->ascent, &status_text, 1);
  /*  XDrawLines(display, window, gc, lines.pts, lines.num_pts, CoordModeOrigin); */
}

static void client_message_handler(XEvent *xev) {
  XClientMessageEvent *cmev = &xev->xclient;

  if (cmev->data.l[0] == XInternAtom(display, "WM_DELETE_WINDOW", False)) {
    cleanup_and_exit(0);
  }
}

static void button_press_handler(XEvent *xev) {
  XButtonEvent *bev = &xev->xbutton;
  int len;

  if (event_in_window(bev)) {
    change_state_to(WINDOW_CLICK_STATE);
  } else {
    XGrabServer(display);
    XCopyArea(display, root_window, pixmap, window_gc,
              0, 0, pixmap_width, pixmap_height,
              0, 0);
  }
  /* Simply throw away any incomplete stroke sequence. (This is for
   * the case when the user left the window with the button pressed
   * and didn't reenter before the button release) */
  stroke_trans(sequence);

  lines.num_pts = 0;
  lines.last_decimated = 0;
  if (state == RECOGNITION_STATE
      || state == (WINDOW_CLICK_STATE && prev_state == RECOGNITION_STATE)) {
    len = snprintf(status_text.chars, MAX_STATUS_LEN, "RECOGNIZING");
    status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
  } else if (state == NULL_STATE
	     || (state == WINDOW_CLICK_STATE && prev_state == NULL_STATE)) {
    len = snprintf(status_text.chars, MAX_STATUS_LEN, "DOING_NOTHING");
    status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
  }
  redraw();
}

static int find_key(Display *display, KeySym keysym, KeyCode *code_ret, int *col_ret)
{
  int col;
  int keycode;
  KeySym k;
  int min_keycode, max_keycode;

  XDisplayKeycodes (display, &min_keycode, &max_keycode);

  for (keycode = min_keycode; keycode <= max_keycode; keycode++) {
    for (col = 0; (k = XKeycodeToKeysym (display, keycode, col)) != NoSymbol; col++)
      if (k == keysym) {
	*code_ret = keycode;
	if (col_ret)
	  *col_ret = col;
	return 1;
      }
  }
  return 0;
}

static void send_key_press_release(char *string) {
  KeySym keysym;
  KeyCode keycode;
  int col;

  keysym = XStringToKeysym(string);
  if (keysym != NoSymbol) {
    if (find_key(display, keysym, &keycode, &col)) {
      if (col & 1)
	XTestFakeKeyEvent (display, shift_keycode, True, 0);
      XTestFakeKeyEvent (display, keycode, True, 0);
      XTestFakeKeyEvent (display, keycode, False, 0);
      if (col & 1)
	XTestFakeKeyEvent (display, shift_keycode, False, 0);
    } else {
      fprintf(stderr, "%s: No key code found for keysym %ld, (from string %s)\n", progname, keysym, string);
    }
  } else {
    fprintf(stderr, "%s: No Keysym found for %s\n", progname, string);
  }
}

static void change_state_to(int new_state) {
  prev_state = state;
  state = new_state;
}

static void toggle_recognition_state() {
  int err, len;

  if (state == RECOGNITION_STATE) {
    XUngrabPointer(display, CurrentTime);
    len = snprintf(status_text.chars, MAX_STATUS_LEN, "DOING_NOTHING");
    status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
    redraw();

    change_state_to(NULL_STATE);
  } else {
    err = XGrabPointer(display, root_window, False,
		       ButtonPressMask | ButtonMotionMask | ButtonReleaseMask,
		       GrabModeAsync, GrabModeAsync, /* Async pointer and keys */
		       None, /* Don't confine to a window */
		       None, /* TODO: Come up with a good cursor here */
		       CurrentTime);
    if (err == GrabSuccess) {
      len = snprintf(status_text.chars, MAX_STATUS_LEN, "RECOGNIZING");
      status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
      redraw();

      change_state_to(RECOGNITION_STATE);
    } else {
      fprintf(stderr, "%s: XGrabPointer failed. :(\n", progname);
    }
  }
}

static void button_release_handler(XEvent *xev) {
  int len;
  /* We'll have to deal with these in one way or another: Button1Mask,
     Button2Mask, Button3Mask, Button4Mask, Button5Mask, ShiftMask,
     LockMask, ControlMask, Mod1Mask, Mod2Mask, Mod3Mask, Mod4Mask,
     Mod5Mask 
     
     Actually, maybe not. The XTestFakeKeyEvent function is nice
     enough to pass the synthetic event through the regular keyboard
     channels so that modifiers are automatically applied.
  */
  XButtonEvent *bev = &xev->xbutton;

  if (window_click && event_in_window(bev)) {
    toggle_recognition_state();
  } else if (! window_click){
    erase_stroke(&lines);
    XUngrabServer(display);
    if (stroke_trans(sequence)) {
      char *string;
      string = lookup_sequence(rec_tree_root, sequence);
      if (string) {
	send_key_press_release(string);
	len = snprintf(status_text.chars, MAX_STATUS_LEN, "%s (%s)", string, sequence);
	status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
	redraw();
	/*
	  fprintf(stderr, "%s => %s\n",sequence, string);
	*/
      } else {
	len = snprintf(status_text.chars, MAX_STATUS_LEN, sequence);
	status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
	redraw();
	fprintf(stdout, "unknown=%s*\n",sequence);
      }
    } else {
      toggle_recognition_state();
      XTestFakeButtonEvent(display, 1, True, CurrentTime);
      XTestFakeButtonEvent(display, 1, False, CurrentTime);
      toggle_recognition_state();
      /*
	fprintf(stderr, "No stroke (%s)\n", sequence);
      */
    }
  }
}

/* Throw out several points to make some more room.  Originally, this
 * function threw out every modth point, but when it was called
 * repeatedly, this stripped a lot more points from the fron of the
 * stroke than the end. I've modified it to remember where it last
 * left off compressing to start from there again and restart at the
 * beginning when the starting point gets close to the end again. This
 * helps, but it's still not perfect for some reason. Oh well...  */
static void decimate_stroke(struct stroke_lines *stroke, int mod) {
  int dest, src;

  if ((MAX_STROKE_POINTS - stroke->last_decimated) < (2 * mod))
    stroke->last_decimated = 0;

  for (dest = stroke->last_decimated, src=dest;
       src < stroke->num_pts; dest++, src++) {
    if ((dest % (mod - 1)) == 0) {
      src++;
      if (src >= stroke->num_pts)
	break;
    }
    stroke->pts[dest].x = stroke->pts[src].x;
    stroke->pts[dest].y = stroke->pts[src].y;
  }
  stroke->num_pts = dest;
  stroke->last_decimated = dest - 1;
}

static void erase_stroke(struct stroke_lines *stroke) {
  XCopyArea(display, pixmap, root_window, window_gc,
            0, 0, pixmap_width, pixmap_height,
            0, 0);
  /*
  XDrawLines(display, root_window, root_gc,
             stroke->pts, stroke->num_pts, CoordModeOrigin);
  int i;
  int penultimate = stroke->num_pts - 2;

  if (stroke->last_decimated > 0) { 
    XDrawLines(display, root_window, root_gc,
	       stroke->pts, stroke->last_decimated + 1, CoordModeOrigin);
    i = stroke->last_decimated - 1;
  } else {
    if (stroke->num_pts > 1) {
      XDrawLines(display, root_window, root_gc,
		 stroke->pts, 2, CoordModeOrigin);
    }
    i = 0;
  }
  
  for (; i < penultimate; i++) {
    XDrawLines(display, root_window, root_gc,
	       stroke->pts + i, 2, CoordModeOrigin);
    XDrawLines(display, root_window, root_gc,
	       stroke->pts + i, 3, CoordModeOrigin);
  }
  */
}

static void draw_stroke(struct stroke_lines *stroke) {
  XDrawLines(display, root_window, root_gc,
	     stroke->pts, stroke->num_pts, CoordModeOrigin);
}

static void motion_notify_handler(XEvent *xev) {
  if (! window_click) {
  XMotionEvent *mev = &xev->xmotion;

  if (state == RECOGNITION_STATE) {
    stroke_record (mev->x, mev->y);

  if (lines.num_pts >= MAX_STROKE_POINTS) {
    /* erase_stroke(&lines); */
    decimate_stroke(&lines, STROKE_DECIMATION);
    /* draw_stroke(&lines); */
  }
  lines.pts[lines.num_pts].x = mev->x;
  lines.pts[lines.num_pts].y = mev->y;
  lines.num_pts++;
  if (lines.num_pts > 2) {
    XDrawLines(display, root_window, root_gc, lines.pts + lines.num_pts -3, 2, CoordModeOrigin);
    XDrawLines(display, root_window, root_gc, lines.pts + lines.num_pts -3, 3, CoordModeOrigin);
  } else {
    /* only two points, just draw it */
    XDrawLines(display, root_window, root_gc, lines.pts, lines.num_pts, CoordModeOrigin);
  }
  }
}

static void configure_notify_handler(XEvent *xev) {
  XConfigureEvent *cev = &xev->xconfigure;

  window_x = cev->x;
  window_x = cev->y;
  window_width = cev->width;
  window_height = cev->height;
}

static void key_press_handler(XEvent *xev) {
  XKeyEvent *kev = &xev->xkey;

  if (XLookupKeysym(kev, 0) == XK_q) {
    cleanup_and_exit(0);
  }
}

static void cleanup_and_exit(int exit_code) {
  if (display) {
    if (state == RECOGNITION_STATE)
      erase_stroke(&lines);
    if (font)
      XFreeFont(display, font);
    if (root_gc)
      XFreeGC(display, root_gc);
    if (window_gc)
      XFreeGC(display, window_gc);
    if (window)
      XDestroyWindow(display, window);
    XCloseDisplay(display);
  }
  free_rec_tree(rec_tree_root);
  /*
    print_node_stats();
  */
  exit(exit_code);
}

