/* 
   xstroke - X based gesture recognition program based on libstroke.

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>

#include "stroke.h"
#include "rec_node.h"
#include "rec_file.h"
#include "xmalloc.h"
#include "rec-interface.h"
#include "stroke_rec.h"

char *progname;

/* Any stroke with fewer pts than MIN_STROKE_PTS will be passed
   through as a single mouse click and not passed to the recognizer */
#define MIN_STROKE_PTS 10
/* This parameter affects only the quality of the recognition, not the
   smoothness of the displayed stroke. */
#define MAX_STROKE_PTS 256
/* When MAX_STOKE_PTS is exceeded, we throw out every Nth
 * point. This is N. */
#define STROKE_DECIMATION 4
/* How many pixels the pointer must be dragged before we start moving
 * the window around */
#define MOVE_THRESHOLD 20

/* Default values. */
/* How wide should the line be that is used to draw the stroke. */
#define DEFAULT_STROKE_WIDTH 3
#define DEFAULT_BORDER_WIDTH 1
#define DEFAULT_STROKE_FILE ".xstrokerc"
#define DEFAULT_STROKE_DIR  "."
#define DEFAULT_FONT        "5x8"
#define DEFAULT_NAME        "xstroke"
#define DEFAULT_CLASS       "Xstroke"
#define DEFAULT_GEOMETRY    "16x8+0+0"
#define DEFAULT_FG_COLOR    "black"
#define DEFAULT_BG_COLOR    "white"

/* State values for the recognizer (one-hot encoding) */
#define NULL_STATE         0
#define RECOGNITION_STATE  1
#define BUTTON_PRESSED     2
#define MOVING_WINDOW      4

struct stroke_lines {
  XPoint pts[MAX_STROKE_PTS];
  int num_pts;
  int last_decimated;
};

static XrmOptionDescRec op_table[] = {
  {"-foreground", "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-background", "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-fg",         "*foreground",  XrmoptionSepArg, (XPointer) NULL},
  {"-bg",         "*background",  XrmoptionSepArg, (XPointer) NULL},
  {"-display",    ".display",     XrmoptionSepArg, (XPointer) NULL},
  {"-strokeWidth",".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-width",      ".strokeWidth", XrmoptionSepArg, (XPointer) NULL},
  {"-fn",         "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-font",       "*font",        XrmoptionSepArg, (XPointer) NULL},
  {"-geometry",   "*geometry",    XrmoptionSepArg, (XPointer) NULL},  
  {"-f",          ".strokeFile",  XrmoptionSepArg, (XPointer) NULL},
  {"-strokeFile", ".strokeFile",  XrmoptionSepArg, (XPointer) NULL}
};

/* Global data (could probably use some cleanup -- maybe organize it
   into appropriate structs) */
/* General X goop */
static Display *display;
static Window root_window;
static Pixmap pixmap;
static int pixmap_width;
static int pixmap_height;
static Window window;
static GC window_gc;
static GC root_gc;
static XFontStruct *font;
static XrmDatabase db;
static int window_width;
static int window_height;
static int border_width;
static int window_depth;
static XSizeHints wm_size_hints;

/* One pointer for the recognizer */
RecInterface rec;

/* Data for drawing */
static struct stroke_lines stroke;
/* static char sequence[MAX_SEQUENCE+1]; */
#define MAX_STATUS_LEN 2*MAX_SEQUENCE
static XTextItem status_text;

/* State information for the program's main logic */
static int state = NULL_STATE;
static int drag_start_x;
static int drag_start_y;
static int drag_start_x_root;
static int drag_start_y_root;

/* Prototypes for private functions */
static void initialize(int *argc, char *argv[]);
static void initializeX(int *argc, char *argv[]);
static void initialize_stroke_rec(int *argc, char *argv[]);
static void main_event_loop(void);
static void cleanup_and_exit(int exit_code);
static void save_root();
static void restore_root();
static void redraw();

static void toggle_recognition_state(int permanent);
static int  event_in_window(XButtonEvent *xev);

static void client_message_handler(XEvent *xev);
static void expose_event_handler(XEvent *xev);
static void button_press_handler(XEvent *xev);
static void button_release_handler(XEvent *xev);
static void perform_stroke_recognition();
static void motion_notify_handler(XEvent *xev);
static void configure_notify_handler(XEvent *xev);
static void visibility_notify_handler(XEvent *xev);
static void key_press_handler(XEvent *xev);

int main(int argc, char *argv[])
{
  progname = argv[0];
  initialize(&argc, argv);

  /*
    print_rec_tree(rec_tree_root);
  */
  /*
    print_node_stats();
  */

  main_event_loop();

  /* This never happens... */
  return 1;
}

static void main_event_loop() {
  XSelectInput(display, window, 
	       ExposureMask
	       | ButtonPressMask
	       | ButtonReleaseMask
	       | ButtonMotionMask
	       | KeyPressMask
               | StructureNotifyMask
	       | VisibilityChangeMask
	       );
  while (1)  {
    XEvent xev;

    XNextEvent(display, &xev);
    switch  (xev.type) {
    case ClientMessage:
      client_message_handler(&xev);
      break;
    case Expose:
      expose_event_handler(&xev);
      break;
    case ButtonPress:
      button_press_handler(&xev);
      break;
    case ButtonRelease:
      button_release_handler(&xev);
      break;
    case MotionNotify:
      motion_notify_handler(&xev);
      break;
    case KeyPress:
      key_press_handler(&xev);
      break;
    case ConfigureNotify:
      configure_notify_handler(&xev);
      break;
    case VisibilityNotify:
      visibility_notify_handler(&xev);
      break;
      /*
    default:
      fprintf(stderr,"%s: unhandled event (type == %d)\n", progname, xev.type);
      break;
      */
    }
  }
}

static void initialize(int *argc, char *argv[]) {
  initializeX(argc, argv);
  
  initialize_stroke_rec(argc, argv);

  stroke.num_pts = 0;
  stroke.last_decimated = 0;
}

/* Ick! This function is getting rather hairy.
*/
static void initializeX(int *argc, char *argv[]) {
  int screen;
  char *displayname = NULL;
  char *fontname = DEFAULT_FONT;
  char *fg_name = DEFAULT_FG_COLOR;
  char *bg_name = DEFAULT_BG_COLOR;
  int stroke_width = DEFAULT_STROKE_WIDTH;
  Colormap colormap;
  XColor fg_color;
  XColor bg_color;
  XrmDatabase cmd_db = NULL;
  char *res_type;
  XrmValue res_value;
  int x, y, width, height, gravity;
  unsigned long gcm;
  XGCValues gcv;
  unsigned long winm;
  XSetWindowAttributes winv;
  int len;

  XrmInitialize();
  XrmParseCommand(&cmd_db,
		  op_table, sizeof(op_table) / sizeof(op_table[0]),
		  DEFAULT_NAME,
		  argc,
		  argv);
  if (XrmGetResource(cmd_db, "xstroke.display", "Xstroke.Display",
		     &res_type, &res_value) && (strcmp(res_type,"String") ==0)) {
    displayname = res_value.addr;
  }

  display = XOpenDisplay(displayname);
  if (display) {
    Atom wm_protocols[] = { XInternAtom(display, "WM_DELETE_WINDOW", False)};
    XClassHint class_hint;
    XWMHints wm_hints;

    screen = DefaultScreen(display);

    /* TODO: Figure out the right way to merge default resources with
       those on the command line. I've looked over the Xlib
       documentation and I'm guessing that this code is correct, but
       XrmGetDatabase is always returning NULL for some reason. */
    db = XrmGetDatabase(display);
    XrmMergeDatabases(cmd_db, &db);
    if (XrmGetResource(db, "xstroke.geometry", "Xstroke.Geometry",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      XWMGeometry(display, screen, NULL, res_value.addr, 0, &wm_size_hints, &x, &y, &width, &height, &gravity);
    } else {
      XWMGeometry(display, screen, NULL, DEFAULT_GEOMETRY, 0, &wm_size_hints, &x, &y, &width, &height, &gravity);
    }
    wm_size_hints.flags |= PWinGravity;
    wm_size_hints.win_gravity = gravity;

    colormap = DefaultColormap(display, screen);
    if (XrmGetResource(db, "xstroke.foreground", "Xstroke.Foreground",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, fg_name, &fg_color)) {
      fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to %s.\n", progname, fg_name, DEFAULT_FG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_FG_COLOR, &fg_color)) {
	fprintf(stderr,"%s: Failed to parse foreground color %s. Falling back to BlackPixel.\n", progname, DEFAULT_FG_COLOR);
	fg_color.pixel = BlackPixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &fg_color);

    if (XrmGetResource(db, "xstroke.background", "Xstroke.Background",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      bg_name = res_value.addr;
    }
    if (! XParseColor(display, colormap, bg_name, &bg_color)) {
      fprintf(stderr,"%s: Failed to parse background color %s. Falling back to %s.\n", progname, bg_name, DEFAULT_BG_COLOR);
      if (! XParseColor(display, colormap, DEFAULT_BG_COLOR, &bg_color)) {
	fprintf(stderr,"%s: Failed to parse background color %s. Falling back to WhitePixel.\n", progname, DEFAULT_BG_COLOR);
	bg_color.pixel = WhitePixel(display, screen);
      }
    }
    XAllocColor(display, colormap, &bg_color);

    if (XrmGetResource(db, "xstroke.strokeWidth", "Xstroke.StrokeWidth",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      stroke_width = atoi(res_value.addr);
    }
    if (stroke_width <= 0) {
      fprintf(stderr, "%s: Invalid stroke_width %s. Using %d\n", progname, res_value.addr, DEFAULT_STROKE_WIDTH);
      stroke_width = DEFAULT_STROKE_WIDTH;
    }

    root_window = DefaultRootWindow(display);
    pixmap_width = DisplayWidth(display, screen);
    pixmap_height = DisplayHeight(display, screen);
    pixmap = XCreatePixmap(display, root_window,
                           pixmap_width,
                           pixmap_height,
                           DefaultDepth(display, screen));

    winm = 0;
    winm |= CWOverrideRedirect; winv.override_redirect = True;
    winm |= CWBackPixel;        winv.background_pixel = bg_color.pixel;
    window = XCreateWindow(display, root_window,
			   x, y, width, height,
			   DEFAULT_BORDER_WIDTH,
			   CopyFromParent,
			   InputOutput,
			   CopyFromParent,
			   winm,
			   &winv);
    /*
    window = XCreateSimpleWindow(display, root_window,
				 x, y, width, height,
				 1,
				 fg_color.pixel, bg_color.pixel);
    */
    XGetGeometry(display, window, &root_window, &x, &y, &window_width, &window_height, &border_width, &window_depth);

    gcm = 0;
    gcm |= GCSubwindowMode; gcv.subwindow_mode = IncludeInferiors;
    gcm |= GCFunction;      gcv.function = GXxor;
    gcm |= GCForeground;    gcv.foreground = 0xffffffff;
    gcm |= GCLineWidth;     gcv.line_width = stroke_width;
    gcm |= GCCapStyle;      gcv.cap_style = CapRound;
    gcm |= GCJoinStyle;     gcv.join_style = JoinRound;
    root_gc = XCreateGC(display, root_window, gcm, &gcv);

    gcm = 0;
    gcm |= GCSubwindowMode; gcv.subwindow_mode = IncludeInferiors;
    gcm |= GCForeground;    gcv.foreground = fg_color.pixel;
    gcm |= GCBackground;    gcv.background = bg_color.pixel;
    window_gc = XCreateGC(display, root_window, gcm, &gcv);

    if (XrmGetResource(db, "xstroke.font", "Xstroke.Font",
		       &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
      fontname = res_value.addr;
    }
    font = XLoadQueryFont(display, fontname);
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Falling back to %s.\n", progname, fontname, DEFAULT_FONT);
      font = XLoadQueryFont(display, DEFAULT_FONT);
    }
    if (font == NULL) {
      fprintf(stderr, "%s: Failed to load font %s. Continuing without a font!\n", progname, DEFAULT_FONT);
    } else {
      XSetFont(display, window_gc, font->fid);
    }
    status_text.chars = xmalloc(MAX_STATUS_LEN);
    status_text.delta = 0;
    status_text.font = None;

    wm_hints.flags = InputHint;
    wm_hints.input = False;
    class_hint.res_name = DEFAULT_NAME;
    class_hint.res_class = DEFAULT_CLASS;
    XmbSetWMProperties(display, window, DEFAULT_NAME, DEFAULT_NAME, argv, *argc,
		       &wm_size_hints,
		       &wm_hints, &class_hint);

    XSetWMProtocols(display, window, wm_protocols, sizeof(wm_protocols) / sizeof(Atom));
    XStoreName(display, window, DEFAULT_NAME);
    XMapWindow(display, window);

    len = snprintf(status_text.chars, MAX_STATUS_LEN, "Off");
    status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
    redraw();

  } else {
    fprintf(stderr, "%s: Could not open display %s\n", progname, displayname ? displayname : getenv("DISPLAY"));
    exit(1);
  }
}

static void initialize_stroke_rec(int *argc, char *argv[]) {
  char *stroke_file = DEFAULT_STROKE_FILE;
  char *stroke_dir = getenv("HOME");
  char *filename;
  char *res_type;
  XrmValue res_value;
  int err;
  int min_kc, max_kc, ks_per_kc;
  KeySym *key_map;

  XDisplayKeycodes(display, &min_kc, &max_kc);
  key_map = XGetKeyboardMapping(display, min_kc,
				(max_kc - min_kc) + 1,
				&ks_per_kc);

  rec = rec_alloc();

  if (stroke_dir == NULL)
    stroke_dir = DEFAULT_STROKE_DIR;

  if (XrmGetResource(db, "xstroke.strokeFile", "Xstroke.StrokeFile",
		     &res_type, &res_value) && (strcmp(res_type,"String") == 0)) {
    filename = res_value.addr;
    err = rec_load_configuration(display, rec, filename);
  } else {
    filename = xmalloc(strlen(stroke_dir) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", stroke_dir, stroke_file);
    err = rec_load_configuration(display, rec, filename);
    free(filename);
  }

  if (err != 0 && strcmp(stroke_dir, DEFAULT_STROKE_DIR)) {
    /* Try again with DEFAULT_STROKE_DIR */
    filename = xmalloc(strlen(DEFAULT_STROKE_DIR) + 1 + strlen(stroke_file) + 1);
    sprintf(filename, "%s/%s", DEFAULT_STROKE_DIR, stroke_file);
    err = rec_load_configuration(display, rec, filename);
    free(filename);
  }

  if (err != 0) {
    fprintf(stderr, "%s: Could not load stroke definition file.\n", progname);
    fprintf(stderr, "%s: Tried %s/%s", progname, stroke_dir, stroke_file);
    if (strcmp(stroke_dir,DEFAULT_STROKE_DIR))
      fprintf(stderr, " and %s/%s", DEFAULT_STROKE_DIR, stroke_file);
    fprintf(stderr, "\n%s: Exiting.\n\n", progname);
    exit(1);
  }
}

/* Is the given event inside the main window */
static int event_in_window(XButtonEvent *xev) {
  int ret_val;
  int dest_x, dest_y;
  Window child;
  XWindowAttributes attr;

  XGetWindowAttributes(display, window, &attr);
  if (attr.map_state != IsViewable)
    return 0;

  XTranslateCoordinates(display, xev->window, window, xev->x, xev->y,
			&dest_x, &dest_y, &child);
  ret_val = ((dest_x >= -border_width)
	     && (dest_y >= -border_width)
	     && (dest_x < window_width + border_width)
	     && (dest_y < window_height + border_width));
  return ret_val;
}

static void redraw() {
  XClearWindow(display, window);
  if (font)
    XDrawText(display, window, window_gc, 1, font->ascent, &status_text, 1);
}

static void expose_event_handler(XEvent *xev) {
  XExposeEvent *eev = &xev->xexpose;
  window_width = eev->width;
  window_height= eev->height;
  if (font)
    XDrawText(display, window, window_gc, 1, font->ascent, &status_text, 1);
}

static void client_message_handler(XEvent *xev) {
  XClientMessageEvent *cmev = &xev->xclient;

  if (cmev->data.l[0] == XInternAtom(display, "WM_DELETE_WINDOW", False)) {
    cleanup_and_exit(0);
  }
}

/* evil. pure and simple. */
static void visibility_notify_handler(XEvent *xev) {
  XVisibilityEvent *vev = &xev->xvisibility;

  if((vev->state == VisibilityFullyObscured) ||
     (vev->state == VisibilityPartiallyObscured)) {
    XRaiseWindow(display, window);
  }
}


static void button_press_handler(XEvent *xev) {
  XButtonEvent *bev = &xev->xbutton;

  if (bev->button > 1) {
    cleanup_and_exit(0);
  }

  if (event_in_window(bev)) {
    Window child;
    state |= BUTTON_PRESSED;
    drag_start_x_root = bev->x_root;
    drag_start_y_root = bev->y_root;
    XTranslateCoordinates(display, bev->window, window, bev->x, bev->y,
			  &drag_start_x, &drag_start_y, &child);
  } else {
    save_root();
  }
}

static void toggle_recognition_state(int permanent) {
  int err, len;

  if (state & RECOGNITION_STATE) {
    XUngrabPointer(display, CurrentTime);
    if (permanent) {
      len = snprintf(status_text.chars, MAX_STATUS_LEN, "Off");
      status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
      redraw();
    }

    state &= ~RECOGNITION_STATE;
  } else {
    err = XGrabPointer(display, root_window, False,
		       PointerMotionMask | ButtonPressMask | ButtonMotionMask | ButtonReleaseMask,
		       GrabModeAsync, GrabModeAsync, /* Async pointer and keys */
		       None, /* Don't confine to a window */
		       None, /* TODO: Come up with a good cursor here */
		       CurrentTime);
    if (err == GrabSuccess) {
      if (permanent) {
	len = snprintf(status_text.chars, MAX_STATUS_LEN, "On");
	status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
	redraw();
      }
      
      state |= RECOGNITION_STATE;
    } else {
      fprintf(stderr, "%s: XGrabPointer failed. :(\n", progname);
    }
  }
}

static void button_release_handler(XEvent *xev) {
  XButtonEvent *bev = &xev->xbutton;

  if (state & MOVING_WINDOW) {

  } else if (state & BUTTON_PRESSED) {
    if (event_in_window(bev)) {
      toggle_recognition_state(1);
    }
  } else if (state & RECOGNITION_STATE) {
    restore_root();
    if (stroke.num_pts >= MIN_STROKE_PTS) {
      perform_stroke_recognition();
    } else {
      /*      fprintf(stderr, "Release at (%d, %d) not in window, sending fake button press.\n", bev->x, bev->y);
       */
      toggle_recognition_state(0);
      XTestFakeButtonEvent(display, 1, True, CurrentTime);
      XTestFakeButtonEvent(display, 1, False, CurrentTime);
      toggle_recognition_state(0);
    }
  }

  state &= ~(BUTTON_PRESSED | MOVING_WINDOW);

  stroke.num_pts = 0;
  stroke.last_decimated = 0;
}

static void perform_stroke_recognition() {
  action_list *alist;
  action *action;
  keycode_action_data *kcad;
  
  alist = rec_recognize_gesture(rec, stroke.pts, stroke.num_pts);
  if (alist) {
    int i;
    for (i=0, action = alist->actions; i < alist->num_actions; i++, action++) {
      if (action == NULL) {
	fprintf(stderr, "%s: Error: received null action data from the recognizer.", progname);
      } else {
	if(action->type == KEYCODE_ACTION) {
	  kcad = action->data;
	  XTestFakeKeyEvent(display, kcad->keycode, kcad->press, CurrentTime);
	} else {
	  fprintf(stderr, "%s: Action code %d not yet implemented.\n", progname, action->type);
	}
      }
    }
  }
  /*
    else {
    len = snprintf(status_text.chars, MAX_STATUS_LEN, sequence);
    status_text.nchars = (len > MAX_STATUS_LEN) ? MAX_STATUS_LEN : len;
    redraw();
    fprintf(stdout, "unknown=%s*\n",sequence);
  }
  */
}

/* Throw out several points to make some more room.  Originally, this
 * function threw out every modth point, but when it was called
 * repeatedly, this stripped a lot more points from the fron of the
 * stroke than the end. I've modified it to remember where it last
 * left off compressing to start from there again and restart at the
 * beginning when the starting point gets close to the end again. This
 * helps, but it's still not perfect for some reason. Oh well...  */
static void decimate_stroke(struct stroke_lines *stroke, int mod) {
  int dest, src;

  if ((MAX_STROKE_PTS - stroke->last_decimated) < (2 * mod))
    stroke->last_decimated = 0;

  for (dest = stroke->last_decimated, src=dest;
       src < stroke->num_pts; dest++, src++) {
    if ((dest % (mod - 1)) == 0) {
      src++;
      if (src >= stroke->num_pts)
	break;
    }
    stroke->pts[dest].x = stroke->pts[src].x;
    stroke->pts[dest].y = stroke->pts[src].y;
  }
  stroke->num_pts = dest;
  stroke->last_decimated = dest - 1;
}

static void save_root() {
  XGrabServer(display);
  XCopyArea(display, root_window, pixmap, window_gc,
	    0, 0, pixmap_width, pixmap_height,
	    0, 0);
}

static void restore_root() {
  XCopyArea(display, pixmap, root_window, window_gc,
            0, 0, pixmap_width, pixmap_height,
            0, 0);
  XUngrabServer(display);
}

static void motion_notify_handler(XEvent *xev) {
  XMotionEvent *mev = &xev->xmotion;

  if (state & BUTTON_PRESSED) {
    if (abs(mev->x_root - drag_start_x_root)
	  + abs(mev->y_root - drag_start_y_root)  > MOVE_THRESHOLD) {
      state |= MOVING_WINDOW;
    }
    if (state & MOVING_WINDOW) {
      XMoveWindow(display, window, mev->x_root - drag_start_x - border_width,
		  mev->y_root - drag_start_y - border_width);
    }
  } else if (state & RECOGNITION_STATE) {
    if (mev->state & (Button1Mask | 
		      Button2Mask |
		      Button3Mask |
		      Button4Mask |
		      Button5Mask)) {
      stroke_record (mev->x, mev->y);
      
      if (stroke.num_pts >= MAX_STROKE_PTS) {
	/* erase_stroke(&lines); */
	decimate_stroke(&stroke, STROKE_DECIMATION);
	/* draw_stroke(&lines); */
      }
      stroke.pts[stroke.num_pts].x = mev->x;
      stroke.pts[stroke.num_pts].y = mev->y;
      stroke.num_pts++;
      if (stroke.num_pts > 2) {
	XDrawLines(display, root_window, root_gc, stroke.pts + stroke.num_pts -3, 2, CoordModeOrigin);
	XDrawLines(display, root_window, root_gc, stroke.pts + stroke.num_pts -3, 3, CoordModeOrigin);
      } else {
	/* only two points, just draw it */
	XDrawLines(display, root_window, root_gc, stroke.pts, stroke.num_pts, CoordModeOrigin);
      }
    } else {
	/*
      toggle_recognition_state(0);
      XTestFakeMotionEvent(display, 1, mev->x, mev->y, CurrentTime);
      toggle_recognition_state(0);
	*/
    }
  }
}

static void configure_notify_handler(XEvent *xev) {
  XConfigureEvent *cev = &xev->xconfigure;

  window_width = cev->width;
  window_height = cev->height;
}

static void key_press_handler(XEvent *xev) {
  XKeyEvent *kev = &xev->xkey;

  if (XLookupKeysym(kev, 0) == XK_q) {
    cleanup_and_exit(0);
  }
}

static void cleanup_and_exit(int exit_code) {
  if (display) {
    if (font)
      XFreeFont(display, font);
    if (root_gc)
      XFreeGC(display, root_gc);
    if (window_gc)
      XFreeGC(display, window_gc);
    if (window)
      XDestroyWindow(display, window);
    XCloseDisplay(display);
  }
  rec_free(rec);
  /*
    print_node_stats();
  */
  exit(exit_code);
}

