%{
/* rec_parse.y -- Grammar for xstroke recognition control file

   Copyright 2001 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "rec.h"
#include "rec_mode.h"
#include "gesture.h"
#include "action.h"
#include "feature.h"
#include "option.h"

#include "rec_lex.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

static int yyerror(char *err);

#define YYPARSE_PARAM rec
#define YYERROR_VERBOSE

#ifdef REC_PARSE_DEBUG
#define YYDEBUG 1
#endif

static feature_list_t FEATURE_ERROR = { -1, NULL };
#define IS_FEATURE_ERROR(f) ( (f).num_features == FEATURE_ERROR.num_features )

%}

%pure_parser

%union {
  int    ival;
  char  *sval;
  double dval;
  rec_mode_t *rec_mode;
  rec_mode_list_t rec_mode_list;
  gesture_list_t gesture_list;
  gesture_t gesture;
  action_t action;
  action_item_t action_item;
  feature_list_t feature_list;
  feature_t feature;
  rec_engine_t *rec_engine;
  option_t option;
  option_list_t option_list;
}

%{
int yylex(YYSTYPE *lvalp);
%}

%token <sval> UNKNOWN_CHARACTER

%token <ival> KEY
%token <ival> BUTTON
%token <ival> MODE_SHIFT
%token <ival> MODE_LOCK
%token <ival> EXEC
%token <ival> ORIENTATION_CORRECTION

%token <ival> MODE
%token <ival> INTEGER
%token <sval> IDENTIFIER
%token <sval> STRING
%token <dval> DOUBLE
%token <ival> ENGINE
%token <ival> OPTION

%type <ival> alphabet
%type <ival> definition_list
%type <ival> definition
%type <rec_mode> mode
%type <rec_mode> mode_decl
%type <rec_mode_list> mode_id_list
%type <rec_mode> mode_id
%type <gesture_list> gesture_list
%type <gesture> gesture
%type <action> action
%type <action_item> action_item
%type <feature_list> feature_group
%type <feature_list> feature_list
%type <feature> feature
%type <rec_engine> engine
%type <option_list> option_list
%type <option> option
%%

alphabet	: /* empty */ { $$ = 0; }
		| definition_list
		;

definition_list		: definition
			| definition_list definition
			;

definition	: mode 
			{ $$ = 0; }
		| gesture
			{
			  rec_mode_add_gesture(((rec_t *) rec)->global_mode,
					       &$1);
			}
		| engine
			{ $$ = 0; }
		;

mode		: mode_decl '{' '}'
			{ $$ = $1; }
		| mode_decl '{' gesture_list '}'
			{
			  $$ = $1;
			  rec_mode_add_gestures($$, &$3);
			}
		;

mode_decl	: MODE STRING
			{
			  $$ = rec_get_mode((rec_t *) rec, $2);
			  free($2);
			}
		| MODE STRING 
			{
			  /* Do this first so the default mode gets set correctly*/
			  $$ = rec_get_mode((rec_t *) rec, $2);
			}
		  ':' mode_id_list
			{
			  $$ = rec_get_mode((rec_t *) rec, $2);
			  rec_mode_add_parents($$, &$5);
			  rec_mode_list_deinit_shallow(&$5);
			  free($2);
			}
		;

mode_id_list	: mode_id
			{
			  rec_mode_list_init(&$$);
			  rec_mode_list_append(&$$, $1);
			}
		| mode_id_list ',' mode_id
			{
			  $$ = $1;
			  rec_mode_list_append(&$$, $3);
			}
		;

mode_id		: STRING
			{
			  $$ = rec_get_mode((rec_t *) rec, $1);
			  free($1);
			}
		;

gesture_list	: gesture
			{
			  gesture_list_init(&$$);
			  gesture_list_append(&$$, &$1);
			}
		| gesture_list gesture
			{
			  $$ = $1;
			  gesture_list_append(&$$, &$2);
			}
		;

gesture		: action '=' feature_group
			{
			  if (IS_FEATURE_ERROR($3)) {
			    char *loc = rec_lex_location_alloc();
			    char *a_str = action_str_alloc(&$1);
			    fprintf(stderr, "%s: Not adding stroke for "
				    "`%s' to database.\n", loc, a_str);
			    free(loc);
			    free(a_str);
			  } else {
			    gesture_init(&$$, $1, $3);
			  }
			}
		;

action		: action_item
			{
			  action_init(&$$);
			  action_add_item(&$$, $1);
			}
		| action ',' action_item
			{
			  $$ = $1;
			  action_add_item(&$$, $3);
			}
		;

action_item	: KEY IDENTIFIER
			{
			  action_item_key_init(&$$, $2, 1);
			  free($2);
			}
		| KEY INTEGER
			{
			  char key[2];
			  key[0] = '0' + $2;
			  key[1] = '\0';
			  action_item_key_init(&$$, key, 1);
			}
		| BUTTON INTEGER
			{
			  action_item_button_init(&$$, $2);
			}
		| MODE_SHIFT STRING
			{
			  action_item_mode_init(&$$,
						rec_get_mode((rec_t *) rec, $2),
						0);
			  free($2);
			}
		| MODE_LOCK STRING
			{
			  action_item_mode_init(&$$,
						rec_get_mode((rec_t *) rec, $2),
						1);
			  free($2);
			}
		| EXEC STRING
			{
			  action_item_exec_init(&$$, $2);
			  free($2);
			}
		| ORIENTATION_CORRECTION INTEGER
			{
			  /* Convert from degrees to radians */
			  double orientation = (M_PI / 180.0) * $2;
			  action_item_orient_init(&$$, orientation);
			}
		| ORIENTATION_CORRECTION DOUBLE
			{
			  /* Convert from degrees to radians */
			  double orientation = (M_PI / 180.0) * $2;
			  action_item_orient_init(&$$, orientation);
			}
		;

feature_group		: feature
				{ rec_lex_newlines_wanted(1); }
			  '\n'
				{ 
				  rec_lex_newlines_wanted(0);
				  feature_list_init(&$$);
				  feature_list_append(&$$, &$1);
				}
			| error
				{ rec_lex_newlines_wanted(1); }
			  '\n'
				{ 
				  rec_lex_newlines_wanted(0);
				  $$ = FEATURE_ERROR;
				  yyerrok;
				}
			| '{' feature_list '}'
				{ $$ = $2; }
			| '{' error '}'
				{ $$ = FEATURE_ERROR; yyerrok; }
			;

feature_list		: feature 
				{
				  feature_list_init(&$$);
				  feature_list_append(&$$, &$1);
				}
			| feature_list ';'
				{
				  $$ = $1;
				}
			| feature_list feature
				{
				  $$ = $1;
				  feature_list_append(&$$, &$2);
				}
			;

feature			: IDENTIFIER '(' STRING ')'
				{
				  feature_init(&$$, (rec_t *) rec, $1, $3);
				  free($1);
				  free($3);
				}
			;

engine		: ENGINE IDENTIFIER '{' '}'
			{
			  $$ = rec_get_engine((rec_t *) rec, $2);
			  free($2);
			}
		| ENGINE IDENTIFIER '{' option_list '}'
			{
			  int i;
			  $$ = rec_get_engine((rec_t *) rec, $2);
			  for (i=0; i < $4.num_options; i++) {
			    rec_engine_set_option($$,
						  $4.options[i].name,
						  $4.options[i].value);
			    free($4.options[i].name);
			    free($4.options[i].value);
			  }
			  option_list_deinit(&$4);
			  free($2);
			}
		;

option_list	: option
			{
			  option_list_init(&$$);
			  option_list_append(&$$, &$1);
			}
		| option_list option
			{
			  $$ = $1;
			  option_list_append(&$$, &$2);
			}
		;

option		: OPTION STRING STRING
			{ option_init(&$$, $2, $3); }
		;
			
%%

static int yyerror(char *err)
{
    char *loc = rec_lex_location_alloc();
    fprintf(stderr, "%s: %s\n", loc, err);
    free(loc);
    return 0;
}

int rec_parse(rec_t *rec, char *filename)
{
    int ret_val;

#ifdef REC_PARSE_DEBUG
    yydebug = 1;
#endif

    rec_lex_initial_file(filename);
    ret_val = yyparse(rec);

    return ret_val;
}
