#!/usr/bin/perl

# A few global parameters
$FILE_PREFIX = "alphabet";
$BOX_SIZE = ".26"; # in units of SIZE_UNITS, (inches by default). See below
$TENSION = 1.25;   # Larger number makes the curves less wild and loopy, but also less smooth.
                   # Values between 1.0 and 1.5 seem to be about right
$CURL = .2; # How much should the line segment to the final point be curled, useful range: 0-1
            # A small value here helps with tight loops like the end of the G stroke

# You shouldn't have to tweak these, but here they are
$H_SIZE = "7.5";
$V_SIZE = "9.9";
$H_OFFSET = "-0.5";  # These offsets are measured from a point that is one to the right
$V_OFFSET = "-0.5";   # and one inch down from the upper-left corner of the page.
$SIZE_UNITS = "in";
$TABLE_COLS = int(($H_SIZE / (3 * $BOX_SIZE)));
$TABLE_ROWS = int(($V_SIZE / (4 * $BOX_SIZE)));

# Now let it begin!
$figure = 0;

open(MP_FILE, ">$FILE_PREFIX.mp") || die "Cannot open $FILE_PREFIX.mp";
open(TEX_FILE, ">$FILE_PREFIX.tex") || die "Cannot open $FILE_PREFIX.tex";

&print_mp_header;
&print_tex_header;
while (<>) {

    next if (/\s*\#/);

    ($character,$sequence) = split(/=/);
    if ($sequence =~ /\*/) {
	$character =~ s/^\s*(\S*)\s*$/\1/;
	$sequence =~ s/^\s*(\S*)\s*$/\1/;

	$make_it_curvy = ($sequence !~ /\*\*/);
	$first_digit = 1;
	for ($i = 0; $i < length($sequence); $i++) {
	    $digit = substr($sequence,$i,1);

	    if ($digit =~ /^[0-9]$/) {
		$point = "z$digit.c";
	    } elsif ($digit eq "[") {
		$digit = substr($sequence,$i+1,1);
		while (substr($sequence,$i,1) ne "]" && $i < length($sequence)) {
		    $i++;
		}
		if ($i == length($sequence)) {
		    print "Error: Unterminated '[' at $ARGV:$.\n";
		    next;
		}
		$second_digit = substr($sequence,$i-1,1);
		$point = "1/2[z$digit.c,z$second_digit.c]";
	    } else {
		next;
	    }

	    if ($first_digit) {
		$filename = $character . "_" . $clash{$character}++ . ".eps";
		print MP_FILE "beginfig($figure); % $filename\n";
		print MP_FILE "  drawgrid;\n";
		print MP_FILE "  label.top(\"$character\", (u,u/2));\n";
		print MP_FILE "  pickup pencircle scaled 5pt;\n";
		print MP_FILE "  draw $point;\n";
		print MP_FILE "  pickup defaultpen;\n";
		print MP_FILE "  drawarrow";
	    } else {
		if ($make_it_curvy) {
		    print MP_FILE " ..tension $TENSION..";
		} else {
		    print MP_FILE " --";
		}
	    }
	    
	    if ($make_it_curvy && $i == length($sequence) - 2) {
		print MP_FILE "{curl $CURL}";
	    }
	    print MP_FILE " $point";

	    $first_digit = 0;
	}
	print MP_FILE ";\nendfig;\n\n";

	if ($figure % ($TABLE_ROWS * $TABLE_COLS) == 0) {
	    if ($figure != 0) {
		&print_tex_table_footer;
	    }
	    &print_tex_table_header;
	} elsif (($figure % ($TABLE_COLS) == 0) && ($figure != 0)) {
	    print TEX_FILE "\\cr\n";
	} else {
	    print TEX_FILE "&";
	}

	print TEX_FILE "\\epsfbox{$FILE_PREFIX.$figure}";
	$figure++;
    }
}
&print_tex_table_footer;
&print_mp_footer;
&print_tex_footer;

exit;

sub print_mp_header {
    print MP_FILE <<EOT;
% Setup the defaultfont for MP and TeX
defaultfont:= "cmr12";
u := $BOX_SIZE $SIZE_UNITS;
ahlength := 1.5 ahlength;

vardef makebox(suffix b) =
  b.tl = b.c + (-u/2,  u/2);
  b.t  = b.c + (   0,  u/2);
  b.tr = b.c + ( u/2,  u/2);
  b.l  = b.c + (-u/2,    0);
  b.r  = b.c + ( u/2,    0);
  b.bl = b.c + (-u/2, -u/2);
  b.b  = b.c + (   0, -u/2);
  b.br = b.c + ( u/2, -u/2);
enddef;

vardef drawgrid =
  z1.c = (0,  0); z2.c = (u,  0); z3.c = (2u,  0);
  z4.c = (0, -u); z5.c = (u, -u); z6.c = (2u, -u);
  z7.c = (0,-2u); z8.c = (u,-2u); z9.c = (2u,-2u);
  makebox(z1); makebox(z2); makebox(z3);
  makebox(z4); makebox(z5); makebox(z6);
  makebox(z7); makebox(z8); makebox(z9);

  draw z1.tl -- z3.tr -- z9.br -- z7.bl -- z1.tl;
  draw z2.tl -- z8.bl;
  draw z2.tr -- z8.br;
  draw z4.tl -- z6.tr;
  draw z4.bl -- z6.br;
  label.lrt("1" infont defaultfont scaled .5, z1.tl);
  label.lrt("2" infont defaultfont scaled .5, z2.tl);
  label.lrt("3" infont defaultfont scaled .5, z3.tl);
  label.lrt("4" infont defaultfont scaled .5, z4.tl);
  label.lrt("5" infont defaultfont scaled .5, z5.tl);
  label.lrt("6" infont defaultfont scaled .5, z6.tl);
  label.lrt("7" infont defaultfont scaled .5, z7.tl);
  label.lrt("8" infont defaultfont scaled .5, z8.tl);
  label.lrt("9" infont defaultfont scaled .5, z9.tl);  
enddef;

EOT
}

sub print_mp_footer {
    print MP_FILE <<EOT;
end;
EOT
}

sub print_tex_header {
print TEX_FILE <<EOT;
\\hsize = $H_SIZE $SIZE_UNITS
\\vsize = $V_SIZE $SIZE_UNITS
\\hoffset = $H_OFFSET $SIZE_UNITS
\\voffset = $V_OFFSET $SIZE_UNITS

\\input epsf
EOT
}

sub print_tex_table_header {
    print TEX_FILE <<EOT;
\\vfil
\$\$\\vbox{
\\offinterlineskip
\\halign{
EOT
    for ($i=0; $i<$TABLE_COLS; $i++) {
	if ($i != 0) {
	    print TEX_FILE "&";
	}
	print TEX_FILE "\\hfil #\\hfil ";
	}
print TEX_FILE "\\cr\n";
EOT
}

sub print_tex_table_footer {
    print TEX_FILE <<EOT;
\\cr
}
}\$\$
\\vfil
\\eject
EOT
}

sub print_tex_footer {
    print TEX_FILE <<EOT;
\\end
EOT
}
