/*
 * FILE: Games.c
 *
 * DESCRIPTION:
 *   This module contains routines for initializing and updating
 *   the game and its levels.
 *
 * HISTORY:
 *   Nov 19, 2000 Created by Dmitry Kaloshin
 */
#include <cywin.h>
#include "Game.h"
#include "Spider.h"
#include "DataTypes.h"

extern spider_direction_type current_dir; // Spider's current direction 

extern cell_type SpiderPositionType; // Spider's current position 
                                                  
extern spider_direction_type last_dir; // Spider's previous direction 

/*
 * FUNCTION: init_level
 *
 * DESCRIPTION: Initialize a new level of the game.
 *
 * PARAMETERS: 
 *   pGame  - pointer to a game struct
 *
 * RETURNS: nothing
 */
void init_level( TGame *pGame )
{
  int i, j;

  int NumKillersOnLevel; // Max number of Killers on current levels
   
  current_dir = last_dir = sdUnknown;

  SpiderPositionType = ctFree;

  for( i = 2; i < MAX_ROWS; i++ )
  {
    for( j = 2; j < MAX_COLS; j++ )
    {
      pGame->Field[ i ][ j ] = ctBusy; // Fill Killer's area
    }
  }
  for( i = 1; i < MAX_COLS + 1; i++ ) // Fill Border   
  {
    pGame->Field[ 1 ][ i ] = ctBorder;
  }
  for( i = 1; i < MAX_COLS + 1; i++ )
  { 
    pGame->Field[ MAX_ROWS ][ i ] = ctBorder;
  }
  for( i = 1; i < MAX_ROWS + 1; i++ )
  {
    pGame->Field[ i ][ 1 ] = ctBorder;
  }
  for( i = 1; i < MAX_ROWS + 1; i++ )
  {
    pGame->Field[ i ][ MAX_COLS ] = ctBorder; 
  }
  for( i = 0; i < MAX_COLS + 2; i++ ) // Fill Spider's area
  {
    pGame->Field[ 0 ][ i ] = ctFree;
  }
  for( i = 0; i < MAX_COLS + 2; i++ )
  {
    pGame->Field[ MAX_ROWS + 1 ][ i ] = ctFree;
  }
  for( i = 0; i < MAX_ROWS + 2; i++ )
  {
    pGame->Field[ i ][ 0 ] = ctFree;
  }
  for( i = 0; i < MAX_ROWS + 2; i++ )
  {
    pGame->Field[ i ][ MAX_COLS + 1 ] = ctFree;
  }
  for( i = 0; i < MAX_KILLERS; i++ ) // Initialize all Killers
  {
    pGame->Killer[ i ].coord.x = 3 + (int) random( MAX_COLS - 4 );

    pGame->Killer[ i ].coord.y = 3 + (int) random( MAX_ROWS - 4 );

    pGame->Killer[ i ].type = ktReserve;

    pGame->Killer[ i ].dir = (killer_direction_type) random( 4 );

    pGame->Killer[ i ].angle = (killer_angle_type) random( 3 );
  }
  // Calculate number of
  // Killers on current level
  NumKillersOnLevel = ( pGame->CurLevel + 2 ) / 2; 
                                                  
  // Calculate Spider's coordinate
  pGame->Spider.coord.x = SPIDER_SIZE + ( MAX_COLS / 2 ) * CELL_SIZE -
    ( SPIDER_SIZE - CELL_SIZE ) / 2;

  pGame->Spider.coord.y = 2;                          

  pGame->Spider.dir = sdUnknown;

  pGame->LastCoordCell.x = ( SPIDER_SIZE + ( MAX_COLS / 2 ) * CELL_SIZE -
    ( SPIDER_SIZE - CELL_SIZE ) / 2 - 2 + SPIDER_SIZE / 2 - 1 ) / CELL_SIZE;

  pGame->LastCoordCell.y = ( SPIDER_SIZE / 2 - 1 ) / CELL_SIZE;

  pGame->Spider.beginCoord.x = 2;

  pGame->Spider.beginCoord.y = pGame->Spider.coord.y;

  pGame->PercentLeft = LEVEL_PERCENT + 7 * ( pGame->CurLevel % 2 );
   
  if( NumKillersOnLevel > 6 ) // Max Killers in game : 6
  {
    NumKillersOnLevel = 6;

    // After 11 level member PercentLeft increase by 1%
    pGame->PercentLeft = LEVEL_PERCENT + 7 + ( pGame->CurLevel - 11 );

    if( pGame->PercentLeft > MAX_LEVEL_PERCENT )  // Threshold of PercentLeft
    {
      pGame->PercentLeft = MAX_LEVEL_PERCENT;
    }
  }
  for( i = 0; i < NumKillersOnLevel; i++ ) // Activate this level's Killers
  {
    pGame->Killer[ i ].type = ktActive;
  }
  // Calculate num of the cell, that the Spider must cut
  pGame->CurCellOnField = ( MAX_ROWS - 2 ) * ( MAX_COLS - 2 );

  pGame->NumCellMustCut = pGame->CurCellOnField * pGame->PercentLeft / 100;
                                                                      
  pGame->Paused = FALSE;
}

/*
 * FUNCTION: init_game
 *
 * DESCRIPTION: Initialize a new game.
 *
 * PARAMETERS: 
 *   pGame  - pointer to a game struct
 *
 * RETURNS: nothing
 */
void init_game( TGame *pGame )
{
  pGame->OriginKiller.x = SPIDER_SIZE - CELL_SIZE - 
    ( KILLER_SIZE - CELL_SIZE ) / 2;

  pGame->OriginKiller.y = SPIDER_SIZE - CELL_SIZE - 
    ( KILLER_SIZE - CELL_SIZE ) / 2;

  pGame->Lifes = SPIDER_LIFES; // Begin Spider's lives

  pGame->CurLevel = 0;
   
  pGame->Status = gsMenu;

  pGame->Score = 0;

  init_level( pGame );
}

/*
 * FUNCTION: update_game_field
 *
 * DESCRIPTION: Update the playfield after a life is lost.
 *
 * PARAMETERS: 
 *   pGame  - pointer to a game struct
 *
 * RETURNS: nothing
 */
void update_game_field( TGame *pGame )
{
  int i, j;

  for( i = 0; i < MAX_ROWS + 2; i++ )
  {
    for( j = 0; j < MAX_COLS + 2; j++ )
    {
      if( pGame->Field[ i ][ j ] == ctActiveBorder )
      {
        pGame->Field[ i ][ j ] = ctBusy;
      }
    }
  }
}
