/*
 * FILE: Killer.c
 *
 * DESCRIPTION:
 *   This module contains routines which describe Killers' movements,
 *   including Spider crushes.
 *
 * HISTORY:
 *   Nov 15, 2000 Created by Dmitry Kaloshin
 */
#include <cywin.h>
#include "Killer.h"
#include "utilsound.h"

int currentPeriod = FALSE;

/*
 * FUNCTION: correct_killer_move
 *
 * DESCRIPTION: Checks that the Killer's next step is correct.
 *
 * PARAMETERS: 
 *   pGame    - pointer to a game struct
 *   pKiller  - pointer to the Killer that checked
 *   angle    - angle of movement
 *   dir      - Killer's direction
 *   newCoord - pointer to struct of next Killer's coordinates,
 *              if move is correct
 *
 * RETURNS: TRUE - next step correct
 */
int  correct_killer_move( TGame *pGame, TKiller *pKiller,
  killer_angle_type angle, killer_direction_type dir,
    PointType *newCoord )
{
  int addX, addY;

  PointType coord;

  coord.x = pKiller->coord.x;

  coord.y = pKiller->coord.y;

  switch( angle ) // Calculate the direction's priority
  {
    case kaX_Y:

      addX = addY = 1;

      break;

    case ka2X_Y:

      addX = 1;

      addY = currentPeriod ? 1 : 0; // May move along y axis

      break;

    case kaX_2Y:

      addY = 1;

      addX = currentPeriod ? 1 : 0; // May move along x axis
  }
  // Calculate new coordinates
  // and if new coordinate is on a border then disallow the move
  switch( dir )
  {
    case kdLeftUp:

      if( pGame->Field[ coord.y - addY ][ coord.x - addX ] != ctBorder && 
        pGame->Field[ coord.y - 1 ][ coord.x - 1 ] != ctBorder )
      {
        newCoord->x = coord.x - addX;

        newCoord->y = coord.y - addY;

        return TRUE;
      }
      else return FALSE;

    case kdRightUp:

      if( pGame->Field[ coord.y - addY ][ coord.x + addX ] != ctBorder &&
        pGame->Field[ coord.y - 1 ][ coord.x + 1 ] != ctBorder )
      {
        newCoord->x = coord.x + addX;

        newCoord->y = coord.y - addY;

        return TRUE;
      }
      else return FALSE;

    case kdRightDown:

      if( pGame->Field[ coord.y + addY ][ coord.x + addX ] != ctBorder &&
        pGame->Field[ coord.y + 1 ][ coord.x + 1 ] != ctBorder )
      {
        newCoord->x = coord.x + addX;

        newCoord->y = coord.y + addY;

        return TRUE;
      }
      else return FALSE;

    case kdLeftDown:

      if( pGame->Field[ coord.y + addY ][ coord.x - addX ] != ctBorder &&
        pGame->Field[ coord.y + 1 ][ coord.x - 1 ] != ctBorder )
      {
        newCoord->x = coord.x - addX;

        newCoord->y = coord.y + addY;

        return TRUE;
      }
      else return FALSE;
  }
}

/*
 * FUNCTION: killer_move
 *
 * DESCRIPTION: 
 *   Moves the Killer. If the intended move is incorrect, 
 *   finds the correct direction to move.
 *
 * PARAMETERS: 
 *   pGame  - pointer to a game struct
 *   kIndex - index the current Killer in an array of Killers
 *
 * RETURNS: nothing
 */
void killer_move( TGame *pGame, int kIndex )
{
  PointType newCoord;

  TKiller *pKiller; // Pointer to a Killer in an array of Killers
   
  pKiller = &( pGame->Killer[ kIndex ] ); // Get pointer of Killer's struct

  if( !correct_killer_move( pGame, pKiller, pKiller->angle, 
    pKiller->dir, &newCoord ))
  {
    // If TRUE then Killer must go the opposite direction
    int backWard = FALSE;

    killer_direction_type nextDir, dir1, dir2; // Possible direction 

    killer_angle_type nextAngle; // Possible angle 

    dir1 = (killer_direction_type) (( pKiller->dir + 1 ) % 4 );

    dir2 = (killer_direction_type) (( pKiller->dir + 3 ) % 4 );

    if( !correct_killer_move( pGame, pKiller, kaX_Y, dir1, &newCoord ) &&
      !correct_killer_move( pGame, pKiller, kaX_Y, dir2, &newCoord ))
    {
      backWard = TRUE;
    }     
    do // Find correct direction
    {
      if( backWard )
      {
        nextDir = (killer_direction_type) (( pKiller->dir + 2 ) % 4 );
      }
      else
      {
        nextDir = (killer_direction_type) 
                    (( pKiller->dir + 1 + ((int) random( 2 ) % 2 ) * 2 ) % 4 );
      }
      nextAngle = (killer_angle_type) ((int) random( 3 ) % 3 );
    }
    while( !correct_killer_move( pGame, pKiller, 
      nextAngle, nextDir, &newCoord ));

    vibrate( 255 );

    sleep( 1 );

    pKiller->angle = nextAngle;

    pKiller->dir = nextDir;

    vibrate( 0 ); 
  }
  // Crush Spider's trace or Spider
  if( pGame->Field[ newCoord.y ][ newCoord.x ] == ctActiveBorder ||
    ( newCoord.y == ( pGame->Spider.coord.y - pGame->Spider.beginCoord.y +
    SPIDER_SIZE/2 - 1 ) / CELL_SIZE && 
    newCoord.x == ( pGame->Spider.coord.x-pGame->Spider.beginCoord.x +
    SPIDER_SIZE/2 - 1 ) / CELL_SIZE )) 
  { 
    pGame->Status = gsKilling;

    play_sound( SOUND_FLY );
  }
  else
  {
    pKiller->coord.x = newCoord.x;

    pKiller->coord.y = newCoord.y;

    pGame->Status = gsPlay;
  }
}

/*
 * FUNCTION: move_killers
 *
 * DESCRIPTION: Moves all Killers onto the playfield.
 *
 * PARAMETERS: 
 *   pGame  - pointer to a game struct
 *
 * RETURNS: nothing
 */
void move_killers( TGame* pGame )
{
  int i;

  for( i = 0; i < MAX_KILLERS; i++ )
  {
    if( pGame->Killer[ i ].type == ktActive )
    {
      killer_move( pGame, i );

      if( pGame->Status == gsKilling ) return;
    }
  }
  pGame->Status = gsPlay;

  currentPeriod = !currentPeriod;
}
