/*
 * FILE: Main.c
 *
 * DESCRIPTION:
 *   This is the main module. It starts the application,
 *   launches events, and stops the application.
 *
 * HISTORY:
 *   Nov 23, 2000 Created by Dmitry Kaloshin
 */
#include <cywin.h>
#include "Main.h"
#include "Spider.h"
#include "Killer.h"
#include "highscore.h"
#include "Menu.h"
#include "utilanimate.h"
#include "utilsound.h"

struct module_t MainModule; // Application's main module

bool exit_game_session; // TRUE if program is finished      
bool exit_current_game; // TRUE if current game is finished

char *strFinished; // Finished game's string

struct MSequence mseq;

spider_direction_type current_dir; // Spider's current direction 

cell_type SpiderPositionType; // Spider's current position 
                                       
spider_direction_type last_dir; // Spider's previous direction 

PointType start_spider_coord; // Spider's starting position 

const char* sz_level =
  "WELCOME TO %d%s LEVEL!\n\nYou must clear %d%% of screen area.";

const char* sz_endings[ 4 ] = { "st", "nd", "rd", "th" };

// What highscore_enter function was returned after the game ended
int end_of_game_status; 

clock_t NextPeriod; // Next period for redraw game

clock_t TimeBetweenCuts; // Current critical time till cut

// Current time till other redraw (used if MainModule loses focus)
static clock_t temp_time_between_cut = CRITICAL_TIME; 
                          
struct BitmapSequence ElemsBmp; // All game images

struct DirectKeyboard* pKeyboard; // Keyboard structure 

TGame Game; // Main structure of the game

struct cDialog MsgDialog; // Cywin dialog object.

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], int start )
{
  struct Message* pMessage;

  struct KeyParam* pKeyParam;
  
  int retDialog;

  long styleDialog;

  char *textDialog;

  int menu_index = 0;

  init_module( &MainModule ); // Initializes application
  
  start_application();

  play_sound( SOUND_TITLE );
  
  while( !exit_game_session )  // The application's main loop
  {
    while( !exit_current_game )
    {
      if( Game.Status == gsMenu )
      {
         MainModule.m_process->HelpContext = 1; // Change help list

         menu_index = enter_menu( &MainModule, menu_index );

         switch( menu_index )
         {
           case 0:

             MainModule.m_process->HelpContext = 0; // Change help list

             menu_index = 2; // Cursor index

             start_new_game();

             break;

           case 1:

             if( highscore_enter( &MainModule, 0, HSM_SHOW ) != HSR_INVITE )
             {
               continue;
             }
             /* fall through */

           default:

             exit_current_game = exit_game_session = TRUE;

             continue;
         }
      }
      pMessage = cWinApp_get_message( MainModule.m_process, 1, 1, MSG_USER );

      if( pMessage )
      {
        switch( pMessage->msgid )
        {
          case MSG_GOTFOCUS: // The process gets the focus (after invitations)

            if( !strFinished )
            {
              Game.Paused = FALSE;

              NextPeriod = clock() + GAME_PERIOD;

              TimeBetweenCuts = clock() + temp_time_between_cut;
            }            
            redraw_game_board();

            break;

          case MSG_LOSTFOCUS:

            Game.Paused = TRUE;

            break;

          case MSG_SHUTUP:   
          case MSG_QUIT:

            exit_current_game = exit_game_session = TRUE;

            highscore_enter( &MainModule, Game.Score, HSM_CHECK );

            break;

          case MSG_KEYDOWN:

            pKeyParam = Message_get_key_param( pMessage );

            switch( pKeyParam->scancode )
            {
              case KEY_ESC:

                styleDialog = mbQuit | mbRestart | mbs5;

                textDialog = str_Really_exit;

                cDialog_ctor( &MsgDialog, NULL, textDialog, styleDialog, 
                  0, MainModule.m_process );

                retDialog = cDialog_ShowModal( &MsgDialog );

                cDialog_dtor( &MsgDialog, LEAVE_MEMORY );
                
                if( retDialog == mrQuit )
                {
                  if( highscore_enter( &MainModule, Game.Score, 
                    HSM_CHECKSHOWTABLE ) == HSR_INVITE )
                  {
                    exit_current_game = exit_game_session = TRUE;
                  }
                  else
                  {
                    Game.Status = gsMenu;
                  }
                  Message_delete( pMessage );

                  continue;
                }
                else
                {
                  if( retDialog == mrRestart )
                  {
                    if( highscore_enter( &MainModule, Game.Score, 
                      HSM_CHECKSHOWTABLE ) == HSR_INVITE )
                    {
                      exit_current_game = exit_game_session = TRUE;
                    }
                    else
                    {
                      start_new_game();
                    }
                    Message_delete( pMessage );

                    continue;
                  }                  
                }                
                if( !Game.Paused )
                {
                  NextPeriod = clock() + GAME_PERIOD;

                  TimeBetweenCuts = clock() + temp_time_between_cut;
                }
                redraw_game_board();

                break;

              case KEY_HELP:

                cWinApp_defproc( MainModule.m_process, pMessage );

                break;

              default:

                // Need processing HELP key
                cWinApp_defproc( MainModule.m_process, pMessage );

                break;
            }
            break;

          default:

            cWinApp_defproc( MainModule.m_process, pMessage );

            break;
        }
        Message_delete( pMessage );
      }
      if( cWinApp_has_focus( MainModule.m_process ) && 
        !exit_current_game && !exit_game_session )
      {
        if( get_next_period( NextPeriod ) == 0  && !Game.Paused )
        {           
          game_process();

          NextPeriod += GAME_PERIOD;
        }
      }
    }
    if( !exit_game_session )
    {
      if( !strcmp( strFinished,str_You_win )) // New level
      {
        Game.CurLevel++;

        init_level( &Game );

        prepare_board();

        redraw_game_board();

        temp_time_between_cut = CRITICAL_TIME;

        show_new_level_dialog();

        exit_current_game = FALSE;

        strFinished=NULL;
      }
      else
      {
        if( end_of_game_status == HSR_RESTART ) // Reinitialize game
        {
          start_new_game();
        }
        else
        {
          if( end_of_game_status == HSR_INVITE )
          {
            exit_game_session = TRUE;
          }
          else
          {
            if( end_of_game_status == HSR_NONE )
            {
              Game.Paused = TRUE;

              exit_current_game = FALSE;

              redraw_game_board();
            }
            else
            {
              if( end_of_game_status == HSR_SHOW )
              {
                 menu_index = 0;
              }
              Game.Status = gsMenu;

              exit_current_game = FALSE;
            }
          }
        }
      }
    }
  }
  stop_application();

  return 0L;
}

/*
 * FUNCTION: start_new_game
 *
 * DESCRIPTION: Initializes struct of game, time, and draws the playfield.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void start_new_game( void )
{
  cEngine_Lock( &Screen );
   
  strFinished = NULL;

  init_game( &Game );

  Game.Status = gsPlay;

  exit_current_game = FALSE;

  NextPeriod = clock() + GAME_PERIOD;

  temp_time_between_cut = CRITICAL_TIME;

  TimeBetweenCuts = clock() + temp_time_between_cut;

  prepare_board();

  redraw_game_board();
   
  cEngine_Unlock( &Screen );

  show_new_level_dialog();
}

/*
 * FUNCTION: redraw_game_board
 *
 * DESCRIPTION: Redraws the CYBIKO screen.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void redraw_game_board( void )
{
  int i;

  DisplayGraphics_page_copy( MainModule.m_gfx, 1, 0, 
    0, 0, 160, 160 );
              
  DisplayGraphics_set_page( MainModule.m_gfx, 0 ); 

  // Set OR draw mode
  DisplayGraphics_set_draw_mode( MainModule.m_gfx, DM_OR );

  //Set bkcolor
  DisplayGraphics_set_bkcolor( MainModule.m_gfx, CLR_WHITE );

  // Draw Spider
  DisplayGraphics_draw_bitmap( MainModule.m_gfx,
    BitmapSequence_get_bitmap( &ElemsBmp, 1 ), Game.Spider.coord.x,
      Game.Spider.coord.y, BM_NORMAL );

  for( i = 0; i < MAX_KILLERS; i++ ) //Draw Killers
  {
    if( Game.Killer[ i ].type == ktActive )
    {
      DisplayGraphics_draw_bitmap( MainModule.m_gfx, 
        BitmapSequence_get_bitmap( &ElemsBmp, 0 ), 
        Game.Killer[ i ].coord.x * CELL_SIZE + Game.OriginKiller.x,
        Game.Killer[ i ].coord.y * CELL_SIZE + Game.OriginKiller.y, 
        BM_NORMAL );
    }
  }
  // Draw progress bar for remaining time till cut
  draw_progress_critical_time();
                                
  DisplayGraphics_show( MainModule.m_gfx ); // Show active page on screen
} 

/*
 * FUNCTION: show_new_level_dialog
 *
 * DESCRIPTION:
 *   Shows "next level" dialog before launching the next level of game play.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void show_new_level_dialog( void )
{
  struct cDialog dlg;

  char* sz_dialog_text;

  sz_dialog_text = (char *) malloc( 100 );

  play_sound( SOUND_NEWLEVEL );

  sprintf( sz_dialog_text, sz_level, Game.CurLevel + 1, 
    ( Game.CurLevel > 3 ) ? sz_endings[ 3 ] : sz_endings[ Game.CurLevel ],
      Game.PercentLeft );

  cDialog_ctor( &dlg, NULL, sz_dialog_text, mbOk, 
    0, MainModule.m_process );

  cDialog_ShowModal( &dlg );

  cDialog_dtor( &dlg, LEAVE_MEMORY );

  free( sz_dialog_text );

  NextPeriod = clock() + GAME_PERIOD;

  TimeBetweenCuts = clock() + temp_time_between_cut;
}

/*
 * FUNCTION: game_process
 *
 * DESCRIPTION: Processes messages and draws the game screen.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void game_process( void )
{
  cell_type tempSpiderPositionType; // Spider's previous position 

  PointType coordCurActiveBoard;

  PointType coordSpiderBeforeMove;
   
  if( SpiderPositionType != ctBusy ) 
  {
    current_dir = sdUnknown;
  }
  DirectKeyboard_scan( pKeyboard );

  if( DirectKeyboard_is_key_pressed( pKeyboard, KEY_UP ))
  {
     current_dir = sdUp;
  }
  else
  {
    if( DirectKeyboard_is_key_pressed( pKeyboard, KEY_DOWN ))
    {
      current_dir = sdDown;
    }
    else
    {
      if( DirectKeyboard_is_key_pressed( pKeyboard, KEY_LEFT ))
      {
        current_dir = sdLeft;
      }
      else
      {
        if( DirectKeyboard_is_key_pressed( pKeyboard, KEY_RIGHT ))
        {
          current_dir = sdRight;
        }
      }
    }
  }  
  tempSpiderPositionType = SpiderPositionType;

  coordSpiderBeforeMove.x = Game.Spider.coord.x;

  coordSpiderBeforeMove.y = Game.Spider.coord.y;
       
  SpiderPositionType = spider_move( &Game, current_dir,
    &coordCurActiveBoard );

  // The Spider crushes himself
  if( last_dir + current_dir == 5 && SpiderPositionType == ctBorder && 
      tempSpiderPositionType == ctBusy )
  {
    Game.Status = gsKilling;

    play_sound( SOUND_TAIL );
  }
  if( SpiderPositionType == ctBusy && 
    tempSpiderPositionType == ctBorder )
  {
    start_spider_coord.x = coordSpiderBeforeMove.x;

    start_spider_coord.y = coordSpiderBeforeMove.y;
  }
  last_dir = current_dir;

  if( Game.Status != gsKilling )
  {
    move_killers( &Game );
  }       
  temp_time_between_cut = get_next_period( TimeBetweenCuts );
       
  if( !temp_time_between_cut )
  {
    Game.Status = gsKilling;

    play_sound( SOUND_END_TIME );
  }       
  if( Game.Status == gsKilling )
  {
    if( --Game.Lifes < 0 ) // End of game
    {
      strFinished = str_You_lose;

      exit_current_game = TRUE;

      Game.Paused = TRUE;

      end_of_game_status =
        highscore_enter( &MainModule, Game.Score, HSM_CHECKSHOWALL );

      return;
    }
    else // Reinitialize level
    {         
      cEngine_Lock( &Screen );

      Game.Status = gsPlay;

      if( !( !temp_time_between_cut && (SpiderPositionType==ctFree ||
        SpiderPositionType==ctBorder || SpiderPositionType==ctUnknown)))
      {
        Game.Spider.coord.x = start_spider_coord.x;

        Game.Spider.coord.y = start_spider_coord.y; 
      }
      Game.LastCoordCell.x = ( SPIDER_SIZE + ( MAX_COLS / 2 ) * CELL_SIZE -
        ( SPIDER_SIZE - CELL_SIZE ) / 2 - 2 +
          SPIDER_SIZE / 2 - 1 ) / CELL_SIZE;

      Game.LastCoordCell.y = ( SPIDER_SIZE / 2 - 1 ) / CELL_SIZE;

      Game.Spider.dir = sdUnknown;

      SpiderPositionType = ctFree;

      last_dir = sdUnknown;

      update_game_field( &Game );

      prepare_board();

      NextPeriod = clock() + GAME_PERIOD;

      TimeBetweenCuts = clock() + CRITICAL_TIME;

      temp_time_between_cut = CRITICAL_TIME;

      redraw_game_board();

      cEngine_Unlock( &Screen );

      return;
    }
  }
  if( SpiderPositionType == ctBusy )
  {
    DisplayGraphics_set_page( MainModule.m_gfx, 1 );

    // Draw Spider's trace
    DisplayGraphics_draw_bitmap( MainModule.m_gfx,
      BitmapSequence_get_bitmap( &ElemsBmp, 2 ), 
      coordCurActiveBoard.x * CELL_SIZE + SPIDER_SIZE - CELL_SIZE,
      coordCurActiveBoard.y * CELL_SIZE +  SPIDER_SIZE - CELL_SIZE,
      BM_NORMAL );
  }
  else
  {
    if( SpiderPositionType == ctBorder && 
      tempSpiderPositionType == ctBusy ) // End of cut                                                                      
    {
      cEngine_Lock( &Screen );

      play_sound( SOUND_CUT );

      repair_board_after_cut();

      prepare_board();

      NextPeriod = clock() + GAME_PERIOD;

      TimeBetweenCuts = clock() + CRITICAL_TIME;

      temp_time_between_cut = CRITICAL_TIME;

      cEngine_Unlock( &Screen );
    }
  }
  redraw_game_board();
}

/*
 * FUNCTION: start_application
 *
 * DESCRIPTION:
 *   Initializes the application. Initializes the random
 *   number generator,  and the main structure of the game.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void start_application( void )
{
  if( cWinApp_has_focus( MainModule.m_process ))
  {
    exit_game_session = exit_current_game = FALSE;

    Game.Status = gsMenu;

    srand( (int) clock());

    // Initializes bitmap list.
    BitmapSequence_ctor_Ex( &ElemsBmp, "elems.pic" );

    DisplayGraphics_set_page( MainModule.m_gfx, 1 );

    // Set PUT draw mode
    DisplayGraphics_set_draw_mode( MainModule.m_gfx, DM_PUT );

    // Draw background
    DisplayGraphics_draw_bitmap( MainModule.m_gfx,
      BitmapSequence_get_bitmap( &ElemsBmp, 3 ), 0, 0, BM_NORMAL );

    // Get pointer to the direct keyboard object.
    pKeyboard = DirectKeyboard_get_instance(); 
                                                
    load_sound();

    load_menu();

    highscore_init( &MainModule );
  }
}

/*
 * FUNCTION: stop_application
 *
 * DESCRIPTION: Releases memory before closing the application.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void stop_application( void )
{
  // Destructor for BitmapSequence
  BitmapSequence_dtor( &ElemsBmp, LEAVE_MEMORY ); 

  // Destructor for DirectKeyboard
  DirectKeyboard_dtor( pKeyboard, FREE_MEMORY );  
                                                  
  release_sound();

  release_menu();

  highscore_cleanup( &MainModule );
}

/*
 * FUNCTION: line_fill
 *
 * DESCRIPTION: Recursively fills the Killer's segment.
 *
 * PARAMETERS: 
 *   x - abscissa of the point from which the fill begins
 *   y - ordinate of the point from which the fill begins
 *   dir - direction of fill
 *   PrevXl - abscissa of the previous fill's left point
 *   PrevXr - abscissa of the previous fill's right point
 *
 * RETURNS: Abscissa of the right border of the fill line
 */
int line_fill( int x, int y, int dir, int PrevXl, int PrevXr )
{
  int c, xl, xr = xl = x;
    
  do // Move to the left of the Killer area
  {
    c = Game.Field[ y ][ --xl ];
  }
  while( c == ctBusy || c == ctNextBusy );
    
  do // Move to the right of the Killer area
  {
    c = Game.Field[ y ][ ++xr ];
  }
  while( c == ctBusy || c == ctNextBusy );

  // A border will go here
  Game.Field[ y ][ xl ] = Game.Field[ y ][ xr ] = ctNextBorder;

  xl++, xr--;

  for( x = xl; x <= xr; x++ ) // Fill segment
  {
    Game.Field[ y ][ x ] = ctNextBusy; // Killer's area will go here
  }
  xl--, xr++;

  for( x = xl; x <= xr; x++ ) // Fill adjacent segments in the same direction
  {
    c = Game.Field[ y + dir ][ x ];

    if( c == ctBusy || c == ctNextBusy )
    {
      x = line_fill( x, y + dir, dir, xl, xr );
    }
    else
    {
      Game.Field[ y + dir ][ x ] = ctNextBorder;
    }
  }
  for( x = xl; x <= PrevXl; x++ )
  {
    c = Game.Field[ y - dir ][ x ];

    if( c == ctBusy || c == ctNextBusy )
    {
      x = line_fill( x, y - dir, -dir, xl, xr );
    }
    else
    {
      Game.Field[ y - dir ][ x ] = ctNextBorder;
    }
  }
  for( x = PrevXr; x <= xr; x++ )
  {
    c = Game.Field[ y - dir ][ x ];

    if( c == ctBusy || c == ctNextBusy )
    {
      x = line_fill( x, y - dir, -dir, xl, xr );
    }
    else
    {
      Game.Field[ y - dir ][ x ] = ctNextBorder;
    }
  }
  return xr;
}

/*
 * FUNCTION: repair_board_after_cut
 *
 * DESCRIPTION:
 *   Calculates the Spider's new area and updates the
 *   playfield after a cut.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void repair_board_after_cut( void )
{
  int i, j, Num = 0;

  for( i = 0; i < MAX_KILLERS; i++ )
  {
    if( Game.Killer[ i ].type == ktActive )
    {
      // If he stands alone
      if( Game.Field[ Game.Killer[ i ].coord.y ]
        [ Game.Killer[ i ].coord.x ] != ctNextBusy )                                                              
      {
        // Calculate the Spider's new area
        line_fill( Game.Killer[ i ].coord.x, Game.Killer[ i ].coord.y, 
          1, Game.Killer[ i ].coord.x, Game.Killer[ i ].coord.x );
      }
    }
  }
  for( i = 1; i < MAX_ROWS + 1; i++ ) // Update the playfield
  {
    for( j = 1; j < MAX_COLS + 1; j++ )
    {
      switch( Game.Field[ i ][ j ] )
      {
        case ctNextBorder:

          Game.Field[ i ][ j ] = ctBorder;

          break;

        case ctNextBusy:

          Game.Field[ i ][ j ] = ctBusy;

          Num++;

          break;

        default:

          Game.Field[ i ][ j ] = ctFree;

          break;
      }
    }
  }
  Game.NumCellMustCut -= ( Game.CurCellOnField - Num );

  Game.Score = Game.Score + ( Game.CurCellOnField - Num ) / 2;

  if( Game.NumCellMustCut <= 0 ) // If the Spider has cut all the cells
  {
    Game.NumCellMustCut = 0;

    strFinished = str_You_win; // Go to the next level

    exit_current_game = TRUE;
  }
  Game.CurCellOnField = Num;
}

/*
 * FUNCTION: prepare_board
 *
 * DESCRIPTION: Prepares the playfield before executing a full draw.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void prepare_board( void )
{
  int i, j;
   
  DisplayGraphics_set_page( MainModule.m_gfx, 1 );

  DisplayGraphics_set_draw_mode( MainModule.m_gfx, DM_PUT );

  DisplayGraphics_fill_screen( MainModule.m_gfx, CLR_WHITE );

  // Draw background
  DisplayGraphics_draw_bitmap( MainModule.m_gfx,
    BitmapSequence_get_bitmap( &ElemsBmp, 3 ), 0, 0, BM_NORMAL );
   
  for( i = 0; i < MAX_ROWS + 2; i++ ) // Draw playfield
  {
    for( j = 0; j < MAX_COLS + 2; j++ )
    {
      if( Game.Field[ i ][ j ] == ctBorder )
      {
        DisplayGraphics_draw_bitmap( MainModule.m_gfx, 
          BitmapSequence_get_bitmap( &ElemsBmp, 2 ),
            j * CELL_SIZE + SPIDER_SIZE - CELL_SIZE, 
              i * CELL_SIZE + SPIDER_SIZE - CELL_SIZE, BM_NORMAL );
      }
      else
      {
        if( Game.Field[ i ][ j ] == ctBusy )
        {
          draw_cell( j, i );
        }
      }
    }
  }
  for( i = 0; i < Game.Lifes; i++ ) // Display number of Spider lives
  {
    DisplayGraphics_draw_bitmap( MainModule.m_gfx, 
      BitmapSequence_get_bitmap( &ElemsBmp, 1 ), 148, 1 + 8 * i, 
        BM_NORMAL );
  }
  DisplayGraphics_draw_bitmap( MainModule.m_gfx,   // Draw background
    BitmapSequence_get_bitmap( &ElemsBmp, 5 ), 125, 78,
      BM_NORMAL );

  draw_statistics(); // Draw game statistics
}

/*
 * FUNCTION: draw_cell
 *
 * DESCRIPTION: Draws a cell from the Killer's area.
 *
 * PARAMETERS:
 *   x - abscissa of cell coordinate
 *   y - ordinate of cell coordinate
 *
 * RETURNS: nothing
 */
void draw_cell( int x, int y )
{
  DisplayGraphics_set_color( MainModule.m_gfx, CLR_WHITE );

  DisplayGraphics_fill_rect( MainModule.m_gfx,
    SPIDER_SIZE - CELL_SIZE + x * CELL_SIZE,
      SPIDER_SIZE - CELL_SIZE + y * CELL_SIZE, CELL_SIZE, CELL_SIZE );
}

/*
 * FUNCTION: draw_progress_bar
 *
 * DESCRIPTION: 
 *   Draws a line to represent how many cells the Spider has left to cut.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_progress_bar( void )
{
  long mustCut = ( MAX_ROWS - 2 ) * ( MAX_COLS - 2 ) * Game.PercentLeft / 100;

  int CurSizeBar;

  PointType barCoord;

  PointType barSize;
   
  barCoord.x = 147;

  barCoord.y = 28;

  barSize.x = 9;

  barSize.y = 48;

  // Calculate size of progress bar
  CurSizeBar = (int) (( barSize.y *
    ( mustCut - Game.NumCellMustCut )) / mustCut );
                                                         
  DisplayGraphics_set_draw_mode( MainModule.m_gfx, DM_XOR );

  if( CurSizeBar ) // If progress bar size equals 0, do not draw
  {
    DisplayGraphics_set_color( MainModule.m_gfx, CLR_BLACK );

    DisplayGraphics_fill_rect( MainModule.m_gfx, barCoord.x + 1, 
      barCoord.y + barSize.y - 1 - CurSizeBar,
        barSize.x - 2, CurSizeBar );
  }
  DisplayGraphics_set_draw_mode( MainModule.m_gfx, DM_PUT );

  DisplayGraphics_set_color( MainModule.m_gfx, CLR_BLACK );

  // Draw border of progress bar
  DisplayGraphics_draw_rect( MainModule.m_gfx,barCoord.x,
    barCoord.y - 2, barSize.x, barSize.y + 2 ); 
}

/*
 * FUNCTION: draw_progress_critical_time
 *
 * DESCRIPTION: 
 *   Draws a line to represent how much time remains before the cut.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_progress_critical_time( void )
{
  DisplayGraphics_set_color( MainModule.m_gfx, CLR_BLACK );

  Graphics_fill_rect( MainModule.m_gfx, 0, 97,
    (int)( temp_time_between_cut / 120 ), 99 );
}

/*
 * FUNCTION: draw_statistics
 *
 * DESCRIPTION: Draws the current game statistics.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_statistics( void )
{
  char OutText[ 6 ];

  int text_width;

  PointType coord;

  coord.x = 143;

  coord.y = 88;

  DisplayGraphics_set_color( MainModule.m_gfx, CLR_BLACK );

  Graphics_set_font( MainModule.m_gfx, mini_normal_font ); // Set mini font

  sprintf( OutText, "%d", Game.Score );

  text_width = DisplayGraphics_string_width( MainModule.m_gfx, OutText );
  
  Graphics_draw_text( MainModule.m_gfx, OutText, 
    coord.x - text_width / 2, coord.y );

  // Draw % symbol on the progress bar
  Graphics_draw_text( MainModule.m_gfx, "\%", 148, 48 ); 
                                                         
  draw_progress_bar(); // Draw progress bar
}
