/*
 * FILE: FoxHunt.c
 *
 * DESCRIPTION:
 *   Main module of Fox_Hunting.app application
 *
 *   Game Play:
 *
 *   Foxes are hidden randomly on the game board.
 *   The player's goal is to recover all of them.
 *   So he chooses a cell and takes a "shot".
 *   If there is a fox in the cell, it is retrieved.
 *   If there is no fox in the cell, a number is displayed in the cell.
 *   Since foxes can spot other foxes, this number represents the number of
 *   foxes you could see from that cell, if you were a fox.
 *   (This means in all directions - horizontal, vertical and both diagonals.)
 *   When the number is displayed, some cells become gray to prevent possible
 *   wrong moves.
 *   The quantity of moves is the basis for calculating the player's skills.
 *   The more moves to retrieve all foxes, the lower the player's skill index.
 *   If the player's skill index is higher than 70%, in each succeeding game
 *   the number of foxes is increased by one.
 *   So, let's begin...
 *
 * HISTORY:
 *   Sep 6, 2000 Created by Aleksey Matiushin
 */
#include "Foxhunt.h"

bool exit_game; // TRUE if current game is finished

bool exit_session; // TRUE if program is finished

bool focused; // TRUE if focused, else FALSE

int game_status; // status - continue/done

struct module_t main_module; // Application's main module

struct Graphics gfx_buf;

struct Bitmap bmp_buf;

struct BitmapSequence pic_seq;

// Board data
// Board fields with markers
// [  Y-rows  ][ X-columns]
int board_field[ BOARD_SIZE ][ BOARD_SIZE ];

// Vertical, horizontal, diagonal data, used to optimize calculations:
// Board_lines - data along Vertical, horizontal and both diagonals
// (total 6*BOARD_SIZE-2 elements in second dimension):
// First dimension:
// [0] - total number of foxes on the line
// [1] - number of found foxes on the line.
// [2] - if 1, this line has already been shown as disabled
// Second dimension:
// [0              : BOARD_SIZE-1]   - verticals,     BOARD_SIZE     elements
// [BOARD_SIZE     : 2*BOARD_SIZE-1] - horizontals,   BOARD_SIZE     elements
// [2*BOARD_SIZE   : 4*BOARD_SIZE-2] - diagonals "/", 2*BOARD_SIZE-1 elements
// [4*BOARD_SIZE-1 : 6*BOARD_SIZE-3] - diagonals "\", 2*BOARD_SIZE-1 elements
int board_lines[ 3 ][ 6 * BOARD_SIZE - 2 ];

int cursor_x, cursor_y; // Cursor coordinates IN CELLS

int fox_total; // Total number of foxes

int found_fox_count; // Number of foxes the user found

int move_number; // Number of moves made

int current_power; // Current power index in %

int current_game_number; // Number of the current game

char best_power[ FOXES_MAX-FOXES_MIN + 1 ];// Best Powers array

struct MSequence common_music; // Music for all

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  // Pointer to keyboard data/parameters
  struct KeyParam* ptr_key_param;

  // Initialize application
  init_module( &main_module );

  // Initialize game/application session
  init_session();

  // Play music intro in the background
  play_music( MUSIC_TITLE, TRUE );

  // Continue the application over game sets
  while( !exit_session )
  {
    init_game(); // Initialize the game set

    while( !exit_game )
    {
      bool esc_press, request_show_result, request_draw_board;

      bool request_set_field, request_set_result;

      struct Message* ptr_message =
        cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );

      // Request flags - down
      esc_press = request_show_result = request_draw_board =
        request_set_field = request_set_result = FALSE;

      switch( ptr_message->msgid )
      {
        case MSG_GOTFOCUS: // The getting of focus

          focused = TRUE;

          request_draw_board = TRUE;

          break;

        case MSG_LOSTFOCUS: // To process losing focus

          focused = FALSE;

          MSequence_stop( &common_music );

          do_vibrate( 0, 0 );

          break;

        case MSG_SHUTUP: // Signals for system exit
        case MSG_QUIT:

          exit_game = exit_session = TRUE;

          focused = FALSE;

          break;

        case MSG_KEYDOWN: // To process keyboard signals.

          ptr_key_param = Message_get_key_param( ptr_message );

          switch( ptr_key_param->scancode )
          {
            case KEY_LEFT:
            case KEY_RIGHT:
            case KEY_DOWN:
            case KEY_UP:

              if( game_status == GAME_CONTINUE &&
                move_cursor( ptr_key_param->scancode,
                  ptr_key_param->mask & KEYMASK_SHIFT ))
              {
                request_draw_board = TRUE;
              }
              break;

            case KEY_ENTER:

              // Miss auto-repeat. Set field
              if( !( ptr_key_param->mask & KEYMASK_AUTOREPEAT ) &&
                game_status == GAME_CONTINUE )
              {
                request_set_field = TRUE;
              }
              break;

            case KEY_ESC:

              // Miss auto-repeat
              if( !( ptr_key_param->mask & KEYMASK_AUTOREPEAT ))
              {
                esc_press = TRUE; // flag-up the esc key pressed
              }
              break;

            default:

              // Do not process this message
              // Process the "Help" key
              cWinApp_defproc( main_module.m_process, ptr_message );
          }
          break;

        default:

          // Do not process this message
          cWinApp_defproc( main_module.m_process, ptr_message );
      }
      // Delete message as soon as possible
      // all "take-long-time" we deal after deletion
      Message_delete( ptr_message );

      // after message deletion we may do anything...
      // Player makes a move
      if( request_set_field && set_field())
      {
        // Info part of the board
        refresh_info();

        // Check the end of the game
        game_status = check_winner();

        // Display "Success" if game is over
        if( game_status == GAME_DONE )
        {
          request_set_result = TRUE;

          say_game_over();
        }
        request_draw_board = TRUE;
      }
      // "Escape" pressed
      if( esc_press )
      {
        if( game_status == GAME_DONE )  // Game has been over before
        {
          request_show_result = TRUE;
        }
        else
        {
          if( make_dialog( str_Really_exit, mbQuit | mbCancel | mbs1, "" ) ==
            mrQuit )
          {
            exit_game = exit_session = TRUE;
          }
        }
      }
      // Display from back to front and reset cursor to a new place
      if( focused && request_draw_board ) draw_board();

      // Set result - "Game Over"
      if( request_set_result )
      {
        set_results();

        request_show_result = TRUE;
      }
      // Show results
      if( focused && request_show_result )
      {
        int answer = show_result();

        if( answer == mrQuit )
        {
          exit_game = exit_session = TRUE;
        }
        else if( answer == mrRestart )
        {
          exit_game = TRUE;
        }
      }
    }
  }
  clear_session(); // Clear session data

  return 0L; // Do not cycle
}

/*
 * FUNCTION: init_session
 *
 * DESCRIPTION: This function initializes the whole game session
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void init_session( void )
{
  focused = exit_session = FALSE;

  current_power = current_game_number = 0;

  fox_total = FOXES_MIN;

  // Fill array with Best Results
  fill_best_powers();

  // Init backstage graphics
  Bitmap_ctor_Ex2( &bmp_buf, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );

  Graphics_ctor_Ex( &gfx_buf, &bmp_buf );

  BitmapSequence_ctor_Ex( &pic_seq, PICTURES_NAME );

  MSequence_ctor( &common_music, MUSIC_TITLE );
}

/*
 * FUNCTION: clear_session
 *
 * DESCRIPTION: Clear session data
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void clear_session( void )
{
  store_best_powers(); // Store Best Results array

  Bitmap_dtor( &bmp_buf, LEAVE_MEMORY );

  Graphics_dtor( &gfx_buf, LEAVE_MEMORY );

  BitmapSequence_dtor( &pic_seq, LEAVE_MEMORY );

  // Destructor of all-purpose MSequence
  MSequence_dtor( &common_music, LEAVE_MEMORY );
}

/*
 * FUNCTION: init_game
 *
 * DESCRIPTION: this function initializes each game
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void init_game( void )
{
  // Increase the number of foxes by 1
  if( got_next_level()) fox_total++; // (obviously not for the 1st game...)

  if( fox_total > FOXES_MAX ) fox_total = FOXES_MAX;

  if( fox_total < FOXES_MIN ) fox_total = FOXES_MIN;

  game_status = GAME_CONTINUE;

  exit_game = FALSE; // Start game

  current_game_number++; // Current game number

  found_fox_count = // Number of foxes found
    move_number = // Move counter
    current_power = 0; // Current power

  // Cursor's coordinates IN CELLS
  cursor_x = cursor_y = ( BOARD_SIZE - 1 ) / 2;

  // Clean the board for a game
  make_clear_board();

  // Clean all game data and init foxes randomly
  clear_game_data();

  // Display status information on a clean board
  refresh_info();

  // Draw the game board
  draw_board();
}

/*
 * FUNCTION: make_clear_board
 *
 * DESCRIPTION: Clear the board for a new game
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void make_clear_board( void )
{
  int i, j;

  // (unlike other bitmaps, this one is not global;
  // it is constructed, used and destroyed here)
  struct Bitmap pic_panel;

  Graphics_set_bkcolor( &gfx_buf, CLR_WHITE );

  Graphics_set_draw_mode( &gfx_buf, DM_PUT );

  Graphics_fill_screen( &gfx_buf, CLR_WHITE );

  // Draw empty cells on the playing board
  for( i = 0; i < BOARD_SIZE; i++ )
  {
    for( j = 0; j < BOARD_SIZE; j++ )
    {
      Graphics_draw_bitmap( &gfx_buf,
        BitmapSequence_get_bitmap( &pic_seq, PIC_CELLFREE ),
          i * CELL_SIZE, j * CELL_SIZE, DM_PUT );
    }
  }
  // Draw the info-panel part of the screen
  Graphics_draw_bitmap( &gfx_buf,
    BitmapSequence_get_bitmap( &pic_seq, PIC_PANEL ),
      BOARD_SIZE * CELL_SIZE + 1, 0, DM_PUT );

  // Draw static messages on the info panel
  Graphics_set_color( &gfx_buf, CLR_BLACK );

  Graphics_set_font( &gfx_buf, mini_bold_font );

  // Total number of foxes
  Graphics_draw_text( &gfx_buf, "Foxes", 108, 11 );

  // "Foxes found" counter
  Graphics_draw_text( &gfx_buf, "Found", 108, 24 );

  // Number of moves
  Graphics_draw_text( &gfx_buf, "Moves", 108, 37 );

  // Current power index
  Graphics_set_font( &gfx_buf, cool_bold_font );

  Graphics_draw_text( &gfx_buf, "Power", 115, 65 );
}

/*
 * FUNCTION: clear_game_data
 *
 * DESCRIPTION: Clean board data in memory and initialize foxes
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void clear_game_data( void )
{
  int i, j, t;

  // Clear the board's field markers
  for( j = 0; j < BOARD_SIZE; j++ )
  {
    for( i = 0; i < BOARD_SIZE; i++ )
    {
      board_field[ j ][ i ] = MARK_EMPTY_FIELD;
    }
  }
  // Clear data about verticals, horizontals, and diagonals
  for( j = 0; j < 3; j++ )
  {
    for( i = 0; i < 6 * BOARD_SIZE - 2; i++ )
    {
      board_lines[ j ][ i ] = 0;
    }
  }
  // Initiate fox_total of fresh, live young foxes
  for( t = 0; t < fox_total; t++ )
  {
    get_random_empty( &i, &j );

    board_field[ j ][ i ] = MARK_ALIVE_FOX;

    // Vertical( x ), Horizontal( y ), diag1( X+Y ), diag2( Y-X+BOARD_SIZE-1 )
    board_lines[ TOTAL_INDEX ][ VER_INDEX( i, j ) ]++;

    board_lines[ TOTAL_INDEX ][ HOR_INDEX( i, j ) ]++;

    board_lines[ TOTAL_INDEX ][ DG1_INDEX( i, j ) ]++;

    board_lines[ TOTAL_INDEX ][ DG2_INDEX( i, j ) ]++;
  }
}

/*
 * FUNCTION: refresh_info
 *
 * DESCRIPTION:
 *   Draw information about player's moves and marks
 *   ( ! ) Note: we do it on the back page!
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void refresh_info( void )
{
  char num_text[ 10 ];

  // Clean old data off the info panel
  Graphics_set_color( &gfx_buf, CLR_LTGRAY );

  Graphics_fill_rect( &gfx_buf, 143, 7, 15, 50 );

  Graphics_fill_rect( &gfx_buf, 107, 76, 47, 19 );

  Graphics_set_color( &gfx_buf, CLR_BLACK );

  Graphics_set_font( &gfx_buf, mini_bold_font );

  // Total number of foxes
  sprintf( num_text, "%d", fox_total );

  Graphics_draw_text( &gfx_buf, num_text, 145, 11 );

  // "Foxes found" counter
  sprintf( num_text, "%d", found_fox_count );

  Graphics_draw_text( &gfx_buf, num_text, 145, 24 );

  // Number of moves
  sprintf( num_text, "%d", move_number );

  Graphics_draw_text( &gfx_buf, num_text, 145, 37 );

  // "Current total" power index
  Graphics_set_font( &gfx_buf, cool_bold_font );

  sprintf( num_text, "%d%%", calc_current_power());

  Graphics_draw_text( &gfx_buf, num_text, 121, 79 );
}

/*
 * FUNCTION: get_random_empty
 *
 * DESCRIPTION: Get a random empty cell
 *
 * PARAMETERS:
 *   x -
 *   y -
 *
 * RETURNS: nothing
 */
void get_random_empty( int *x, int *y )
{
  do
  {
    *x = (int) random( BOARD_SIZE );

    *y = (int) random( BOARD_SIZE );

    if( *x > BOARD_SIZE - 1 ) *x = BOARD_SIZE - 1;

    if( *y > BOARD_SIZE - 1 ) *y = BOARD_SIZE - 1;
  }
  while( board_field[ *y ][ *x ] != MARK_EMPTY_FIELD );
}

/*
 * FUNCTION: move_cursor
 *
 * DESCRIPTION: to move the player's cursor in the chosen direction
 *
 * PARAMETERS:
 *   direction -
 *   fastmode -
 *
 * RETURNS: -
 */
bool move_cursor( int direction, int fastmode )
{
  int k, x = cursor_x, y = cursor_y, dx, dy = dx = 0;

  bool place_is_found = FALSE;

  // do not touch the cursor if don't need to.
  if( !cWinApp_has_focus( main_module.m_process )
    || exit_game || exit_session )
  {
    return FALSE;
  }
  // Set changing delta of x( columns ) and y( rows )
  switch( direction )
  {
    case KEY_LEFT:

      dx = -1;

      break;

    case KEY_UP:

      dy = -1;

      break;

    case KEY_RIGHT:

      dx = 1;

      break;

    case KEY_DOWN:

      dy = 1;

      break;
  }
  // Find an empty place in the chosen direction, max BOARD_SIZE-1 times
  for( k = 0; !place_is_found && k < BOARD_SIZE - 1; k++ )
  {
    x += dx;

    y += dy;

    if( x < 0 ) x = BOARD_SIZE - 1;

    if( x >= BOARD_SIZE ) x = 0;

    if( y < 0 ) y = BOARD_SIZE - 1;

    if( y >= BOARD_SIZE ) y = 0;

    place_is_found = ( board_field[ y ][ x ] == MARK_EMPTY_FIELD ||
      board_field[ y ][ x ] == MARK_ALIVE_FOX || !fastmode )
        ? TRUE : FALSE;

    if( k == 0 ) // If an empty place is not found, it will be TRUE
    {
      cursor_x = x;

      cursor_y = y;
    }
  }
  // If an empty place is found - mostly for fastmode
  if( place_is_found )
  {
    cursor_x = x;

    cursor_y = y;
  }
  return TRUE;
}

/*
 * FUNCTION: set_field
 *
 * DESCRIPTION:
 *   look up the field the user has chosen and set it up in an
 *   appropriate manner
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
bool set_field( void )
{
  int *field;

  if( cursor_x < 0 || cursor_y < 0 ||
    cursor_x > BOARD_SIZE - 1 || cursor_y > BOARD_SIZE - 1 )
  {
    return FALSE;
  }
  // To simplify usage, we refer to a single cell as a  'field'
  field = board_field[ cursor_y ] + cursor_x;

  if( *field != MARK_EMPTY_FIELD && *field != MARK_ALIVE_FOX )
  {
    return FALSE;
  }
  if( *field == MARK_EMPTY_FIELD )
  {
    char num_text[ 10 ];

    // Calculate how many foxes are visible from here
    *field = board_lines[ TOTAL_INDEX ][ VER_INDEX( cursor_x, cursor_y ) ] +
      board_lines[ TOTAL_INDEX ][ HOR_INDEX( cursor_x, cursor_y ) ] +
        board_lines[ TOTAL_INDEX ][ DG1_INDEX( cursor_x, cursor_y ) ] +
          board_lines[ TOTAL_INDEX ][ DG2_INDEX( cursor_x, cursor_y ) ];

    // Fill the field with a "disabled" picture
    Graphics_draw_bitmap( &gfx_buf,
      BitmapSequence_get_bitmap( &pic_seq, PIC_CELLDISB ),
        ( cursor_x * CELL_SIZE ) + 1, ( cursor_y * CELL_SIZE ) + 1, DM_PUT );

    // Show this number in the cell
    sprintf( num_text, "%d", *field );

    Graphics_set_font( &gfx_buf, mini_bold_font );

    Graphics_set_color( &gfx_buf, CLR_BLACK );

    Graphics_draw_text( &gfx_buf, num_text, cursor_x * CELL_SIZE + 3,
       cursor_y * CELL_SIZE + 2 );
  }
  else
  {
    // Mark the field
    *field = MARK_DEAD_FOX;

    // Display fox on the desk
    Graphics_draw_bitmap( &gfx_buf,
      BitmapSequence_get_bitmap( &pic_seq, PIC_CELLFOX ),
        cursor_x * CELL_SIZE, cursor_y * CELL_SIZE, DM_PUT );

    // Increase the counters in the "found foxes" fields
    //  in the board_lines array
    board_lines[ FOUND_INDEX ][ VER_INDEX( cursor_x, cursor_y ) ]++;

    board_lines[ FOUND_INDEX ][ HOR_INDEX( cursor_x, cursor_y ) ]++;

    board_lines[ FOUND_INDEX ][ DG1_INDEX( cursor_x, cursor_y ) ]++;

    board_lines[ FOUND_INDEX ][ DG2_INDEX( cursor_x, cursor_y ) ]++;

    // Increase the counter by the number of foxes that have been found
    // and check the end of the game
    ++found_fox_count;

    // Music for "Found Fox" event
    play_music( MUSIC_ON_FOX, FALSE );

    // Do short vibration
    do_vibrate( 3, 150 );
  }
  move_number++; // Counter for number of moves in this game

  // Search a board to disable some lines on it
  try_disable_lines( cursor_x, cursor_y );

  return TRUE;
}

/*
 * FUNCTION: say_game_over
 *
 * DESCRIPTION: Display "success" on the game board
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void say_game_over( void )
{
  if( game_status == GAME_DONE )
  {
    Graphics_set_color( &gfx_buf, CLR_BLACK );

    Graphics_fill_rect( &gfx_buf, 104, 47, 54, 12 );

    Graphics_set_color( &gfx_buf, CLR_WHITE );

    Graphics_set_font( &gfx_buf, cool_bold_font );

    Graphics_draw_text( &gfx_buf, "Success", 107, 47 );
  }
}

/*
 * FUNCTION: do_vibrate
 *
 * DESCRIPTION:
 *   Vibrate
 *   Levels 1-7
 *   Duration in milliseconds
 *
 * PARAMETERS:
 *   level -
 *   duration -
 *
 * RETURNS: nothing
 */
void do_vibrate( int level, int duration )
{
  if( !focused || level < 0 || duration < 0 ) return;

  if( level > 7 ) level = 7;

  // Vibrate() uses interval [ 0...255 ] and calls set_vibrate with index/32
  vibrate( level * 32 );

  cWinApp_pause( main_module.m_process, duration );

  vibrate( 0 );
}

/*
 * FUNCTION: try_disable_lines
 *
 * DESCRIPTION:
 *   Search the board and disable appropriate lines if they should be disabled
 *   We name this function in one of two ways:
 *   either we put a number or a fox-mark within the cell( x, y ).
 *   THIS FUNC IS CALLED FOR BACK PAGE ONLY
 *
 * PARAMETERS:
 *   x -
 *   y -
 *
 * RETURNS: nothing
 */
void try_disable_lines( int x, int y )
{
  bool was_disabled = FALSE;

  if( board_field[ y ][ x ] == MARK_DEAD_FOX )
  {
    int i, cx, cy, dx, dy;

    // Look through vert, horiz, and both diagonal directions
    for( i = 0; i < 4; i++ )
    {
      // to prepare for "looking" direction
      if( set_search_param( x, y, i, &cx, &cy, &dx, &dy ))
      {
        // and "looking thru" direction
        for( ; cx >= 0 && cy >= 0 && cx < BOARD_SIZE &&
          cy < BOARD_SIZE; cx += dx, cy += dy )
        {
          if( board_field[ cy ][ cx ] > MARK_EMPTY_FIELD
            && search_cell( cx, cy ))
          {
            was_disabled = TRUE;
          }
        }
      }
    }
  }
  else was_disabled = search_cell( x, y );

  if( was_disabled ) play_music( MUSIC_ON_DISABLE, FALSE );
}

/*
 * FUNCTION: set_search_param
 *
 * DESCRIPTION: Set searching parameters for line
 *
 * PARAMETERS:
 *   x, y( in ) : cell coordinates
 *   drct( in ) : 0 - vert; 1 - horiz; 2 - diag"/"; 3 - diag2"\"
 *   sx-start x; sy-start y; lx-step x; ly-step y
 *
 * RETURNS: -
 */
bool set_search_param( int px, int py, int drct,
  int *sx, int *sy, int *lx, int *ly )
{
  bool ok = TRUE;

  switch( drct )
  {
    case 0: // Vertical

      *sx = px;

      *sy = *lx = 0;

      *ly = 1;

      break;

    case 1: // Horizontal

      *sx = *ly = 0;

      *sy = py;

      *lx = 1;

      break;

    case 2: // Diag 1 "/"

      *sx = imax( 0, px + py - BOARD_SIZE + 1 );

      *sy = imin( BOARD_SIZE - 1, px + py );

      *lx = 1;

      *ly = -1;

      break;

    case 3:   // Diag 2 "\"

      *sx = imax( 0, px - py );

      *sy = imax( 0, py - px );

      *lx = *ly = 1;

      break;

    default:

      *sx = *sy = BOARD_SIZE;

      *lx = *ly = 0;

      ok = FALSE;
  }
  return ok;
}

/*
 * FUNCTION: search_cell
 *
 * DESCRIPTION:
 *   Search the concrete cell to disable any lines that cross it
 *   ( this func does not change meaningful data; it just serves the user )
 *   executed only within Back page, already set
 *
 * PARAMETERS:
 *   x -
 *   y -
 *
 * RETURNS: -
 */
bool search_cell( int x, int y )
{
  int ii[ 4 ], thisvalue = board_field[ y ][ x ], totalopened;

  bool waswas = FALSE;

  if( thisvalue < 0 )
  {
    return FALSE;
  }
  else
  {
    // Get indexes of appropriate places in all( 4 ) directions
    ii[ 0 ] = VER_INDEX( x, y );

    ii[ 1 ] = HOR_INDEX( x, y );

    ii[ 2 ] = DG1_INDEX( x, y );

    ii[ 3 ] = DG2_INDEX( x, y );

    // Count how many foxes ALREADY FOUND are visible from this point
    // If this value is equal to the number of ALL visible foxes,
    // gray-down the empty cells that cross this point in all directions
    // ... note that values may be zero!
    totalopened = board_lines[ FOUND_INDEX ][ ii[ 0 ]] +
      board_lines[ FOUND_INDEX ][ ii[ 1 ]] +
        board_lines[ FOUND_INDEX ][ ii[ 2 ]] +
          board_lines[ FOUND_INDEX ][ ii[ 3 ]];

    if( thisvalue == totalopened )
    {
      int cx, cy, dx, dy , k, l;

      // If scan is still enabled, set disabling marks
      // on all 4 directions
      for( k = 0; k < 4; k++ )
      {
        // If scan is still not disabled, prepare to look in
        //the direction of the scan
        if( board_lines[ DISABLE_INDEX ][ ii[ k ]] == 0  &&
          set_search_param( x, y, k, &cx, &cy, &dx, &dy ))
        { // Search along the direction
          for( ; cx >= 0 && cy >= 0 && cx < BOARD_SIZE &&
            cy < BOARD_SIZE; cx += dx, cy += dy )
          {
            if( board_field[ cy ][ cx ] == MARK_EMPTY_FIELD ) // Cell is empty
            {
              // Fill the field with light gray
              Graphics_draw_bitmap( &gfx_buf,
                BitmapSequence_get_bitmap( &pic_seq, PIC_CELLDISB ),
                  ( cx * CELL_SIZE ) + 1, ( cy * CELL_SIZE ) + 1, DM_PUT );

              waswas = TRUE;
            }
          }
          // Mark the direction as disabled
          board_lines[ DISABLE_INDEX ][ ii[ k ]] = 1;
        }
      }
    }
  }
  return waswas;
}

/*
 * FUNCTION: calc_current_power
 *
 * DESCRIPTION: -
 *
 * PARAMETERS: none
 *
 * RETURNS: the current power of user's mind
 */
int calc_current_power( void )
{
  long s0, s1, pwr;

  if( move_number > 0 )
  {
    s0 = BOARD_SIZE * BOARD_SIZE - move_number -
      fox_total + found_fox_count;

    s1 = BOARD_SIZE * BOARD_SIZE - fox_total - 1;

    pwr = ( s0 * s0 * 100 ) / ( s1 * s1 );

    if( pwr > 100 )
    {
      pwr = 100;
    }
    else if( pwr < 0 )
    {
      pwr = 0;
    }
  }
  else pwr = 0;

  return (int) pwr;
}

/*
 * FUNCTION: draw_board
 *
 * DESCRIPTION:
 *   Redrawing the screen
 *   !!! NOTE: ONLY THIS FUNCTION COPIES THE BACK PAGE TO THE FRONT PAGE !!!
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_board( void )
{
  // Set front page
  DisplayGraphics_set_page( main_module.m_gfx, FRONT_PAGE );

  if( focused && !exit_session && cWinApp_has_focus( main_module.m_process ))
  {
    // DIRECT! FAST! Copy from back graphics page to front graphics page.
    DisplayGraphics_put_background( main_module.m_gfx,
      Graphics_get_buf_addr( &gfx_buf ));

    // Draw a cursor on the current graphics page. The cursor is the icon
    // that is transparent(white) in color
    DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_WHITE );

    DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

    if( cWinApp_has_focus( main_module.m_process ))
    {
      draw_lib( DEFAULT_PICTURES, PIC_CURSOR,
        cursor_x * CELL_SIZE, cursor_y * CELL_SIZE, BM_NORMAL );

      // Show the current page
      if( cWinApp_has_focus( main_module.m_process ))
      {
        DisplayGraphics_show( main_module.m_gfx );
      }
    }
  }
}

/*
 * FUNCTION: check_winner
 *
 * DESCRIPTION: Look for the game result
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
int check_winner( void )
{
  return found_fox_count == fox_total ? GAME_DONE : GAME_CONTINUE;
}

/*
 * FUNCTION: set_results
 *
 * DESCRIPTION: Set a result for the current game
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void set_results( void )
{
  current_power = calc_current_power(); // This game's power index
}

/*
 * FUNCTION: show_result
 *
 * DESCRIPTION: Draw the current game's result
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
int show_result( void )
{
  int answ, nextfoxes;

  char sbest[ 50 ], snext[ 50 ];

  bool isbest = FALSE;

  char* resmes = malloc( 100 );

  // If the best power level is reached
  if( current_power > (int) best_power[ fox_total - FOXES_MIN ] )
  {
    best_power[ fox_total - FOXES_MIN ] = (char) current_power;

    sprintf( sbest, "%d%% is the BEST RESULT for %d Foxes!\n",
      current_power, fox_total );

    isbest = TRUE;

    play_music( MUSIC_SUCCESS, FALSE );
  }
  else sbest[ 0 ] = 0;

  if( got_next_level()) // Reach next level or not?
  {
    if( !isbest ) play_music( MUSIC_SUCCESS, FALSE );

    nextfoxes = imin( fox_total + 1, FOXES_MAX );

    if( nextfoxes > fox_total )
    {
      sprintf( snext, "%s LEVEL is reached:\n%d Foxes",
        ( nextfoxes == FOXES_MAX ) ? "HIGHEST" : "NEXT", nextfoxes );
    }
    do_vibrate( 5, 300 );
  }
  else
  {
    if( !isbest ) play_music( MUSIC_SUCCESS, FALSE );

    sprintf( snext, fox_total == FOXES_MAX ?
      "You have been on the HIGHEST level already..." :
      "Try the same level again..." );
  }
  sprintf( resmes, "%s%s", sbest, snext );

  answ = (int) make_dialog( resmes, mbRestart | mbQuit | mbReverseOrder, "" );

  free( resmes );

  return answ; // mrQuit, mrRestart may be returned
}

/*
 * FUNCTION: got_next_level
 *
 * DESCRIPTION: Check if the user got to the next level
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
bool got_next_level( void )
{
  return current_power > NEXT_LEVEL_POWER && fox_total < FOXES_MAX;
}

/*
 * FUNCTION: make_dialog
 *
 * DESCRIPTION: Standard Dialogue
 *
 * PARAMETERS:
 *   prompt -
 *   style  -
 *   musics -
 *
 * RETURNS: -
 */
int make_dialog( char *prompt, long style, char* musics )
{
  struct cDialog dialog;

  int dlg_res;

  if( *musics ) play_music( musics, TRUE ); // Play musical accompaniment

  // Set to front page
  DisplayGraphics_set_page( main_module.m_gfx, FRONT_PAGE );

  // Constructor
  cDialog_ctor( &dialog, NULL, prompt, style, 0, main_module.m_process );

  // Execute a dialogue
  dlg_res = cDialog_ShowModal( &dialog );

  // Destructor
  cDialog_dtor( &dialog , LEAVE_MEMORY );

  // Clear dialogue window from the screen
  draw_board();

  return dlg_res;
}

/*
 * FUNCTION: play_music
 *
 * DESCRIPTION: Play music
 *
 * PARAMETERS:
 *   musfile -
 *   inback  -
 *
 * RETURNS: nothing
 */
void play_music( char *musfile, bool inback )
{
  MSequence_dtor( &common_music , LEAVE_MEMORY );

  MSequence_ctor( &common_music, musfile );

  if( inback )
  {
    MSequence_play_background( &common_music );
  }
  else
  {
    MSequence_play( &common_music );
  }
}

/*
 * FUNCTION: fill_best_powers
 *
 * DESCRIPTION: Fill array with Best Results
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void fill_best_powers( void )
{
  int i, n = FOXES_MAX - FOXES_MIN + 1; // Max size of Best Results array

  struct Input* inp;

  // init ALL Best Scores with 0
  for ( i = 0; i < n; i++ ) best_power[ i ] = 0;

  inp = Archive_open_Ex( main_module.m_process->module->archive,
    BESTS_ARCNAME );

  Input_read( inp, best_power, n );

  Input_dtor( inp, FREE_MEMORY );
}

/*
 * FUNCTION: store_best_powers
 *
 * DESCRIPTION: Store array of Best Results to app's archive
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void store_best_powers( void )
{
  int i, n = FOXES_MAX-FOXES_MIN + 1; // Max size of Best Results array

  struct Output* out =
    Archive_open_write_Ex( main_module.m_process->module->archive,
    BESTS_ARCNAME );

  Output_write( out, best_power, n );

  Output_dtor( out, FREE_MEMORY );
}
