/*
 * FILE: Explosion.c
 *
 * DESCRIPTION: Implementation of the big explosion effect
 *
 * HISTORY:
 *   Nov 17, 2000 Created by Aleksey Slesarev
 */
#include "Explosion.h"
#include "Utils.h"
#include "Screen.h"
#include "Vehicles.h"
#include "Main.h"

struct BigExplosion big_explosion;

struct BitmapSequence  big_explosion_images; // Images of the big explosion

/*
 * FUNCTION: init_big_explosion
 *
 * DESCRIPTION:
 *   Initializes the big explosion (after collision with the tank truck)
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_big_explosion( void )
{
  big_explosion.current_state = 0;

  big_explosion.w = big_explosion.h = 64;

  BitmapSequence_ctor_Ex( &big_explosion_images, "big_explosion.pic" );
}

/*
 * FUNCTION: release_big_explosion
 *
 * DESCRIPTION:
 *   Releases the resources of the big explosion
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_big_explosion( void )
{
  BitmapSequence_dtor( &big_explosion_images, LEAVE_MEMORY );
}

/*
 * FUNCTION: perform_big_explosion
 *
 * DESCRIPTION: Puts the big explosion on the screen
 *
 * PARAMETERS:
 *   ptr_vehicle_destination -
 *
 * RETURNS: nothing
 */
void perform_big_explosion( struct rect_t* ptr_vehicle_destination )
{
  if( !big_explosion.current_state )
  {
    play_sound( SOUND_EXPLOSION );

    big_explosion.current_state = 1;

    big_explosion.x = ptr_vehicle_destination->x - 7;

    big_explosion.y = ptr_vehicle_destination->y - 21;
  }
}

/*
 * FUNCTION: calculate_big_explosion
 *
 * DESCRIPTION: 
 *   Updates the form and coordinates of the big explosion
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void calculate_big_explosion( void )
{
  if( big_explosion.current_state )
  {
    struct rect_t test_rectangle;

    big_explosion.x -= scroll_velocity;

    // Checks if the big explosion interacts with player's vehicle
    if( rect_and( &test_rectangle, &big_explosion, &player_vehicle ))
    {
      explosion( &player_vehicle );
    }
    // Moves the big explosion
    big_explosion.current_state++;

    big_explosion.current_state %= 11; 
  }
}

/*
 * FUNCTION: draw_big_explosion
 *
 * DESCRIPTION: Draws the big explosion on the screen
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_big_explosion( void )
{
  if( big_explosion.current_state )
  {
    struct Bitmap* ptr_current_bitmap = 
      BitmapSequence_get_bitmap( &big_explosion_images,
       ( big_explosion.current_state - 1 ) / 2 );
    
    // Draws one quarter of the picture
    DisplayGraphics_draw_bitmap( main_module.m_gfx, ptr_current_bitmap,
      big_explosion.x, big_explosion.y, BM_NORMAL );

    // Flips one quarter of the picture over the X-axis and draws it
    DisplayGraphics_draw_bitmap( main_module.m_gfx, ptr_current_bitmap,
      big_explosion.x + 32, big_explosion.y, BM_FLIP_X );

    // Flips one quarter of the picture over the Y-axis and draws it
    DisplayGraphics_draw_bitmap( main_module.m_gfx, ptr_current_bitmap,
      big_explosion.x, big_explosion.y + 32, BM_FLIP_Y );

    // Flips one quarter of the picture over bouth the X-axis and 
    // the Y-axis  and draws it
    DisplayGraphics_draw_bitmap( main_module.m_gfx, ptr_current_bitmap,
      big_explosion.x + 32, big_explosion.y + 32, BM_FLIP_Y | BM_FLIP_X );
  }
}
