/*
 * FILE: Main.c
 *
 * DESCRIPTION: Application's main module
 *
 * HISTORY:
 *   Nov 17, 2000 Created by Aleksey Slesarev
 */
#include "Main.h"
#include "Utils.h"
#include "Vehicles.h"
#include "Screen.h"  

static const char* sz_main_menu_text[ 3 ] = 
{ 
  "Start Game", "High Scores", "Exit" 
};

static const char* sz_month[ 12 ] =
{
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"  
};

struct module_t main_module; // Application's main module

struct BitmapSequence menu_images; // Intro screen images

struct cList main_menu; // Main menu for the game

struct score_t high_scores[ 10 ]; // High scores for the game

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  bool exit_game = FALSE;

  // Initializes the application
  init_module( &main_module );

  // Initializes the main menu  
  init_menu();

  // Initializes the game scene
  init_scene();

  // Initializes the High Scores archive
  init_high_scores();

  // Initializes sound effects
  load_sound();

  while( !exit_game )
  {
    // Main menu for the game
    switch( select_action())
    {
      case MENU_HIGH_SCORES: // Show the high scores

        show_high_scores();

        break;

      case MENU_EXIT: // Exit the game 

        exit_game = TRUE;

        break;

      default:

        exit_game = game_process(); // Start the game
     }
  }
  // Performs cleanup
  release_sound();

  release_high_scores();

  release_scene();

  release_menu();

  return 0L;
}

/*
 * FUNCTION: init_menu
 *
 * DESCRIPTION: Initializes the main menu for the game.
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_menu( void )
{
  int index;

  // Loads menu images from the "menu.pic" archive
  BitmapSequence_ctor_Ex( &menu_images, "menu.pic" );

  // Creates a menu object
  cList_ctor( &main_menu, 73 );

  // Fills menu object with item objects
  for( index = 0; index < 3; index++ ) 
  {
    struct cItem* ptr_menu_item = malloc( sizeof *ptr_menu_item );

    cItem_ctor( ptr_menu_item, 73, sz_main_menu_text[ index ],
      FALSE, NULL, NULL );

    cList_AddItem( &main_menu, ptr_menu_item );
  }
}

/*
 * FUNCTION: release_menu
 *
 * DESCRIPTION: Releases the resources of the main menu
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_menu( void )
{
  cList_dtor( &main_menu, LEAVE_MEMORY );

  BitmapSequence_dtor( &menu_images, LEAVE_MEMORY );
}

/*
 * FUNCTION: select_action
 *
 * DESCRIPTION: Shows the main menu for the game
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
int select_action( void )
{  
  int result = MENU_EXIT;

  int road_coordinate = 0;

  int builds_coordinate = 0;

  bool exit_menu = FALSE;
  
  // Turns on the menu melody
  play_sound( SOUND_MENU );

  // Adds a menu object to the screen
  cWinApp_AddObj( main_module.m_process, &main_menu, 44, 16 );

  // Message processing
  while( !exit_menu )
  {
    int index;

    struct Message* ptr_message =
      cWinApp_get_message( main_module.m_process, 1, 1, MSG_USER );

    if( ptr_message ) // Real message received 
    {
      switch( ptr_message->msgid )
      {
        case MSG_SHUTUP: // Processes system exit signals
        case MSG_QUIT:

          exit_menu = TRUE;

          break;

        case MSG_LOSTFOCUS: // Application has lost focus

          MSequence_stop( game_sounds + background_sound );

          cList_Hide( &main_menu );

          break;

        case MSG_GOTFOCUS: // Application has focus

          MSequence_play_background( game_sounds + background_sound );

          cList_Show( &main_menu );

          break;

        case MSG_KEYDOWN: // Processes the keyboard

          if( Message_get_key_param( ptr_message )->mask & KEYMASK_AUTOREPEAT )
          {
            break;
          }
          switch( Message_get_key_param( ptr_message )->scancode )
          {
            case KEY_ENTER:

              result = cList_Sel( &main_menu );

            case KEY_ESC:

              exit_menu = TRUE;
          }
        default:

          // Sends message to menu object
          if( !cList_proc( &main_menu, ptr_message ))
          {
            // Sends message to application object
            cWinApp_defproc( main_module.m_process, ptr_message );
          }
      }
      Message_delete( ptr_message );
    }
    else // if( ptr_message ) It was timeout.
    {
      if( has_focus())
      {
        // Runs animation of the road
        DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

        DisplayGraphics_draw_bitmap( main_module.m_gfx, 
          BitmapSequence_get_bitmap( &menu_images, 0 ), 0, 0, BM_NORMAL );

        DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );
        
        for( index = 0; index < 45; index += 15 ) 
        {
          DisplayGraphics_fill_rect( main_module.m_gfx, 76,
            65 + (( index + road_coordinate ) * 
            ( index + road_coordinate )) / 45, 8,
            ( index + road_coordinate ) / 5 + 2 );
        }
        road_coordinate += 5;

        road_coordinate %= 16;

        // Runs animation of the buildings
        DisplayGraphics_set_clip( main_module.m_gfx, 0, 20, 35, 60 );

        DisplayGraphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( &menu_images, 1 ), -builds_coordinate, 
            42, BM_NORMAL );

        DisplayGraphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( &menu_images, 1 ), 33 - builds_coordinate, 
            42, BM_NORMAL );

        DisplayGraphics_set_clip( main_module.m_gfx, 125, 20, 160, 60 );
        
        DisplayGraphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( &menu_images, 2 ), 93 + builds_coordinate,
            42, BM_NORMAL );

        DisplayGraphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( &menu_images, 2 ), 126 + builds_coordinate,
            42, BM_NORMAL );
   
        DisplayGraphics_set_clip( main_module.m_gfx, 0, 0, 160, 100 );

        builds_coordinate++; 

        builds_coordinate %= 33;

        // Redraws the menu object and the screen
        cList_update( &main_menu );
      }
      // Sleeps for 40 milliseconds
      sleep( 40 );
    }
  } // while(!exit_menu) 
  // Removes menu object from the screen
  cWinApp_RemObj( main_module.m_process, &main_menu );

  return result;
}

/*
 * FUNCTION: game_process
 *
 * DESCRIPTION: Processes messages and draws the game screen
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
bool game_process( void )
{
  struct DirectKeyboard* ptr_keyboard;

  struct KeyParam* ptr_key_param;

  bool exit_current_game = FALSE;

  bool exit_game = FALSE;

  // Disables keyboard's clicks
  bool click_enable = get_clicks_enabled();

  set_clicks_enabled( FALSE );
 
  // Gets pointer to the direct keyboard object
  ptr_keyboard = DirectKeyboard_get_instance();

  // Initializes the new game
  init_new_game();

  while( !exit_current_game )
  {
    struct Message* ptr_message = 
      cWinApp_get_message( main_module.m_process, 1, 1, MSG_USER );

    // Real message received : it was not a timeout
    if( ptr_message )
    {
      // Processes messages
      switch( ptr_message->msgid )
      {
        case MSG_LOSTFOCUS: // Application has lost focus

          set_clicks_enabled( TRUE );

          MSequence_stop( game_sounds + background_sound );

          break;

        case MSG_GOTFOCUS: // Application has focus

          set_clicks_enabled( FALSE );

          if( !game_over && background_sound != SOUND_EXPLOSION )
          {
            MSequence_play_background( game_sounds + background_sound );
          }
          break;

        case MSG_SHUTUP: // Processes system exit signals
        case MSG_QUIT:

          exit_game = exit_current_game = TRUE;

        case MSG_KEYDOWN: // Processes ESC key

          if( Message_get_key_param( ptr_message )->scancode == KEY_ESC )
          {
            switch( show_dialog( DIALOG_EXIT ))
            {
              case EXIT_GAME:

                exit_game = TRUE;

              case RESTART_GAME:

                exit_current_game = TRUE;
            }
          }
        default:

        // Sends message to the application object
        // Need to process HELP key
        cWinApp_defproc( main_module.m_process, ptr_message );
      }
      // Releases message
      Message_delete( ptr_message );
    }
    // Draws the game's next scene
    if( cWinApp_has_focus( main_module.m_process ))
    { 
      // Processes the keyboard 
      if( !player_vehicle.explosion )
      {
        DirectKeyboard_scan( ptr_keyboard );

        if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_UP ))
        {
          player_vehicle.vertical_velocity = -PLAYER_Y_VELOCITY;
        }
        if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_DOWN ))
        {
          player_vehicle.vertical_velocity = PLAYER_Y_VELOCITY;
        }
        if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_LEFT ))
        {
          player_vehicle.horizontal_velocity = -PLAYER_X_VELOCITY;
        }
        if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_RIGHT ))
        {
          player_vehicle.horizontal_velocity = PLAYER_X_VELOCITY;
        }
        if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_ENTER ))
        {
          shoot();
        }
      } // if(!player_vehicle.exlosion)
      // Calculates new scene
      if( calculate_scene())
      {
        // Game over
        draw_scene();

        switch( show_dialog( DIALOG_GAME_OVER ))
        {
          case EXIT_GAME:

            exit_game = TRUE;

          case RESTART_GAME:

            exit_current_game = TRUE;

            break;
        }
      }
      else
      {
        // Draws new scene
        draw_scene();
      }
    }
  }
  if( high_scores[ 9 ].score < score )
  {
    // Saves high score
    update_high_scores();
  }
  // Performs cleanup   
  DirectKeyboard_dtor( ptr_keyboard, FREE_MEMORY );

  // Restores keybaord's clicks
  set_clicks_enabled( click_enable );

  return exit_game;
}

/*
 * FUNCTION: show_high_scores
 *
 * DESCRIPTION: Shows the best scores of the game
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void show_high_scores( void )
{
  int index;

  char* sz_item_text;

  struct cList high_score;

  struct Time high_score_time;

  struct cFrameForm* ptr_score_form; 

  struct Message* ptr_message;

  struct cSItem* ptr_list_item;

  struct BitmapSequence score_images;

  play_sound( SOUND_WIN );

  // Allocates resources  
  sz_item_text = (char*) malloc( 48 );

  BitmapSequence_ctor_Ex( &score_images, "high_score.pic" );

  ptr_score_form = (struct cFrameForm*) malloc( sizeof *ptr_score_form );

  cFrameForm_ctor( ptr_score_form, "High Scores",
    main_module.m_process );

  cList_ctor( &high_score, 156 );

  // Fills the menu's items
  for( index = 0; index < 10; index++ )
  {
    Time_decode( &high_score_time, high_scores[ index ].time );

    sprintf( sz_item_text, "%04d......%02d %s %04d",
      high_scores[ index ].score, high_score_time.day,
        sz_month[ high_score_time.month - 1 ],
          high_score_time.year + 1900 );

    ptr_list_item = malloc( sizeof *ptr_list_item );

    cSItem_ctor( ptr_list_item, 156, 
      high_scores[ index ].score ? sz_item_text : NULL, 
        BitmapSequence_get_bitmap( &score_images, index ));

    cList_AddItem( &high_score, ptr_list_item );
  }
  // Shows modal CyWin FrameForm
  cFrameForm_AddObj( ptr_score_form, &high_score, 0, 0 );

  cFrameForm_ShowModal( ptr_score_form );

  // Performs cleanup
  cList_dtor( &high_score, LEAVE_MEMORY );

  cCustomForm_dtor( ptr_score_form, FREE_MEMORY );

  BitmapSequence_dtor( &score_images, LEAVE_MEMORY );

  free( sz_item_text );
}

/*
 * FUNCTION: init_high_scores
 *
 * DESCRIPTION: Performs initialization of the high scores archive
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_high_scores( void )
{
  struct Input* ptr_input = 
    Archive_open_Ex( main_module.m_process->module->archive, "score.inf" );

  // Loads high scores from the "score.inf"
  Input_read( ptr_input, high_scores, sizeof( high_scores ));

  Input_dtor( ptr_input, FREE_MEMORY );
}

/*
 * FUNCTION: update_high_scores
 *
 * DESCRIPTION: Updates current high scores
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void update_high_scores( void )
{
  int index;

  struct score_t* high_scores_buffer =
    malloc( 9 * sizeof *high_scores_buffer );

  for( index = 0; index < 10; index++ )
  {
    if( score > high_scores[ index ].score )
    {
      // Shifts old high scores
      if( index != 9 )
      {
        memcpy( high_scores_buffer, high_scores,
          9 * sizeof *high_scores_buffer );

        memcpy((struct score_t*) high_scores + index + 1, 
          (struct score_t*) high_scores_buffer + index,
            ( 9 - index ) * sizeof *high_scores_buffer );
      }
      // Stores new high score
      high_scores[ index ].score = score;

      high_scores[ index ].cyid = get_own_id();

      high_scores[ index ].time = time();

      strncpy( high_scores[ index ].nickname,
        ptr_Finder->mf.f_nick, NICKNAMESIZE ); 
  
      break;
    }
  }
  free( high_scores_buffer );
}

/*
 * FUNCTION: release_high_scores
 *
 * DESCRIPTION: Releases allocated resources of the high scores archive
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_high_scores( void )
{
  struct Output* ptr_output = 
    Archive_open_write_Ex( main_module.m_process->module->archive,
      "score.inf");

  // Stores high scores in the "score.inf" archive
  Output_write( ptr_output, high_scores, sizeof( high_scores ));

  Output_dtor( ptr_output, FREE_MEMORY );
}
