/*
 * FILE: Screen.c
 *
 * DESCRIPTION: Drawing of the game scene
 *
 * HISTORY:
 *   Nov 17, 2000 Created by Aleksey Slesarev
 */
#include "Screen.h"
#include "Vehicles.h"
#include "Utils.h"
#include "Main.h"

// Bounding rectangle of the road
struct rect_t road_edge = { 0, 13, 160, 53 };

struct Bitmap initial_screen; // Bitmap of the empty field

struct Graphics virtual_graphics; // Destination for drawing initial_screen

struct BitmapSequence game_images; // Images of the game's objects

bool game_over; // TRUE if the game is over

int scroll_shift; // Shift of the road from the initial position

int scroll_velocity; // Current scroll velocity

int generation_delay; // Delay in generating the first enemy at this level

int max_speed; // Maximum scrolling velocity for this level

int level; // Current level

int lives; // Remaining lives

int extra_lives; // Number of extra lives

int score; // Current score

int remining_enemes; // Remaining enemies to destroy at this level

/*
 * FUNCTION: init_scene
 *
 * DESCRIPTION:
 *   Initializes the game scene's objects and renders the initial scene
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_scene( void )
{
  int index;

  // Sets seed for pseudorandom sequence
  srand( (int) clock());

  // Initializes bitmap list
  BitmapSequence_ctor_Ex( &game_images, "highway_hitman.pic" );

  // Initializes scene objects
  init_bullets();

  init_vehicles();

  init_big_explosion();
 
  // Initializes empty bitmap 
  Bitmap_ctor_Ex2( &initial_screen, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );

  // Sets bitmap as the destination for the drawing
  Graphics_ctor_Ex( &virtual_graphics, &initial_screen );

  // Draws the road
  Graphics_set_draw_mode( &virtual_graphics, DM_PUT );

  Graphics_set_color( &virtual_graphics, CLR_BLACK );

  Graphics_fill_rect( &virtual_graphics, 0, 0, 160, 100 );

  // Draws the road markings
  Graphics_set_color( &virtual_graphics, CLR_WHITE );

  for( index = 0; index < 4; index++ )
  {
    Graphics_fill_rect( &virtual_graphics, index * 40, 38, 24, 4 );
  }
  // Draws the roadside scenery
  Graphics_draw_bitmap( &virtual_graphics, 
    BitmapSequence_get_bitmap( &game_images, 13 ), 0, 0 , BM_NORMAL );

  Graphics_draw_bitmap( &virtual_graphics, 
    BitmapSequence_get_bitmap( &game_images, 13 ), 0, 54,
    BM_FLIP_Y | BM_FLIP_X );
}

/*
 * FUNCTION: release_scene
 *
 * DESCRIPTION:
 *   Releases the resources allocated by the scene's objects
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_scene( void )
{
  release_vehicles();

  release_big_explosion();

  Graphics_dtor( &virtual_graphics, LEAVE_MEMORY );

  Bitmap_dtor( &initial_screen, LEAVE_MEMORY );

  BitmapSequence_dtor( &game_images, LEAVE_MEMORY );
}

/*
 * FUNCTION: init_new_game
 *
 * DESCRIPTION: Initializes the game
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_new_game( void )
{
  level = 1;

  lives = 3;

  extra_lives = score = 0;

  game_over = current_enemy.active = FALSE;

  init_player_vehicle();

  init_level();
}

/*
 * FUNCTION: init_level
 *
 * DESCRIPTION: Initializes the level
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_level( void )
{
  max_speed = level + 14;

  generation_delay = 20;

  remining_enemes = ( level + 1 ) * 5;

  player_vehicle.vertical_velocity = player_vehicle.horizontal_velocity = 0;             

  draw_info();

  draw_scene();

  show_dialog( DIALOG_LEVEL );

  play_sound( SOUND_DRIVE );
}

/*
 * FUNCTION: draw_scene
 *
 * DESCRIPTION: Draws the scene's object on the screen
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void draw_scene( void )
{
  // Draws part of the road
  DisplayGraphics_set_page( main_module.m_gfx, 1 );

  DisplayGraphics_put_background( main_module.m_gfx,
    Graphics_get_buf_addr( &virtual_graphics ));

  DisplayGraphics_scroll( main_module.m_gfx, 0, 0, 160, 78, - scroll_shift, 0 );

  DisplayGraphics_set_page( main_module.m_gfx, 0 );

  DisplayGraphics_put_background( main_module.m_gfx,
    Graphics_get_buf_addr( &virtual_graphics ));

  DisplayGraphics_scroll( main_module.m_gfx, 0, 0, 160, 78, 
    160 - scroll_shift, 0 );

  DisplayGraphics_page_copy( main_module.m_gfx, 1, 0, 0, 0, 
    160 - scroll_shift, 78 );

  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

  // Draws the player's vehicle  
  draw_vehicle( &player_vehicle );

  // Draws the enemy's vehicle
  if( current_enemy.active == TRUE )
  {
    draw_vehicle( &current_enemy );
  }
  // Draws other objects of the scene
  draw_bullets();
  
  draw_big_explosion();

  DisplayGraphics_show( main_module.m_gfx );
}

/*
 * FUNCTION: calculate_scene
 *
 * DESCRIPTION: Updates coordinates of the scene's objects
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
bool calculate_scene( void )
{
  // Moves bullets
  move_bullets();
  
  // Moves the player's vehicle
  move_player_vehicle();

  // Moves the enemy
  move_enemy();

  // Generates a new enemy
  generate_enemy();

  // Updates the coordinates of the big explosion (if one has occurred)
  calculate_big_explosion();
 
  // Checks the extra life bonus
  if( extra_lives != ( score / 1000 ))
  {
    extra_lives++;

    lives++;

    draw_info();
  }
  if( !player_vehicle.explosion ) // Checks if the player's vehicle has moved
  {
    if( scroll_velocity < max_speed )
    {
      scroll_velocity++;
    }
    // Scrolls the road.
    scroll_shift += scroll_velocity;

    scroll_shift %= 160;
  }
  else if( player_vehicle.explosion > 10 ) // has player's vehicle exploded?
  {
    if( game_over ) return FALSE; 

    lives--; // If player has used all his lives, sets game_over flag

    if( !lives )
    {
      game_over = TRUE;

      draw_info();

      return TRUE;
    }
    init_player_vehicle();

    init_level(); // Initializes level
  }
  else scroll_velocity = 0;

  if( remining_enemes < 1 ) // Checks if player passed the level
  {
    if( ++level > 10 ) return TRUE;

    init_level();
  }
  return FALSE;
}

/*
 * FUNCTION: draw_info
 *
 * DESCRIPTION:
 *   Draws game information at the bottom of the game screen
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void draw_info( void )
{
  char sz_info_text[ 16 ];

  // Draws image of the small car 
  Graphics_set_color( &virtual_graphics, CLR_WHITE );

  Graphics_fill_rect( &virtual_graphics, 1, 80, 158, 19 );

  Graphics_draw_bitmap( &virtual_graphics, 
    BitmapSequence_get_bitmap( &game_images, 12 ), 126, 81, BM_NORMAL );

  // Draws text of game play status (lives, score, remaining enemy to destroy)
  Graphics_set_font( &virtual_graphics, cool_normal_font );

  Graphics_set_color( &virtual_graphics, CLR_BLACK );

  sprintf( sz_info_text, "X %d", lives );

  Graphics_draw_text( &virtual_graphics, sz_info_text, 143, 80 );

  sprintf( sz_info_text, "Score:%04d", score );

  Graphics_draw_text( &virtual_graphics, sz_info_text, 4, 80 );

  sprintf( sz_info_text, "%d remaining", remining_enemes );

  Graphics_draw_text( &virtual_graphics, sz_info_text, 65, 86 );
}
