/*
 * FILE: Utils.cpp
 *
 * DESCRIPTION: Helper functions
 *
 * HISTORY:
 *   Nov 17, 2000 Created by Aleksey Slesarev
 */
#include "Screen.h"
#include "Utils.h"
#include "Main.h"

static const char* sz_level = "WELCOME TO %d%s LEVEL!";

static const char* sz_endings[ 4 ] =
{
  "st", "nd", "rd", "th" 
};

static const char* sz_game_over = "GAME OVER!";

static const char* sz_mission_is_completed = "MISSION IS COMPLETED!";

static const char* sz_high_score = "\nHIGH SCORE!";

static const char* sz_sound_files[ 10 ] =
{
  "title.mus", "drive.mus", "speed_up.mus", "slow_down.mus", "explosion.mus",
  "knock.mus", "shoot.mus", "level.mus", "win.mus", "lose.mus"
};

int background_sound; // Current background music

struct MSequence game_sounds[ 10 ]; // Sound effects for the game

/*
 * FUNCTION: load_sound
 *
 * DESCRIPTION:
 *   Loads sound effects from the application archive
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void load_sound( void )
{
  int index;

  for( index = 0; index < 10; index++ )
  {
    MSequence_ctor( game_sounds + index, sz_sound_files[ index ] );
  }
  background_sound = SOUND_WIN;     
}

/*
 * FUNCTION: release_sound
 *
 * DESCRIPTION:
 *   Releases the resources allocated for sound effects
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_sound( void )
{
  int index;

  for( index = 0; index < 10; index++ )
  {
    MSequence_dtor( game_sounds + index, LEAVE_MEMORY );
  }
}

/*
 * FUNCTION: play_sound
 *
 * DESCRIPTION:
 *   Plays specified sound effect or melody
 *
 * PARAMETERS:
 *   sound_id -
 *
 * RETURNS: nothing
 */
void play_sound( int sound_id )
{
  switch( sound_id )
  {
    case SOUND_LEVEL:
    case SOUND_SHOOT:

      MSequence_play( game_sounds + sound_id );

      break;

    default:

      background_sound = sound_id;

      if( !MSequence_is_playing( game_sounds + sound_id ))
      {
        MSequence_play_background( game_sounds + sound_id );
      }
  }
}

/*
 * FUNCTION: show_dialog
 *
 * DESCRIPTION:
 *   Shows the model form for the specified dialog
 *
 * PARAMETERS:
 *   dialog_id -
 *
 * RETURNS: nothing
 */
int show_dialog( int dialog_id )
{
  char* sz_dialog_text;

  struct cDialog player_dialog;

  struct Message* ptr_message;

  long dialog_mode = mbQuit | mbRestart;

  int result = CONTINUE_GAME;

  sz_dialog_text = (char*) malloc( 48 );

  // Stops playing background sound effects
  MSequence_stop( game_sounds + background_sound );

  // Chooses dialog mode and dialog text
  switchfast( dialog_id )
  {
    case DIALOG_EXIT: // Do you really want to exit the game ?

      strcpy( sz_dialog_text, str_Really_exit );

      dialog_mode |= mbs5;

      break;

    case DIALOG_LEVEL:

      dialog_mode = mbOk; // Welcome to the next level

      sprintf( sz_dialog_text, sz_level, level, ( level > 3 )    ? 
        sz_endings[ 3 ] : sz_endings[ level - 1 ] );

      play_sound( SOUND_LEVEL );

      break;

    case DIALOG_GAME_OVER: // Game result dialog

      dialog_mode |= mbReverseOrder;

      if( level > 10 )
      {
        strcpy( sz_dialog_text, sz_mission_is_completed );

        play_sound( SOUND_WIN );
      }
      else
      {
        strcpy( sz_dialog_text, sz_game_over );

        play_sound( SOUND_LOSE );
      }
      if( high_scores[ 9 ].score < score )
      {
        strcpy( sz_dialog_text + strlen( sz_dialog_text ), sz_high_score);
      }
  }
  // Creates CyWin dialog
  cDialog_ctor( &player_dialog, NULL, sz_dialog_text,
    dialog_mode, 0, main_module.m_process );

  set_clicks_enabled( TRUE );

  // Deletes keyboard events from the message queue
  cWinApp_pause( main_module.m_process, 250 );

  while( ptr_message = cWinApp_peek_message( main_module.m_process,
    TRUE, MSG_KEYDOWN, MSG_KEYDOWN ))
  {
    Message_delete( ptr_message );
  }
  // Shows CyWin dialog in modal form
  switch( cDialog_ShowModal( &player_dialog )) 
  {
    case mrQuit:

      result = EXIT_GAME;

      break;

    case mrRestart:

      result = RESTART_GAME;

      break;
  }
  set_clicks_enabled( FALSE );

  cDialog_dtor( &player_dialog, LEAVE_MEMORY );

  // Restores the sound effect that the dialog interrupted 
  if( dialog_id == DIALOG_EXIT && !game_over && 
    background_sound != SOUND_EXPLOSION )
  {
    play_sound( background_sound );
  }
  free( sz_dialog_text );

  return result;
}
