/*
 * FILE: Vehicles.c
 *
 * DESCRIPTION: Implementation of the game's vehicles
 *
 * HISTORY:
 *   Nov 17, 2000 Created by Aleksey Slesarev
 */
#include "Vehicles.h"
#include "Bullet.h"
#include "Screen.h"
#include "Explosion.h"
#include "Utils.h"
#include "Main.h"

// Enemy's hit points
static const int enemy_hit_points[ 10 ] = 
{ 
  1, 3, 4, 1, 3, 6, 8, 10, 10, 20 
};

struct BitmapSequence explosion_images; // Images of the explosion

struct EnemyType enemy_vehicle[ 10 ];                   

struct PlayerVehicle player_vehicle;

struct EnemyVehicle current_enemy; 

/*
 * FUNCTION: init_vehicles
 *
 * DESCRIPTION:
 *   Initializes vehicles' objects and explosions of the object
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_vehicles( void )
{
  int index; 

  // Loads explosion images from the "explosion.pic" archive
  BitmapSequence_ctor_Ex( &explosion_images, "explosion.pic" );

  // Initializes player's vehicle
  player_vehicle.ptr_image = BitmapSequence_get_bitmap( &game_images, 0 );

  player_vehicle.w = 38;

  player_vehicle.h = 17;

  // Initializes enemy's vehicle
  for( index = 0; index < 10; index++ )
  {
    enemy_vehicle[ index ].hit_points = enemy_hit_points[ index ];

    enemy_vehicle[ index ].move_param = 0;

    enemy_vehicle[ index ].type = index;

    enemy_vehicle[ index ].score = ( index + 1 ) * 5;

    enemy_vehicle[ index ].ptr_image = 
      BitmapSequence_get_bitmap( &game_images, index + 1 );

    enemy_vehicle[ index ].w = enemy_vehicle[ index ].ptr_image->w;

    enemy_vehicle[ index ].h = enemy_vehicle[ index ].ptr_image->h;
  }
}

/*
 * FUNCTION: init_player_vehicle
 *
 * DESCRIPTION:
 *   Initializes parameters of the player's vehicle
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_player_vehicle( void )
{
  scroll_velocity = 0;

  player_vehicle.x = 0;

  player_vehicle.y = 31;

  player_vehicle.explosion = 0;

  player_bullet[ 0 ].active = FALSE;

  player_bullet[ 1 ].active = FALSE;

  player_bullet[ 2 ].active = FALSE;
}

/*
 * FUNCTION: release_vehicles
 *
 * DESCRIPTION:
 *   Releases the resources allocated for the vehicles and the explosion
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void release_vehicles( void )
{
  BitmapSequence_dtor( &explosion_images, LEAVE_MEMORY );
}

/*
 * FUNCTION: draw_vehicle
 *
 * DESCRIPTION:
 *   Draws a specified vehicle on the screen
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void draw_vehicle( struct Vehicle* ptr_vehicle )
{
  // Sets transparent color for the images
  DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );

  if( !ptr_vehicle->explosion )
  {
    // Draws the vehicle image if it has not exploded
    DisplayGraphics_draw_bitmap( main_module.m_gfx, 
      ptr_vehicle->ptr_image, ptr_vehicle->x, ptr_vehicle->y, BM_NORMAL );
  }
  else 
    if( ptr_vehicle->explosion < 11 )
    {
      // Draws explosion
      DisplayGraphics_draw_bitmap( main_module.m_gfx, 
        BitmapSequence_get_bitmap( &explosion_images,
          ( ptr_vehicle->explosion - 1 ) / 2 ),
            ptr_vehicle->x, ptr_vehicle->y, BM_NORMAL );

      ptr_vehicle->explosion++;
    }
}

/*
 * FUNCTION: explosion
 *
 * DESCRIPTION: Blows up the specified vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void explosion( struct Vehicle* ptr_vehicle )
{
  if((struct Vehicle*) ptr_vehicle == (struct Vehicle*) &player_vehicle )
  {
    // Knocking sound effect  
    play_sound( SOUND_KNOCK );

    cWinApp_pause( main_module.m_process, 200 );
  }
  if( !ptr_vehicle->explosion )
  {
    // Explosion sound effect
    play_sound( SOUND_EXPLOSION );

    ptr_vehicle->explosion = 1;
  }
}

/*
 * FUNCTION: generate_enemy
 *
 * DESCRIPTION: Generates enemy vehicle on screen
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void generate_enemy( void )
{
  if( generation_delay > 0 )
  {
    generation_delay--;

    return;
  }
  if( random( 20 ) <= level )
  { 
    if( !current_enemy.active && !big_explosion.current_state )
    {
      memcpy( &current_enemy, enemy_vehicle + random( level ),
        sizeof( struct EnemyType ));

      current_enemy.active = TRUE;

      current_enemy.x = 160;

      current_enemy.y = 13 + (int) random( 54 - current_enemy.h );
    }
  }
}

/*
 * FUNCTION: move_enemy
 *
 * DESCRIPTION: Updates coordinates of enemy vehicles
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void move_enemy( void ) 
{
  struct rect_t test_rectangle;

  if( !current_enemy.active )
  {
    return;
  }
  if( current_enemy.explosion ) // Blows up the enemy
  {
    if( current_enemy.explosion > 10 )
    {
      current_enemy.active = FALSE;
    }
    else
    {
      // If the enemy is a tank truck, then execute the big explosion
      if( current_enemy.type == 3 )
      {
        current_enemy.active = FALSE;

        perform_big_explosion( &current_enemy );
      }
      else
      {
        current_enemy.x -= scroll_velocity / 4;
      }
    }
  }
  else // Enemy has not exploded
  {
    if( player_vehicle.explosion ) // Blows-up the player's vehicle
    {
      current_enemy.x += 20;

      draw_vehicle( &current_enemy );

      if( current_enemy.x > 160 )
      {
        current_enemy.active = FALSE;
      }
    } // Moves enemy 
    else
    {
      switchfast( current_enemy.type )
      {
        case 0:

          move_vehicle_1( &current_enemy );

          break;

        case 1:

          move_vehicle_2( &current_enemy );

          break;

        case 2:

          move_vehicle_3( &current_enemy );

          break;

        case 3:
          move_vehicle_4( &current_enemy );

          break;

        case 4:

          move_vehicle_5( &current_enemy );

          break;

        case 5:

          move_vehicle_6( &current_enemy );

          break;

        case 6:

          move_vehicle_7( &current_enemy );

          break;

        case 7:

          move_vehicle_8( &current_enemy );

          break;

        case 8:

          move_vehicle_9( &current_enemy );

          break;

        case 9:

          move_vehicle_10( &current_enemy );

          break;
      }
      // Checks if enemy vehicle leaves the screen
      if( current_enemy.x < -current_enemy.w )
      {
        current_enemy.active = FALSE;
      }
    }
    // Checks if player's vehicle collides with the enemy
    if( rect_and( &test_rectangle, &current_enemy, &player_vehicle ))
    {
      explosion( &current_enemy );

      explosion( &player_vehicle );
    }  
  }
}

/*
 * FUNCTION: move_player_vehicle
 *
 * DESCRIPTION: Updates coordinates of the player's vehicle
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void move_player_vehicle( void )
{
  if( player_vehicle.horizontal_velocity )
  {
    // Sound effects for speeding up and slowing down
    play_sound(( player_vehicle.horizontal_velocity > 0 ) ? 
      SOUND_SPEED_UP :  SOUND_SLOW_DOWN );

    // Moves player's vehicle horizontally
    player_vehicle.x += player_vehicle.horizontal_velocity;

    if( player_vehicle.x < 0 )
    {
      player_vehicle.x = 0;
    }
    else
      if( player_vehicle.x > 122 )
      {
        player_vehicle.x = 122;
      } 
    player_vehicle.horizontal_velocity = 0; 
  }
  else
  {
    // Turns on moving vehicle sound effect
    if( background_sound != SOUND_DRIVE && 
      !MSequence_is_playing( game_sounds + SOUND_EXPLOSION ) && 
        !player_vehicle.explosion )
    {
      play_sound( SOUND_DRIVE );
    }
  }
  // Moves player's vehicle horizontally
  if( player_vehicle.vertical_velocity )
  {
    player_vehicle.y += player_vehicle.vertical_velocity;

    if( player_vehicle.y < 13 )
    {
      explosion( &player_vehicle );
    }
    else 
      if( player_vehicle.y > 49 )
      {
        explosion( &player_vehicle );
      }
    player_vehicle.vertical_velocity = 0; 
  }
}

/*
 * FUNCTION: move_vehicle_1
 *
 * DESCRIPTION: Behavior of the 1st enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_1( struct EnemyVehicle* ptr_vehicle )
{
  ptr_vehicle->x -= scroll_velocity / 4;
}

/*
 * FUNCTION: move_vehicle_2
 *
 * DESCRIPTION: Behavior of the 2nd enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_2( struct EnemyVehicle* ptr_vehicle )
{
  while( !ptr_vehicle->move_param )
  {
    ptr_vehicle->move_param = 1 - (int) random( 3 );
  }
  ptr_vehicle->x -= scroll_velocity / 4;

  ptr_vehicle->y += ( PLAYER_Y_VELOCITY * ptr_vehicle->move_param ) / 2; 

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;

    ptr_vehicle->move_param = 1;
  }
  else 
    if( ptr_vehicle->y > 49 )
    {
      ptr_vehicle->y = 49;

      ptr_vehicle->move_param = -1;
    }
}

/*
 * FUNCTION: move_vehicle_3
 *
 * DESCRIPTION: Behavior of the 3rd enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_3( struct EnemyVehicle* ptr_vehicle )
{
  ptr_vehicle->x -= scroll_velocity / 4;

  ptr_vehicle->y += 
    ( PLAYER_Y_VELOCITY / 2 - (int) random( PLAYER_Y_VELOCITY + 1 ));

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;
  }
  else 
    if( ptr_vehicle->y > 47 )
    {
      ptr_vehicle->y = 47;
    }
}

/*
 * FUNCTION: move_vehicle_4
 *
 * DESCRIPTION: Behavior of the 4th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_4( struct EnemyVehicle* ptr_vehicle )
{
  if( ptr_vehicle->y == 29 )
  {
    ptr_vehicle->y++;
  }
  ptr_vehicle->x -= scroll_velocity / 4;  
}

/*
 * FUNCTION: move_vehicle_5
 *
 * DESCRIPTION: Behavior of the 5th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_5( struct EnemyVehicle* ptr_vehicle )
{
  while( !ptr_vehicle->move_param )
  {
    ptr_vehicle->move_param = 1 - (int) random( 3 );
  }
  ptr_vehicle->x -= scroll_velocity / 4;

  ptr_vehicle->y += ( PLAYER_Y_VELOCITY * ptr_vehicle->move_param );

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;

    ptr_vehicle->move_param = 1;
  }
  else 
    if( ptr_vehicle->y > 49 )
    {
      ptr_vehicle->y = 49;

      ptr_vehicle->move_param = -1;
    }
}

/*
 * FUNCTION: move_vehicle_6
 *
 * DESCRIPTION: Behavior of the 6th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_6( struct EnemyVehicle* ptr_vehicle )
{
  while( !ptr_vehicle->move_param )
  {
    ptr_vehicle->move_param = 1 - (int) random( 3 );
  }
  ptr_vehicle->x -= scroll_velocity / 4;

  ptr_vehicle->y += ( PLAYER_Y_VELOCITY * ptr_vehicle->move_param ) / 2; 

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;

    ptr_vehicle->move_param = 1;
  }
  else 
    if( ptr_vehicle->y > 49 )
    {
      ptr_vehicle->y = 49;

      ptr_vehicle->move_param = -1;
    }
}

/*
 * FUNCTION: move_vehicle_7
 *
 * DESCRIPTION: Behavior of the 7th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_7( struct EnemyVehicle* ptr_vehicle )
{
  if( ptr_vehicle->hit_points > 3 )
  {
    ptr_vehicle->x -= scroll_velocity / 4;
  }
  else
  {
    ptr_vehicle->x -= scroll_velocity / 2;
  }
}

/*
 * FUNCTION: move_vehicle_8
 *
 * DESCRIPTION: Behavior of the 8th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_8( struct EnemyVehicle* ptr_vehicle )
{
  while( !ptr_vehicle->move_param )
  {
    ptr_vehicle->move_param = 1 - (int) random( 3 );
  }
  ptr_vehicle->x -= scroll_velocity / 4;

  ptr_vehicle->y += ( PLAYER_Y_VELOCITY * ptr_vehicle->move_param ) / 2;

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;

    ptr_vehicle->move_param = 1;
  }
  else 
    if( ptr_vehicle->y > 45 )
    {
      ptr_vehicle->y = 45;

      ptr_vehicle->move_param = -1;
    }
}

/*
 * FUNCTION: move_vehicle_9
 *
 * DESCRIPTION: Behavior of the 9th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_9( struct EnemyVehicle* ptr_vehicle )
{
  while( !ptr_vehicle->move_param )
  {
    ptr_vehicle->move_param = 1 - (int) random( 3 );
  }
  ptr_vehicle->x -= scroll_velocity / 6;

  ptr_vehicle->y += ( PLAYER_Y_VELOCITY * ptr_vehicle->move_param );

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;

    ptr_vehicle->move_param = 1;
  }
  else 
    if( ptr_vehicle->y > 45 )
    {
      ptr_vehicle->y = 45;

      ptr_vehicle->move_param = -1;
    }
}

/*
 * FUNCTION: move_vehicle_10
 *
 * DESCRIPTION: Behavior of the 10th enemy vehicle
 *
 * PARAMETERS:
 *   ptr_vehicle -
 *
 * RETURNS: nothing
 */
void move_vehicle_10( struct EnemyVehicle* ptr_vehicle )
{
  ptr_vehicle->x -= scroll_velocity / 8;

  ptr_vehicle->y += 
    ( PLAYER_Y_VELOCITY / 2 - (int) random( PLAYER_Y_VELOCITY + 1 ));

  if( ptr_vehicle->y < 13 )
  {
    ptr_vehicle->y = 13;
  }
  else if( ptr_vehicle->y > 41 )
  {
    ptr_vehicle->y = 41;
  }
}
