/*
 * FILE: actors.c
 *
 * DESCRIPTION:
 *   This module contains routines that describe the actors' actions.
 *
 * HISTORY:
 *   May 29, 2001 Created by Dmitry Kaloshin
 */
#include "DataTypes.h"
#include "Maps.h"
#include "main.h"
#include "creature.h"
#include "menu.h"
#include "highscore.h"
#include "utils.h"

actor struct System cybiko_system = { R_INITIALIZE };

actor struct SplashScreen level_splash = { R_INITIALIZE };

actor struct Barricades barricades = { R_DO_NOTHING, 0, 0, 0, 0 };

actor struct Bananas bananas = { R_DO_NOTHING, 0, 0 };

actor struct Fires fires = { R_DO_NOTHING, 0, 0 };

actor struct Stairs stairs = { R_DO_NOTHING, 0, 0 };

actor struct Hero hero = { R_INITIALIZE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
  0, 0, MAX_LIVES, MAX_START_BULLETS, 0, 0, 0 };

actor struct HeroField game_field = { R_INITIALIZE, 0, 0, 0 };

actor struct Doors doors = { R_DO_NOTHING, 0, 0 };

actor struct Monsters monsters = { R_DO_NOTHING, 0, 0 };

actor struct Info info;

actor struct Final final = { R_INITIALIZE };

actor struct MusicPlayer media_player = { R_INITIALIZE, TITLE_MUS, NO_SOUND, 
  FALSE };

actor struct Menu menu = { R_INITIALIZE, 0 };

struct MSequence music_seq[ NUM_MUS_FILE ];

char* sz_file_music_name[ NUM_MUS_FILE ] =
{
  "level.music", "boot.music", "berry.music", "bonus.music",

  "lose.music", "eat.music", "key.music", "button.music", "end_bonus.music",

  "banana.music", "gate.music", "close_barricade.music", "stairs.music",

  "title.music", "win.music"
};

struct MediaFile music[ NUM_MUS_FILE ] = 
{
  { 0, TRUE, TRUE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, FALSE, FALSE },

  { 0, TRUE, FALSE },

  { 0, FALSE, FALSE },
};

bool score_flag = FALSE;

bool terminate = FALSE;

int splash_frame = 1;

int prev_visible = 1;

int can_fire;

int loading = 0;

static struct Bitmap* ptr_field_bitmap = 0;

static struct BitmapSequence* ptr_info_images = 0;

static struct BitmapSequence* ptr_digits = 0;

static struct BitmapSequence* ptr_level_digits = 0;

static struct BitmapSequence* ptr_final_image = 0;

static struct BitmapSequence* ptr_level_images = 0;

/*
 * FUNCTION: global_release
 *
 * DESCRIPTION: Releases memory allocated by the actors.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void global_release( void )
{
  int index;

  if( !terminate )
  {
    highscore_enter( &main_module, hero.score, HSM_CHECK );
  }
  if( barricades.ptr_barricades )
  {
    free( barricades.ptr_barricades );
  }
  if( barricades.ptr_buttons )
  {
    free( barricades.ptr_buttons );
  }
  if( doors.ptr_doors )
  {
    free( doors.ptr_doors );
  }
  if( hero.ptr_bullets )
  {
     free( hero.ptr_bullets );
  }
  if( hero.ptr_keyboard )
  {
    DirectKeyboard_dtor( hero.ptr_keyboard, FREE_MEMORY );
  }
  if( monsters.ptr_monsters )
  {
    free( monsters.ptr_monsters );
  }
  if( ptr_virtual_graphics )
  {
    Graphics_dtor( ptr_virtual_graphics, FREE_MEMORY );
  }
  if( ptr_field_bitmap )
  {
    Bitmap_dtor( ptr_field_bitmap, FREE_MEMORY );
  }
  if( ptr_game_images )
  {
    BitmapSequence_dtor( ptr_game_images, FREE_MEMORY );
  }
  if( fires.ptr_fires )
  {
    free( fires.ptr_fires );
  }
  if( bananas.ptr_bananas )
  {
    free( bananas.ptr_bananas );
  }
  for( index = 0; index < NUM_MUS_FILE; index ++ )
  {
    MSequence_dtor( &music_seq[ NUM_MUS_FILE - 1 - index ], LEAVE_MEMORY );
  }
  if( ptr_info_images )
  {
    BitmapSequence_dtor( ptr_info_images, FREE_MEMORY );
  }
  if( ptr_digits )
  {
    BitmapSequence_dtor( ptr_digits, FREE_MEMORY );
  }
  if( ptr_level_digits )
  {
    BitmapSequence_dtor( ptr_level_digits, FREE_MEMORY );
  }
  if( loading )
  {
    release_menu();
  }
  if( ptr_final_image )
  {
    BitmapSequence_dtor( ptr_final_image, FREE_MEMORY );
  }
  if( ptr_level_images )
  {
    BitmapSequence_dtor( ptr_level_images, FREE_MEMORY );
  }  
  cleanup();

  if( loading )
  {
    highscore_cleanup( &main_module );
  }
}

/*
 * FUNCTION: prepare_restart
 *
 * DESCRIPTION: Preparation before restart.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
static void prepare_restart( void )
{
  int i;

  game_field.level = 0;

  hero.lives = MAX_LIVES;

  hero.score = 0;

  splash_frame = 1;

  hero.current_bullets_num = MAX_START_BULLETS;

  game_field.request = R_INIT_LEVEL;

  for( i = 0; i < MAX_BULLETS_NUM; i++ )
  {
    hero.ptr_bullets[ i ].request = R_PASSIVE;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: actor-function for the actor barricades
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Barricades::proc()
{
  int col, row, index;

  char field_type;

  switch( request )
  {
    case R_INITIALIZE:

      if( ptr_barricades )
      {
        free( ptr_barricades );
      }
      ptr_barricades = malloc( number_barricades * sizeof *ptr_barricades );

      if( ptr_buttons )
      {
        free( ptr_buttons );
      }
      ptr_buttons = malloc( number_buttons * sizeof *ptr_buttons );

      for( row = 0; row < ptr_current_level->height; row++ )
      {
        for( col = 0; col < ptr_current_level->width; col++ )
        {
          field_type = 
            ptr_current_level->map_fields[ ptr_current_level->height * col + 
              row ];

          // initialize the buttons and barricades
          if( field_type >= BUTTON && field_type < BARRICADE )
          {
            ptr_buttons[ field_type - BUTTON ].x = col;

            ptr_buttons[ field_type - BUTTON ].y = row;

            ptr_buttons[ field_type - BUTTON ].request = R_PASSIVE;
          }
          else if( field_type >= BARRICADE && field_type < DOOR )
          {
            ptr_barricades[ field_type - BARRICADE ].x = col;

            ptr_barricades[ field_type - BARRICADE ].y = row;

            ptr_barricades[ field_type - BARRICADE ].request = R_PASSIVE;

            ptr_barricades[ field_type - BARRICADE ].timeout = 0;
          }
        }
      }
      request = R_DO_NOTHING;

      break;

    case R_DO_NOTHING:

      break;

    case R_ACTIVE:

      if( main_module.m_visible )
      {
        int barricade_x, barricade_y;

        DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );

        DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );
        
        for( index = 0; index < number_barricades; index++ )
        {
          barricade_x = ptr_barricades[ index ].x * ELEMENT_SIZE;

          barricade_y = ptr_barricades[ index ].y * ELEMENT_SIZE;

          if( ptr_barricades[ index ].request == R_PASSIVE )
          {
            Graphics_draw_bitmap( ptr_virtual_graphics,
              BitmapSequence_get_bitmap( ptr_game_images, 16 ), barricade_x, 
                barricade_y, BM_NORMAL );
          }
          else
          {
            fill_black( barricade_x, barricade_y );

            if( --ptr_barricades[ index ].timeout == 0 )
            {
              // nobody stays on the barricade's place
              if( is_clear_field( barricade_x, barricade_y ))
              {
                ptr_barricades[ index ].request = R_PASSIVE;

                ptr_buttons[ index ].request = R_PASSIVE;

                media_player.melody_index = CLOSE_BARRICADE_MUS;

                ptr_current_level->map_fields[ ptr_current_level->height * 
                  ptr_barricades[ index ].x + ptr_barricades[ index ].y ] = 
                    BARRICADE;
              
                Graphics_draw_bitmap( ptr_virtual_graphics,
                  BitmapSequence_get_bitmap( ptr_game_images, 16 ), barricade_x,
                    barricade_y, BM_NORMAL );
              }
              // somebody stays on the barricade's place
              if( ptr_barricades[ index ].request != R_PASSIVE )
              {
                ptr_barricades[ index ].timeout++;
              }
            }
          }
        }
        // draw butons
        for( index = 0; index < number_buttons; index++ )
        {
          Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images,
              17 + R_PASSIVE - ptr_buttons[ index ].request ), 
                ptr_buttons[ index ].x * ELEMENT_SIZE, 
                  ptr_buttons[ index ].y * ELEMENT_SIZE, BM_NORMAL );
        }
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor doors
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Doors::proc()
{
  int col, row, index;  

  switch( request )
  {
    case R_INITIALIZE:

      if( ptr_doors )
      {
        free( ptr_doors );
      }
      ptr_doors = malloc( number * sizeof *ptr_doors );

      for( row = 0; row < ptr_current_level->height; row++ )
      {
        for( col = 0; col < ptr_current_level->width; col++ )
        {
          char field_type = 
            ptr_current_level->map_fields[ ptr_current_level->height * col + 
              row ];

          // initialize the keys
          if( field_type >= DOOR && field_type < KEY )
          {
            ptr_doors[ field_type - DOOR ].x = col;

            ptr_doors[ field_type - DOOR ].y = row;

            ptr_doors[ field_type - DOOR ].type = R_CLOSE;
          }
        }
      }
      request = R_DO_NOTHING;

      break;

    case R_DO_NOTHING:

      break;

    case R_ACTIVE:

      if( main_module.m_visible )
      {
        for( index = 0; index < number; index++ )
        {
          DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

          DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );

          // draw closed door
          if( ptr_doors[ index ].type == R_CLOSE )
          {
            DisplayGraphics_draw_bitmap( main_module.m_gfx,
              BitmapSequence_get_bitmap( ptr_game_images, 20 ),
                ptr_doors[ index ].x * ELEMENT_SIZE + game_field.x, 
                ptr_doors[ index ].y * ELEMENT_SIZE + game_field.y, BM_NORMAL );
          }
          else // draw open door
          {
            DisplayGraphics_draw_bitmap( main_module.m_gfx,
              BitmapSequence_get_bitmap( ptr_game_images, 21 ),
                ptr_doors[ index ].x * ELEMENT_SIZE + game_field.x, 
                ptr_doors[ index ].y * ELEMENT_SIZE + game_field.y, BM_NORMAL );
          }
        }
      }
      break;
  }
}

/*
 * FUNCTION: check_highscore
 *
 * DESCRIPTION: Checks high score after the end of the game.
 *
 * PARAMETERS: none
 *
 * RETURNS: result of the check-up
 */
static int check_highscore()
{
  int result;  

  menu.index_menu = 0;

  cEngine_Unlock( &Screen );

  switch(( result = highscore_enter( &main_module, hero.score, 
    HSM_CHECKSHOWALL )))
  {
    case HSR_RESTART:

      rest();

      prepare_restart();

      break;

    case HSR_INVITE:

      rest();

      cybiko_system.request = R_RELEASE;

      break;

    case HSR_NONE:

      hero.request = R_DO_NOTHING;

      hero.x = -5;

      media_player.request = R_DO_NOTHING;

      break;

    case HSR_SHOW:      

      // fall through

    default:

      rest();

      menu.request = R_ACTIVE;

      media_player.request = R_PLAY_SOUND;

      media_player.background_index = TITLE_MUS;

      media_player.melody_index = NO_SOUND;

      break;
  }
  score_flag = FALSE;

  cEngine_Lock( &Screen );

  return result;
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor hero
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Hero::proc()
{
  static int frame = 0, last_direction;
  
  static int climb_into = 0;

  int index;

  char field_type;

  switch( request )
  {
    case R_DO_NOTHING:

      break;

    case R_INITIALIZE:

      ptr_keyboard = DirectKeyboard_get_instance();

      if( ptr_bullets )
      {
         free( ptr_bullets );
      }
      ptr_bullets = malloc( MAX_BULLETS_NUM * sizeof *ptr_bullets );

      // initialize the bullets
      for( index = 0; index < MAX_BULLETS_NUM; index++ )
      {
        ptr_bullets[ index ].request = R_PASSIVE;
      }
      request = R_DO_NOTHING;

      break;

    case R_DIE:

      if( main_module.m_visible )
      {
        Graphics_set_draw_mode( ptr_virtual_graphics, DM_OR );

        Graphics_set_bkcolor( ptr_virtual_graphics, CLR_BLACK );

        Graphics_draw_bitmap( ptr_virtual_graphics,
           BitmapSequence_get_bitmap( ptr_game_images,
             12 + die_frame ), x * ELEMENT_SIZE + x_step, 
               y * ELEMENT_SIZE + y_step, BM_NORMAL );

        if( ++die_frame == HERO_DIE_FRAMES )
        {
          if( --lives == 0 )
          {   
            int i;

            for( i = 0; i < MAX_BULLETS_NUM; i++ )
            {
              ptr_bullets[ i ].request = R_PASSIVE;
            }
            fill_black( x * ELEMENT_SIZE + x_step, y * ELEMENT_SIZE + y_step );
            
            request = R_DO_NOTHING;

            score_flag = TRUE;
          }
          else
          {
            // prepare after losing a life
            fill_black( x * ELEMENT_SIZE + x_step, y * ELEMENT_SIZE + y_step );

            x = start_x;

            y = start_y;

            x_step = y_step = 0;

            direction = 0;

            fall = 0;

            type = HERO_MAD;

            timeout = TIMEOUT_MAD_AFTER_DIE;

            request = R_SHOW_FRAME;

            for( index = 0; index < MAX_BULLETS_NUM; index++ )
            {
              ptr_bullets[ index ].request = R_PASSIVE;
            }
          }
        }
      }
      if( request != R_SHOW_FRAME ) break;

    case R_SHOW_FRAME:

      if( main_module.m_visible )
      {         
         if( !climb_into )
         {
           DirectKeyboard_scan( ptr_keyboard );

           if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_LEFT ))
           {
             direction = DIRECTION_LEFT;
           }
           else if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_RIGHT ))
           {
             direction = DIRECTION_RIGHT;
           }
           else if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_UP ))
           {
             direction = DIRECTION_UP;
           }
           else if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_DOWN ))
           {
             direction = DIRECTION_DOWN;
           }           
           last_direction = direction;
         }
         // there is a hero on the banana
         if( !fall )
         {
           field_type = move_hero();
         }
         if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_ENTER ) && 
           current_bullets_num > 0 && can_fire )
         {
           if( init_bullet( ptr_bullets, direction, x * ELEMENT_SIZE + x_step, 
             y * ELEMENT_SIZE + y_step ))
           {
             current_bullets_num--; 
           }
         }
         else can_fire = 1;

         move_bullets( ptr_bullets );

         if( field_type == BANANA && !x_step && !y_step )
         {
           fall = 3;

           fall_frame = FALL_FRAMES;

           media_player.melody_index = BANANA_MUS;

           if( --score < 0 ) score = 0;
         }
         // end of the level
         if( field_type == STAIRS && !x_step && !y_step )
         {
           media_player.melody_index = STAIRS_MUS;
         }
         if( climb_into )
         { 
           direction = last_direction;

           climb_into = 0;
         }           
         if( last_direction != direction )
         {
           climb_into++;
         }
         if( field_type == BERRY || field_type == BONUS || 
           field_type == KEY || field_type == BOOT )
         {
           fill_black( ( x + SIGN( x_step )) * ELEMENT_SIZE, 
             ( y + SIGN( y_step )) * ELEMENT_SIZE );
         }         
         Graphics_set_draw_mode( ptr_virtual_graphics, DM_OR );

         Graphics_set_bkcolor( ptr_virtual_graphics, CLR_BLACK );

         if( fall )
         {
           Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images,
              4 + type + (( fall_frame - 1 ) / 2 ) * 2 ), 
                x * ELEMENT_SIZE, y * ELEMENT_SIZE, ( fall_frame - 1 == 3 ) ? 
                  BM_FLIP_Y : ( fall_frame - 1 == 1 ) ? BM_FLIP_X : BM_NORMAL );
           
           if( --fall_frame == 0 )
           {
             --fall;

             fall_frame = FALL_FRAMES;
           }
         }
         else
         {
           Graphics_draw_bitmap( ptr_virtual_graphics,
             BitmapSequence_get_bitmap( ptr_game_images,
               4 + type + (( direction - 1 ) / 2) * 2 + frame ), 
                 x * ELEMENT_SIZE + x_step, y * ELEMENT_SIZE + y_step, 
                    ( direction - 1 == 3 ) ? BM_FLIP_Y : 
                       ( direction - 1 == 1 ) ? BM_FLIP_X : BM_NORMAL );
         }
         // correct focus of the game field
         game_field.x = adjust_visible_part( 160, x * ELEMENT_SIZE,
           x_step, ptr_current_level->width * ELEMENT_SIZE );

         game_field.y = adjust_visible_part( 100, y * ELEMENT_SIZE, 
           y_step, ptr_current_level->height * ELEMENT_SIZE );

         frame = 1 - frame;

         // the hero is mad
         if( type )
         {
           if( --timeout == 0 )
           {
             type = 0;

             media_player.melody_index = END_BONUS_MUS;
           }
         }         
      }
      break;
  }
}

/*
 * FUNCTION: draw_bullets
 *
 * DESCRIPTION: Draws bullets
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
static void draw_bullets()
{
  int i;

  for( i = 0; i < MAX_BULLETS_NUM; i++ )
  {
    if( hero.ptr_bullets[ i ].request == R_ACTIVE )
    {
      DisplayGraphics_draw_bitmap( main_module.m_gfx,
        BitmapSequence_get_bitmap( ptr_game_images,
          29 ), hero.ptr_bullets[ i ].x + game_field.x, 
            hero.ptr_bullets[ i ].y + game_field.y, BM_NORMAL );
    }
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor monsters
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Monsters::proc()
{
  int row, col, current_monster = 0;

  char field_type;

  switch( request )
  {
    case R_DO_NOTHING:

      break;

    case R_INITIALIZE:

      if( ptr_monsters ) 
      {
        free( ptr_monsters );
      }
      ptr_monsters = malloc( number * sizeof *ptr_monsters );

      for( row = 0; row < ptr_current_level->height; row++ )
      {
        for( col = 0; col < ptr_current_level->width; col++ )
        {
          field_type =  
            ptr_current_level->map_fields[ ptr_current_level->height * col + 
              row ];

          // initialize the monsters
          if( field_type == MONSTER )
          {
            init_monster( &ptr_monsters[ current_monster ], col, row );

            ptr_monsters[ current_monster ].start_x = col;

            ptr_monsters[ current_monster ].start_y = row;

            current_monster ++;
          }
        }
      }
      request = R_DO_NOTHING;

      break;

    case R_MOVE_MONSTERS:

      if( main_module.m_visible )
      {
        if( hero.request != R_DIE )
        {
          int hero_x = hero.x * ELEMENT_SIZE + hero.x_step, 
              hero_y = hero.y * ELEMENT_SIZE + hero.y_step,
              monster_x, monster_y;
          
          DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );

          DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

          for( current_monster = 0; current_monster < number; 
            current_monster++ )
          {
            bool kill_monster = FALSE;

            int bullet_index;

            ptr_monsters[ current_monster ].direction = 
              new_monster_direction( &ptr_monsters[ current_monster ] );
            
            if( game_field.request != R_INIT_LEVEL && 
              hero.request != R_DO_NOTHING )
            {
              move_monster( current_monster );
            }
            monster_x = 
              ptr_monsters[ current_monster ].x * ELEMENT_SIZE + 
                 ptr_monsters[ current_monster ].x_step;

            monster_y = 
              ptr_monsters[ current_monster ].y * ELEMENT_SIZE + 
                ptr_monsters[ current_monster ].y_step;

            for( bullet_index = 0; bullet_index < MAX_BULLETS_NUM; 
              bullet_index++ )
            {
              if( hero.ptr_bullets[ bullet_index ].request == R_ACTIVE )
              {
                if( check_kill( hero.ptr_bullets[ bullet_index ].x, 
                  hero.ptr_bullets[ bullet_index ].y, monster_x, monster_y ))
                {
                  kill_monster = TRUE;

                  hero.ptr_bullets[ bullet_index ].request = R_PASSIVE;
                }                    
              }
            }
            if( !kill_monster && 
              check_intersect( monster_x, monster_y, hero_x, hero_y ))
            {
              if( hero.type != HERO_MAD )
              {
                hero.request = R_DIE;

                hero.die_frame = 0;

                media_player.melody_index = LOSE_MUS;
              }
              else
              {
                kill_monster = TRUE;
              }              
            }  
            if( kill_monster )
            {
              init_monster( &ptr_monsters[ current_monster ], 
                ptr_monsters[ current_monster ].start_x, 
                  ptr_monsters[ current_monster ].start_y );

              hero.score += HERO_KILL_MONSTER;

              media_player.melody_index = EAT_MUS;
            }
            DisplayGraphics_draw_bitmap( main_module.m_gfx,
              BitmapSequence_get_bitmap( ptr_game_images,
                MONSTER ), monster_x + game_field.x, 
                  monster_y + game_field.y, BM_NORMAL );
          }
          if( game_field.request != R_INIT_LEVEL && 
            hero.request != R_DO_NOTHING )
          {
            draw_bullets();
          }
        }
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor cybiko_system
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void System::proc()
{  
  int show_result;

  if( main_module.m_key == KEY_ESC && splash_frame != 1 && 
      prev_visible && has_focus())
  {
    switch(( show_result = show_dialog( DIALOG_EXIT )))
    {
      case EXIT_GAME:        

        if( highscore_enter( &main_module, hero.score, 
           HSM_CHECKSHOWTABLE ) == HSR_INVITE )
        {
           request = R_RELEASE;
        }
        else
        {
          rest();

          menu.request = R_ACTIVE;

          menu.index_menu = 2;

          media_player.request = R_PLAY_SOUND;

          media_player.background_index = TITLE_MUS;
        }
        break;

      case RESTART_GAME:

        if( highscore_enter( &main_module, hero.score, 
          HSM_CHECKSHOWTABLE ) == HSR_INVITE )
        {
           request = R_RELEASE;
        }
        else
        {
          rest();

          prepare_restart();

          DisplayGraphics_fill_screen( main_module.m_gfx, CLR_BLACK );
        }
        break;
    }
    if( show_result != CONTINUE_GAME && final.request == R_ACTIVE )
    {
      final.request = R_DO_NOTHING;
    }
  }
  else if( main_module.m_key == KEY_SELECT )
  {
    if( info.request == R_DO_NOTHING && game_field.request == R_SHOW_FRAME ) 
    {
      info.request = R_ACTIVE;
    }
    else info.request = R_DO_NOTHING;
  }
  if( main_module.m_quit == TRUE )
  {
    request = R_RELEASE;    
  }
  // stop application
  if( request == R_RELEASE )
  {
    rest();
    
    final.request = R_DO_NOTHING;

    menu.request = R_DO_NOTHING;

    if( !main_module.m_quit )
    {
       terminate = TRUE;
    }
    main_module.m_quit = TRUE;
  }
  prev_visible = main_module.m_visible;
}
 
/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor game_field
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void HeroField::proc()
{   
  switch( request )
  {
    case R_DO_NOTHING:

      break;

    case R_INITIALIZE:

      srand((int) clock());

      ptr_game_images = malloc( sizeof *ptr_game_images );      

      BitmapSequence_ctor_Ex( ptr_game_images, "images.pic" );      

      request = R_DO_NOTHING;

      break;

    case R_INIT_LEVEL:      

      init_map( &ptr_field_bitmap, level );

      monsters.request = R_INITIALIZE;

      barricades.request = R_INITIALIZE;

      bananas.request = R_INITIALIZE;

      fires.request = R_INITIALIZE;

      doors.request = R_INITIALIZE;

      hero.request = R_DO_NOTHING;

      info.request = R_DO_NOTHING;
      
      if( media_player.request != R_INITIALIZE )
      {
        media_player.background_index = LEVEL_MUS;

        media_player.request = R_PLAY_SOUND; 
      }
      request = R_DO_NOTHING;

      break;

    case R_SHOW_FRAME:

      // end of the level
      if( media_player.melody_index == STAIRS_MUS )
      {
        level++;

        hero.score += 10;

        // end of the game
        if( level >= MAX_LEVEL )
        {
          rest();

          final.request = R_ACTIVE;          

          media_player.request = R_PLAY_SOUND;

          media_player.melody_index = WIN_MUS;
        }
        else request = R_INIT_LEVEL;
      }
      if( main_module.m_visible )
      {        
        DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

        DisplayGraphics_draw_bitmap( main_module.m_gfx, ptr_field_bitmap,
          x, y, BM_NORMAL );

        fill_black( hero.x * ELEMENT_SIZE + hero.x_step, 
          hero.y * ELEMENT_SIZE + hero.y_step );
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor level_splash
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void SplashScreen::proc()
{  
  static char sz_intro_text[ 10 ];

  cEngine_Lock( &Screen );
 
  switch( request )
  {
    case R_DO_NOTHING:

      break;

    case R_INITIALIZE:

      info.request = R_INITIALIZE;

      request = R_DO_NOTHING;

      ptr_level_images = malloc( sizeof *ptr_level_images );

      BitmapSequence_ctor_Ex( ptr_level_images, "level.pic" );

      ptr_level_digits = malloc( sizeof *ptr_level_digits );

      BitmapSequence_ctor_Ex( ptr_level_digits, "level_digits.pic" );

      break;      

    case R_SHOW_SPLASH:

      splash_frame = 0;

      request = R_SHOW_FRAME;      

    case R_SHOW_FRAME:

      if( splash_frame && 
        ( main_module.m_key || splash_frame < -60 ))
      {
        request = R_DO_NOTHING;

        game_field.request = R_SHOW_FRAME;

        hero.request = R_SHOW_FRAME;

        monsters.request = R_MOVE_MONSTERS;

        barricades.request = R_ACTIVE;

        doors.request = R_ACTIVE;

        fires.request = R_ACTIVE;

        bananas.request = R_ACTIVE;

        info.request = R_ACTIVE;

        stairs.request = R_ACTIVE;

        MSequence_stop( music[ media_player.background_index ].ptr_msequence );
        
        media_player.background_index = NO_SOUND;

        can_fire = 0;
      }
      else
      {
        if( main_module.m_visible ) 
        {
          DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

          DisplayGraphics_fill_screen( main_module.m_gfx, CLR_BLACK );

          draw_lib( 0, 4 + splash_frame % 2, splash_frame + 60, 48, BM_NORMAL );

          Graphics_draw_bitmap( main_module.m_gfx,
            BitmapSequence_get_bitmap( ptr_level_digits,
              game_field.level + 1 ), splash_frame + 164, 42, BM_NORMAL );

          Graphics_draw_bitmap( main_module.m_gfx,
            BitmapSequence_get_bitmap( ptr_level_images,
              0 ), splash_frame + 78, 40, BM_NORMAL );

          splash_frame--;
        }
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor fires
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Fires::proc()
{
  int col, row, index;  

  switch( request )
  {
    case R_INITIALIZE:

      if( ptr_fires )
      {
        free( ptr_fires );
      }
      ptr_fires = malloc( number * sizeof *ptr_fires );

      index = 0;

      for( row = 0; row < ptr_current_level->height; row++ )
      {
        for( col = 0; col < ptr_current_level->width; col++ )
        {
          char field_type = 
            ptr_current_level->map_fields[ ptr_current_level->height * col + 
              row ];

          // initialize firing
          if( field_type == FIRE )
          {
            ptr_fires[ index ].x = col;

            ptr_fires[ index ].y = row;

            ptr_fires[ index ].frame = 0;
            
            index++;
          }
        }
      }
      level_splash.request = R_SHOW_SPLASH;

      request = R_DO_NOTHING;

      break;

    case R_DO_NOTHING:

      break;

    case R_ACTIVE:

      if( main_module.m_visible )
      {
        DisplayGraphics_set_draw_mode( ptr_virtual_graphics, DM_PUT );

        for( index = 0; index < number; index++ )
        {
          Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images,
              23 + ptr_fires[ index ].frame ), 
                ptr_fires[ index ].x * ELEMENT_SIZE, 
                  ptr_fires[ index ].y * ELEMENT_SIZE, BM_NORMAL );

          if( ++ptr_fires[ index ].frame == FIRE_FRAMES )
          {
            ptr_fires[ index ].frame = 0;
          }
        }
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor bananas
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Bananas::proc()
{
  int col, row, index;  

  switch( request )
  {
    case R_INITIALIZE:

      if( ptr_bananas )
      {
        free( ptr_bananas );
      }
      ptr_bananas = malloc( number * sizeof *ptr_bananas );

      index = 0;

      for( row = 0; row < ptr_current_level->height; row++ )
      {
        for( col = 0; col < ptr_current_level->width; col++ )
        {
          char field_type = 
            ptr_current_level->map_fields[ ptr_current_level->height * col + 
              row ];

          // initialize the bananas
          if( field_type == BANANA )
          {
            ptr_bananas[ index ].x = col;

            ptr_bananas[ index ].y = row;

            index++;
          }
        }
      }      
      request = R_DO_NOTHING;

      break;

      case R_DO_NOTHING:

        break;

      case R_ACTIVE:
        
        if( main_module.m_visible )
        {
          for( index = 0; index < number; index++ )
          {
            Graphics_draw_bitmap( ptr_virtual_graphics,
              BitmapSequence_get_bitmap( ptr_game_images,
                28 ), ptr_bananas[ index ].x * ELEMENT_SIZE, 
                  ptr_bananas[ index ].y * ELEMENT_SIZE, BM_NORMAL );
          }
        }
        break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor media_player
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void MusicPlayer::proc()
{
  int index;  

  switch(request)
  {
    case R_INITIALIZE:

      for( index = 0; index < NUM_MUS_FILE; index++ )
      {
        MSequence_ctor( &music_seq[ index ], 
          sz_file_music_name[ index ] );

        music[ index ].ptr_msequence = &music_seq[ index ];
      }
      request = R_PLAY_SOUND;

      // fall through

    case R_PLAY_SOUND:

      if( score_flag && has_focus() && main_module.m_visible )
      {     
        check_highscore();        
      }
      if( background_index != NO_SOUND )
      {
        if( main_module.m_visible )
        {
          if( !MSequence_is_playing( music[ background_index ].ptr_msequence ))
          {
            MSequence_play_background( 
              music[ background_index ].ptr_msequence );
          }
        }
        else
        {
          MSequence_stop( music[ background_index ].ptr_msequence );

          if( !music[ background_index ].repeat )
          {
            background_index = NO_SOUND;
          }
        }
      }
      if( melody_index != NO_SOUND )
      {
        if( music[ melody_index ].background )
        {
          background_index = melody_index;
        }
        else
        {
          MSequence_play( music[ melody_index ].ptr_msequence );
        }
        melody_index = NO_SOUND;
      }            
      break;
  }
  cEngine_Unlock( &Screen );
}

/*
 * FUNCTION: draw_num
 *
 * DESCRIPTION: Draws a number on the screen
 *
 * PARAMETERS:
 *   digits - array for storing numbers
 *   score - number to be drawn
 *   ptr_digits - sequence that contains bitmaps of the numbers
 *   info_y - y coordinate for drawing
 *
 * RETURNS: nothing
 */
static void draw_num( char *digits, int score, 
  struct BitmapSequence* ptr_digits, int info_y )
{
  int index = 0, i;

  while( score )
  {
    digits[ index ] = (char)( score % 10 );

    score /= 10;

    index++;
  }  
  if( index )
  {
    for( i = 0; i < index; i++ )
    {
      Graphics_draw_bitmap( main_module.m_gfx,
        BitmapSequence_get_bitmap( ptr_digits,
          digits[ index - i - 1 ] ), INFO_X + 30 - 6 * ( index - i ), 
            info_y, BM_NORMAL );
    }
  }
  else
  {
    Graphics_draw_bitmap( main_module.m_gfx,
      BitmapSequence_get_bitmap( ptr_digits, 0 ), 
        INFO_X + 24, info_y, BM_NORMAL );
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor info
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Info::proc()
{   
  switch( request )
  {
    case R_INITIALIZE:

      if( ptr_info_images )
      {
        free( ptr_info_images );
      }
      ptr_info_images = malloc( sizeof *ptr_info_images );

      BitmapSequence_ctor_Ex( ptr_info_images, "info.pic" );

      if( ptr_digits )
      {
        free( ptr_digits );
      }
      ptr_digits = malloc( sizeof *ptr_digits );

      BitmapSequence_ctor_Ex( ptr_digits, "digits.pic" );

      memset( digits, 0, 10 );

      request = R_DO_NOTHING;

      break;

    case R_DO_NOTHING:

      break;

    case R_ACTIVE:
      
      if( main_module.m_visible )
      {
        DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

        DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );
        
        Graphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( ptr_info_images, 0 ), 
            INFO_X, 0, BM_NORMAL );

        Graphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( ptr_info_images, 1 ), 
            INFO_X, 24, BM_NORMAL );

        Graphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( ptr_digits, hero.lives ), 
            INFO_X + 24, 36, BM_NORMAL );

        Graphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( ptr_game_images, 29 ), 
            INFO_X + 24, 48, BM_NORMAL );

        draw_num( digits, hero.score, ptr_digits, 12 );

        draw_num( digits, hero.current_bullets_num, ptr_digits, 60 );

        main_module.m_visible = has_focus();
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor stairs
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Stairs::proc()
{
  switch( request )
  {
    case R_DO_NOTHING:

      break;

    case R_ACTIVE:

      if( main_module.m_visible )
      {
        DisplayGraphics_set_draw_mode( ptr_virtual_graphics, DM_PUT );

        Graphics_draw_bitmap( ptr_virtual_graphics,
          BitmapSequence_get_bitmap( ptr_game_images, 30 ), 
            x * ELEMENT_SIZE, y * ELEMENT_SIZE, BM_NORMAL );
      }
      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor menu
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Menu::proc()
{
  bool exit_menu = FALSE;

  switch( request )
  {
    case R_INITIALIZE:

      load_menu();

      highscore_init( &main_module );

      loading = 1;

      request = R_ACTIVE;

    case R_ACTIVE:

      if( main_module.m_visible )
      {     
        main_module.m_process->HelpContext = 1;

        index_menu = enter_menu( &main_module, index_menu ); 

        switch( index_menu )
        {
         case 0:
           
           prepare_restart();

           main_module.m_process->HelpContext = 0;
           
           request = R_PASSIVE;

           break;

         case 1:

           if( highscore_enter( &main_module, 0, HSM_SHOW ) == HSR_INVITE )
           {
             cybiko_system.request = R_RELEASE;
           }
           media_player.background_index = NO_SOUND;

           break;

         default:

           splash_frame = 1;

           cybiko_system.request = R_RELEASE;

           request = R_PASSIVE;

           break;
        }
      }
      break;

    case R_PASSIVE:

      break;
  }
}

/*
 * FUNCTION: proc
 *
 * DESCRIPTION: Actor-function for the actor final
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
actor void Final::proc()
{
  switch( request )
  { 
    case R_INITIALIZE:

      ptr_final_image = malloc( sizeof *ptr_final_image );

      BitmapSequence_ctor_Ex( ptr_final_image, "final.pic" );

      request = R_DO_NOTHING;

      break;

    case R_DO_NOTHING:

      break;

    case R_ACTIVE:      

      if( main_module.m_visible )
      {
        DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

        DisplayGraphics_draw_bitmap( main_module.m_gfx,
          BitmapSequence_get_bitmap( ptr_final_image, 0 ), 0, 0, BM_NORMAL );
 
        if( main_module.m_key == KEY_ENTER )
        {              
          if( check_highscore() != HSR_NONE ) request = R_DO_NOTHING;
        }
      }
      break;
  }
}
