/*
 * FILE: creature.c
 *
 * DESCRIPTION:
 *   This module contains routines that describe creatures' actions.
 *
 * HISTORY:
 *   May 29, 2001 Created by Dmitry Kaloshin
 */
#include "DataTypes.h"
#include "Maps.h"
#include "main.h"
#include "actors.h"

/*
 * FUNCTION: fill_black
 *
 * DESCRIPTION: fill the black color square ELEMENT_SIZE * ELEMENT_SIZE
 *
 * PARAMETERS:
 *   x - x coordinate of the square
 *   y - y coordinate of the square
 *
 * RETURNS: nothing
 */
void fill_black( int x, int y )
{
  Graphics_set_color( ptr_virtual_graphics, CLR_BLACK );

  Graphics_fill_rect( ptr_virtual_graphics, x, y,
    ELEMENT_SIZE, ELEMENT_SIZE );
}

/*
 * FUNCTION: get_cell_type
 *
 * DESCRIPTION: Returns the game cell's type
 *
 * PARAMETERS:
 *   col - x coordinate
 *   row - y coordinate
 *
 * RETURNS: The game cell's type
 */
static char get_cell_type( int col, int row )
{
  char field_type;

  if( col < 0 || col >= ptr_current_level->width ||
    row < 0 || row >= ptr_current_level->height )
  {
    field_type = EMPTY; // TELEPORT
  }
  else
  {
    field_type = 
      ptr_current_level->map_fields[ ptr_current_level->height * col + row ];

    if( field_type >= BARRICADE && field_type < DOOR )
    {
      field_type = BRICK;
    }
    else if( field_type >= DOOR && field_type < KEY )
    {
      if( doors.ptr_doors[ field_type - DOOR].type == R_CLOSE )
      {
        field_type = BRICK;
      }
    }
  }
  return field_type;
}

/*
 * FUNCTION: get_intersection
 *
 * DESCRIPTION: Returns the game cell's type
 *
 * PARAMETERS:
 *   col - x absolute coordinate
 *   row - y absolute coordinate
 *
 * RETURNS: The game cell's type
 */
static char get_intersection( int x, int y )
{  
  return get_cell_type( x / ELEMENT_SIZE, y / ELEMENT_SIZE );
}

/*
 * FUNCTION: adjust_visible_part
 *
 * DESCRIPTION: Corrects the focus of the game
 *
 * PARAMETERS:
 *   visible_part - visible part
 *   value - absolute focus' coordinate
 *   step - step of the moving focus
 *   max_value - max value of the field
 *
 * RETURNS: Focus' new coordinate
 */
int adjust_visible_part( int visible_part, int value, int step, int max_value )
{
  int half_visible_part = visible_part / 2;

  if( value + step <= half_visible_part )
  {
    return 0;
  }  
  if( value + step > ( max_value -  half_visible_part ))
  {
    return visible_part - max_value;
  }                                        
  return half_visible_part - value - step;
}

/*
 * FUNCTION: climb_into_hole
 *
 * DESCRIPTION: Check whether hero can climb into the hole 
 *
 * PARAMETERS:
 *   front_x - front x coordinate
 *   front_y - front y coordinate
 *   simple_step - hero's simple step 
 *   direction - hero's direction
 *
 * RETURNS: Hero's new direction
 */
static int climb_into_hole( int front_x, int front_y, int simple_step, 
  int direction )
{
  int new_direction = direction;

  char field_type;

  switch( direction )
  {
    case DIRECTION_LEFT:
    case DIRECTION_RIGHT:

      if(( field_type = get_intersection( front_x, front_y + simple_step )) != 
        BRICK && !( field_type >= TL_GATE && field_type <= BR_GATE ))
      {
        new_direction = DIRECTION_DOWN;
      }
      else if(( field_type = get_intersection( front_x, front_y - 
        simple_step )) != BRICK && !( field_type >= TL_GATE && 
          field_type <= BR_GATE ))
      {
        new_direction = DIRECTION_UP;
      }
      break;

    case DIRECTION_UP:
    case DIRECTION_DOWN:

      if((field_type = get_intersection( front_x + simple_step, front_y )) != 
        BRICK && !( field_type >= TL_GATE && field_type <= BR_GATE ))
      {
        new_direction = DIRECTION_RIGHT;
      }
      else if((field_type = get_intersection( front_x - simple_step, 
        front_y )) != BRICK && !( field_type >= TL_GATE && 
          field_type <= BR_GATE ))
      {
        new_direction = DIRECTION_LEFT;
      }
      break;
  }
  return new_direction;
}

/*
 * FUNCTION: teleporting
 *
 * DESCRIPTION: Checks whether hero could to teleport 
 *
 * PARAMETERS:
 *   ptr_creature - pointer to the creature
 *   x - new x coordinate
 *   y - new y coordinate
 *   simple_step - creature's simple step
 *
 * RETURNS: 1 - if creature can be teleported
 */
static int teleporting( struct Creature *ptr_creature, int x, int y, 
  int simple_step )
{
  int ret_teleporting = 0;

  if( x < 0 )
  {
    ptr_creature->x = ptr_current_level->width - 1;

    ptr_creature->x_step = ELEMENT_SIZE - simple_step;

    ret_teleporting = 1;
  }
  else if( y < 0 )
  {
    ptr_creature->y = ptr_current_level->height - 1;

    ptr_creature->y_step = ELEMENT_SIZE - simple_step;

    ret_teleporting = 1;
  }
  else if( x >= ptr_current_level->width )
  {
    ptr_creature->x = 0;

    ptr_creature->x_step = simple_step - ELEMENT_SIZE;

    ret_teleporting = 1;
  }
  else if( y >= ptr_current_level->height )
  {
    ptr_creature->y = 0;
     
    ptr_creature->y_step = simple_step - ELEMENT_SIZE;

    ret_teleporting = 1;
  }
  return ret_teleporting;
}

/*
 * FUNCTION: open_gate
 *
 * DESCRIPTION: Checks whether hero could to go through the gate 
 *
 * PARAMETERS:
 *   direction - creature's direction 
 *   field_type - type of game cell
 *   x - x coordinate
 *   y - y coordinate 
 *
 * RETURNS: 1 - if gate was opened
 */
static bool open_gate( int direction, char field_type, int x, int y )
{
  bool ret_open = FALSE;

  char gate_type;

  int new_x, new_y;

  switch( field_type )
  {
    case TL_GATE:

     if( direction == DIRECTION_RIGHT )
     {
       if( y > 0 && is_clear_field( x * ELEMENT_SIZE, 
         ( y - 1 ) * ELEMENT_SIZE ))
       {
         new_x = x, new_y = y - 1, gate_type = BL_GATE, ret_open = TRUE;
       }
     }
     else if( direction == DIRECTION_DOWN )
     {
       if( x > 0 && is_clear_field(( x - 1 ) * ELEMENT_SIZE, y * ELEMENT_SIZE ))
       {
         new_x = x - 1, new_y = y, gate_type = TR_GATE, ret_open = TRUE;
       }
     }
     break;

    case TR_GATE:

     if( direction == DIRECTION_LEFT )
     {
       if( y > 0 && is_clear_field( x * ELEMENT_SIZE, ( y - 1 ) * ELEMENT_SIZE ))
       {
         new_x = x, new_y = y - 1, gate_type = BR_GATE, ret_open = TRUE;         
       }
     }
     else if( direction == DIRECTION_DOWN )
     {
       if( x < ptr_current_level->width - 1 && 
          is_clear_field(( x + 1 ) * ELEMENT_SIZE, y * ELEMENT_SIZE ) )
       {
         new_x = x + 1, new_y = y, gate_type = TL_GATE, ret_open = TRUE;
       }
     }
     break;

    case BL_GATE:

     if( direction == DIRECTION_RIGHT )
     {
       if( y < ptr_current_level->height - 1 && 
         is_clear_field( x * ELEMENT_SIZE, ( y + 1 ) * ELEMENT_SIZE ))
       {
         new_x = x, new_y = y + 1, gate_type = TL_GATE, ret_open = TRUE;         
       }
     }
     else if( direction == DIRECTION_UP )
     {
       if( x > 0 && is_clear_field(( x - 1 ) * ELEMENT_SIZE, y * ELEMENT_SIZE ))
       {
         new_x = x - 1, new_y = y, gate_type = BR_GATE, ret_open = TRUE;
       }
     }
     break;

    case BR_GATE:

     if( direction == DIRECTION_LEFT )
     {
       if( y < ptr_current_level->height - 1 && 
         is_clear_field( x * ELEMENT_SIZE, ( y + 1 ) * ELEMENT_SIZE ))
       {
         new_x = x, new_y = y + 1, gate_type = TR_GATE, ret_open = TRUE;         
       }
     }
     else if( direction == DIRECTION_UP )
     {
       if( x < ptr_current_level->width - 1 && 
         is_clear_field(( x + 1 ) * ELEMENT_SIZE, y * ELEMENT_SIZE ) )
       {
         new_x = x + 1, new_y = y, gate_type = BL_GATE, ret_open = TRUE;
       }
     }
     break;
  }
  if( ret_open && ptr_current_level->map_fields[ ptr_current_level->height * 
    new_x + new_y ] == EMPTY )
  {
    // rotate the gate
    ptr_current_level->map_fields[ ptr_current_level->height * x + y ] = EMPTY;

    ptr_current_level->map_fields[ ptr_current_level->height * new_x + 
      new_y ] = gate_type;

    draw_gate( gate_type, new_x, new_y );

    fill_black( x * ELEMENT_SIZE, y * ELEMENT_SIZE );
  }
  else ret_open = FALSE;

  return ret_open;
}


/*
 * FUNCTION: go_back
 *
 * DESCRIPTION: Turns back a monster
 *
 * PARAMETERS:
 *   direction - pointer to a monster direction
 *
 * RETURNS: nothing
 */
static void go_back( int *direction )
{
  switch( *direction )
  {
    case DIRECTION_LEFT:
      
      *direction = DIRECTION_RIGHT;

      break;

    case DIRECTION_RIGHT:

      *direction = DIRECTION_LEFT;

      break;
    
    case DIRECTION_UP:

      *direction = DIRECTION_DOWN;

      break;
    
    case DIRECTION_DOWN:

      *direction = DIRECTION_UP;

      break;
  }
}

/*
 * FUNCTION: move_creature
 *
 * DESCRIPTION: Moves creature
 *
 * PARAMETERS:
 *   ptr_creature - pointer to a creature object
 *   simple_step - creature's simple step
 *   type_creature - type of creature
 *
 * RETURNS: Type of new game cell
 */
static char move_creature( struct Creature *ptr_creature, 
  int simple_step, int type_creature )
{
  int x = ptr_creature->x, y = ptr_creature->y, 
      x_step = ptr_creature->x_step, y_step = ptr_creature->y_step;

  int front_x1, front_y1, front_x2, front_y2;

  char field_type1, field_type2 = EMPTY;

  switch( ptr_creature->direction )
  {
    case DIRECTION_LEFT:

      x_step -= simple_step;

      if( -x_step == ELEMENT_SIZE )
      {
        x--;

        x_step = 0;
      }
      front_x1 = front_x2 = x * ELEMENT_SIZE + x_step + 1;

      front_y2 = ( y + 1 ) * ELEMENT_SIZE + y_step - 1;

      front_y1 = y * ELEMENT_SIZE + y_step + 1;

      break;

    case DIRECTION_RIGHT:

      x_step += simple_step;

      if( x_step == ELEMENT_SIZE )
      {
        x++;

        x_step = 0;
      }
      front_x1 = front_x2 = ( x + 1 ) * ELEMENT_SIZE + x_step - 1;

      front_y2 = ( y + 1 ) * ELEMENT_SIZE + y_step - 1;

      front_y1 = y * ELEMENT_SIZE + y_step + 1;

      break;

    case DIRECTION_UP:

      y_step -= simple_step;

      if( -y_step == ELEMENT_SIZE )
      {
        y--;

        y_step = 0;
      }      
      front_y1 = front_y2 = y * ELEMENT_SIZE + y_step + 1;

      front_x2 = ( x + 1 ) * ELEMENT_SIZE + x_step - 1;

      front_x1 = x * ELEMENT_SIZE + x_step + 1;      

      break;

    case DIRECTION_DOWN:

      y_step += simple_step;

      if( y_step == ELEMENT_SIZE )
      {
        y++;

        y_step = 0;
      }      
      front_y1 = front_y2 = ( y + 1 ) * ELEMENT_SIZE + y_step - 1;

      front_x2 = ( x + 1 ) * ELEMENT_SIZE + x_step - 1;

      front_x1 = x * ELEMENT_SIZE + x_step + 1;

      break;
  }
  if( !teleporting( ptr_creature, x, y, simple_step ))
  {
    field_type1 = get_intersection( front_x1, front_y1 );
    
    field_type2 = get_intersection( front_x2, front_y2 );

    if( field_type1 != BRICK && field_type2 != BRICK  && 
      field_type2 == field_type1 )
    { 
      if( field_type2 >= TL_GATE && field_type2 <= BR_GATE )
      {  
        if( field_type2 == field_type1 && 
          open_gate( ptr_creature->direction, field_type2, 
            front_x2 / ELEMENT_SIZE, front_y2 / ELEMENT_SIZE ))
        {
          ptr_creature->x = x;

          ptr_creature->y = y;

          ptr_creature->x_step = x_step;

          ptr_creature->y_step = y_step;

          media_player.melody_index = GATE_MUS;
        }
        else if( type_creature == MONSTER )
        {
          go_back( &ptr_creature->direction );
        }
      }
      else if( field_type1 == FIRE )
      {
         if( type_creature == HERO && hero.boot || 
           type_creature == MONSTER )
         {
           ptr_creature->x = x;

           ptr_creature->y = y;

           ptr_creature->x_step = x_step;

           ptr_creature->y_step = y_step;
         }
      }
      else
      {
        ptr_creature->x = x;

        ptr_creature->y = y;

        ptr_creature->x_step = x_step;

        ptr_creature->y_step = y_step;

        if( type_creature == HERO )
        {
          // take a ammo
          if( field_type2 == BERRY )
          {
            if(( ABS( x_step + y_step )) == INTERSECT_BERRY_STEP )
            {
              ptr_current_level->map_fields[ ptr_current_level->height * 
                ( x + SIGN( x_step )) + y + SIGN( y_step ) ] = EMPTY;

              hero.current_bullets_num++;

              hero.score++;
              
              media_player.melody_index = BERRY_MUS;
            }
            else
            {
              field_type2 = EMPTY;        
            }
          }
          // take a bonus
          else if( field_type2 == BONUS )
          {
            ptr_current_level->map_fields[ ptr_current_level->height * 
              ( x + SIGN( x_step )) + y + SIGN( y_step ) ] = EMPTY;

            hero.type = HERO_MAD;

            hero.timeout = TIMEOUT_MAD;

            media_player.melody_index = BONUS_MUS;
          }
          // take a boot
          else if( field_type2 == BOOT )
          {
            hero.boot = 1; 

            ptr_current_level->map_fields[ ptr_current_level->height * 
              ( x + SIGN( x_step )) + y + SIGN( y_step ) ] = EMPTY;

            media_player.melody_index = BOOT_MUS;
          }
          // take a key
          else if( field_type2 >= KEY &&  field_type2 < EMPTY )
          {
            ptr_current_level->map_fields[ ptr_current_level->height * 
              ( x + SIGN( x_step )) + y + SIGN( y_step ) ] = EMPTY;

            doors.ptr_doors[ field_type2 - KEY ].type = R_OPEN;

            field_type2 = KEY;
            
            media_player.melody_index = KEY_MUS;
          }
        }
        // push a button
        if( field_type2 >= BUTTON && field_type2 < BARRICADE )
        {
          barricades.ptr_buttons[ field_type2 - BUTTON ].request = R_ACTIVE;

          barricades.ptr_barricades[ field_type2 - BUTTON ].request = R_ACTIVE;

          barricades.ptr_barricades[ field_type2 - BUTTON ].timeout = 
            TIMEOUT_BARRICADE;

          ptr_current_level->map_fields[ ptr_current_level->height * 
            barricades.ptr_barricades[ field_type2 - BUTTON ].x + 
              barricades.ptr_barricades[ field_type2 - BUTTON ].y ] = EMPTY;

          media_player.melody_index = BUTTON_MUS;
        }
      }
    }
    else if( type_creature == HERO )
    { 
      if( field_type1 != BRICK && 
        ( field_type2 == BRICK || ( field_type2 >= TL_GATE && field_type2 <= 
          BR_GATE )))
      {
        hero.direction = 
          climb_into_hole( front_x2, front_y2, simple_step, 
            ptr_creature->direction );
      }
      else if( field_type2 != BRICK && 
        ( field_type1 == BRICK || ( field_type1 >= TL_GATE && field_type1 <= 
          BR_GATE )))
      {
        hero.direction = 
          climb_into_hole( front_x1, front_y1, simple_step, 
            ptr_creature->direction );
      }
    }
  }
  return field_type2;
}

/*
 * FUNCTION: move_hero
 *
 * DESCRIPTION: Moves the hero
 *
 * PARAMETERS: none
 *
 * RETURNS: type of new game cell, where the hero will stay 
 */
char move_hero( void )
{
  return move_creature((struct Creature *)&hero, HERO_SIMPLE_STEP, HERO );
}

/*
 * FUNCTION: move_monster
 *
 * DESCRIPTION: Move the monster
 *
 * PARAMETERS:
 *   index_monster - index of a monster in the monsters's array
 *
 * RETURNS: type of new game cell, where the monster will stay 
 */
char move_monster( int index_monster )
{
  return move_creature((struct Creature *)&monsters.ptr_monsters[ index_monster ], 
    MONSTER_SIMPLE_STEP, MONSTER );
}

/*
 * FUNCTION: available_direction
 *
 * DESCRIPTION: Searches for a new available direction
 *
 * PARAMETERS:
 *   direction - monster's direction
 *   x - x monster coord
 *   y - y monster coord
 *
 * RETURNS: 1 - there is a direction available
 */
static int available_direction( int direction, int x, int y )
{
  int available = 0;

  switch( direction )
  {
    case DIRECTION_LEFT:

      if( x > 0 && get_cell_type( x - 1 , y ) != BRICK )
      {
        available = DIRECTION_LEFT;
      }
      break;

    case DIRECTION_RIGHT:

      if( x < ptr_current_level->width - 1  &&
        get_cell_type( x + 1 , y ) != BRICK )
      { 
        available = DIRECTION_RIGHT;
      }
      break;

    case DIRECTION_UP:

      if( y > 0 && get_cell_type( x, y - 1 ) != BRICK )
      {
        available = DIRECTION_UP;
      }
      break;

    case DIRECTION_DOWN:
      
      if( y < ptr_current_level->height - 1 && 
        get_cell_type( x, y + 1 ) != BRICK )
      {
        available = DIRECTION_DOWN;
      }
  }
  return available;
}

/*
 * FUNCTION: new_monster_direction
 *
 * DESCRIPTION: Searches for a new direction for the monster
 *
 * PARAMETERS:
 *   ptr_monster - pointer to a monster
 *
 * RETURNS: new direction
 */
int new_monster_direction( struct Monster *ptr_monster )
{
  int new_direction = ptr_monster->direction,
      mask_available_direction[ 3 ], qty_available_dir = 0, 
      back_dir;

  if( !ptr_monster->x_step && !ptr_monster->y_step )
  { 
    if( get_cell_type( ptr_monster->x , ptr_monster->y ) == TELEPORT )
    {
      mask_available_direction[ qty_available_dir++ ] = new_direction;
    }
    else
    {
      switch( ptr_monster->direction )
      {
        case DIRECTION_LEFT:

          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_LEFT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_UP, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_DOWN, ptr_monster->x, 
              ptr_monster->y )))
          {
             qty_available_dir++;
          }
          back_dir = DIRECTION_RIGHT;

          break;        

        case DIRECTION_RIGHT:

          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_RIGHT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_UP, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_DOWN, ptr_monster->x, 
              ptr_monster->y )))
          {
             qty_available_dir++;
          }
          back_dir = DIRECTION_LEFT;

          break;

        case DIRECTION_UP:

          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_UP, ptr_monster->x, 
              ptr_monster->y )))
          {
             qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_LEFT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_RIGHT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          back_dir = DIRECTION_DOWN;

          break;

        case DIRECTION_DOWN:

          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_DOWN, ptr_monster->x, 
              ptr_monster->y )))
          {
             qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_LEFT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          if(( mask_available_direction[ qty_available_dir ] = 
            available_direction( DIRECTION_RIGHT, ptr_monster->x, 
              ptr_monster->y )))
          {
            qty_available_dir++;
          }
          back_dir = DIRECTION_UP;

          break;
      }
    }
    if( qty_available_dir )
    {
      new_direction = 
        mask_available_direction[ (int) random( qty_available_dir) ];
    }
    else
    {
      new_direction = back_dir;
    }
  }
  return new_direction;
}

/*
 * FUNCTION: init_monster
 *
 * DESCRIPTION: Initializes a monster
 *
 * PARAMETERS:
 *   ptr_monster - pointer to a monster
 *   col - new monster's x coordinate
 *   row - new monster's y coordinate
 *
 * RETURNS: new direction
 */
void init_monster( struct Monster *ptr_monster, int col, int row )
{
  ptr_monster->x = col;

  ptr_monster->y = row;

  ptr_monster->x_step = 0;
            
  ptr_monster->y_step = 0;

  ptr_monster->direction = (int) random( 4 ) + 1;
}

/*
 * FUNCTION: init_bullet
 *
 * DESCRIPTION: Initializes a bullet
 *
 * PARAMETERS:
 *   ptr_bullets - pointer to a bullet object
 *   col - new monster's x coordinate
 *   row - new monster's y coordinate
 *
 * RETURNS: TRUE if initialization was successful
 */
bool init_bullet( struct Creature *ptr_bullets, int direction, int x, int y )
{  
  int index;

  bool ret_init = TRUE;

  for( index = 0; index < MAX_BULLETS_NUM; index++ )
  {
    if( ptr_bullets[ index ].request == R_PASSIVE ) break;
  }
  if( index != MAX_BULLETS_NUM )
  {
    switch( direction )
    {
      case DIRECTION_RIGHT:

        x += BULLET_SIZE;

        // fall through

      case DIRECTION_LEFT:

        y += QUARTER_ELEMENT_SIZE;

        break;

      case DIRECTION_DOWN:

        y += BULLET_SIZE;

        // fall through

      case DIRECTION_UP:

        x += QUARTER_ELEMENT_SIZE;

        break;
        
      default:

        y += QUARTER_ELEMENT_SIZE;

        direction = DIRECTION_LEFT;

        break;
    }
    ptr_bullets[ index ].direction = direction;

    ptr_bullets[ index ].x = x;

    ptr_bullets[ index ].y = y;

    ptr_bullets[ index ].request = R_ACTIVE;
  }
  else ret_init = FALSE;

  return ret_init;
}

/*
 * FUNCTION: move_bullet
 *
 * DESCRIPTION: Moves bullet
 *
 * PARAMETERS:
 *   ptr_bullets - pointer to a bullet object
 *   simple_step - simple bullet's step
 *
 * RETURNS: nothing
 */
static void move_bullet( struct Creature *ptr_bullets, int simple_step )
{
  char field_type1, field_type2;

  int new_coord_x1 = 0, new_coord_y1 = 0, new_coord_x2 = 0, new_coord_y2 = 0;

  switch( ptr_bullets->direction )
  {
    case DIRECTION_RIGHT:

      ptr_bullets->x += simple_step;

      new_coord_x1 += BULLET_SIZE;

      new_coord_x2 += BULLET_SIZE;

      new_coord_y2 += BULLET_SIZE;      

      break;

    case DIRECTION_LEFT:

      ptr_bullets->x -= simple_step;

      new_coord_y2 += BULLET_SIZE;

      break;

    case DIRECTION_DOWN:

      ptr_bullets->y += simple_step;

      new_coord_y1 += BULLET_SIZE;

      new_coord_y2 += BULLET_SIZE;

      new_coord_x2 += BULLET_SIZE;

      break;

    case DIRECTION_UP:

      ptr_bullets->y -= simple_step;

      new_coord_x2 += BULLET_SIZE;

      break;
  }
  if( ptr_bullets->x < 0 || ptr_bullets->x >= ELEMENT_SIZE * 
    ptr_current_level->width || ptr_bullets->y < 0 || ptr_bullets->y >= 
      ELEMENT_SIZE * ptr_current_level->height )
  {
    ptr_bullets->request = R_PASSIVE;
  }
  else
  {
    new_coord_x1 += ptr_bullets->x;

    new_coord_y1 += ptr_bullets->y;

    new_coord_x2 += ptr_bullets->x;

    new_coord_y2 += ptr_bullets->y;

    field_type1 = get_intersection( new_coord_x1, new_coord_y1 );

    field_type2 = get_intersection( new_coord_x2, new_coord_y2 );

    if( field_type1 != field_type2 || field_type1 == BRICK )
    {
      ptr_bullets->request = R_PASSIVE;
    }
  }
}

/*
 * FUNCTION: move_bullets
 *
 * DESCRIPTION: Moves bullets
 *
 * PARAMETERS:
 *   ptr_bullets - pointer to a array of bullet's objects
 *
 * RETURNS: nothing
 */
void move_bullets( struct Creature *ptr_bullets )
{
  int i;

  char field_type;

  for( i = 0; i < MAX_BULLETS_NUM; i++ )
  {
     if( ptr_bullets[ i ].request == R_ACTIVE )
     {
       move_bullet( &ptr_bullets[ i ], BULLET_SIMPLE_STEP );
     }
  }
}
