/*
 * FILE: main.c
 *
 * DESCRIPTION:
 *   This is the main module.
 *
 * HISTORY:
 *   May 29, 2001 Created by Dmitry Kaloshin
 */
#include "DataTypes.h"
#include "Maps.h"
#include "actors.h"

struct module_t main_module;  // Application's main module

struct LevelMap* ptr_current_level = 0; // pointer to the levels' maps

struct Graphics* ptr_virtual_graphics = 0; // game field graphics page

struct BitmapSequence* ptr_game_images = 0;

/*
 * FUNCTION: check_intersect
 *
 * DESCRIPTION: check whether 2 objects 
 *     ( having size ELEMENT_SIZE * ELEMENT_SIZE) intersect
 *
 * PARAMETERS:
 *   x1 - x coordinate of the first object
 *   y1 - y coordinate of the first object
 *   x2 - x coordinate of the second object
 *   y2 - y coordinate of the second object
 *
 * RETURNS: TRUE - if objects intersect
 */
bool check_intersect( int x1, int y1, int x2, int y2 )
{
  if( ABS( x1 - x2 ) < ELEMENT_SIZE && ABS( y1 - y2 ) < ELEMENT_SIZE )
    return TRUE;
  else
    return FALSE;
}

/*
 * FUNCTION: check_kill
 *
 * DESCRIPTION: check whether bullet intersects monster 
 *
 * PARAMETERS:
 *   bullet_x - x coordinate of a bullet 
 *   bullet_y - y coordinate of a bullet
 *   monster_x - x coordinate of a monster
 *   monster_y - y coordinate of a monster
 *
 * RETURNS: TRUE - if bullet and monster intersect
 */
bool check_kill( int bullet_x, int bullet_y, int monster_x, int monster_y )
{
  if( bullet_x - monster_x >= ELEMENT_SIZE || 
    monster_x - bullet_x >= BULLET_SIZE ) 
  {
    return FALSE;
  }
  if( bullet_y - monster_y >= ELEMENT_SIZE || 
    monster_y - bullet_y >= BULLET_SIZE ) 
  {
    return FALSE;
  }
  return TRUE;
}

/*
 * FUNCTION: is_clear_field
 *
 * DESCRIPTION: check whether anybody stays on this cell 
 *
 * PARAMETERS:
 *   x - x coordinate of a game field cell
 *   y - y coordinate of a game field cell
 *
 * RETURNS:
 */
bool is_clear_field( int x, int y )
{
  bool ret = TRUE;

  if( !check_intersect( hero.x * ELEMENT_SIZE + hero.x_step, 
      hero.y * ELEMENT_SIZE + hero.y_step, 
       x, y ))
  {
    int current_monster;

    for( current_monster = 0; current_monster < monsters.number; 
      current_monster++ )
    {
       if( check_intersect( monsters.ptr_monsters[ current_monster ].x * 
         ELEMENT_SIZE + monsters.ptr_monsters[ current_monster ].x_step, 
           monsters.ptr_monsters[ current_monster ].y * ELEMENT_SIZE + 
             monsters.ptr_monsters[ current_monster ].y_step, x, y ))
       {
         break;
       }
    }
    if( current_monster != monsters.number )
    {
      ret = FALSE;
    }
  }
  else ret = FALSE;

  return ret;
}

/*
 * FUNCTION: cleanup
 *
 * DESCRIPTION: prepare every actor for release
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void cleanup( void )
{  
  if( ptr_current_level )
  {
    free( ptr_current_level->map_fields );

    free( ptr_current_level );    
  }
}

/*
 * FUNCTION: rest
 *
 * DESCRIPTION: prepare every actor for the rest
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void rest( void )
{
  hero.request = R_DO_NOTHING; 

  game_field.request = R_DO_NOTHING;

  level_splash.request = R_DO_NOTHING;

  media_player.request = R_DO_NOTHING;

  monsters.request = R_DO_NOTHING;

  barricades.request = R_DO_NOTHING;

  doors.request = R_DO_NOTHING;

  fires.request = R_DO_NOTHING;

  bananas.request = R_DO_NOTHING;

  info.request = R_DO_NOTHING;

  stairs.request = R_DO_NOTHING;  
}

/*
 * FUNCTION: draw_gate
 *
 * DESCRIPTION: Drawings the gate
 *
 * PARAMETERS:
 *   field_type - cell type
 *   col - x coord of a cell
 *   row - y coord of a cell
 *
 * RETURNS: nothing
 */
void draw_gate( char field_type, int col, int row )
{
  int style;

  switch( field_type )
  {
    case TL_GATE:
      
      style = BM_NORMAL;

      break;

    case TR_GATE:
      
      style = BM_FLIP_X;

      break;

    case BL_GATE:
      
      style = BM_FLIP_Y;

      break;

    case BR_GATE:                            

      style = BM_FLIP_X | BM_FLIP_Y;

      break;
  }
  Graphics_draw_bitmap( ptr_virtual_graphics,
    BitmapSequence_get_bitmap( ptr_game_images, 19 ), 
      col * ELEMENT_SIZE, row * ELEMENT_SIZE, style );
}

/*
 * FUNCTION: create_buffer
 *
 * DESCRIPTION: Creates a buffer for using and modifying the current level 
 *
 * PARAMETERS:
 *   level - level number
 *
 * RETURNS: nothing
 */
static void create_buffer( int level )
{
  if( !ptr_current_level )
  {
    ptr_current_level = malloc( sizeof * ptr_current_level );

    ptr_current_level->map_fields = malloc( 21 * 17 );
  }  
  ptr_current_level->height = game_maps[ level ].height;

  ptr_current_level->width = game_maps[ level ].width;

  memcpy( ptr_current_level->map_fields, game_maps[ level ].map_fields, 
    21 * 17 );
}

/*
 * FUNCTION: init_map
 *
 * DESCRIPTION: Initializes the current level's map
 *
 * PARAMETERS:
 *   ptr_field_bitmap - pointer to the bitmap's pointer for the game 
 *   level - current level
 *
 * RETURNS: nothing
 */
void init_map( struct Bitmap** ptr_field_bitmap, int level )
{
  int row, col;

  char field_type;  

  create_buffer( level );

  if( *ptr_field_bitmap )
  {
    Bitmap_dtor( *ptr_field_bitmap, FREE_MEMORY );
  }
  *ptr_field_bitmap = malloc( sizeof **ptr_field_bitmap );  

  Bitmap_ctor_Ex2( *ptr_field_bitmap, ELEMENT_SIZE * ptr_current_level->width,
    ELEMENT_SIZE * ptr_current_level->height, 2 );

  if( ptr_virtual_graphics )
  {
    Graphics_dtor( ptr_virtual_graphics, FREE_MEMORY );
  }
  ptr_virtual_graphics = malloc( sizeof *ptr_virtual_graphics );

  Graphics_ctor_Ex( ptr_virtual_graphics, *ptr_field_bitmap ); 

  Graphics_set_draw_mode( ptr_virtual_graphics, DM_PUT );

  Graphics_fill_screen( ptr_virtual_graphics, CLR_BLACK );

  monsters.number = fires.number = bananas.number = 0;
   
  barricades.number_buttons = barricades.number_barricades = 0;

  doors.number = 0;

  media_player.mute = FALSE;

  for( row = 0; row < ptr_current_level->height; row++ )
  { 
    for( col = 0; col < ptr_current_level->width; col++ )
    {
      field_type = 
        ptr_current_level->map_fields[ ptr_current_level->height * col + row ];

      if( field_type < EMPTY )
      {
        if( field_type == MONSTER )
        {
          monsters.number++;
        }
        else if( field_type == FIRE )
        {
          fires.number++;
        }
        else if( field_type == BOOT )
        {
          Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images, 27 ),
              col * ELEMENT_SIZE, row * ELEMENT_SIZE, BM_NORMAL );
        }
        else if( field_type == BANANA )
        {
          bananas.number++;
        }
        else if( field_type == STAIRS )
        {
          stairs.x = col;

          stairs.y = row;
        }
        else if( field_type >= TL_GATE && field_type <= BR_GATE )
        {
          draw_gate( field_type, col, row );
        }
        else if( field_type >= BUTTON && field_type < BARRICADE )
        {
          barricades.number_buttons++; 
        }
        else if( field_type >= BARRICADE && field_type < DOOR )
        {
          barricades.number_barricades++;
        }
        else if( field_type >= DOOR && field_type < KEY )
        {
          doors.number++;
        }
        else if( field_type >= KEY )
        {
          Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images, 22 ),
              col * ELEMENT_SIZE, row * ELEMENT_SIZE, BM_NORMAL );
        }
        else
        {
          Graphics_draw_bitmap( ptr_virtual_graphics,
            BitmapSequence_get_bitmap( ptr_game_images, field_type ),
              col * ELEMENT_SIZE, row * ELEMENT_SIZE, BM_NORMAL );
        }
      }
      else if( field_type == HERO )
      {
        hero.x = hero.start_x = col;

        hero.y = hero.start_y = row;

        hero.x_step = hero.y_step = 0;

        hero.direction = 0;

        hero.type = 0;

        hero.boot = 0;

        hero.fall = 0;
      }
    }
  }
}

/*
 * FUNCTION: main
 *
 * DESCRIPTION: Program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  main_module.m_tick_period = 60;

  if( !start ) global_release();

  return (long)&main_module;
} 
