/*
 * FILE: CybikoPlayer.c
 *
 * DESCRIPTION: Implements the Cybiko's moves
 *
 * HISTORY:
 *   Sep 6, 2000 Created by Aleksey Slesarev
 */
#include "TicTacToe.h"
#include "CybikoPlayer.h"

// Moves the ratings
enum
{ 
  NOT_AVAILABLE = 0, // Move not available

  WORST_MOVE = 1, // Bad move!

  SIMPLE_MOVE = 2, // Simple move

  PRE_WIN = 3, // Open 2

  WILL_FORK = 4, // I'll win in a round!

  BEST_MOVE = 12 // I win!
};    

static const int WIN_DIRECTIONS[ 24 ] = 
{ 
  0, 1, 2, // 1 horizontal

  3, 4, 5, // 2 horizontal

  6, 7, 8, // 3 horizontal

  0, 3, 6, // 1 vertical

  1, 4, 7, // 2 vertical

  2, 5, 8, // 3 vertical

  0, 4, 8, // 1 diagonal

  2, 4, 6  // 2 diagonal
}; 

int first_move; // Index of the game's first move

int second_move; // Index of the game's second move

/*
 * FUNCTION: make_cybiko_move
 *
 * DESCRIPTION: Draws moves made by the CYBIKO computer
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void make_cybiko_move( void )
{
  int index;

  int answer_index;

  int answer_rating;

  int current_rating = 0;

  int best_rating = 0;

  switch( move_number )
  {
    case 0: // First move

      cursor_index = make_first_move();

      break;

    case 1: // Second move

      cursor_index = make_second_move();

      break;

    case 2: // Third move

      cursor_index = make_third_move();

      break;

    default: // Fourth moves and higher

      for( index = 0; index < 9; index++ )
      {
        // Checks move ratings
        current_rating = check_rating( index, move_number % 2 );

        if( current_rating > best_rating )
        {
          // Checks the answer rating
          board_field[ index ] =  move_number % 2;

          for( answer_index = 0; answer_index < 9; answer_index++ )
          {
            answer_rating = check_rating( answer_index, 1 - move_number % 2 );

            if( answer_rating > current_rating && answer_rating >= WILL_FORK )
            {                    
              break;
            }
          }                                 
          if( answer_index == 9 )
          {
            best_rating = current_rating;

            cursor_index = index;
          }          
          board_field[ index ] =  EMPTY_FIELD;
        }
      }
      break;
  }
  set_mark(); // Draws the mark
}

/*
 * FUNCTION: make_first_move
 *
 * DESCRIPTION: Calculates the first move made by the CYBIKO computer
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
int make_first_move( void )
{
  return (int)random( 4 ) * 2;
}

/*
 * FUNCTION: make_second_move
 *
 * DESCRIPTION: Calculates the second move made by the CYBIKO device.
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
int make_second_move( void )
{
  int index = 4;

  if( board_field[ 4 ] != EMPTY_FIELD )
  {
    do 
    {
      index = (int)random( 4 ) * 2;
    }
    while( index == 4 );
  }
  return index;
}

/*
 * FUNCTION: make_third_move
 *
 * DESCRIPTION: Calculates the third move made by the CYBIKO computer
 *
 * PARAMETERS: none 
 *
 * RETURNS: -
 */
int make_third_move( void )
{
  int index;

  if( board_field[ 4 ] == MARK_X ) // "X" mark in the center square
  {
    if( second_move % 2 )  
    { 
      return 0;
    } 
    else
    {
      return 8 - second_move;
    }
  } 
  else if( board_field[ 4 ] == MARK_0 ) // "0" mark in the center square
  {
    return 8 - first_move;
  } 
  else // Center square is empty
  {
    if( second_move % 2 )  
    {
      return 4;
    } 
    else // if(second_move%2) 
    {
      for( index = 0; index < 10; index += 2)
      {
        if( index != 4 && board_field[ index ] == EMPTY_FIELD )
        {
          return index;
        }
      }
    }
  }     
}

/*
 * FUNCTION: check_rating
 *
 * DESCRIPTION: Calculates the rating of the moves after the third round
 *
 * PARAMETERS:
 *   index -
 *   mark_type -
 *
 * RETURNS: -
 */
int check_rating( int index, int mark_type )
{
  int i, j;

  int your_win_direction[ 8 ];

  int partner_win_direction[ 8 ];

  int result  = SIMPLE_MOVE;

  if( board_field[ index ] != EMPTY_FIELD ) // Field not empty
  {
    return NOT_AVAILABLE;
  }
  // Calculate the number of marks in wining directions
  for( i = 0; i < 8; i++ )
  {
    your_win_direction[ i ] = 0;

    partner_win_direction[ i ] = 0;

    for( j = 0; j < 3; j++ )
    {
      int current_mark = board_field[ WIN_DIRECTIONS[ 3 * i + j ]];

      if( current_mark == mark_type || WIN_DIRECTIONS[ 3 * i + j ] == index )
      {
        your_win_direction[ i ]++;
      } 
      else 
        if( current_mark != EMPTY_FIELD )
        {
          partner_win_direction[ i ]++;
        }
    }
  }
  for( i = 0; i < 8; i++ )
  {
    if( your_win_direction[ i ] == 3 )
    {
      return BEST_MOVE; // I win !
    } 
    else if( your_win_direction[ i ] == 2 && !partner_win_direction[ i ] )
    { 
      if( result == PRE_WIN )
      {
        result = WILL_FORK; // It is a fork !
      } 
      else if( result == SIMPLE_MOVE )
      {
        result = PRE_WIN; // Open 2
      }
    } 
    else if( partner_win_direction[ i ] == 2 && !your_win_direction[ i ] )
    {
      result = WORST_MOVE;
    }
  }  
  return result;
}

/*
 * FUNCTION: get_result
 *
 * DESCRIPTION: Gets the current result of the game
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
int get_result( void )
{
  int i;

  for( i = 0; i < 8; i++ )
  {
    int current_mark = board_field[ WIN_DIRECTIONS[ 3 * i ]];

    if( current_mark == board_field[ WIN_DIRECTIONS[ 3 * i + 1 ]]  &&
      current_mark == board_field[ WIN_DIRECTIONS[ 3 * i + 2 ]])
    {
      if( current_mark == player_mark_type )
      {
        return RESULT_YOU;
      } 
      else if( current_mark != EMPTY_FIELD )
      {
        return RESULT_CYBIKO; 
      }
    }
  }
  if( move_number < 9 )
  {
    return RESULT_CONTINUE;
  }
  return RESULT_DRAW;
}
