/*
 * FILE: TicTacToe.c
 *
 * DESCRIPTION: Application's main module
 *
 * HISTORY:
 *   Sep 6, 2000 Created by Aleksey Slesarev
 */
#include "TicTacToe.h"
#include "CybikoPlayer.h"

#define CURSOR_WIDTH 26 // Cursor width (in pixels)

#define CURSOR_HEIGHT 26 // Cursor height (in pixels)  

#define ABSCISSA( index ) ( index % 3 ) // Calculate cursor's x-position

#define ORDINATE( index ) ( index / 3 ) // Calculate cursor's y-position

struct module_t main_module; // Application's main module

struct cDialog player_dialog; // CyWin dialog object

bool game_over; // TRUE if current game is finished

bool exit_game_session; // TRUE if program is finished

int your_score; // Player's score

int partner_score; // Cybiko's score

int cursor_index; // Cursor coordinate

int player_mark_type; // Type of player's mark 

int board_field[ 9 ]; // Board's fields

char* ptr_background; // Background image

int move_number; // Current move number
                                       
/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  int temp_cursor_index;

  bool exit_current_game;

  struct MSequence mseq_title;

  struct KeyParam* ptr_key_param;

  init_module( &main_module );

  // Plays title music from "titel.mus" resource file
  MSequence_ctor( &mseq_title, "title.mus" );

  MSequence_play_background( &mseq_title );

  init_session();

  while( !exit_game_session ) // Application's main loop
  {
    exit_current_game = FALSE;

    init_game();
                  
    draw_board();
  
    if( player_mark_type != MARK_X )
    {
      make_cybiko_move();
    }   
    while( !exit_current_game )
    {
      struct Message* ptr_message = 
        cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );

      switch( ptr_message->msgid )
      {
        case MSG_GOTFOCUS: // This process gets the focus (after invitations)

          draw_board();

          break;

        case MSG_SHUTUP:   
        case MSG_QUIT:

          exit_current_game = exit_game_session = TRUE;

          break;

        case MSG_KEYDOWN:

           ptr_key_param = Message_get_key_param( ptr_message );

           switch( ptr_key_param->scancode )
           {
             case KEY_LEFT:
             case KEY_RIGHT:
             case KEY_DOWN:
             case KEY_UP:

               move_cursor( ptr_key_param->scancode );

               break;

             case KEY_ENTER:

               if( set_mark()) // Player's move
               {
                 if( get_result() != RESULT_CONTINUE )
                 {
                   game_over = TRUE;

                   exit_current_game = show_score();                            
                 } 
                 else // Cybiko's answer
                 {
                   temp_cursor_index = cursor_index;

                   make_cybiko_move();

                   if( get_result() != RESULT_CONTINUE )
                   {  
                     game_over = TRUE;

                     exit_current_game = show_score();                            
                   }
                   cursor_index = temp_cursor_index;
                 }
                 draw_board();
               }
               break;

             case KEY_ESC:

               // Shows CyWin modal dialog:
               // "Do you really want to exit the game?"
               cDialog_ctor( &player_dialog, NULL, str_Really_exit,
                 mbRestart | mbQuit | mbsEsc, 0, main_module.m_process );

               switch( cDialog_ShowModal( &player_dialog ))
               {
                 case mrQuit:

                   exit_game_session = TRUE; /* fall through */

                 case mrRestart:

                   exit_current_game = TRUE;
               }
               cDialog_dtor( &player_dialog, LEAVE_MEMORY );

               draw_board();

               break;

             default:
               
               // Processes previously unprocessed messages
               // Need to process the HELP key
               cWinApp_defproc( main_module.m_process, ptr_message ); 
           }
           break;

        default:
          
          // Processes previously unprocessed messages.
          cWinApp_defproc(main_module.m_process, ptr_message);
      }
      // Deletes message                
      Message_delete(ptr_message);
    }
  }
  // Performs cleanup
  MSequence_dtor( &mseq_title, LEAVE_MEMORY );

  free( ptr_background );

  return 0L;
}

/*
 * FUNCTION: init_session
 *
 * DESCRIPTION: Initializes the game session
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_session( void )
{
  int i;

  exit_game_session = FALSE;       

  your_score = partner_score = 0;            

  srand( (int) clock());

  player_mark_type = random( 1 ) ? MARK_X : MARK_0;

  // Draws background at 1 graphics page
  ptr_background = 
    (char*) malloc( DisplayGraphics_get_bytes_total( main_module.m_gfx ));

  DisplayGraphics_set_page( main_module.m_gfx, 1 ); 
               
  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

  DisplayGraphics_fill_screen( main_module.m_gfx, CLR_WHITE );

  DisplayGraphics_set_color( main_module.m_gfx, CLR_LTGRAY );

  // Half-tone (100x100)
  for( i = 0 ; i < 100; i += 2 )
  {
    DisplayGraphics_draw_line( main_module.m_gfx, i, 0, 100, (100 - i));
    
    DisplayGraphics_draw_line( main_module.m_gfx, 0, i, (100 - i), 100 );
  }
  // 3D border around #
  draw_3d_frame( 29 , 3, 8, 94 );

  draw_3d_frame( 63, 3, 8, 94 );

  draw_3d_frame( 3, 29 , 94, 8 );

  draw_3d_frame( 3, 63 , 94, 8 );

  DisplayGraphics_set_color( main_module.m_gfx, CLR_DKGRAY );

  // Draws #
  DisplayGraphics_fill_rect( main_module.m_gfx, 30, 4, 6, 92 );

  DisplayGraphics_fill_rect( main_module.m_gfx, 64, 4, 6, 92 );

  DisplayGraphics_fill_rect( main_module.m_gfx, 4, 30 , 92, 6 );

  DisplayGraphics_fill_rect( main_module.m_gfx, 4, 64 , 92, 6 );

  DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );

  DisplayGraphics_fill_rect( main_module.m_gfx, 101, 0, 59, 100 );

  draw_info_screen();

  DisplayGraphics_set_page( main_module.m_gfx, 0 );

  // Copies the background image to memory
  memcpy( ptr_background,
    DisplayGraphics_get_page_ptr( main_module.m_gfx, 1 ), 
      DisplayGraphics_get_bytes_total( main_module.m_gfx ));

  draw_board();
}                                    

/*
 * FUNCTION: init_game
 *
 * DESCRIPTION: Initializes each game
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void init_game( void )
{
  int i;

  cursor_index = 4;

  // Clears the board
  for( i = 0; i < 9 ; i++ )
  {
    board_field[ i ] = EMPTY_FIELD;
  }
  player_mark_type = 1 - player_mark_type;

  game_over = FALSE;

  move_number = 0;  

  // Draws background at 1 graphics page
  DisplayGraphics_set_page( main_module.m_gfx, 1 );

  DisplayGraphics_set_draw_mode( main_module.m_gfx , DM_PUT );

  DisplayGraphics_put_background( main_module.m_gfx, ptr_background );
               
  draw_info_screen();
  
  DisplayGraphics_set_page( main_module.m_gfx, 0 );
}

/*
 * FUNCTION: move_cursor
 *
 * DESCRIPTION: Moves the player's cursor in the specified direction
 *
 * PARAMETERS:
 *   direction -
 *
 * RETURNS: nothing
 */
void move_cursor( int direction )
{
  if( cWinApp_has_focus( main_module.m_process ) && !game_over )
  {
    switch( direction )
    {
      case KEY_LEFT:

        if( ABSCISSA( cursor_index ) > 0)
        {
          cursor_index--;
        } 
        else
        {
          cursor_index += 2; // Scrolls the cursor.
        }
        break;

      case KEY_UP:

        if( ORDINATE( cursor_index ) > 0)
        { 
          cursor_index -= 3;
        } 
        else 
        {
          cursor_index += 6; // Scrolls the cursor.   
        }
        break;

      case KEY_RIGHT:

        if( ABSCISSA( cursor_index ) < 2)
        {
          cursor_index++;
        } 
        else
        {
          cursor_index -= ABSCISSA( cursor_index ); // Scrolls the cursor.
        }
        break;

      default:

        if( ORDINATE( cursor_index ) < 2)
        {
          cursor_index += 3;
        } 
        else
        {
          cursor_index = ABSCISSA( cursor_index ); // Scrolls the cursor.
        }
    }
    draw_board();
  }
}

/*
 * FUNCTION: set_mark
 *
 * DESCRIPTION: Sets the player's mark in the specified field
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
bool set_mark( void )
{
  if( board_field[ cursor_index ] == EMPTY_FIELD && !game_over )
  {
    int delay;

    struct MSequence mseq_mark;

    MSequence_ctor( &mseq_mark, "mark.mus" );

    MSequence_play( &mseq_mark );

    board_field[ cursor_index ] = move_number % 2;

    if( !move_number )
    {
      first_move = cursor_index;          
    } 
    else if( move_number == 1 )
    {
      second_move = cursor_index;
    }
    // Selects 1 graphics page
    DisplayGraphics_set_page( main_module.m_gfx, 1 );

    DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

    // Sets transparent color for bitmap
    DisplayGraphics_set_bkcolor( main_module.m_gfx, CLR_WHITE );

    // Draws a bitmap of the mark on 1 graphics page
    draw_lib( 0, move_number % 2, 3 + ABSCISSA( cursor_index ) * 34,
      3 + ORDINATE( cursor_index ) * 34, BM_NORMAL );

    move_number++;

    draw_info_screen();

    DisplayGraphics_set_page( main_module.m_gfx, 0 );
    
    draw_board();

    MSequence_dtor( &mseq_mark, LEAVE_MEMORY );

    return TRUE;
  }
  return FALSE;
}

/*
 * FUNCTION: draw_board
 *
 * DESCRIPTION: Redraws the Cybiko screen
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_board( void )
{
  if( cWinApp_has_focus( main_module.m_process ))
  {
    Graphics_set_draw_mode( main_module.m_gfx, DM_PUT );

    // Copies from 1 graphics page to current graphics page.
    DisplayGraphics_page_copy( main_module.m_gfx, 1, 0, 0, 0, 160, 100 );

    DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );

    // Draws the cursor on the current graphics page
    if( !game_over )
    {
      DisplayGraphics_draw_rect( main_module.m_gfx,
        3 + ABSCISSA( cursor_index ) * 34,
        3 + ORDINATE( cursor_index ) * 34,
        CURSOR_WIDTH, CURSOR_HEIGHT );
    }
    // Shows the current graphics page.
    DisplayGraphics_show( main_module.m_gfx );
  }
} 

/*
 * FUNCTION: draw_info_screen
 *
 * DESCRIPTION: Draws information about the player's moves and marks
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_info_screen( void ) 
{
  bool up_field = ( player_mark_type == move_number % 2 );

  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

  DisplayGraphics_set_color( main_module.m_gfx, 
    up_field ? CLR_WHITE : CLR_LTGRAY );

  DisplayGraphics_fill_rect( main_module.m_gfx, 103, 2, 55, 47 );

  DisplayGraphics_set_color( main_module.m_gfx, 
    up_field ? CLR_LTGRAY : CLR_WHITE );

  DisplayGraphics_fill_rect( main_module.m_gfx, 103, 51, 55, 47 );

  DisplayGraphics_set_font( main_module.m_gfx, cool_normal_font );

  DisplayGraphics_set_color( main_module.m_gfx, CLR_DKGRAY );

  DisplayGraphics_draw_text( main_module.m_gfx,
    "PLAYER", 113, 6 );

  DisplayGraphics_draw_text( main_module.m_gfx,
    "CYBIKO", 113, 55 );

  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_OR );

  DisplayGraphics_set_bkcolor(main_module.m_gfx, CLR_WHITE);

  // Draws  bitmaps of the marks on the right side of the screen.
  draw_lib( 0, player_mark_type, 117, 20, BM_NORMAL );

  draw_lib( 0, 1 - player_mark_type, 117, 69, BM_NORMAL );
}

/*
 * FUNCTION: draw_3d_frame
 *
 * DESCRIPTION: Draws light and dark parts of the 3D frame
 *
 * PARAMETERS:
 *   x0 -
 *   y0 -
 *   width -
 *   height -
 *
 * RETURNS: nothing
 */
void draw_3d_frame( int x0, int y0, int width , int height )
{
  DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );

  DisplayGraphics_draw_rect( main_module.m_gfx, x0, y0, width, height );

  DisplayGraphics_set_color( main_module.m_gfx, CLR_WHITE );

  DisplayGraphics_draw_rect( main_module.m_gfx, x0, y0,
    width - 1, height - 1 );
}

/*
 * FUNCTION: show_score
 *
 * DESCRIPTION: Draws the result of the current game
 *
 * PARAMETERS: none
 *
 * RETURNS: -
 */
bool show_score( void )
{
  int delay;

  struct MSequence mseq_result;

  struct Message* ptr_message;

  char* sz_header_text;

  char sz_result_text[ 64 ];

  bool result = FALSE;

  int game_result = get_result();

  if( game_result == RESULT_YOU )
  {
    your_score++;

    sz_header_text = str_You_win;

    MSequence_ctor( &mseq_result, "win.mus" );
  } 
  else 
    if( game_result == RESULT_CYBIKO )
    {
      partner_score++;

      sz_header_text = str_You_lose;

      MSequence_ctor( &mseq_result, "loss.mus" );
    } 
    else
    {
      sz_header_text = str_Draw;

      MSequence_ctor( &mseq_result, "draw.mus" );
    }
  // Sound effect
  MSequence_play_background( &mseq_result );

  // Waits 3 seconds
  cWinApp_pause( main_module.m_process, 3000 );

  // Deletes all keyboard messages from the message queue
  while( ptr_message = 
    cWinApp_peek_message( main_module.m_process, 
      TRUE, MSG_KEYDOWN, MSG_KEYDOWN ))
  {
    Message_delete( ptr_message );
  }
  // Shows a modal CyWin dialog with the result of the game
  sprintf( sz_result_text, "%s\nPLAYER %d Games Won\nCYBIKO %d Games Won",
    sz_header_text, your_score, partner_score );

  cDialog_ctor( &player_dialog, NULL, sz_result_text, 
    mbRestart | mbQuit | mbReverseOrder | mbsEsc, 0, main_module.m_process );

  switch( cDialog_ShowModal( &player_dialog ))
  {
    case mrQuit:

      exit_game_session = TRUE;

    case mrRestart:

      result = TRUE;
  }        
  cDialog_dtor( &player_dialog, LEAVE_MEMORY );

  MSequence_dtor( &mseq_result, LEAVE_MEMORY );

  return result;
}
