/*
 * FILE: 3d_demo.c
 *
 * DESCRIPTION:
 *   ATTENTION: the 'cy3d.dl' library is required for the example to work.
 *   If this library is absent on your Cybiko computer, the example will
 *   lock up. You can find this library in the 'lib' folder of the SDK
 *   installation.
 *
 * HISTORY:
 *   Feb 9, 2001 Created by Aleksey Slesarev
 */
#include <cywin.h>
#include "walls.h"
#include "objects.h"

/*
 * FUNCTION: is_position_acceptable
 *
 * DESCRIPTION: Checks whether the camera locates in a valid place.
 *
 * PARAMETERS:
 *   p -
 *
 * RETURNS: -
 */
bool is_position_acceptable( point_t p )
{
  // Checks whether the camera is in the room.
  if( p.p_x > 500 || p.p_x < -500 || p.p_y > 1000 || p.p_y < -1000 )
  {
    return FALSE;
  }
  // Checks whether the camera is too close to the sprite.
  if( cy3d_distance2( p, sign_location[ 0 ] ) < 64
     || cy3d_distance2( p, sign_location[ 1 ] ) < 64 )
  {
    return FALSE;
  }
  return TRUE;
}

/*
 * FUNCTION: message_loop
 *
 * DESCRIPTION: Processes the application's messages
 *
 * PARAMETERS:
 *   ptr_main_module -
 *
 * RETURNS: nothing
 */
void message_loop( struct module_t* ptr_main_module )
{
  bool click_enable;

  fixed_t* z_buffer = (fixed_t*) malloc( 160 * ( sizeof *z_buffer ));

  point_t camera;

  struct DirectKeyboard* ptr_keyboard;

  int direction = 90;

  bool exit_application = FALSE;

  char* ptr_video_page = 
    DisplayGraphics_get_page_ptr( ptr_main_module->m_gfx, 0 );

  camera.p_x = 0;

  camera.p_y = 0;

  // Disables keyboard clicks
  click_enable = get_clicks_enabled();

  set_clicks_enabled( FALSE );
 
  // Gets the pointer to the DirectKeyboard object
  ptr_keyboard = DirectKeyboard_get_instance();

  while( !exit_application )
  {
    struct cDialog* ptr_dialog;

    point_t new_position;

    struct Message* ptr_message = 
      cWinApp_get_message( ptr_main_module->m_process, 1, 1, MSG_USER );

    // Renders the next frame
    if( has_focus())
    { 
      // Processes the keyboard input
      DirectKeyboard_scan( ptr_keyboard );

      if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_LEFT ))
      {
        direction += 10;

        direction %= 360;
      }
      if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_RIGHT ))
      {
        direction -= 10;

        direction %= 360;
      }
      if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_UP ))
      {
        new_position = cy3d_move( camera, 64, direction );

        if( is_position_acceptable( new_position ))
        {
          camera = new_position;        
        } 
      }
      if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_DOWN ))
      {
        new_position = cy3d_move( camera, -64, direction );

        if( is_position_acceptable( new_position ))
        {
          camera = new_position;
        }
      }
      // Performs drawing
      cy3d_draw_sky( ptr_video_page, CLR_LTGRAY );

      render_walls( ptr_video_page, camera, direction, z_buffer );

      cy3d_mirror_buffer( ptr_video_page );

      render_objects( ptr_video_page, camera, direction, z_buffer );

      DisplayGraphics_show( ptr_main_module->m_gfx );
    }
    switch( ptr_message->msgid )
    {
      case MSG_LOSTFOCUS: // Application loses focus
        
        set_clicks_enabled( TRUE );

        break;

      case MSG_GOTFOCUS: // Application gets focus
        
        set_clicks_enabled( FALSE );

        break;

      case MSG_SHUTUP: // Processes system exit signal
      case MSG_QUIT:

        exit_application = TRUE;

        break;

      case MSG_KEYDOWN: // Processes keyboard messages

        switch( Message_get_key_param( ptr_message )->scancode )
        { 
          case KEY_ESC:

            set_clicks_enabled( TRUE );

            ptr_dialog = (struct cDialog*) malloc( sizeof *ptr_dialog );

            cDialog_ctor( ptr_dialog, NULL, str_Really_exit,
              mbYes | mbNo | mbsEsc, 0, ptr_main_module->m_process );

            if( cDialog_ShowModal( ptr_dialog ) == mrYes )
            {
              exit_application = TRUE;
            }
            cDialog_dtor( ptr_dialog, FREE_MEMORY );

            set_clicks_enabled( FALSE );

            break;
        }
      default: // Takes care of all unprocessed messages

        cWinApp_defproc( ptr_main_module->m_process, ptr_message );

        break;
    } 
    Message_delete( ptr_message );
  }
  // Performs cleanup
  free( z_buffer );

  DirectKeyboard_dtor( ptr_keyboard, FREE_MEMORY );
  
  // Enables keyboard clicks
  set_clicks_enabled( click_enable );
}
  
/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  long index;

  struct module_t main_module;

  init_module( &main_module );

  // Initializes walls and objects
  init_walls();

  init_objects();

  message_loop( &main_module );

  // Performs cleanup
  release_walls();

  release_objects();

  return 0L; 
}
