/*
 * FILE: objects.c
 *
 * DESCRIPTION: Renders the objects
 *
 * HISTORY:
 *   Feb 9, 2001 Created by Aleksey Slesarev
 */
#include "objects.h"

#define MAX_SIGN 2 // Amount of the Cybiko signs (sprites) in the scene.

#define MAX_COLUMN 4 // Amount of the columns in the scene.

static int sign_frame; // Current frame of the animated Cybiko sign (sprite).

static raster_t* ptr_sign; // Images of the Cybiko sign.

// Coordinates of the Cybiko sign (sprite) in the scene.
point_t sign_location[ 2 ] = 
{ 
  { 0, 512 }, { 0, -512 }
};

static raster_t* ptr_column; // Image of the column.

// Coordinates of the columns in the scene.
static point_t column_location[ 4 ] =
{
  { -512, 1024 }, { 512, 1024 }, { 512, -1024 }, { -512, -1024 }
};

/*
 * FUNCTION: init_objects
 *
 * DESCRIPTION: Initializes images of the scene objects.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void init_objects( void )
{
  sign_frame = 0;

  ptr_sign = cy3d_load( "sign.spr" );

  ptr_column = cy3d_load( "column.spr" );
}

/*
 * FUNCTION: render_objects
 *
 * DESCRIPTION: Performs rendering of the objects.
 *
 * PARAMETERS:
 *   ptr_gfx_buffer -
 *   camera         -
 *   direction      -
 *   z_buffer       -
 *
 * RETURNS: nothing
 */
void render_objects( char* ptr_gfx_buffer, 
  point_t camera, int direction, int* z_buffer )
{
  int index;  

  // Transforms and renders column.
  for( index = 0; index < MAX_COLUMN; index++ )
  {
    cy3d_draw_sprite( ptr_gfx_buffer,
      cy3d_warp( column_location[ index ], camera, direction ),
        ptr_column, 0, z_buffer );
  }
  // Transforms and renders signs.
  for( index = 0; index < MAX_SIGN; index++ )
  {
    int effective_index = ( camera.p_y > 0 ) ? 1 - index : index ;

    cy3d_draw_sprite( ptr_gfx_buffer,
      cy3d_warp( sign_location[ effective_index ], camera, direction ),
        ptr_sign, sign_frame, z_buffer );
  }
  sign_frame++;

  sign_frame %= 19;
}

/*
 * FUNCTION: release_objects
 *
 * DESCRIPTION: Releases images of the scene objects.
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void release_objects( void )
{
  free( ptr_sign );

  free( ptr_column );
}
