/*
 * FILE: Menu.c
 *
 * DESCRIPTION:
 *   This module describes routines for working with the Start menu
 *
 * HISTORY:
 *   Jan 21, 2001 Created by Oleg Diakov
 */
#include <highscore.h>
#include "Menu.h"

struct BitmapSequence images_menu;

struct BitmapSequence images_cursor;

int full_offset_y; // Cursor's current offset

/*
 * FUNCTION: paint_cursor
 *
 * DESCRIPTION: Shows the menu's cursor on the Cybiko display screen
 *
 * PARAMETERS:
 *   ptr_main_module - pointer to an application's main module
 *
 * RETURNS: nothing
 */
void paint_cursor( struct module_t* ptr_main_module )
{
  static int count = 0;  

  DisplayGraphics_set_draw_mode( ptr_main_module->m_gfx, DM_OR );

  DisplayGraphics_set_bkcolor( ptr_main_module->m_gfx, CLR_WHITE );

  DisplayGraphics_draw_bitmap( ptr_main_module->m_gfx,
    BitmapSequence_get_bitmap( &images_cursor, count / 2 ),
    CURSOR_X, CURSOR_Y + full_offset_y, BM_NORMAL );

  count = ( count + 1 ) % ( NUM_OF_CURSOR_BITMAPS * 2 );
}

/*
 * FUNCTION: paint_menu
 *
 * DESCRIPTION: Shows the specified menu on the Cybiko display screen
 *
 * PARAMETERS:
 *   ptr_main_module - pointer to an application's main module
 *
 * RETURNS: nothing
 */
void paint_menu( struct module_t* ptr_main_module )
{
  DisplayGraphics_set_draw_mode( ptr_main_module->m_gfx, DM_PUT );

  DisplayGraphics_draw_bitmap( ptr_main_module->m_gfx,
    BitmapSequence_get_bitmap( &images_menu, 0 ), 0, 0, BM_NORMAL );

  paint_cursor( ptr_main_module );

  DisplayGraphics_show( ptr_main_module->m_gfx );
}

/*
 * FUNCTION: load_menu
 *
 * DESCRIPTION: Loads pictures for the menu and the cursor drawing
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void load_menu( void )
{
  BitmapSequence_ctor_Ex( &images_menu, "menu.pic" );

  BitmapSequence_ctor_Ex( &images_cursor, "cursors.pic" );
}

/*
 * FUNCTION: release_menu
 *
 * DESCRIPTION: Releases resources allocated for images
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void release_menu( void )
{
  BitmapSequence_dtor( &images_cursor, LEAVE_MEMORY );

  BitmapSequence_dtor( &images_menu, LEAVE_MEMORY );
}

/*
 * FUNCTION: get_new_offset
 *
 * DESCRIPTION:
 *   Get the offset from the menu cursor's original position
 *   and store it in members full_offset_y
 *
 * PARAMETERS:
 *   direction - value of the current key pressed
 *
 * RETURNS: nothing
 */
void get_new_offset( int direction )
{
  if( direction == KEY_UP )
  {
    if( full_offset_y ) // first position
    {
      full_offset_y -= OFFSET_Y;
    }
    else
    {
      full_offset_y = ( NUM_OF_MENU - 1 ) * OFFSET_Y;
    }
  }
  else
  {
    if( direction == KEY_DOWN )
    {
      if( full_offset_y != ( NUM_OF_MENU - 1 ) * OFFSET_Y )
      {
        full_offset_y += OFFSET_Y;
      }
      else 
      {
        full_offset_y = 0; // last position
      }
    }
  }
}

/*
 * FUNCTION: enter_menu
 *
 * DESCRIPTION:
 *   This is the entry point for working with the menu. This function
 *   starts the menu, dispatches events, and exits from menu.
 *
 * PARAMETERS:
 *   ptr_main_module - pointer to an application's main module
 *   begin_index     - cursor's starting point within the menu
 *
 * RETURNS: Index of the menu's cursor at the time the menu was closed
 */
int enter_menu( struct module_t* ptr_main_module, int begin_index )
{
  bool exit_menu = FALSE;

  int menu_index;

  full_offset_y = begin_index * OFFSET_Y;

  paint_menu( ptr_main_module );

  while( !exit_menu )
  {
    struct Message* pMessage; 

    struct KeyParam* pKeyParam;

    pMessage = cWinApp_get_message( ptr_main_module->m_process,
      1, 1, MSG_USER );

    switch( pMessage->msgid )
    {
       case MSG_GOTFOCUS: // This process gets the focus (after invitations)

         paint_menu( ptr_main_module );

         break;

       case MSG_SHUTUP:   
       case MSG_QUIT:

         exit_menu = TRUE;

         menu_index = -1;

         Message_delete( pMessage );

         continue;

       case MSG_KEYDOWN:

         pKeyParam = Message_get_key_param( pMessage );

         switch( pKeyParam->scancode )
         {
           case KEY_ESC:

             menu_index = -1;

             exit_menu = TRUE;

             break;

           case KEY_UP:
           case KEY_DOWN:

             if ( pKeyParam->mask != KEYMASK_AUTOREPEAT )
             {
               get_new_offset( pKeyParam->scancode );
             }
             break;

           case KEY_ENTER:

             menu_index = full_offset_y / OFFSET_Y;

             exit_menu = TRUE;

             break;

           default:

             // Processes previously unprocessed messages
             // Need to process the HELP key
             cWinApp_defproc( ptr_main_module->m_process, pMessage );

             break;
         }
         break;

       default:
         
         //  Processes previously unprocessed message
         cWinApp_defproc( ptr_main_module->m_process, pMessage );

         break;
     }
     Message_delete( pMessage );

     if( cWinApp_has_focus( ptr_main_module->m_process ))
     {
       paint_menu( ptr_main_module );
     }
  }
  return menu_index;
}
