/*
 * FILE: Multimedia.c
 *
 * DESCRIPTION: This tutorial shows how to work with multimedia
 *
 * HISTORY:
 *   Feb 14, 2001 Created by Aleksey Slesarev
 */
#include <cywin.h>

struct module_t main_module; // Application's main module

struct MSequence cybiko_music; // Cybiko sound effect object

struct cFrameForm* ptr_main_form; // Application's main form

struct cList* ptr_main_menu; // Application's main menu

int bitmap_x; // x-coordinate for the Cybiko flag's bitmap

int bitmap_y; // y-coordinate for the Cybiko flag's bitmap

/*
 * FUNCTION: cMainForm_proc
 *
 * DESCRIPTION: Processes messages from the main form
 *
 * PARAMETERS:
 *   ptr_form    -
 *   ptr_message -
 *
 * RETURNS: -
 */
bool cMainForm_proc( struct cFrameForm* ptr_form, struct Message* ptr_message )
{
  switch( ptr_message->msgid )
  {
    case MSG_SHUTUP: // Processes the system exit signal
    case MSG_QUIT:

       ptr_form->ModalResult = mrQuit;

       break;

    case MSG_KEYDOWN: // Processes the keyboard

      switch( Message_get_key_param( ptr_message )->scancode )
      {
        case KEY_ESC:

          ptr_form->ModalResult = mrQuit;

          return TRUE;

        case KEY_ENTER:

          // ModalResult containing the selected menu item
          ptr_form->ModalResult = mrOk;

          return TRUE;
      }
  }
  return cFrameForm_proc( ptr_form, ptr_message );
}

/*
 * FUNCTION: cMainForm_ShowModal
 *
 * DESCRIPTION: Shows the main form of the application
 *
 * PARAMETERS:
 *   ptr_form -
 *
 * RETURNS: -
 */
int cMainForm_ShowModal( struct cFrameForm* ptr_form )
{
  ptr_form->HelpContext = 0;

  cFrameForm_Show( ptr_form );

  ptr_form->ModalResult = mrNone;

  while( ptr_form->ModalResult == mrNone )
  {
    struct Message* ptr_message = 
      cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );

    cMainForm_proc( ptr_form, ptr_message );

    cFrameForm_update( ptr_form );

    Message_delete( ptr_message );
  }
  return ptr_form->ModalResult;
}

/*
 * FUNCTION: cMenuList_ctor
 *
 * DESCRIPTION: Creates the main menu of the application
 *
 * PARAMETERS:
 *   ptr_menu -
 *   width    -
 *
 * RETURNS: -
 */
struct cList* cMenuList_ctor( struct cList* ptr_menu, int width )
{
  int index;

  static char* sz_menu_text[ 6 ] = { "Pixels", "Lines", "Rectangles",
    "Bitmaps&Keyboard", "Sound&Vibration", "Quit" };
 
  // Creates a main menu list
  cList_ctor( ptr_menu, width );

  // Fills the menu with items
  for( index = 0; index < 6; index++ )
  {
    struct cItem* ptr_menu_item = 
      (struct cItem*) malloc( sizeof *ptr_menu_item );

    cItem_ctor( ptr_menu_item, width, sz_menu_text[ index ], TRUE, NULL, NULL );

    cList_AddItem( ptr_menu, ptr_menu_item );
  }
  return ptr_menu;
}

/*
 * FUNCTION: draw_pixel
 *
 * DESCRIPTION: Demonstrates pixel drawing
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_pixel( void )
{
  // Places a black pixel at random on the Cybiko display
  DisplayGraphics_set_pixel( main_module.m_gfx, 
    (int) random( SCREEN_WIDTH ), (int) random( SCREEN_HEIGHT ), CLR_BLACK );

  // Shows the Cybiko display
  DisplayGraphics_show( main_module.m_gfx );
}

/*
 * FUNCTION: draw_line
 *
 * DESCRIPTION: Demonstrates lines drawing
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_line( void )
{
  // Draws black lines
  DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );

  // Draws lines with the "XOR" draw mode
  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_XOR );
  
  // Draws a line in a random location
  DisplayGraphics_draw_line( main_module.m_gfx, (int) random( SCREEN_WIDTH ),
    (int) random( SCREEN_HEIGHT ), (int) random( SCREEN_WIDTH ), 
      (int) random( SCREEN_HEIGHT ));

  // Shows the Cybiko display
  DisplayGraphics_show( main_module.m_gfx );
}

/*
 * FUNCTION: draw_rectangle
 *
 * DESCRIPTION: Demonstrates rectangles drawing
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_rectangle( void )
{
  // Sets rectangle color and draw mode
  DisplayGraphics_set_color( main_module.m_gfx, CLR_BLACK );

  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_XOR );

  if( random( 2 ))
  {
    // Draws a random rectangle (frame only)
    DisplayGraphics_draw_rect( main_module.m_gfx, (int) random( SCREEN_WIDTH ),
      (int) random( SCREEN_HEIGHT ), (int) random( SCREEN_WIDTH ),
        (int) random( SCREEN_HEIGHT ));
  }
  else
  {
    // Draws a random rectangle (filled)
    DisplayGraphics_fill_rect( main_module.m_gfx, (int) random( SCREEN_WIDTH ),
      (int) random( SCREEN_HEIGHT ), (int) random( SCREEN_WIDTH ),
         (int) random( SCREEN_HEIGHT ));
  }
  // Shows the Cybiko display
  DisplayGraphics_show( main_module.m_gfx );
}

/*
 * FUNCTION: draw_picture
 *
 * DESCRIPTION: Demonstrates bitmap drawing
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void draw_picture( void )
{
  static int animation_cadr = 0;

  // Clears the screen
  DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT );

  DisplayGraphics_fill_screen( main_module.m_gfx, DM_PUT );

  // Draws a bitmap with image of the Cybiko flag
  draw_lib( 0, ( animation_cadr++ ) % 7, bitmap_x, bitmap_y, BM_NORMAL );

  // Shows the buffer
  DisplayGraphics_show( main_module.m_gfx );
}

/*
 * FUNCTION: play_effects
 *
 * DESCRIPTION: Demonstrates sound effects and vibration effects
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void play_effects( void )
{
  // Plays the music sequence in the background
  if( !MSequence_is_playing( &cybiko_music ))
  {
    MSequence_play_background( &cybiko_music );
  }
  // Sets or resets vibrations
  if( random( 10 ) > 7 )
  {
    vibrate( 255 );
  }
  else 
  {
    vibrate( 0 );
  }
}

/*
 * FUNCTION: on_key
 *
 * DESCRIPTION: Demonstrates Cybiko direct keyboard object usage
 *
 * PARAMETERS:
 *   ptr_keyboard -
 *
 * RETURNS: nothing
 */
void on_key( struct DirectKeyboard* ptr_keyboard )
{
  // Scans the keyboard 
  DirectKeyboard_scan( ptr_keyboard );

  // Calculates the new position of the bitmap object
  if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_LEFT )
    && bitmap_x > 3 )
  { 
    bitmap_x -= 4;
  }
  if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_RIGHT )
    && bitmap_x < ( SCREEN_WIDTH - 51 ))
  { 
    bitmap_x += 4;
  }
  if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_UP )
     && bitmap_y > 3 )
  { 
    bitmap_y -= 4;
  }
  if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_DOWN )
     && bitmap_y < ( SCREEN_HEIGHT - 44 ))
  {
    bitmap_y += 4;
  }
}

/*
 * FUNCTION: cMyApp_ctor
 *
 * DESCRIPTION: Initializes the application's objects
 *
 * PARAMETERS:
 *   ptr_win_app - 
 *
 * RETURNS: -
 */
struct cWinApp* cMyApp_ctor( struct cWinApp* ptr_win_app ) 
{
  bitmap_x = 0;

  bitmap_y = 0;

  // Initializes the music sequence from the "cybiko.mus" file
  MSequence_ctor( &cybiko_music, "cybiko.mus" );

  // Creates the application's main form
  // with the "Graphics Tutorial" caption
  ptr_main_form = malloc( sizeof *ptr_main_form );

  cFrameForm_ctor( ptr_main_form, "Multimedia Tutorial", ptr_win_app );

  // Creates the application's main menu
  ptr_main_menu = (struct cList*) malloc( sizeof *ptr_main_menu );

  cMenuList_ctor( ptr_main_menu, SCREEN_WIDTH - 4 );

  // Adds the menu to the main form
  cFrameForm_AddObj( ptr_main_form, ptr_main_menu, 0, 0 );

  return ptr_win_app;
}

/*
 * FUNCTION: cMyApp_dtor
 *
 * DESCRIPTION: Releases the application's objects
 *
 * PARAMETERS:
 *   ptr_win_app -
 *
 * RETURNS: nothing
 */
void cMyApp_dtor( struct cWinApp* ptr_win_app )
{
  // Releases all allocated objects
  if( ptr_main_form )
  {
    cFrameForm_dtor( ptr_main_form, FREE_MEMORY );
  }
  MSequence_dtor( &cybiko_music, LEAVE_MEMORY );
}

/*
 * FUNCTION: cMyApp_apprun
 *
 * DESCRIPTION: Processes the application's messages
 *
 * PARAMETERS:
 *   ptr_win_app -
 *
 * RETURNS: nothing
 */
void cMyApp_apprun( struct cWinApp* ptr_win_app )
{
  struct DirectKeyboard* ptr_keyboard;

  bool clicks_enabled;

  // Initializes the direct keyboard object
  ptr_keyboard = DirectKeyboard_get_instance();
  
  // Stores in the "clicks_enabled" variable keyboard clicks state
  clicks_enabled = get_clicks_enabled();

  while( ptr_main_form->ModalResult != mrQuit )
  {
    int effect_type;

    // Turns on the keyboard clicks
    set_clicks_enabled( FALSE );

    // Selects the effect
    cMainForm_ShowModal( ptr_main_form );

    cFrameForm_Hide( ptr_main_form );

    effect_type = cList_Sel( ptr_main_menu );

    // Sets 1.help as a help file
    ptr_win_app->HelpContext = 1;

    // Turns off the keyboard clicks
    set_clicks_enabled( FALSE );

    // Clears the screen
    cWinApp_clear_screen();

    while( ptr_main_form->ModalResult == mrOk )
    {
      struct Message* ptr_message = 
        cWinApp_get_message( ptr_win_app, 5, 1, MSG_USER );

      // If this process does not have the focus - do nothing
      if( cWinApp_has_focus( ptr_win_app ))
      {
        switch( effect_type )
        {             
          case 0: // Pixels

            draw_pixel();

            break;

          case 1: // Lines

            draw_line();

            break;

          case 2: // Rectangles

            draw_rectangle();

            break;

          case 3: // Keyboard + bitmap

            on_key( ptr_keyboard );

            draw_picture();

            break;

          case 4: // Music + Vibration

            play_effects();

            break;

          case 5:  // Exit

            ptr_main_form->ModalResult = mrQuit;
        }
      }
      // This was a real message (it was not timed out)
      if( !ptr_message )
      {
        continue;
      }
      switch( ptr_message->msgid )
      {
        case MSG_LOSTFOCUS: // Application has lost its focus

          MSequence_stop( &cybiko_music );

          vibrate( 0 );

          break;

        case MSG_GOTFOCUS: // Application has the focused

          cWinApp_clear_screen();

          break;

        case MSG_QUIT: // Processes system exit signal
        case MSG_SHUTUP:

          ptr_main_form->ModalResult = mrQuit;

          break;

        case MSG_KEYDOWN: // Processes keyboard

          if( Message_get_key_param( ptr_message )->scancode == KEY_ESC )
          {
            vibrate( 0 );

            MSequence_stop( &cybiko_music );

            ptr_main_form->ModalResult = mrCancel;

            break;
          }
        default:

          cWinApp_defproc( main_module.m_process, ptr_message );
      }
      // Deletes the processed message
      Message_delete( ptr_message );
    }
  }
  // Restores clicks and stops vibrations
  vibrate( 0 );

  set_clicks_enabled( clicks_enabled );

  // Releases the direct keyboard object
  DirectKeyboard_dtor( ptr_keyboard, FREE_MEMORY );
}

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  // Initializes application
  init_module( &main_module );

  cMyApp_ctor( main_module.m_process );

  cMyApp_apprun( main_module.m_process );

  // Deletes all allocated objects
  cMyApp_dtor( main_module.m_process );
  
  return 0L;
}
