/*
 * FILE: Private_Chat.c
 *
 * DESCRIPTION: This tutorial shows how to work with Network
 *
 * HISTORY:
 *   Sep 6, 2000 Created by Aleksey Slesarev
 */
#include <cywin.h>

#define mrExit ( mrUser + 1 )

#define mrShutUp ( mrUser + 2 )

#define MSG_PRIVATE_CHAT ( MSG_USER + 1 )

#define MSG_PARTNER_QUIT ( MSG_USER + 2 )

struct module_t main_module; // Application's main module      
                     
cyid_t partner_id; // Cyber ID ofthe chat partner's device

char sz_nickname[ NICKNAMESIZE + 1 ]; // Device owner's nickname

char sz_partner_nickname[ NICKNAMESIZE + 1 ]; // Chat partner's nickname

struct Message* ptr_chat_message; // Chat message.

struct cChatForm
{
  struct cFrameForm* ptr_frame_form; // Application's frame form

  struct cEdit* ptr_send_edit; // String to send

  struct cEdit* ptr_messages_edit; // All chat messages

  struct cClip* ptr_send_clip; // Clipping region of the string to send

  struct cClip* ptr_messages_clip; // Clipping region of the all chat messages

  struct cBevel* ptr_send_bevel; // Bevel around the string to send

  struct cBevel* ptr_messages_bevel; // Bevel around the all chat messages
};

/*
 * FUNCTION: notify_partner
 *
 * DESCRIPTION: Notifies the chat partner about the application has closed
 *
 * PARAMETERS: none 
 *
 * RETURNS: nothing
 */
void notify_partner( void )
{
  struct Message* ptr_quit_message;

  ptr_quit_message = Message_new( sizeof *ptr_quit_message );

  ptr_quit_message->msgid = MSG_PARTNER_QUIT;

  Message_deliver( ptr_quit_message, "Private_Chat",
    partner_id, 20000 | MSG_AUTODELETE);
}

/*
 * FUNCTION: send_chat_message
 *
 * DESCRIPTION: Sends a message with a specified buffer to the chat partner
 *
 * PARAMETERS:
 *   sz_text_buffer -
 *
 * RETURNS: nothing
 */
void send_chat_message( char* sz_text_buffer )
{
  struct Buffer* ptr_message_buffer;

  // Creates a buffer object
  ptr_message_buffer = (struct Buffer*) malloc( sizeof *ptr_message_buffer );
  
  Buffer_ctor( ptr_message_buffer, 20, 10 );

  Buffer_store_string( ptr_message_buffer, sz_text_buffer, 0 );

  // Creates a message
  ptr_chat_message = Message_new( sizeof *ptr_chat_message );

  ptr_chat_message->msgid = MSG_PRIVATE_CHAT;

  // Attaches a buffer with a string to the message
  Message_attach_buffer( ptr_chat_message, ptr_message_buffer );

  // Posts the message to the chat partner
  Message_deliver( ptr_chat_message, "Private_Chat", partner_id, 60000 );
}

/*
 * FUNCTION: cChatForm_add_string
 *
 * DESCRIPTION: Adds the string to the list of all messages
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *   sz_text_buffer -
 *
 * RETURNS: nothing
 */
void cChatForm_add_string( struct cChatForm* ptr_chat_form,
  char* sz_text_buffer )
{
  char* ptr_messages_text;

  char* ptr_new_text;

  int current_scroll = -1;

  // Removes old messages if the combined length of the messages exceeds
  // the buffer capacity
  while( strlen( sz_text_buffer ) + 
    cEdit_GetTextLength( ptr_chat_form->ptr_messages_edit ) > 1022 )
  {
    ptr_messages_text = 
      (char*) malloc( cEdit_GetTextLength( ptr_chat_form->ptr_messages_edit ));

    cEdit_GetText( ptr_chat_form->ptr_messages_edit, ptr_messages_text );

    ptr_new_text = ptr_messages_text;

    while( ptr_new_text &&  *ptr_new_text != '\n' )
    {
      ptr_new_text++;
    }
    cEdit_SetText( ptr_chat_form->ptr_messages_edit, ptr_new_text + 1 );

    free( ptr_messages_text );
  }
  // Appends the message's text to the message list
  if( cEdit_GetTextLength( ptr_chat_form->ptr_messages_edit ))
  {
    cEdit_AppendText( ptr_chat_form->ptr_messages_edit, "\n" );
  }
  cEdit_AppendText( ptr_chat_form->ptr_messages_edit, sz_text_buffer );

  // Scrolls the message list down to the last message
  while( cClip_GetShifty( ptr_chat_form->ptr_messages_clip ) != current_scroll )
  {
    current_scroll = cClip_GetShifty( ptr_chat_form->ptr_messages_clip );

    cClip_Scroll_Ex( ptr_chat_form->ptr_messages_clip, 0, 12 );
  }
}

/*
 * FUNCTION: cChatForm_check_chat_message
 *
 * DESCRIPTION: 
 *   Checks the delivery status of the message sent to the chat partner
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *
 * RETURNS: nothing
 */
void cChatForm_check_chat_message( struct cChatForm* ptr_chat_form )
{
  struct cDialog* ptr_dialog;

  if( !ptr_chat_message )
  {
    return;
  }
  switch( Message_check_delivery( ptr_chat_message ))
  {
    // Message was not delivered to the chat partner
    case DL_ABORT:
    case DL_TIMEOUT:

      ptr_dialog = (struct cDialog*) malloc( sizeof *ptr_dialog );

      cDialog_ctor( ptr_dialog, NULL, 
        str_Link_off, mbOk | mbs4, 0, main_module.m_process );

      cDialog_ShowModal( ptr_dialog );

      cDialog_dtor( ptr_dialog, FREE_MEMORY );

      ptr_chat_form->ptr_frame_form->ModalResult = mrExit;

    // Message was delivered to the chat partner successfully
    case DL_SUCCESS:

      cEdit_SetReadOnly( ptr_chat_form->ptr_send_edit, FALSE );

      Message_delete( ptr_chat_message );

      ptr_chat_message = NULL;
  }
}

/*
 * FUNCTION: cChatForm_ctor
 *
 * DESCRIPTION: Initializes the main form of the application.
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *   sz_caption     -
 *   ptr_win_app    -
 *
 * RETURNS: -
 */
struct cChatForm* cChatForm_ctor( struct cChatForm* ptr_chat_form, 
  char* sz_caption, struct cWinApp* ptr_win_app )
{
  // Allocates memory
  ptr_chat_form->ptr_frame_form = 
    (struct cFrameForm*) malloc(sizeof *ptr_chat_form->ptr_frame_form );

  ptr_chat_form->ptr_send_edit = 
    (struct cEdit*) malloc( sizeof *ptr_chat_form->ptr_send_edit );

  ptr_chat_form->ptr_messages_edit = 
    (struct cEdit*) malloc( sizeof *ptr_chat_form->ptr_messages_edit );

  ptr_chat_form->ptr_send_clip = 
    (struct cClip*) malloc( sizeof *ptr_chat_form->ptr_send_clip );

  ptr_chat_form->ptr_messages_clip = 
    (struct cClip*) malloc( sizeof *ptr_chat_form->ptr_messages_clip );

  ptr_chat_form->ptr_send_bevel = 
    (struct cBevel*) malloc( sizeof *ptr_chat_form->ptr_send_bevel );

  ptr_chat_form->ptr_messages_bevel = 
    (struct cBevel*) malloc( sizeof *ptr_chat_form->ptr_messages_bevel );

  // Creates CyWin objects
  cFrameForm_ctor( ptr_chat_form->ptr_frame_form, sz_caption, ptr_win_app );

  ptr_chat_form->ptr_frame_form->HelpContext = 0;

  cEdit_ctor( ptr_chat_form->ptr_send_edit, cool_normal_font,
    64, es_normal, CLR_BLACK, -1 );

  cEdit_SetText( ptr_chat_form->ptr_send_edit, "Hello!" );

  cEdit_ctor( ptr_chat_form->ptr_messages_edit, cool_normal_font, 
    1024, es_readonly, CLR_BLACK, 140 );

  cClip_ctor( ptr_chat_form->ptr_send_clip, 151, 12 );

  cClip_ctor( ptr_chat_form->ptr_messages_clip, 151, 63 );

  cBevel_ctor( ptr_chat_form->ptr_send_bevel, 154, 14, CLR_BLACK, All );

  cBevel_ctor( ptr_chat_form->ptr_messages_bevel, 154, 65, CLR_BLACK, All );

  // Adds objects to clipping regions
  cClip_AddObj( ptr_chat_form->ptr_send_clip, 
    ptr_chat_form->ptr_send_edit, 0, 0 );

  cClip_AddObj( ptr_chat_form->ptr_messages_clip, 
    ptr_chat_form->ptr_messages_edit, 0, 0 );

  // Adds objects to the frame form
  cFrameForm_AddObj( ptr_chat_form->ptr_frame_form, 
    ptr_chat_form->ptr_send_clip, 2, 70 );

  cFrameForm_AddObj( ptr_chat_form->ptr_frame_form, 
    ptr_chat_form->ptr_send_bevel, 1, 69 );

  cFrameForm_AddObj( ptr_chat_form->ptr_frame_form, 
    ptr_chat_form->ptr_messages_clip, 2, 3 );

  cFrameForm_AddObj( ptr_chat_form->ptr_frame_form, 
    ptr_chat_form->ptr_messages_bevel, 1, 2 );

  cCustomForm_Show( ptr_chat_form->ptr_frame_form );

  return ptr_chat_form; 
}

/*
 * FUNCTION: cChatForm_dtor
 *
 * DESCRIPTION: Releases the main form of the application
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *   memory_flag   -
 *
 * RETURNS: nothing
 */
void cChatForm_dtor( struct cChatForm* ptr_chat_form, int memory_flag )
{
  cEdit_dtor( ptr_chat_form->ptr_send_edit, FREE_MEMORY );

  cEdit_dtor( ptr_chat_form->ptr_messages_edit, FREE_MEMORY );

  cClip_dtor( ptr_chat_form->ptr_send_clip, FREE_MEMORY );

  cClip_dtor( ptr_chat_form->ptr_messages_clip, FREE_MEMORY );

  cBevel_dtor( ptr_chat_form->ptr_send_bevel, FREE_MEMORY ); 

  cBevel_dtor( ptr_chat_form->ptr_messages_bevel, FREE_MEMORY );

  cFrameForm_dtor( ptr_chat_form->ptr_frame_form, FREE_MEMORY );

  if( memory_flag == FREE_MEMORY )
  {
    free( ptr_chat_form );
  }
}

/*
 * FUNCTION: cChatForm_proc
 *
 * DESCRIPTION: Processes messages for the application.
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *   ptr_message   -
 *
 * RETURNS: -
 */
bool cChatForm_proc( struct cChatForm* ptr_chat_form,
  struct Message* ptr_message )
{
  struct cDialog* ptr_dialog;
  
  char sz_message_text[ 80 ];

  switch( ptr_message->msgid )
  {
    case MSG_SHUTUP: 
    case MSG_QUIT:

      // Processes exit message. 
      notify_partner();

      ptr_chat_form->ptr_frame_form->ModalResult = mrShutUp;

      return TRUE;

    case MSG_PRIVATE_CHAT:

      // Processes the chat partner's message
      Buffer_load( Message_get_buffer( ptr_message ), sz_message_text, 0, 0 );

      cChatForm_add_string( ptr_chat_form, sz_message_text );

      break;

    case MSG_PARTNER_QUIT:

      // Processes the chat partner's exit message
      ptr_dialog = (struct cDialog*) malloc( sizeof *ptr_dialog );

      cDialog_ctor( ptr_dialog, NULL, "Partner left private chat.", 
        mbOk | mbs4, 0, main_module.m_process );

      cDialog_ShowModal( ptr_dialog );

      cDialog_dtor( ptr_dialog, FREE_MEMORY );

      ptr_chat_form->ptr_frame_form->ModalResult = mrExit;

      break;

    case MSG_KEYDOWN:
      
      // Processes keyboard input
      switch( Message_get_key_param( ptr_message )->scancode )
      {
        case KEY_ESC:

          // Do you really want to exit? 
          ptr_dialog = (struct cDialog*) malloc( sizeof *ptr_dialog );

          cDialog_ctor( ptr_dialog, NULL, str_Really_exit,
            mbQuit | mbCancel | mbs1, 0, main_module.m_process );

          if( cDialog_ShowModal( ptr_dialog ) == mrQuit )
          {
            notify_partner();

            ptr_chat_form->ptr_frame_form->ModalResult = mrExit;
          }
          cDialog_dtor( ptr_dialog, FREE_MEMORY );

          return TRUE;

        case KEY_ENTER:
          
          // Sends message to the chat partner
          strcpy( sz_message_text, sz_nickname );

          strcat( sz_message_text, ": " );

          if(( cEdit_GetText( ptr_chat_form->ptr_send_edit, sz_message_text + 
            strlen( sz_message_text ))) && 
              (cClip_GetSelectedObject( 
                 ptr_chat_form->ptr_frame_form->ClientArea ) == 
                   (struct cObject*) ptr_chat_form->ptr_send_clip ))
          {
            cChatForm_add_string( ptr_chat_form, sz_message_text );

            cEdit_SetText( ptr_chat_form->ptr_send_edit, "" );

            cEdit_SetReadOnly( ptr_chat_form->ptr_send_edit, TRUE );

            send_chat_message( sz_message_text );
          }
          break;

        case KEY_UP:
        case KEY_DOWN:

          cClip_proc( ptr_chat_form->ptr_messages_clip, ptr_message );

          return TRUE;
      }
  }
  return cFrameForm_proc( ptr_chat_form->ptr_frame_form, ptr_message );
}

/*
 * FUNCTION: cChatForm_ShowModal
 *
 * DESCRIPTION: Shows the main form of the application.
 *
 * PARAMETERS:
 *   ptr_chat_form  -
 *
 * RETURNS: -
 */
int cChatForm_ShowModal( struct cChatForm* ptr_chat_form )
{ 
  ptr_chat_form->ptr_frame_form->ModalResult = mrNone;

  while( ptr_chat_form->ptr_frame_form->ModalResult == mrNone )
  {
    struct Message* ptr_message = 
      cWinApp_get_message( main_module.m_process, 250, 1, MSG_USER + 2 );

    // Checks chat message delivery
    cChatForm_check_chat_message( ptr_chat_form );

    if( !ptr_message )
    {
      continue;
    }
    // Processes system messages
    cChatForm_proc( ptr_chat_form, ptr_message );

    cFrameForm_update( ptr_chat_form->ptr_frame_form );

    Message_delete( ptr_message );
  }
  return ptr_chat_form->ptr_frame_form->ModalResult;
}

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  struct cChatForm* ptr_chat_form;

  char sz_message_text[ 48 ];

  init_module( &main_module );

  ptr_chat_message = NULL;

  strcpy( sz_nickname, finder.mf.f_nick );

  // Initializes the main form of the appliaction
  ptr_chat_form = (struct cChatForm*) malloc( sizeof *ptr_chat_form );

  cChatForm_ctor( ptr_chat_form, 
    "Private Chat Room", main_module.m_process );

  if( argc != 1 ) 
  {
    // Indicates that a remote partner called the device
    partner_id = atoul( argv[ 1 ] );
    
    strcpy( sz_partner_nickname, argv[ 2 ] );
  }
  else
  {
    // Invites the partner to private chat
    sprintf( sz_message_text, 
      "%s invites you to private chat.", sz_nickname );

    partner_id = select_app_partner( main_module.m_process,
      sz_message_text, SGP_NONE, sz_partner_nickname, "Avialable Partners",
        IM_CHAT, NULL );

    if( !partner_id )
    {
      return 0;
    } 
  }
  // Shows the main form of the application  
  cChatForm_ShowModal( ptr_chat_form );

  cChatForm_dtor( ptr_chat_form, FREE_MEMORY );

  // Deletes any remaining chat message
  if( ptr_chat_message )
  {
    Message_delete( ptr_chat_message );
  }
  return 0L;
}
