/*
 * FILE: serial.c
 *
 * DESCRIPTION:
 *   Serial Port tutorial
 *
 *   This example reads the exact number of READ_MAX bytes
 *   from the serial port and echoes them back
 *   excluding system data.
 *   At the same time, it parses the read data
 *   and reports '*wow*' each time it encounters
 *   the word 'test'.
 *
 *   This example doesn't contain any components
 *   required for a Cybiko GUI application,
 *   such as root.inf or root.ico. Thus the only
 *   way to run it is to use the Cybiko SDK console.
 *
 *   ATTENTION: the 'comport.dl' library is required 
 *   for the example to work. If this library is absent 
 *   on your Cybiko computer, the example will lock up. 
 *   You can find this library in the 'lib' folder of the SDK installation.
 *
 *   Run the console and send the compiled example
 *   to the Cybiko computer. Then type 'serial' in the
 *   console's command line box and press <Enter>.
 *   Wait a bit for the 'ready to read data'
 *   prompt and then start typing whatever
 *   you want, remembering to press <Enter>
 *   at times. Anything you type will
 *   be interpreted by the example (not more
 *   than READ_MAX characters though).
 *
 * HISTORY:
 *   Dec 19, 2000 Created by Sergey Gromov
 */
#include <cybiko.h>

#define READ_MAX 50

#define DETECT_BUFFER_SIZE 16

#define TEST_STRING "tset" // 'test' in reverse order

#define REPLY_STRING " *wow* "

#define USER_PROMPT "\nready to read data\n"

/*
 * FUNCTION: my_com_print
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   port -
 *   str  -
 *
 * RETURNS: nothing
 */
static void my_com_print( com_t port, const char* str )
{
  int i;

  for( i = 0; str[ i ]; i++ )
  {
    com_write( port, str[ i ], 0 );
  }
}

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  com_t port;

  char buf[ DETECT_BUFFER_SIZE ];

  memset( buf, 0, sizeof buf );

  // open the default serial port, wait a second for availability
  port = com_open( COMM_DEV_DEFAULT, 1000 );

  if( port )
  {
    int i, j, data;

    my_com_print( port, USER_PROMPT );

    for( i = 0; i < READ_MAX; i++ )
    {
      // read a byte, wait forever
      data = com_read( port, 0 );

      if( data >= 0 ) // data received
      {
        // put the char back in the port, wait forever
        com_write( port, data, 0 );

        if( data == 0xa )
        {
          // The Cybiko console always sends two additional
          // bytes of system data after the line feed; skip them
          com_read( port, 0 );

          com_read( port, 0 );
        }
        // search for the test string
        memmove( buf + 1, buf, DETECT_BUFFER_SIZE - 1 );

        buf[ 0 ] = (char) data;

        if( strncmp( buf, TEST_STRING, strlen( TEST_STRING )) == 0 )
        {
          my_com_print( port, REPLY_STRING );
        }
      }
    }
    // close the port
    com_close( port );

    TRACE( "done reading" );
  }
  else TRACE( "cannot open the port" );

  return 0L;
}
