/*
 * FILE: Step5.c
 *
 * DESCRIPTION: Interface tutorial, Step 5.
 *
 * HISTORY:
 *   Sep 6, 2000 Created by Aleksey Slesarev
 */
#include <cywin.h>

struct module_t main_module; // Application's main module

struct cCustomForm* ptr_menu_form; // Main menu form

struct cList* ptr_menu; // Application's main menu

/*
 * FUNCTION: cListForm_proc
 *
 * DESCRIPTION: Processes messages for the list form
 *
 * PARAMETERS:
 *   ptr_form    -
 *   ptr_message -
 *
 * RETURNS: -
 */
bool cListForm_proc( struct cFrameForm* ptr_form, struct Message* ptr_message )
{
  switch( ptr_message->msgid )
  {
    case MSG_SHUTUP:    // Processes the system exit signal
    case MSG_QUIT:

      ptr_form->ModalResult = mrQuit;

      ptr_menu_form->ModalResult = mrQuit;

      return TRUE;

    case MSG_KEYDOWN:   // Processes keyboard input

      if( Message_get_key_param( ptr_message )->scancode == KEY_ENTER )
      {
        struct cDialog* ptr_edit_dialog;

        char sz_item_text[ 16 ];

        struct cList* ptr_list_object;

        struct cItem* ptr_selected_item;

        int current_position;

        // Retrieves the pointer to the list object
        ptr_list_object = cClip_GetSelectedObject( ptr_form->ClientArea );

        current_position = cList_Sel( ptr_list_object );

        // Retrieves the pointer to the list item currently selected
        ptr_selected_item = cList_GetSelectedObject( ptr_list_object );

        // Initializes a dialog box with the edit line
        ptr_edit_dialog = 
          (struct cDialog*) malloc( sizeof *ptr_edit_dialog );

        cDialog_ctor( ptr_edit_dialog, 0, "Edit Text",
          mbEdit | mbOk | mbCancel | mbs1, 15, ptr_form->CurrApplication );

        cDialog_SetEditText( ptr_edit_dialog, 
          cItem_GetTitle( ptr_selected_item ));

        // If the user has pressed the <OK> button
        if( cDialog_ShowModal( ptr_edit_dialog ) == mbOk )
        {
          // Retrieves the text from the dialog's edit line
          cDialog_GetEditText( ptr_edit_dialog, sz_item_text );

          // Removes an old item from the list
          cList_RemItem( ptr_list_object, ptr_selected_item );

          cItem_dtor( ptr_selected_item, FREE_MEMORY );

          // Inserts a new item in the same position
          ptr_selected_item = 
            (struct cItem*) malloc( sizeof *ptr_selected_item );

          cItem_ctor( ptr_selected_item, SCREEN_WIDTH - 4,
            sz_item_text, FALSE, NULL, NULL );

          cList_AddItem_Ex( ptr_list_object, ptr_selected_item,
            current_position );
        }
        // Releases the dialog
        cDialog_dtor( ptr_edit_dialog, FREE_MEMORY );

        return TRUE;
      }
  }
  // You must call the base object cFrameForm_proc() function
  // to handle the other messages (for example, when the 
  // user changes selection with the <Arrows>
  return cCustomForm_proc( ptr_form, ptr_message );
}

/*
 * FUNCTION: cListForm_ShowModal
 *
 * DESCRIPTION: Shows the form with the list object
 *
 * PARAMETERS:
 *   ptr_form    -
 *
 * RETURNS: -
 */
int cListForm_ShowModal( struct cFrameForm* ptr_form )
{
  cFrameForm_Show( ptr_form );

  ptr_form->ModalResult = mrNone;

  while( ptr_form->ModalResult == mrNone )
  {
    struct Message* ptr_message = 
      cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );

    cListForm_proc( ptr_form, ptr_message );

    cCustomForm_update( ptr_form );

    Message_delete( ptr_message );
  }
  return ptr_form->ModalResult;
}

/*
 * FUNCTION: cMyApp_on_menu1
 *
 * DESCRIPTION: Shows how to use of the CyWin list
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void cMyApp_on_menu1( void )
{
  int index;

  struct cFrameForm* ptr_frame_form;

  struct cList* ptr_menu_list;

  // Initializes the form for the list
  ptr_frame_form = (struct cFrameForm*) malloc( sizeof *ptr_frame_form );

  cFrameForm_ctor( ptr_frame_form, "List Example", main_module.m_process );

  ptr_frame_form->HelpContext = 0;

  // Initializes the list object and fills it with items
  ptr_menu_list = (struct cList*) malloc( sizeof *ptr_menu_list );

  cList_ctor( ptr_menu_list, SCREEN_WIDTH - 4 );
  
  for( index = 1; index < 11; index++ )
  {
    char sz_item_text[ 16 ];

    struct cItem* ptr_menu_item = 
      (struct cItem*) malloc( sizeof *ptr_menu_item );

    sprintf( sz_item_text, "Element %d", index );

    cItem_ctor( ptr_menu_item, SCREEN_WIDTH - 4, sz_item_text,
      FALSE, NULL, NULL );

    cList_AddItem( ptr_menu_list, ptr_menu_item );
  }
  // Adds the list object to the form and shows the form
  cFrameForm_AddObj( ptr_frame_form, ptr_menu_list, 0, 0 );

  cListForm_ShowModal( ptr_frame_form );
  
  // Performs cleanup; all objects connected to the form will be
  // released automatically
  cFrameForm_dtor( ptr_frame_form, FREE_MEMORY );
}

/*
 * FUNCTION: cMyApp_on_menu2
 *
 * DESCRIPTION: Shows how to use CyWin menu
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void cMyApp_on_menu2( void )
{
  int index;

  struct BitmapSequence* ptr_images;

  struct cFrameForm* ptr_frame_form;

  struct cList* ptr_menu_list;

  // Initializes the bitmap sequence object
  ptr_images = (struct BitmapSequence*) malloc( sizeof *ptr_images );

  BitmapSequence_ctor_Ex( ptr_images, "digits.pic" );

  // Initializes the form for the list 
  ptr_frame_form = (struct cFrameForm*) malloc( sizeof *ptr_frame_form );

  cFrameForm_ctor( ptr_frame_form, "Menu Example", main_module.m_process );

  ptr_frame_form->HelpContext = 0;

  // Initializes the list object and fills it with items
  ptr_menu_list = (struct cList*) malloc( sizeof *ptr_menu_list );

  cList_ctor( ptr_menu_list, SCREEN_WIDTH - 4 );
  
  for( index = 1; index < 11; index++ )
  {
    char sz_item_text[ 16 ];
    
    struct cSItem* ptr_menu_item = 
      (struct cSItem*) malloc( sizeof *ptr_menu_item );

    sprintf( sz_item_text, "Element %d", index );

    cSItem_ctor( ptr_menu_item, SCREEN_WIDTH - 4, sz_item_text,
      BitmapSequence_get_bitmap( ptr_images, index - 1 ));

    cList_AddItem( ptr_menu_list, ptr_menu_item );
  }
  // Adds the list object to the form
  cFrameForm_AddObj( ptr_frame_form, ptr_menu_list, 0, 0 );

  // Shows the form
  cFrameForm_ShowModal( ptr_frame_form );

  // Performs cleanup; all objects connected to the form will be
  // released automatically
  cFrameForm_dtor( ptr_frame_form, FREE_MEMORY );

  BitmapSequence_dtor( ptr_images, FREE_MEMORY );
}

/*
 * FUNCTION: cMyApp_on_menu3
 *
 * DESCRIPTION: Shows how to use the CyWin dialog
 *
 * PARAMETERS: none
 *
 * RETURNS: nothing
 */
void cMyApp_on_menu3( void )
{
  struct cDialog* ptr_dialog =
    (struct cDialog*) malloc( sizeof *ptr_dialog );

  cDialog_ctor( ptr_dialog, 0, "dummy dialog 3", 
    mbQuit | mbCancel | mbEdit | mbs6, 10, main_module.m_process );

  cDialog_ShowModal( ptr_dialog );
  
  cDialog_dtor( ptr_dialog, FREE_MEMORY );
}

/*
 * FUNCTION: cMainMenu_proc
 *
 * DESCRIPTION: Processes messages for the main menu
 *
 * PARAMETERS:
 *   ptr_form    -
 *   ptr_message -
 *
 * RETURNS: -
 */
bool cMainMenu_proc( struct cCustomForm* ptr_form,
  struct Message* ptr_message )
{
  switch( ptr_message->msgid )
  {
    case MSG_SHUTUP: // Processes the system exit signal
    case MSG_QUIT:

      ptr_form->ModalResult = mrQuit;

      return TRUE;

    case MSG_KEYDOWN: // Processes keyboard input

      if( Message_get_key_param( ptr_message )->scancode == KEY_ENTER )
      {
        switch( cList_Sel( ptr_menu ))
        {
          case 0:

            cMyApp_on_menu1();

            break;

          case 1:

            cMyApp_on_menu2();

            break;

          case 2:

            cMyApp_on_menu3();
        }
        return TRUE;
      }
  }
  return cCustomForm_proc( ptr_form, ptr_message );
}

/*
 * FUNCTION: cMainMenu_ShowModal
 *
 * DESCRIPTION: Shows the main menu of the application
 *
 * PARAMETERS:
 *   ptr_form    -
 *
 * RETURNS: -
 */
int cMainMenu_ShowModal( struct cCustomForm* ptr_form )
{
  cCustomForm_Show( ptr_form );

  ptr_form->ModalResult = mrNone;

  while( ptr_form->ModalResult == mrNone )
  {
    struct Message* ptr_message = 
      cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );

    cMainMenu_proc( ptr_form, ptr_message );

    cCustomForm_update( ptr_form );

    Message_delete( ptr_message );
  }
  return ptr_form->ModalResult;
}

/*
 * FUNCTION: cMainForm_ctor
 *
 * DESCRIPTION: Initializes the application's main form
 *
 * PARAMETERS:
 *   ptr_form    -
 *   sz_caption  -
 *   ptr_win_app -
 *
 * RETURNS: -
 */
struct cFrameForm* cMainForm_ctor( struct cFrameForm* ptr_form,
  char* sz_caption, struct cWinApp* ptr_win_app )
{
  int index;

  struct cText* ptr_text;

  struct rect_t form_rect;

  // Initializes the application's main form
  cFrameForm_ctor( ptr_form, sz_caption, ptr_win_app );

  ptr_form->HelpContext = 0;

  ptr_form->ModalResult = mrNone;

  // Creates a menu
  ptr_menu = (struct cList*) malloc( sizeof *ptr_menu );

  cList_ctor( ptr_menu, 80 );
  
  // Adds items to the menu
  for( index = 1; index < 4; index++ )
  {
    char sz_menu_text[ 8 ];

    struct cItem* ptr_menu_item = 
      (struct cItem*) malloc( sizeof *ptr_menu_item );

    sprintf( sz_menu_text, "Item %d", index );

    cItem_ctor( ptr_menu_item, 80, sz_menu_text, TRUE, NULL, NULL );

    cList_AddItem( ptr_menu, ptr_menu_item );
  }
  
  // Creates the form for the menu
  ptr_menu_form = (struct cCustomForm*) malloc( sizeof *ptr_menu_form );

  form_rect.x = 40;

  form_rect.y = 20;

  form_rect.w = 86;

  form_rect.h = 60;

  cCustomForm_ctor( ptr_menu_form, &form_rect,
    "Main Menu", TRUE, ptr_win_app );

  ptr_menu_form->HelpContext = 0;

  ptr_menu_form->ModalResult = mrNone;

  // Adds the menu to the menu form
  cCustomForm_AddObj( ptr_menu_form, ptr_menu, 0, 0 );

  cCustomForm_Hide( ptr_menu_form );
  
  // Creates text
  ptr_text = (struct cText*) malloc( sizeof *ptr_text );

  cText_ctor( ptr_text, "Press <Enter>", cool_normal_font, CLR_BLACK );

  // Adds the text to the main form
  cFrameForm_AddObj( ptr_form, ptr_text, 42, 35 );
}

/*
 * FUNCTION: cMainForm_proc
 *
 * DESCRIPTION: Processes messages for the main form
 *
 * PARAMETERS:
 *   ptr_form     -
 *   ptr_message  -
 *
 * RETURNS: -
 */
bool cMainForm_proc( struct cFrameForm* ptr_form, struct Message* ptr_message )
{
  struct KeyParam* ptr_key_param;

  struct cDialog* ptr_quit_dialog;
  
  switch( ptr_message->msgid )
  {
    case MSG_SHUTUP: // Processes the system exit signal.
    case MSG_QUIT:

      ptr_form->ModalResult = mrQuit;

      break;

    case MSG_KEYDOWN: // Processes keyboard input

      ptr_key_param = Message_get_key_param( ptr_message );

      switch( ptr_key_param->scancode )
      {
        case KEY_ESC:

          // Creates the dialog box with <Quit> and <Cancel> buttons
          ptr_quit_dialog = 
            (struct cDialog*) malloc( sizeof *ptr_quit_dialog );

          cDialog_ctor( ptr_quit_dialog, 0, str_Really_exit,
            mbQuit | mbCancel | mbs1, 0, ptr_form->CurrApplication );

          // The cDialog_ShowModal() function shows the dialog
          // and returns the code of the button pressed
          // If user pressed the <Quit> button, the application will close
          // Otherwise it will continue processing messages
          if( cDialog_ShowModal( ptr_quit_dialog ) == mrQuit )
          {
            ptr_form->ModalResult = mrQuit;
          }
          cDialog_dtor( ptr_quit_dialog, FREE_MEMORY );
    
          return TRUE;

        case KEY_ENTER:

          // When user presses the <Enter> key in the application's main form, 
          // the application creates a main menu
          cMainMenu_ShowModal( ptr_menu_form );

          // Hides the menu
          cCustomForm_Hide( ptr_menu_form );

          return TRUE;
      }
  }
  return cFrameForm_proc( ptr_form, ptr_message );
}

/*
 * FUNCTION: cMainForm_ShowModal
 *
 * DESCRIPTION: Shows the main form for the application
 *
 * PARAMETERS:
 *   ptr_form -
 *
 * RETURNS: -
 */
int cMainForm_ShowModal( struct cFrameForm* ptr_form )
{ 
  cFrameForm_Show( ptr_form );

  ptr_form->ModalResult = mrNone;

  while( ptr_form->ModalResult == mrNone && 
    ptr_menu_form->ModalResult != mrQuit )
  {
    struct Message *ptr_message = 
      cWinApp_get_message( ptr_form->CurrApplication, 0, 1, MSG_USER );

    cMainForm_proc( ptr_form, ptr_message );

    Message_delete( ptr_message );
  }
  return ptr_form->ModalResult;
}

/*
 * FUNCTION: main
 *
 * DESCRIPTION: program entry point
 *
 * PARAMETERS:
 *   argc - number of arguments
 *   argv - array of 'argc' arguments passed to the application
 *   start - TRUE if the application is being initialized, FALSE otherwise
 *
 * RETURNS: 0L
 */
long main( int argc, char* argv[], bool start )
{
  struct cFrameForm* ptr_main_form;

  // Initializes the application
  init_module( &main_module );
  
  // Creates the application's main form
  // with the "Interface Tutorial" caption
  ptr_main_form = (struct cFrameForm*) malloc( sizeof *ptr_main_form );

  cMainForm_ctor( ptr_main_form, "Interface Tutorial", main_module.m_process );
  
  // Shows the main form
  cMainForm_ShowModal( ptr_main_form );

  // Performs cleanup
  cFrameForm_dtor( ptr_main_form, FREE_MEMORY );

  return 0L;
}
