/** 
 *  @file appgen.h
 *  Hooks to CyWin class AppGeneric.
 */

#ifndef __APPGEN_H_INCLUDE__
#define __APPGEN_H_INCLUDE__

//============================
//=                          =
//=    object AppGeneric     =
//=  The application object. =
//=                          =
//============================

/**
 *   @ingroup application_group
 *   @brief the base structure for applications.
 *
 *   AppGeneric is a simple application structure. In other words, it is 
 *   the base structure for applications. It implements 
 *   such functions as: receiving the graphics context for 
 *   application needs, cleaning the screen, processing messages and
 *   canceling MSG_SHUTUP events. This object cannot be instantiated!
 *   It can be obtained using init_module().
 *   @see application_group
 */
struct AppGeneric : public Process
{
#ifndef _SKIP_DOC
  bool cls;
  bool exit_to_root;
  bool enable_section_keys;
  bool PADDING_1;
  int HelpContext;
  int  PADDING_2;
};
#endif

/**
 *  Obtains the graphics context.
 *  @return A pointer to a DisplayGraphics object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DisplayGraphics* ptr_gfx;
 *       ...
 *       // Clears the screen.
 *       AppGeneric_clear_screen();
 *       ...
 *       // Draws a filled rectangle. 
 *       ptr_gfx = AppGeneric_init_display();
 *       DisplayGraphics_set_color( ptr_gfx , CLR_BLACK );
 *       DisplayGraphics_fill_rect( ptr_gfx , 5, 5, 30, 30 );
 *       DisplayGraphics_show( ptr_gfx );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 4
#endif
struct DisplayGraphics* AppGeneric_init_display(void);

/**
 *  Clears the screen with a white color.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct DisplayGraphics* ptr_gfx;
 *       ...
 *       //  Clears the screen.
 *       AppGeneric_clear_screen();
 *       ...
 *       //  Draws a filled rectangle. 
 *       ptr_gfx = AppGeneric_init_display();
 *       DisplayGraphics_set_color( ptr_gfx , CLR_BLACK );
 *       DisplayGraphics_fill_rect( ptr_gfx , 5, 5, 30, 30 );
 *       DisplayGraphics_show( ptr_gfx );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 5
#endif
void AppGeneric_clear_screen(void);

/**
 *  Provides generic message processing. Should be called on every message 
 *  that the user doesn't process.
 *  @param ptr_app_generic A pointer to the initialized AppGeneric object
 *  @param ptr_message A pointer to aprocessed Message
 *  @return TRUE if the message was processed
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Sets the resource "0.help" as the help file
 *       main_module.m_process->HelpContext = 0;
 *       ptr_message = AppGeneric_get_message( main_module.m_process, 
 *                                             0, 
 *                                             1, 
 *                                             MSG_USER );
 *       //  The help screen appears if the user presses the '?' button
 *       AppGeneric_proc( main_module.m_process, ptr_message );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 7
#endif
bool AppGeneric_proc(struct AppGeneric* ptr_app_generic, struct Message* ptr_message);

/**
 *  Cancels the MSG_SHUTUP message.
 *  Cancels a task switch operation previously launched
 *  by another application in the task manager, and deletes any MSG_SHUTUP
 *  messages from the message queue. Requires the AppGeneric object pointer.
 *  @param ptr_app_generic A pointer to initialized AppGeneric object.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       struct Message* ptr_message;
 *       ...
 *       ptr_message = AppGeneric_get_message( main_module.m_process, 
 *                                             0, 
 *                                             1, 
 *                                             MSG_USER );
 *       if( ptr_message->msgid == MSG_SHUTUP )
 *       {                                    
 *         //  We do not need to exit.
 *         //  Cancels shutup and deletes all MSG_SHUTUP messages.
 *         AppGeneric_cancel_shutup(  main_module.m_process );
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see MSG_SHUTUP
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 9
#endif
void AppGeneric_cancel_shutup(struct AppGeneric* ptr_app_generic);

/**
 *  Cancels the MGS_SHUTUP message statically.
 *  Used to cancel a task switch operation previously launched
 *  by another application in the task manager.
 *  Doesn't require the AppGeneric object pointer,
 *  but doesn't delete any MSG_SHUTUP messages.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       struct Message* ptr_message;
 *       ...
 *       ptr_message = AppGeneric_get_message( main_module.m_process, 
 *                                             0, 
 *                                             1, 
 *                                             MSG_USER );
 *       if( ptr_message->msgid == MSG_SHUTUP )
 *       {
 *         //  We do not need to exit.
 *         //  Cancels shutup.
 *         AppGeneric_ext_cancel_shutup();
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see MSG_SHUTUP
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 307
#endif
void AppGeneric_ext_cancel_shutup(void);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_pause          SystemThread_pause
#else
/**
 *  Pauses execution of the application.
 *  @param ptr_app_generic A pointer to an initialized AppGeneric object 
 *  @param timeout The time span for sleeping
 *  @return FALSE if timeout has passed
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       play_tone( 30 );
 *       AppGeneric_pause( main_module.m_process, 200 );
 *       play_tone( -1 );
 *       ...
 *  @endcode
 */
bool AppGeneric_pause(struct AppGeneric* ptr_app_generic, clock_t timeout);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_request_focus  Process_request_focus
#else
/**
 *  Requests that the application receive the focus. The application with the focus
 *  has actual access to the screen and receives the 
 *  keyboard and pointing device events. When the system gets this call, it 
 *  passes the focus to the process that requested it and sends it the message MSG_GOTFOCUS.
 *  After that, the system sends the message MSG_LOSTFOCUS to the process that had the focus 
 *  before this call.
 *  @param ptr_app_generic A pointer to the initialized AppGeneric object
 *  @return A pointer to the Process object that had the focus before this call
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      AppGeneric_request_focus( main_module.m_process );
 *      ...
 *      if( AppGeneric_has_focus( main_module.m_process ) )
 *      {
 *        //  Draws something.
 *        ...
 *      }  
 *      ... 
 *  @endcode
 *  @see MSG_GOTFOCUS, MSG_LOSTFOCUS
 */
struct Process* AppGeneric_request_focus(struct Process* ptr_app_generic);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_has_focus      Process_has_focus 
#else
/**
 *  Tests whether the process has the device's focus.
 *  @param ptr_app_generic A pointer to an initialized AppGeneric object
 *  @return TRUE if the application has the focus
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      AppGeneric_request_focus( main_module.m_process );
 *      ...
 *      if( AppGeneric_has_focus( main_module.m_process ) )
 *      {
 *        // Draws something.
 *        ...
 *      }
 *      ... 
 *  @endcode
 */
bool AppGeneric_has_focus(struct AppGeneric* ptr_app_generic);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_get_display    Process_get_display
#else
/**
 *  Retrieves DisplayGraphics object to access the Cybiko display.
 *  @return A pointer to the DisplayGraphics object
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      struct Bitmap bmp;
 *      ...
 *      //  Creates a bitmap from the file "root.ico".
 *      Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *      ...
 *      //  Draws bitmap.
 *      DisplayGraphics_draw_bitmap( AppGeneric_get_display(), 
 *                                   &bmp, 
 *                                   30, 
 *                                   40, 
 *                                   BM_NORMAL );
 *      DisplayGraphics_show( AppGeneric_get_display() );
 *      ...
 *      Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *      ... 
 *  @endcode
 */
struct DisplayGraphics* AppGeneric_get_display(void);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_peek_message(ptr_app_generic, remove, min, max) \
        Queue_peek_message((char*)(ptr_app_generic) + 108, (remove), (min), (max)) ///<  peeks or get message from queue. It calls Queue_peek_message.
#else
/**
 *  Peeks or gets a message from the queue. 
 *  If 'remove' is TRUE, the message will be removed. This function does not wait for a message 
 *  to be available; it returns 0 immediately if there are no messages in 
 *  the specified range. Don't use it unless you really need to.
 *  @param ptr_app_generic A pointer to the initialized AppGeneric object
 *  @param remove If TRUE, the message will be removed from the message queue
 *  @param min Specifies the integer value of the lowest message value to be retrieved
 *  @param max Specifies the integer value of the highest message value to be retrieved
 *  @return Pointer to the Message if there is one in the process' queue; otherwise 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Removes all keyboard messages from the queue.
 *       while( ptr_message = AppGeneric_peek_message( main_module.m_process, 
 *                                                     TRUE, 
 *                                                     MSG_KEYDOWN, 
 *                                                     MSG_KEYDOWN ) )
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see AppGeneric_get_message.
 */
struct Message* AppGeneric_peek_message(struct AppGeneric* ptr_app_generic, bool remove, int min, int max);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_get_message(ptr_app_generic, timeout, min, max) \
        Queue_get_message((char*)(ptr_app_generic) + 108, (timeout), (min), (max))
#else
/**
 *  Returns the message for the specified interval. The calling thread waits 
 *  until the first message appears in the queue. Be careful, not to overload the queue.
 *  Delete the message when you are finished with it.
 *  @param ptr_app_generic A pointer to initialized AppGeneric object
 *  @param timeout Timeout value(in milliseconds)
 *  @param min Deprecated
 *  @param max Deprecated  
 *  @return Pointer to the Message if there is one in the process' queue,
 *  or 0 if a timeout occurs
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = AppGeneric_get_message( main_module.m_process, 
 *                                             0, 
 *                                             1, 
 *                                             MSG_USER );
 *       if( ptr_message->msgid == MSG_KEYDOWN )
 *       {
 *         if( Message_get_key_param( ptr_message )->scancode == MSG_ESC )
 *         {
 *           //  Processes 'Esc' key.
 *         }
 *       }
 *       ...
 *       Message_delete( ptr_message );
 *       ...
 *  @endcode
 *  @see AppGeneric_put_message, AppGeneric_peek_message.
 */
struct Message* AppGeneric_get_message (struct AppGeneric* ptr_app_generic, long timeout, int min, int max);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_put_message(ptr_app_generic, ptr_message) Queue_put_message((char*)(ptr_app_generic) + 108, (ptr_message))
#else
/**
 *  Puts a Message in the application's Queue.
 *  @param ptr_app_generic A pointer to the initialized AppGeneric object
 *  @param ptr_message A pointer to the message to send
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_MAKE_TASK  MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = AppGeneric_get_message( main_module.m_process, 
 *                                             0, 
 *                                             1, 
 *                                             MSG_MAKE_TASK );
 *       if( ptr_message->msgid == MSG_MAKE_TASK )
 *       {
 *         ...
 *         //  Performs some periodical calculation.
 *         ...
 *         sleep( 250 );
 *         AppGeneric_put_message( main_module.m_process, ptr_message );
 *       }
 *       else
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see Process_get_message, Process_peek_message.
 */
void AppGeneric_put_message(struct AppGeneric* ptr_app_generic, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define AppGeneric_get_name(ptr_app_generic)  Queue_get_name((char*)(ptr_app_generic) + 108)
#else
/**
 *  Retrieves the name of the application.
 *  @param ptr_app_generic A pointer to an initialized AppGeneric object
 *  @return The name of the application
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       TRACE( "Hello from the %s application", 
 *             AppGeneric_get_name( main_module.m_process ) );
 *       ...
 *  @endcode
 */
char* AppGeneric_get_name(struct AppGeneric* ptr_app_generic);
#endif

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __APPGEN_H_INCLUDE__


