/** 
 *  @file archive.h
 *  CyOS archives.
 *
 */
#ifndef __ARCHIVE_H_INCLUDED__
#define __ARCHIVE_H_INCLUDED__

/** 
 *  Archive storage.
 *  @ingroup filesystem_overview
 *  @brief A file that holds more files inside, each one of may be compressed.
 * 
 *  An archive is a file that holds more files inside it, with possible compression of every one of them.
 *  Right now only one compression method is supported,
 *  but space for 254 more is reserved. You get standard Input* when opening 
 *  a file from inside the archive, and you should handle it like regular input. If 
 *  files are not compressed, random access methods can be used for their Input
 *  objects.
 *  <br>
 *  <br>
 *  Currently, compressed streams do not support random access.
 *  Using archives is the preferred way to handle large groups of files. Getting files from
 *  an archive might be faster than opening a file, especially if there are a lot 
 *  of them.
 *  <br>
 *  <br>
 *  You must call the Archive_ctor() function before retrieving a file from an archive and Archive_dtor() afterward. The pointer to the archive of the current application can be obtained 
 *  in the following manner:
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      struct module_t main_module;
 *      ...
 *      init_module( &main_module );
 *      //  main_module.m_process->module->archive is the pointer to the
 *      //  Archive of the current application.
 *      ...
 *  @endcode
 *  @see filesystem_overview
 */
struct Archive
{
#ifndef _SKIP_DOC
  char* name;
  void* index;
  short total_entries;
  short PADDING;
};
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Archive_ctor(ptr_archive, archive_name) {  \
        (ptr_archive)->name = strdup(archive_name);\
        (ptr_archive)->index = 0;                  \
        Archive_init(ptr_archive); } 
#else
/**
 *  Initializes an Archive object for a specified archive file.
 *  @param ptr_archive A pointer to an initialized Archive object
 *  @param sz_archive_name The name of the archive file
 *  @return None
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        } 
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_dtor
 */
void Archive_ctor(struct Archive* ptr_archive, char* sz_archive_name);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Archive_dtor(ptr_archive, memory_flag) { \
        free((ptr_archive)->name);  \
        free((ptr_archive)->index); \
        if((memory_flag) == 3) free((ptr_archive));} 
#else
/**
 *  Destructor of an Archive object. It frees all used resources.
 *  @param ptr_archive A pointer to the Archive object to be destroyed
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If malloc() was used to allocate the object's memory, use FREE_MEMORY.
 *  If the object was static or allocated in stack, use LEAVE_MEMORY.
 *  @return None
 *  @code
 *      #include <cybiko.h>       
 *      ...
 *      {     
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        }
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
void Archive_dtor( struct Archive* ptr_archive, int memory_flag);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Archive_is_good(ptr_archive) ((ptr_archive)->index != 0) 
#else
/**
 *  Checks whether the Archive object is valid.
 *  @param ptr_archive A pointer to the Archive object to be checked
 *  @return TRUE if the Archive object is valid
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        }
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *     }
 *  @endcode
 */
bool Archive_is_good(struct Archive* ptr_archive);
#endif

#ifndef _SKIP_DOC
__cyos 789
void Archive_init(struct Archive* ptr_archive);
#endif

/**
 *  Opens an Input for the specified resource index.
 *  All files in the archive are numbered. These numbers are often referenced as
 *  resource IDs.
 *  Archive_open constructs the right kind of stream for an entry and returns it. 
 *  Use it the same way as a regular stream. Important: don't forget to delete it when you're done with it
 *  @param ptr_archive A pointer to an initialized Archive object
 *  @param entry The entry number (resource id)
 *  @return Input object to read the specified file from the archive, or NULL if there
 *  is no such Id or not enough memory for the operation
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        struct BitmapSequence movie_frames;
 *        struct Archive resources;
 *        struct Input* ptr_input;
 *        ...
 *        Archive_ctor( &resources, "movie.dat" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          ptr_input = Archive_open( &resources, 0 );
 *          BitmapSequence_ctor( &movie_frames );
 *          BitmapSequence_load( &movie_frames, ptr_input );
 *          ...
 *        }
 *        ...
 *        BitmapSequence_dtor( &movie_frames, LEAVE_MEMORY );
 *        Input_dtor( ptr_input, FREE_MEMORY );
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_open_Ex.
 */
#ifndef _SKIP_DOC
__cyos 916
#endif
struct Input* Archive_open(struct Archive* ptr_archive, int entry);

/**
 *  Opens an Input from the archive for the file named.
 *  It works just like Archive_open() but looks for files by name.
 *  @param ptr_archive A pointer to an initialized Archive object
 *  @param sz_name The name of the file inside the archive
 *  @return A pointer to a corresponding Input object, or NULL if the file is not found or
 *  there is not enough memory to open it.
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        struct BitmapSequence movie_frames;
 *        struct Archive resources;
 *        struct Input* ptr_input;
 *        ...
 *        Archive_ctor( &resources, "movie.dat" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          ptr_input = Archive_open_Ex( &resources, "frames.pic" );
 *          BitmapSequence_ctor( &movie_frames );
 *          BitmapSequence_load( &movie_frames, ptr_input );
 *          ...
 *        }
 *        ...
 *        BitmapSequence_dtor( &movie_frames, LEAVE_MEMORY );
 *        Input_dtor( ptr_input, FREE_MEMORY );
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_open.
 */
#ifndef _SKIP_DOC
__cyos 915
#endif
struct Input* Archive_open_Ex(struct Archive* ptr_archive, char* sz_name);

/**
 *  All files in the archive are numbered. These numbers are often referenced as
 *  resource IDs. Archive_open_write constructs the right kind of stream for 
 *  an entry, and returns it. Use it the same way as a regular stream. Important: don't forget to delete 
 *  it when you're done with it! FILE MUST NOT BE COMPRESSED!
 *  @param ptr_archive A pointer to an initialized Archive object.
 *  @param  entry The entry number (resource ID).
 *  @return Output object to write to the specified archive file; or NULL if 
 *  there is no such ID, not enough memory for the operation, or the file is compressed
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        struct Archive resources;
 *        struct Output* ptr_output;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        { 
 *          ptr_output = Archive_open_write( &resources, 0);
 *          Output_write( ptr_output, 
 *                        game_information, 
 *                        sizeof( struct GameInformation ) );
 *          ...
 *        }
 *        ...
 *        Output_dtor( ptr_output, FREE_MEMORY );
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_open_write_Ex.
 */
#ifndef _SKIP_DOC
__cyos 1334
#endif
struct Output* Archive_open_write(struct Archive* ptr_archive, int entry);

/**
 *  Opens an Output for the archive file named.
 *  It works just like Archive_open_write, but looks for files by name.
 *  Archive_open_write_Ex constructs the right kind of stream for an entry, and 
 *  returns it. Use it the same way as a regular stream. Important: don't forget to delete it when you're 
 *  done with it! FILE MUST NOT BE COMPRESSED!
 *  @param ptr_archive A pointer to an initialized Archive object.
 *  @param sz_name The name of the file inside the archive
 *  @return A pointer to a corresponding Output object; or NULL if the file is not found,
 *  there is not enough memory to open it, or the file is compressed.
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        struct Archive resources;
 *        struct Output* ptr_output;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          ptr_output = Archive_open_write_Ex( &resources, "game_1" );
 *          Output_write( ptr_output, 
 *                        game_information, 
 *                        sizeof( struct GameInformation ) );
 *          ...
 *        }
 *        ...
 *        Output_dtor( ptr_output, FREE_MEMORY );
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 *  @see Archive_open_write.
 */
#ifndef _SKIP_DOC
__cyos 1333
#endif
struct Output* Archive_open_write_Ex(struct Archive* ptr_archive, char* sz_name);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Archive_count_entries(ptr_archive) ((ptr_archive)->total_entries) 
#else
/**
 *  Returns the number of entries (resources) in the archive.
 *  @param ptr_archive A pointer to an initialized Archive object
 *  @return The number of entries (resources) in the archive
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        }
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 */
int Archive_count_entries(struct Archive* ptr_archive);
#endif

/**
 *  Retrieves the name of the Nth entry.
 *  @param ptr_archive A pointer to the initialized Archive object.
 *  @param entry The element's index in the Archive
 *  @return Name of the entry with the specified index
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        }
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 693
#endif
char* Archive_get_name(struct Archive* ptr_archive, int entry);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Archive_archive_name(ptr_archive) ((ptr_archive)->name) 
#else
/**
 *  Returns the archive name. 
 *  @param ptr_archive A pointer to an initialized Archive object
 *  @return Name of the archive
 *  @code
 *      #include <cybiko.h>
 *      ...
 *      {
 *        int index;
 *        int max_entries;
 *        struct Archive resources;
 *        ...
 *        Archive_ctor( &resources, "renju.sav" );
 *        if( Archive_is_good( &resources ) )
 *        {
 *          TRACE( "Archive :%s Entries:", Archive_archive_name( &resources ) );
 *          max_entries = Archive_count_entries( &resources );
 *          for( index = 0; index < max_entries ; index ++ )
 *          {
 *            TRACE( "%d   %s", index + 1, Archive_get_name( &resources, index ) );
 *          }
 *          ...
 *        }
 *        ...
 *        Archive_dtor( &resources, LEAVE_MEMORY );
 *      }
 *  @endcode
 */
char* Archive_archive_name(struct Archive* ptr_archive);
#endif

#ifdef _SKIP_DOC
};
#endif


#endif  //  #ifndef __ARCHIVE_H_INCLUDED__
