/**
 *  @file around.h
 *  Hooks to cywin Select Partner API.
 */

#ifndef __AROUND_H_INCLUDED__
#define __AROUND_H_INCLUDED__
                             
/**
 *  @ingroup selecting_remote_partner_group
 *  "Select partner" dialog styles.
 */
enum
{
  /**
   *  No additional strings
   */
  SGP_NONE            = 0x0,

  /**
   *  Allows user to select a game in hot seat mode on one CYBIKO computer
   */
  SGP_HOT_SEAT        = 0x4,

  /**
   *  Allows the user to select the Cybiko computer as a game play partner
   */
  SGP_CYBIKO          = 0x2,

  /**
   *  Allows the user to select the Cybiko computer as an Easy game play partner
   */
  SGP_CYBIKO_EASY     = 0x10,

  /**
   *  Allows the user to select the Cybiko computer as a Hard game partner
   */
  SGP_CYBIKO_HARD     = 0x20,

  /**
   *  Allows the user to play the game without a partner
   */
  SGP_SINGLE_GAME     = 0x40,

  /**
   *  Shows CYPETNAMEs instead of NICKNAMEs in the "select partner" dialog 
   */
  SGP_CYLANDIA        = 0x100,  

  /**
   *  Does not invite a partner; 
   *  only selects nickname from list and retrieves the partner's Cy ID
   */
  SGP_DONT_INVITE     = 0x8000,

};

/**
 *  @ingroup selecting_remote_partner_group
 *  Results of the "select partner" functions.
 */
enum
{
  /**
   *  Game in hot seat mode on one CYBIKO computer
   */
  SGP_RES_HOT_SEAT      = -3L,

  /**
   *  Game with the CYBIKO computer as a partner
   */
  SGP_RES_CYBIKO        = -2L,

  /**
   *  Game with the CYBIKO computer as an Easy partner
   */
  SGP_RES_CYBIKO_EASY   = -5L,

  /**
   *  Game with the CYBIKO computer as a Hard partner
   */
  SGP_RES_CYBIKO_HARD   = -6L,

  /**
   *  Game without a partner
   */
  SGP_RES_SINGLE_GAME   = -7L,
};

/**
 *  @ingroup selecting_remote_partner_group
 *  Modes of invitation.
 */
enum invite_mode_t
{
  /**
   *  Shows "Yes" and "No" buttons
   */
  IM_DEFAULT           =  0x1000000,

  /**
   *  Makes dialog text to look like "<NICKNAME> wants to play <GAME>. Play?" 
   *  and shows "Yes" and "No" buttons
   */
  IM_GAME              =  0x02000000,

  /**
   *  Shows "Chat" and "Cancel" buttons
   */
  IM_CHAT              =  0x3000000,

  /**
   *  Makes dialog text to look like "<NICKNAME> wishes to send you a B-Card. 
   *  Accept?" and shows "Yes" and "No" buttons
   */
  IM_PHONEBOOK         =  0x4000000,

};

/**
 *  @ingroup selecting_remote_partner_group
 *  Selects a partner.
 *  @param ptr_process The pointer to the Process object, that calls the function
 *  @param style The dialog style. This parameter can be any combination 
 *  of the following values: SGP_NONE,  SGP_HOT_SEAT, SGP_CYBIKO, 
 *  SGP_CYBIKO_EASY, SGP_CYBIKO_HARD, SGP_SINGLE_GAME, SGP_CYLANDIA, 
 *  SGP_DONT_INVITE
 *  @param sz_partner_name A pointer to the buffer in which the partner's Cy Name will be stored
 *  @return If the function succeeds, ID of the partner's computer, or special ID ( SGP_RES_HOT_SEAT, SGP_RES_CYBIKO, SGP_RES_CYBIKO_EASY, 
 *  SGP_RES_CYBIKO_HARD, SGP_RES_SINGLE_GAME ). If the function fails or player presses "Escape", the return value is 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       char sz_partner_name[9];
 *       cyid_t partner_id;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       partner_id = select_partner( main_module.m_process, 
 *                                    SGP_SINGLE_GAME, 
 *                                    sz_partner_name );
 *       if( ( partner_id ) 
 *          && ( partner_id != SGP_RES_SINGLE_GAME ) )
 *       {
 *         ...
 *         TRACE("Partner name: %s, partner computer id: %ld", sz_partner_name, partner_id );
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see select_partner_Ex.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 23
#endif
cyid_t select_partner(struct Process* ptr_process, int style, char* sz_partner_name);

/**
 *  @ingroup selecting_remote_partner_group
 *  Extended version of the select_partner function.
 *  @param ptr_process The pointer to the Process object, that calls the function
 *  @param sz_partner_name A pointer to the buffer in which the partner's Cy Name will be stored
 *  @param style The dialog style. This parameter can be any combination 
 *  of the following values:
 *  SGP_NONE,  SGP_HOT_SEAT, SGP_CYBIKO, SGP_CYBIKO_EASY, SGP_CYBIKO_HARD,
 *  SGP_SINGLE_GAME, SGP_CYLANDIA, SGP_DONT_INVITE
 *  @param sz_caption The "select partner" dialog caption
 *  @param exclude_cyids NULL-terminated array of the Cybiko computer's IDs, whose 
 *  Cy Names will be excluded from the "select partner" dialog
 *  @return If the function succeeds, Cy ID of the partner computer or special
 *  ID ( SGP_RES_HOT_SEAT, SGP_RES_CYBIKO, SGP_RES_CYBIKO_EASY, SGP_RES_CYBIKO_HARD,
 *  SGP_RES_SINGLE_GAME ). If the function fails or player presses "Escape", 
 *  the return value is 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       char partner_name[9];
 *       cyid_t sz_partner_id;
 *       cyid_t exclude_list[2] = { 110, NULL };
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Nickname of the computer with Cy ID 110 will be excluded from the
 *       //  list of available partners.
 *       partner_id = select_partner_Ex( main_module.m_process,
 *                                       SGP_SINGLE_GAME,
 *                                       sz_partner_name,
 *                                       "Available Partners",
 *                                       exclude_list );
 *       if( ( partner_id ) && ( partner_id != SGP_RES_SINGLE_GAME ) )
 *       {
 *         ...
 *         TRACE("Partner name: %s, partner computer id: %ld", sz_partner_name, partner_id );
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see select_partner.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 313
#endif
cyid_t select_partner_Ex(struct Process* ptr_process, int style, char* sz_partner_name, char* sz_caption, cyid_t* exclude_cyids);

/**
 *  @ingroup selecting_remote_partner_group
 *  Invites a partner to launch a specified application. If the user's partner accepts 
 *  the invitation, the application will launch on his CYBIKO computer.
 *  ulong module_main (uint argc, char* argv[], bool start) of the invited 
 *  application will have 2 parameters : 
 *  \arg string with the partner's computer Cy ID 
 *  \arg string with the partner's Cy Name
 *  @param ptr_process The pointer to the Process object that calls the function
 *  @param sz_text The invitation text or game name if the invite_mode is IM_GAME
 *  @param style The dialog style. This parameter can be any 
 *  combination of the following values: SGP_NONE,  SGP_HOT_SEAT, SGP_CYBIKO, 
 *  SGP_CYBIKO_EASY, SGP_CYBIKO_HARD, SGP_SINGLE_GAME, SGP_CYLANDIA, 
 *  SGP_DONT_INVITE
 *  @param sz_partner_name A pointer to the buffer in which the partner's Cy Name will be stored
 *  @param sz_caption The "select partner" dialog caption
 *  @param invite_move An invitation mode. This parameter can be one of the 
 *  following values: IM_DEFAULT, IM_CHAT, IM_GAME, IM_PHONEBOOK
 *  @param exclude_cyids An array of computer's Cy IDs, whose Cy Names 
 *  will be excluded from the "select partner" dialog
 *  @return If the function succeeds, the Cy ID of the partner's  computer, or special
 *  ID ( SGP_RES_HOT_SEAT, SGP_RES_CYBIKO, SGP_RES_CYBIKO_EASY, SGP_RES_CYBIKO_HARD,
 *  SGP_RES_SINGLE_GAME ). If the function fails or player presses "Escape", 
 *  the return value is 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       ulong module_main(uint argc, char* argv[], bool start)
 *       {
 *         ...
 *         struct module_t main_module;
 *         char sz_partner_name[9];
 *         cyid_t partner_id;
 *         cyid_t exclude_list[2] = { 110, NULL };
 *         ...
 *         init_module( &main_module );
 *         ...
 *         if(argc != 1)       //  Application was invited.
 *         {
 *           partner_id = atoul(argv[1]);
 *           strcpy(sz_partner_name, argv[2]);
 *         }
 *         else               //  Application will invite another application.
 *         {
 *           //  Nickname of the computer with Cy ID 110 will be excluded from the
 *           //  list of available partners.
 *           partner_id = select_app_partner( main_module.m_process,
 *                                            "Enter private chat?",
 *                                            SGP_NONE,
 *                                            sz_partner_name,
 *                                            "Available Partners",
 *                                            IM_CHAT,
 *                                            exclude_list );
 *         }
 *         ...
 *         TRACE("Partner name: %s, partner computer id: %ld", sz_partner_name, partner_id );
 *         ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 314
#endif
cyid_t select_app_partner(struct Process* ptr_process, char* sz_text, int style, char* sz_partner_name, char* sz_caption, long invite_mode, cyid_t* exclude_cyids);

/**
 *  @ingroup selecting_remote_partner_group
 *  Select a partner for single or multi-player games.
 *  If the user's partner accepts the invitation, the game will launch on his CYBIKO 
 *  computer.
 *  ulong module_main (uint argc, char* argv[], bool start) of the invited 
 *  application will have 2 parameters : 
 *  \arg string with partner's computer's Cy ID 
 *  \arg string with partner's Cy Name
 *  @param ptr_process The pointer to the Process object, that calls the function
 *  @param sz_game_name The name of the game the user wants his partner to play with him 
 *  @param style The dialog style. This parameter can be any combination 
 *  of the following values: SGP_NONE,  SGP_HOT_SEAT, SGP_CYBIKO, 
 *  SGP_CYBIKO_EASY, SGP_CYBIKO_HARD, SGP_SINGLE_GAME, SGP_CYLANDIA, 
 *  SGP_DONT_INVITE
 *  @param sz_partner_name A pointer to the buffer in which the partner's Cy Name of your will be stored
 *  @return If the function succeeds, Cy ID of the partner's computer, or special
 *  ID ( SGP_RES_HOT_SEAT, SGP_RES_CYBIKO, SGP_RES_CYBIKO_EASY, SGP_RES_CYBIKO_HARD,
 *  SGP_RES_SINGLE_GAME ). If the function fails or the player presses "Escape", 
 *  the return value is 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       ulong module_main(uint argc, char* argv[], bool start)
 *       {
 *         ...
 *         struct module_t main_module;
 *         char sz_partner_name[9];
 *         cyid_t partner_id;
 *         ...
 *         init_module( &main_module );
 *         ...
 *         if(argc != 1)       //  Application was invited.
 *         {
 *           partner_id = atoul(argv[1]); 
 *           strcpy(sz_partner_name, argv[2]);
 *         }
 *         else               //  Application will invite another application.
 *         {
 *           partner_id = select_game_partner( main_module.m_process,
 *                                             "Billiard",
 *                                             SGP_CYBIKO | SGP_HOT_SEAT,
 *                                             sz_partner_name );
 *         }
 *         ...
 *         TRACE("Partner name: %s, partner computer id: %ld", sz_partner_name, partner_id );
 *         ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 25
#endif
cyid_t select_game_partner(struct Process* ptr_process, char* sz_game_name, long style, char* sz_partner_name);

#endif  //  #ifndef __AROUND_H_INCLUDED__
