
/** @file comport.h
 * Interface for Cybiko serial port
 */

#ifndef __COMPORT_H_INCLUDED__
#define __COMPORT_H_INCLUDED__

#ifndef EOF
#define EOF -1
#endif

/*
 * @typedef typedef void *com_t;
 * @brief communication device handle.
 * @ingroup comport_overview
 */
typedef void *com_t;

/**
 * @brief Available serial ports.
 * @ingroup comport_overview
 */
typedef enum
{
  COMM_DEV_DEFAULT ///< Default serial port
}
com_dev_t;

/**
 *  @brief Baud Rates
 *  @ingroup comport_overview
 */
enum
{
  COMM_BAUD_110,    ///< 100 baud
  COMM_BAUD_300,    ///< 300 baud
  COMM_BAUD_600,    ///< 600 baud
  COMM_BAUD_1200,   ///< 1200 baud
  COMM_BAUD_2400,   ///< 2400 baud
  COMM_BAUD_4800,   ///< 4800 baud
  COMM_BAUD_7200,   ///< 7200 baud
  COMM_BAUD_9600,   ///< 9600 baud
  COMM_BAUD_14400,  ///< 14400 baud
  COMM_BAUD_19200,  ///< 19200 baud
  COMM_BAUD_38400,  ///< 38400 baud
  COMM_BAUD_57600,  ///< 57600 baud (default)
  COMM_BAUD_115200, ///< 115200 baud
};

/**
 *  @brief Number of data bits per byte.
 *  @ingroup comport_overview
 */
enum
{
  COMM_DATABITS_5, ///< 5 bits
  COMM_DATABITS_6, ///< 6 bits
  COMM_DATABITS_7, ///< 7 bits
  COMM_DATABITS_8, ///< 8 bits
};

/**
 *  @brief Number of stop bits after byte
 *  @ingroup comport_overview
 */
enum
{
  COMM_STOPBITS_10, ///< 1 bit
  COMM_STOPBITS_15, ///< 1.5 bits
  COMM_STOPBITS_20, ///< 2 bits
};

/**
 *  @brief Parity check.
 *  @ingroup comport_overview
 */
enum
{
  COMM_PARITY_NONE,  ///< No flow control
  COMM_PARITY_ODD,   ///< Odd
  COMM_PARITY_EVEN,  ///< Even
  COMM_PARITY_MARK,  ///< Mark
  COMM_PARITY_SPACE, ///< Space
};

/**
 *  @brief Flow control types.
 *  @ingroup comport_overview
 */
enum
{
  COMM_FC_NONE =0x00,           ///< No flow control
  COMM_FC_XONXOFF_INPUT =0x01,  ///< XON/XOFF control of input data
  COMM_FC_XONXOFF_OUTPUT =0x02, ///< XON/XOFF control of output data
  COMM_FC_RTSCTS_INPUT =0x04,   ///< RTS/CTS control of input data
  COMM_FC_RTSCTS_OUTPUT =0x08,  ///< RTS/CTS control of output data
  COMM_FC_DSRDTR_INPUT =0x10,   ///< DSR/DTR control of input data
  COMM_FC_DSRDTR_OUTPUT =0x20,  ///< DSR/DTR control of output data
};

/**
 *  @brief com_wait_for_data() function results
 *  @ingroup comport_overview
 */
enum WaitResult
{
  COMM_WAIT_TIMEOUT,   ///< Timeout expired
  COMM_WAIT_DATAREADY, ///< Data ready
  COMM_WAIT_ERROR,     ///< General error (e.g. port is absent)
};

/**
 *  @brief Receive errors & special values.
 *  @ingroup comport_overview
 */
enum
{
  /// General error
  COMM_ERROR_GENERAL = 0x01,
  /// Overrun error
  COMM_ERROR_OVERRUN = 0x02,
  /// Parity error
  COMM_ERROR_PARITY  = 0x04,
  /// Framing error
  COMM_ERROR_FRAMING = 0x08,
  /// Timeout expired
  COMM_ERROR_TIMEOUT = 0x10,
  /// State pins changed (CTS, DSR or RI). New bit values are specified in data field.
  COMM_STATE_CHANGED = 0x80,
};

/**
 * @brief COM port configuration parameters
 * @ingroup comport_overview
 * @brief COM port configuration parameters.
 * @see comport_overview
 */
struct COMMConfig
{
/**
 * @brief Placeholder for COMM_BAUD_* values.
*/
  char baud_rate;
/**
 * @brief Placeholder for COMM_DATABITS_* values.
*/
  char data_bits;
/**
 * @brief Placeholder for COMM_STOPBITS_* values.
*/
  char stop_bits;
/**
 * @brief Placeholder for COMM_PARITY_* values.
*/
  char parity_type;
/**
 * @brief Placeholder for COMM_FC_* values.
*/
  char flow_control;
/**
 * @brief Character used as XON.
*/
  char XON_char;
/**
 * @brief Character used as XOFF.
*/
  char XOFF_char;
/**
 * @brief Reserved.
*/
  char reserved;
/**
 * @brief Maximum timeout (in ms) between write() call and actual byte sending.
*/
  clock_t timeout;
};

/**
 * @ingroup comport_overview
 * @brief Opens a serial port for read / write access.
 * @param device a port to open
 * @param timeout the time to allow for the port to open, in milliseconds.
 * @return the handle to an opened port,
 * or 0 if the port is already opened.
 * @code
 *  #include <cybiko.h>
 *   ...
 *   com_t port;
 *   int data;
 *   ...
 *   // open the default serial port, wait a second for availability
 *   port = com_open( COMM_DEV_DEFAULT, 1000 );
 *   ...
 *   // read a byte, wait forever
 *   data = com_read( port, 0 );
 *   ...
 *   // close the port
 *   com_close( port );
 *   ...
 * @endcode
 * @see com_close
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 1
#endif
com_t com_open( com_dev_t device, long timeout );

/**
 * @ingroup comport_overview
 * @brief Closes an opened serial port.
 * @param port the handle of a port to close.
 * @return TRUE on success; FALSE otherwise
 * @code
 *  #include <cybiko.h>
 *   ...
 *   com_t port;
 *   int data;
 *   ...
 *   // open the default serial port, wait a second for availability
 *   port = com_open( COMM_DEV_DEFAULT, 1000 );
 *   ...
 *   // read a byte, wait forever
 *   data = com_read( port, 0 );
 *   ...
 *   // close the port
 *   com_close( port );
 *   ...
 * @endcode
 * @see com_open
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 2
#endif
bool com_close( com_t port );

/**
 * @ingroup comport_overview
 * @brief Reads a byte from a serial port.
 * @param port the handle of a port to read from
 * @param timeout the time to allow for the byte to be read, in milliseconds.
 * @return on success, the function returns non-negative
 * value of the specified byte; on error, some negative value is returned.
 * Though it can be only -1 for now, this may change in the future
 * @code
 *  #include <cybiko.h>
 *   ...
 *   com_t port;
 *   int data;
 *   ...
 *   // open the default serial port, wait a second for availability
 *   port = com_open( COMM_DEV_DEFAULT, 1000 );
 *   ...
 *   // read a byte, wait forever
 *   data = com_read( port, 0 );
 *   ...
 *   // close the port
 *   com_close( port );
 *   ...
 * @endcode
 * @see com_write
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 3
#endif
int com_read( com_t port, long timeout );

/**
 * @ingroup comport_overview
 * @brief Writes a byte to a serial port.
 * @param port the handle of the port to which the byte should be written.
 * @param value the value of the byte to be written; low-order byte is used.
 * @return TRUE on success; FALSE otherwise.
 * @code
 *  #include <cybiko.h>
 *   ...
 *   static void my_com_print( com_t port, const char* str )
 *   {
 *     int i;
 *     for( i = 0; str[ i ]; i++ )
 *     {
 *       com_write( port, str[ i ], 0 );
 *     }
 *   }
 *   ...
 * @endcode
 * @see com_read
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 4
#endif
bool com_write( com_t port, int value, long timeout );

/**
 * @ingroup comport_overview
 * @brief Waits for data from a serial port.
 * @param port the handle of the serial port in which to look in
 * @param timeout the time to allow for the byte to read, in milliseconds.
 * @return one of the WaitResult constants.
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 5
#endif
int com_wait_for_data( com_t port, long timeout );

/**
 * @ingroup comport_overview
 * @brief Sets the serial port's configuration.
 * @param port the handle of a serial port.
 * @param config COMMConfig structure containing the configuration data.
 * @return TRUE on success; FALSE otherwise.
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 6
#endif
bool com_set_config( com_t port, struct COMMConfig* config );

/**
 * @ingroup comport_overview
 * @brief Retrieves current configuration of a serial port.
 * @param port the handle of a serial port.
 * @param config COMMConfig structure that is to receive configuration data
 * @return TRUE on success; FALSE otherwise
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 7
#endif
bool com_get_config( com_t port, struct COMMConfig* config );

/**
 * @ingroup comport_overview
 * @brief Retrieves current state of a serial port.
 * @param port the handle of a serial port.
 * @return either of these constants: COMM_ERROR_* or COMM_STATE_CHANGED.
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 8
#endif
int com_get_state( com_t port );

/**
 * @ingroup comport_overview
 * @brief Flushes serial port I/O buffers.
 * @param port the handle of a serial port.
 * @return TRUE on success; FALSE otherwise
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 9
#endif
bool com_flush( com_t port );

/**
 * @ingroup comport_overview
 * @brief Sets serial port's DTR state.
 * @param port the handle of a serial port.
 * @param state desired DTR state
 * @return TRUE on success; FALSE otherwise
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 10
#endif
bool com_set_DTR( com_t port, bool state );

/**
 * @ingroup comport_overview
 * @brief Sets serial port's RTS state.
 * @param port the handle of a serial port.
 * @param state the desired RTS state.
 * @return TRUE on success; FALSE otherwise.
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 11
#endif
bool com_set_RTS( com_t port, bool state );

/**
 * @ingroup comport_overview
 * @brief Gets the serial port's current DTR state.
 * @param port the handle of a serial port.
 * @return current DTR state
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 12
#endif
bool com_get_DTR( com_t port );

/**
 * @ingroup comport_overview
 * @brief Gets serial port's current RTS state.
 * @param port the handle of a serial port.
 * @return current RTS state
 */
#ifndef _SKIP_DOC
__cyos "ComPort.dl" 13
#endif
bool com_get_RTS( com_t port );

#endif /* #ifndef __COMPORT_H_INCLUDED__ */
