/** 
 *  @file controls.h
 *  Hooks to cywin cBox, cBevel, cBitmap, cEdit and cProgressBar classes.
 */

#ifndef __CONTROLS_H_INCLUDE__
#define __CONTROLS_H_INCLUDE__

//============================
//=                          =
//=      object cBox         =
//=    Filled rectangle      =
//=                          =
//============================

/**
 *  @brief cBox is simply a filled rectangle.
 *  @ingroup controls_group
 *  A filled rectangle. To put a cBox on the Cybiko screen, use 
 *  the cWinApp_AddObj() function.
 *   @see controls_group, cWinApp_AddObj
 */
struct cBox : public cObject
{
#ifndef _SKIP_DOC
  char cBox_Members_STUB[4];
};
#endif

/** 
 *  Constructor. Creates an empty cBox object.
 *  @param ptr_cbox A pointer to the initialized cBox object
 *  @param w The box width
 *  @param h The box height
 *  @param color The box color
 *  @param transparent Transparent mode. If TRUE, the cBox is transparent
 *  @return A pointer to the initialized cBox object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBox cbox;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       cBox_ctor( &cbox, 100, 100, CLR_LTGRAY, TRUE );
 *       // Puts the cBox object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbox, 10, 40 );
 *       ...
 *       cBox_dtor( &cbox, LEAVE_MEMORY );
 *  @endcode
 *  @see cBox_dtor.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 74
#endif
struct cBox* cBox_ctor(struct cBox* ptr_cbox, int w, int h, color_t color, bool transparent);

/**
 *  Destructor. Deletes the cBox.
 *  @param ptr_cbox A pointer the initialized cBox object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated
 *  for the object by malloc(), use FREE_MEMORY to free it.
 *  Use LEAVE_MEMORY if object was static or allocated in stack.
 *  @return None.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBox cbox;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cBox_ctor( &cbox, 100, 100, CLR_LTGRAY, TRUE );
 *       // Puts the cBox object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbox, 10, 40 );
 *       ...
 *       cBox_dtor( &cbox, LEAVE_MEMORY );
 *  @endcode
 *  @see cBox_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 67
#endif
void cBox_dtor(struct cBox* ptr_cbox, int memory_flag);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_proc         cObject_proc
#else
/**
 *  The Message-processing function. 
 *  @param ptr_cbox A pointer to the initialized cBox object
 *  @param ptr_message A pointer to the processed message
 *  @return TRUE if the message was processed, otherwise FALSE
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBox cbox;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       cBox_ctor( &cbox, 100, 100, CLR_LTGRAY, TRUE );
 *       // Puts the cBox object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbox, 10, 40 );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 
 *                                          0,
 *                                          1, 
 *                                          MSG_USER );
 *       //  Processes messages using the cBox message handler.
 *       cBox_proc( &cbox, ptr_message );
 *       ...
 *       cBox_dtor( &cbox, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
bool cBox_proc(struct cBox* ptr_cbox, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Disconnect   cObject_Disconnect
#else
/**
 *  Disconnects a cBox object from its parent object.
 *  @param ptr_cbox A pointer to the initialized cBox object
 *  @return None
 *  @code
 *      #include <cywin.h>
 *      ...
 *      struct cBox cbox;
 *      struct module_t main_module;
 *      ...
 *      init_module( &main_module );
 *      ...
 *      cBox_ctor( &cbox, 100, 100, CLR_LTGRAY, TRUE );
 *      // Puts the cBox object on the Cybiko screen.
 *      cWinApp_AddObj( main_module.m_process, &cbox, 10, 40 );
 *      ...
 *      // Removes the cBox object from the Cybiko screen.
 *      cBox_Disconnect( &cbmp );
 *      ...
 *      // Puts the cBox object on the Cybiko screen again.
 *      cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *      ...
 *      cBox_dtor( &cbox, LEAVE_MEMORY );
 *      ...
 *  @endcode
 */
void cBox_Disconnect(struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Select       cObject_Select
#else
/**
 *   Selects a cBox object.
 *   @param ptr_cbox A pointer to the initialized cBox object
 *   @return TRUE if object was selected, otherwise FALSE.
 *   @code
 *       #include <cywin.h>
 *       ...
 *       struct cBox cbox;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // It's the same as using the cButton_Select( &button ) function.
 *       // Selects the button.
 *       cObject_Select( &button );
 *       ...
 *       cButton_dtor( &button, LEAVE_MEMORY ); 
 *   @endcode
 */
bool cBox_Select (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_update       cObject_update
#else
/**
    Updates a cBox object.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    cWinApp_clear_screen();
    // It's the same as using the cButton_update( &button ) function.
    // Redraws the button.
    cObject_update( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
void cBox_update (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_GetParent    cObject_GetParent
#else
/**
    Returns a pointer to the parent object's cClip.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return A pointer to the parent object
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_GetParent( &button ) function.
    // Checks for the right parent relation.
    parent = cObject_GetParent( &button );
    if ( parent == main_module.m_process )
    {
      ...
    }
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
struct cClip* cBox_GetParent (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Hide         cObject_Hide
#else
/**
    Hides a cBox object.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBox_Show.
 */
void cBox_Hide (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Show         cObject_Show   
#else
/**
    Shows a cBox object.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBox_Hide.
 */
void cBox_Show (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Disable      cObject_Disable
#else
/**
    Disables a  cBox object so that it cannot be selected.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBox_Enable.
 */
void cBox_Disable (struct cBox* ptr_cbox);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBox_Enable       cObject_Enable
#else
/**
    Enables a cBox object - so that it may be selected.
    @param ptr_cbox A pointer to the initialized cBox structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBox_Disable.
 */
void cBox_Enable (struct cBox* ptr_cbox);
#endif

#ifdef _SKIP_DOC
};
#endif

/** The visible styles for creating a bevel's sides.
 *   @ingroup controls_group
*/
enum tBevelSidesTypes
{
  /**
   * the top side of a bevel
   */
  Top = 1,

  /**
   * the bottom side of a bevel
   */
  Bottom = 2,

  /**
   * the left side of a bevel
   */
  Left = 4,

  /**
   * the right side of a bevel
   */
  Right = 8,

  /**
   * show all of the bevel's sides
   */
  All = 15
};

/**
 *   @brief cBevel object is an outlined empty rectangle.
 *   @ingroup controls_group
 *   The cBevel object is an outlined empty rectangle. The cBevel structure lets you put beveled lines,
 *   boxes, or frames on the Cybiko screen. Any side of the rectangle can be visible or invisible.
 *  <br>
 *  <br>
 *   The default is for all sides to be visible. The visibility values for all sides are defined in the tBevelSidesTypes enumeration
 *   and described in the 'Enums' section.
 *   @see controls_group
 */
struct cBevel : public cObject
{
#ifndef _SKIP_DOC
  char cBevel_Members_STUB[4];
};
#endif

//============================
//=                          =
//=      object cBevel.      =
//= Outlined empty rectangle.=
//=                          =
//============================

/** @fn struct cBevel* cBevel_ctor(struct cBevel* ptr_cbevel, int w, int h, color_t color, char sides)
 *  Constructor. Creates an empty cBevel object.
 *  @param ptr_cbevel A pointer to the initialized cBevel object
 *  @param w The box width
 *  @param h The box height
 *  @param color The object's color
 *  @param sides The visibility state of the box sides. See 'tBevelSidesTypes' enumeration in the 'Enums' section
 *  @return A pointer to the initialized cBevel object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBevel cbevel;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cBevel_ctor( &cbox, 100, 100, CLR_LTGRAY, Left | Right );
 *       // Puts the cBevel object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbevel, 10, 40 );
 *       ...
 *       cBox_dtor( &cbevel, LEAVE_MEMORY );
 *  @endcode
 *  @see cBevel_dtor.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 73
#endif
struct cBevel* cBevel_ctor(struct cBevel*, int, int, color_t, char);

/** @fn void cBevel_dtor(struct cBevel* ptr_cbevel, int memory_flag)
 *  Destructor. Deletes an object.
 *  @param ptr_cbevel A pointer to the initialized cBevel object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in stack 
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBevel cbevel;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cBevel_ctor( &cbox, 100, 100, CLR_LTGRAY, Left | Right );
 *       // Puts the cBevel object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbevel, 10, 40 );
 *       ...
 *       cBox_dtor( &cbevel, LEAVE_MEMORY );
 *  @endcode
 *  @see cBevel_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 63
#endif
void cBevel_dtor(struct cBevel*, int);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_proc         cObject_proc
#else
/**
    The Message-processing function.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @param ptr_message A pointer to the processed message
    @return TRUE if the message was processed
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    struct Message* ptr_message;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
    // It's the same as using the cObject_proc( &cbmp, ptr_message ) function.
    // Processes messages that manage the cBitmap control.
    cBitmap_proc( &cbmp, ptr_message );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
bool cBevel_proc (struct cBevel* ptr_cbevel, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Disconnect   cObject_Disconnect
#else
/**
    Disconnects cBevel from its parent object.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Removes the object from the Cybiko screen.
    // It's the same as using the cObject_Disconnect( &cbmp ) function.
    cBitmap_Disconnect( &cbmp );
    ...
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode    
 */
void cBevel_Disconnect (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Select       cObject_Select
#else
/**
    Selects cBevel object.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return TRUE if the object was selected
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Select( &button ) function.
    // Selects the button.
    cObject_Select( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
bool cBevel_Select (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_update       cObject_update
#else
/**
    Updates a cBevel object.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    cWinApp_clear_screen();
    // It's the same as using the cButton_update( &button ) function.
    // Redraws the button.
    cObject_update( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode 
 */
void cBevel_update (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_GetParent    cObject_GetParent
#else
/**
    Returns a pointer to the parent object's cClip
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return A pointer to the parent object's cClip
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_GetParent( &button ) function.
    // Checks for the right parent relation.
    parent = cObject_GetParent( &button );
    if ( parent == main_module.m_process )
    {
      ...
    }
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
struct cClip* cBevel_GetParent (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Hide         cObject_Hide
#else
/**
    Hides the cBevel object.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBevel_Show.
 */
void cBevel_Hide (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Show         cObject_Show
#else
/**
    Shows the cBevel object.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBevel_Hide.
 */
void cBevel_Show (struct cBevel* ptr_cbevel);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Disable      cObject_Disable
#else
/**
    Disables the cBevel object so that it cannot be selected.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBevel_Enable.
 */
void cBevel_Disable (struct cBevel* ptr_cbevel);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBevel_Enable       cObject_Enable
#else
/**
    Enables the cBevel object so that it may be selected.
    @param ptr_cbevel A pointer to the initialized cBevel structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBevel_Disable.
 */
void cBevel_Enable (struct cBevel* ptr_cbevel);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  @brief use this structure to place a simple picture on the screen.
 *  @ingroup controls_group
 *  Use this structure to put a simple picture on the screen.
 *   @see controls_group
 */
struct cBitmap : public cObject
{
#ifndef _SKIP_DOC
  char cBitmap_Members_STUB[36];
};
#endif

//============================
//=                          =
//=      object cBitmap.     =
//=        The picture.      =
//=                          =
//============================

/** @fn struct cBitmap* cBitmap_ctor(struct cBitmap* ptr_cbitmap, struct Bitmap* ptr_pic)
 *  Constructor for a non-transparent picture.
 *  @param ptr_cbitmap A pointer to the cBitmap object
 *  @param ptr_pic A pointer to a template cBitmap object
 *  @return A pointer to the initialized cBitmap object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       cBitmap_ctor( &cbmp, &bmp );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see cBitmap_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 32
#endif
struct cBitmap* cBitmap_ctor(struct cBitmap*, struct Bitmap*);

/** @fn struct cBitmap* cBitmap_ctor_Ex(struct cBitmap* ptr_cbitmap, struct Bitmap* ptr_pic, color_t color)
 *  Extended version of the cBitmap_ctor function. Constructs a transparent picture.
 *  @param ptr_cbitmap A pointer to the cBitmap struct
 *  @param ptr_pic A pointer to the template cBitmap object
 *  @param color A transparent color
 *  @return A pointer to the initialized cBitmap object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       // The color value may be following: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cBitmap_ctor_Ex( &cbmp, &bmp, CLR_LTGRAY );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see cBitmap_ctor.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 30
#endif
struct cBitmap* cBitmap_ctor_Ex(struct cBitmap*, struct Bitmap*, color_t);

/** @fn void cBitmap_dtor(struct cBitmap* ptr_cbitmap, int memory_flag)
 *  Destructor. Deletes an object.
 *  @param ptr_cbitmap A pointer to the initialized cBitmap object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If memory was 
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. 
 *  Use LEAVE_MEMORY if the object was static or allocated in stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       // The color value should have as follows: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cBitmap_ctor_Ex( &cbmp, &bmp, CLR_LTGRAY );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 65
#endif
void cBitmap_dtor(struct cBitmap*, int);

/** @fn void cBitmap_SetFlips(struct cBitmap* ptr_cbitmap, short flip)
 *  Changes the 'flip' state.
 *  @param ptr_cbitmap A pointer to the initialized cBitmap object
 *  @param flip A flip state flag. It may be any combination of the following: BM_NORMAL, BM_INVERSE, BM_FLIP_X, BM_FLIP_Y
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cBitmap cbmp;
 *       struct Bitmap bmp;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // Creates a bitmap from the file "root.ico".
 *       Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cBitmap_ctor_Ex( &cbmp, &bmp, CLR_LTGRAY );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
 *       ...
 *       cBitmap_SetFlips( &cbmp, BM_FLIP_X );
 *       ...
 *       cBitmap_dtor( &cbmp, LEAVE_MEMORY );
 *       Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see BM_NORMAL , BM_INVERSE , BM_FLIP_X , BM_FLIP_Y
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 72
#endif
void cBitmap_SetFlips(struct cBitmap*, short);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_proc         cObject_proc
#else
/**
    The Message-processing function.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @param ptr_message A pointer to the processed message
    @return TRUE if the message was processed
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    struct Message* ptr_message;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
    // It's the same as using the cObject_proc( &cbmp, ptr_message ) function.
    // Processes mesages that manages the cBitmap control.
    cBitmap_proc( &cbmp, ptr_message );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
bool cBitmap_proc (struct cBitmap* ptr_cbitmap, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Disconnect   cObject_Disconnect
#else
/**
    Disconnects a cBitmap from its parent object.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Removes the object from the Cybiko screen. 
    //It's the same as using the cObject_Disconnect( &cbmp ) function.
    cBitmap_Disconnect( &cbmp );
    ...
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
void cBitmap_Disconnect (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Select       cObject_Select
#else
/**
    Selects a cBitmap object.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return TRUE if the object was selected
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Select( &button ) function.
    // Selects the button.
    cObject_Select( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
bool cBitmap_Select (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_update       cObject_update
#else
/**
    Updates a cBitmap object.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    cWinApp_clear_screen();
    // It's the same as using the cButton_update( &button ) function.
    // Redraws the button.
    cObject_update( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
void cBitmap_update (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_GetParent    cObject_GetParent
#else
/**
    Returns a pointer to the parent object's cClip.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return A pointer to the parent object's cClip
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_GetParent( &button ) function.
    // Checks for the right parent relation.
    parent = cObject_GetParent( &button );
    if ( parent == main_module.m_process )
    {
      ...
    }
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
struct cClip* cBitmap_GetParent (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Hide         cObject_Hide
#else
/**
    Hides a cBitmap object.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBitmap_Show.
 */
void cBitmap_Hide (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Show         cObject_Show
#else
/**
    Shows a cBitmap object.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBitmap_Hide.
 */
void cBitmap_Show (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Disable      cObject_Disable  
#else
/**
    Disables a cBitmap object so that it cannot be selected.
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBitmap_Enable.
 */
void cBitmap_Disable (struct cBitmap* ptr_cbitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cBitmap_Enable       cObject_Enable
#else
/**
    Enables a cBitmap object so that it may be selected
    @param ptr_cbitmap A pointer to the initialized cBitmap object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cBitmap_Disable.
 */
void cBitmap_Enable (struct cBitmap* ptr_cbitmap);
#endif

#ifdef _SKIP_DOC
};
#endif

/** Edit box styles.
 *  @ingroup controls_group
*/
enum tEditStyles
{
  /// show the text as entered.
  es_normal = 0,

  /// show text as read-only; no editing allowed.
  es_readonly = 1,

  /// show all characters as '*' onscreen regardless of what characters are entered (however, in this mode the actual text is stored with correct characters, which may be accessed via password).
  es_password = 2,

  /// center each line within the given rectangle.
  es_centered = 4
};

/** Commands for cursor positioning.
 *  @ingroup controls_group
*/
enum tEditCursorCommand
{
  /// move the cursor one line up.
  mc_lineup = 0,

  /// move the cursor one line down.
  mc_linedown = 1,

  /// move the cursor one page up.
  mc_pageup = 2,

  /// move the cursor one page down.
  mc_pagedown = 3,

  /// move the cursor to the beginning of the string.
  mc_home = 4,

  /// move the cursor to the end of the string.
  mc_end = 5
};

/**
 *   @brief this structure puts a standard edit control on the Cybiko screen.
 *   @ingroup controls_group
 *   Use a cEdit object to put a standard edit control on the Cybiko screen. cEdit controls are used to
 *   retrieve the text that users type. cEdit controls can also display text to the user.
 *   <br>
 *   <br>
 *   When only displaying text to the user, choose an edit control to allow users to select text and copy it to the buffer.
 *   @see controls_group
 */
struct cEdit : public cObject
{
#ifndef _SKIP_DOC
  char cEdit_Members_STUB[64];
};
#endif
//============================
//=                          =
//=        object cEdit.     =
//=       The text editor.   =
//=                          =
//============================

/**
 *  Constructor.
 *  @param ptr_cedit A pointer to the cEdit struct
 *  @param ptr_cfont The font for drawing text
 *  @param buffer_size Maximum text length (not including finishing '\0')
 *  @param style Style state bits
 *  @param color Text color.
 *  @param width Width of the edit window (-1 means single-line style with unlimited width and horizontal scrolling)
 *  @return A pointer to the initialized cEdit object.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_dtor.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 34
#endif
struct cEdit* cEdit_ctor(struct cEdit* ptr_cedit, struct Font* ptr_cfont, int buffer_size, int style, color_t color, int width);

/**
 *  Destructor. Deletes an object.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. Use
 *  LEAVE_MEMORY if the object was static or allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 37
#endif
void cEdit_dtor(struct cEdit* ptr_cedit, int memory_flag);

/** @fn void cEdit_HighliteText(struct cEdit* ptr_cedit, bool flag)
 *  Sets a highlighted state.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param flag Sets the highlighted state
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *        if( cEdit_IsHighlited( &cedit ) )
 *            cEdit_HighliteText( &cedit, FALSE );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 39
#endif
void cEdit_HighliteText(struct cEdit*, bool flag);

/**
 *  Sets a read-only state.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param state TRUE if the read-only state needs to be set
 *  @return The previous read-only state
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the password's character limit. A password can't be longer than 8 symbols.
 *       if ( cEdit_GetCursorPos( &cedit ) >= 8 )
 *            cEdit_SetReadOnly( &cedit, TRUE );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 44
#endif
bool cEdit_SetReadOnly(struct cEdit* ptr_cedit, bool state);

/**
 *  Sets the page length (number of lines to move text when using cursor pageup/down commands).
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param  lines The number of lines
 *  @return The previous page line count
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       sruct Message* ptr_message;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_normal,
 *                   CLR_LTGRAY,
 *                   100 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *       if( ( ptr_message->msgid == MSG_KEYDOWN ) &&
 *           ( Message_get_key_param(ptr_message)->scancode == KEY_U ))
 *       {
 *            // Page will contain 20 lines.
 *            cEdit_SetPageLineCount( &cedit, 20 );
 *            cEdit_MoveCursor( &cedit, mc_pageup );
 *       }
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 45
#endif
int cEdit_SetPageLineCount(struct cEdit* ptr_cedit, int lines);

/**
 *  Highlighted state.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @return TRUE if the text is highlighted
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *        if( cEdit_IsHighlited( &cedit ) )
 *            cEdit_HighliteText( &cedit, FALSE );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 47
#endif
bool cEdit_IsHighlited(struct cEdit* ptr_cedit);

/**
 *  Moves the cursor to a given position relative to the beginning of a string or block of text.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param position The new cursor position
 *  @return The resulting cursor position
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the password's character limit. A password can't be longer than 8 characters.
 *       if ( cEdit_GetCursorPos( &cedit ) >= 8 )
 *       {
 *         cEdit_SetReadOnly( &cedit, TRUE );
 *         cEdit_SetCursorPos( &cedit, 0 );
 *       }
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_GetCursorPos.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 48
#endif
int cEdit_SetCursorPos(struct cEdit* ptr_cedit, int position);

/**
 *  Uses one of these defined methods for moving the cursor: one line up = mc_lineup; one line down = mc_linedown; one page up = mc_pageup;
 *  one page down = mc_pagedown (the number of lines per page is defined in page_line_count).
 *  Move to the beginning of the current line = mc_home; move to the end of the current line = mc_end.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param reg The method of moving.  See "Remarks"
 *  @return The resulting cursor position
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       sruct Message* ptr_message;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_normal,
 *                   CLR_LTGRAY,
 *                   100 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *       if( ( ptr_message->msgid == MSG_KEYDOWN ) &&
 *           ( Message_get_key_param(ptr_message)->scancode == KEY_U ))
 *            cEdit_MoveCursor( &cedit, mc_pageup );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see mc_lineup , mc_linedown , mc_pageup , mc_pagedown , mc_home , mc_end.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 49
#endif
int cEdit_MoveCursor(struct cEdit* ptr_cedit, int reg);

/**
 *  Returns the current cursor position relative to the beginning of a piece of text.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @return Resulting cursor position
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       int    curPos;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );            
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the password's character limit. A password can't be longer than 8 characters.
 *       if ( cEdit_GetCursorPos( &cedit ) >= 8 )
 *       {
 *         cEdit_SetReadOnly( &cedit, TRUE );
 *         cEdit_SetCursorPos( &cedit, 0 );
 *       }
 *       ...
 *       curPos = cEdit_GetCursorPos( &cedit );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_SetCursorPos.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 50
#endif
int cEdit_GetCursorPos(struct cEdit* ptr_cedit);

/**
 *  Loads text from an external buffer
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param text The text to set
 *  @return The number of characters in the text string
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       int    numSimb;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_normal,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       numSimb = cEdit_SetText( &cedit, "This text will be loaded" );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_SetText_Ex, cEdit_GetText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 51
#endif
int cEdit_SetText(struct cEdit* ptr_cedit, char* text);

/**
 *  Appends text from external buffer.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param text The text to append
 *  @return Number of characters in the text string
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       char   buffer[100];
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_normal,
 *                   CLR_LTGRAY,
 *                   100 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       cEdit_GetText( &cedit, buffer );
 *       cEdit_AppendText( &cedit, ";" );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 52
#endif
int cEdit_AppendText(struct cEdit* ptr_cedit, char* text);

/**
 *  Loads text from an external input stream.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param ptr_input A pointer to the initialized Input object
 *  @return The number of characters in the text string
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       struct FileInput file_doc;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       FileInput_ctor_Ex( &file_doc, "doc.txt" );
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       cEdit_SetText_Ex( &cedit, &file_doc );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *       FileInput_dtor( &file_doc, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_SetText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 53
#endif
int cEdit_SetText_Ex(struct cEdit* ptr_cedit, struct Input* ptr_input);

/**
 *  Saves "text" to the external output stream.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param ptr_output A pointer to the initialized Output object
 *  @return Number of characters in the text string
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       struct FileOutput file_doc;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       FileOutput_ctor_Ex( &file_doc, "doc.txt", TRUE );
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *         cEdit_GetText_Ex( &cedit, &file_doc );
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *       FileOutput_dtor( &file_doc, LEAVE_MEMORY );
 *   @endcode
 *   @see cEdit_GetText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 54
#endif
int cEdit_GetText_Ex(struct cEdit* ptr_cedit, struct Output* ptr_output);

/**
 *  Checks whether the text has been modified
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @return TRUE if the text was modified
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       char   buffer[8];
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the password's character limit. A password can't be longer than 8 characters.
 *       if ( cEdit_IsModified( &cedit ) && if ( cEdit_GetTextLength( &cedit ) >= 8 )
 *       {
 *         cEdit_SetReadOnly( &cedit, TRUE );
 *         cEdit_SetCursorPos( &cedit, 0 );
 *         cEdit_GetText( &cedit, buffer );
 *       }
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 68
#endif
bool cEdit_IsModified(struct cEdit* ptr_cedit);

/**
 *  Returns the current text length (not including finishing '\0').
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @return Current text length
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       char   buffer[8];
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the limit on password. Password can't be larger than 8 symbols.
 *       if ( cEdit_GetTextLength( &cedit ) >= 8 )
 *       {
 *         cEdit_SetReadOnly( &cedit, TRUE );
 *         cEdit_SetCursorPos( &cedit, 0 );
 *         cEdit_GetText( &cedit, buffer );
 *       }
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_GetText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 69
#endif
int cEdit_GetTextLength(struct cEdit* ptr_cedit);

/** @fn char* cEdit_GetBufferPtr(struct cEdit* ptr_cedit)
 *  Returns a pointer to the internal buffer.
 *  @param ptr_cedit A pointer to the initialized cEdit object.
 *  @return A pointer to a cEdit text string.
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_normal,
 *                   CLR_LTGRAY,
 *                   100 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       TRACE( "%s", cEdit_GetBufferPtr( &cedit ) );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 70
#endif
char* cEdit_GetBufferPtr(struct cEdit*);

/** @fn int cEdit_GetText(struct cEdit* ptr_cedit, char* buffer)
 *  Saves "text" to the external buffer.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param buffer A string buffer
 *  @return Number of characters in the "text"
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       char   buffer[8];
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   oool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       // Sets the password's character limit. A password can't be longer than 8 characters.
 *       if ( cEdit_GetTextLength( &cedit ) >= 8 )
 *       {
 *         cEdit_SetReadOnly( &cedit, TRUE );
 *         cEdit_SetCursorPos( &cedit, 0 );
 *         cEdit_GetText( &cedit, buffer );
 *       }
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cEdit_GetText_Ex, cEdit_SetText.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 71
#endif
int cEdit_GetText(struct cEdit*, char*);

/**
 *  Message-processing function.
 *  @param ptr_cedit A pointer to the initialized cEdit object
 *  @param ptr_message A pointer to the Message struct
 *  @return TRUE if the message was processed
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cEdit cedit;
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY,
 *       //                         CLR_WHITE, CLR_BLACK.
 *       cEdit_ctor( &cedit,
 *                   cool_normal_font,
 *                   100,
 *                   es_password,
 *                   CLR_LTGRAY,
 *                   -1 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cedit, 10, 40 );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
 *       // Processes the messages that the cEdit control manages.
 *       cEdit_proc( &cedit, ptr_message );
 *       ...
 *       cEdit_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 57
#endif
bool cEdit_proc(struct cEdit* ptr_cedit, struct Message* ptr_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Disconnect   cObject_Disconnect
#else
/**
    Disconnects a cEdit from its parent object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Removes the object from the Cybiko screen. 
    //It's the same as using the cObject_Disconnect( &cbmp ) function.
    cBitmap_Disconnect( &cbmp );
    ...
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
void cEdit_Disconnect (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Select       cObject_Select 
#else
/**
    Selects a cEdit object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return TRUE if the object was selected
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the "button" object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Select( &button ) function.
    // Selects the button.
    cObject_Select( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
bool cEdit_Select (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_update       cObject_update
#else
/**
    Updates a cEdit object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the "button" object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    cWinApp_clear_screen();
    // It's the same as using the cButton_update( &button ) function.
    // Redraws the button.
    cObject_update( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode   
 */
void cEdit_update (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_GetParent    cObject_GetParent 
#else
/**
    Returns a pointer to the cClip of the parent object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return A pointer to the cClip of the parent object
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_GetParent( &button ) function.
    // Checks for the right parent relation.
    parent = cObject_GetParent( &button );
    if ( parent == main_module.m_process )
    {
      ...
    }
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
struct cClip* cEdit_GetParent (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Hide         cObject_Hide
#else
/**
    Hides a cEdit object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cEdit_Show.
 */
void cEdit_Hide (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Show         cObject_Show
#else
/**
    Shows a cEdit object.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cEdit_Hide.
 */
void cEdit_Show (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Disable      cObject_Disable
#else
/**
    Disables the cEdit object so that it cannot be selected.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cEdit_Enable.
 */
void cEdit_Disable (struct cEdit* ptr_cedit);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cEdit_Enable       cObject_Enable
#else
/**
    Enables a cEdit object so that it may be selected.
    @param ptr_cedit A pointer to the initialized cEdit object
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cEdit_Disable.
 */
void cEdit_Enable (struct cEdit* ptr_cedit);
#endif

#ifdef _SKIP_DOC
};
#endif


/** 
 *   @ingroup controls_group
 *   @brief Adds a progress bar to your application.
 *
 *   Use cProgressBar to add a progress bar to your application. Progress bars provide users with visual feedback
 *   about the progress of a procedure within an application. As the procedure progresses, the rectangular progress
 *   bar gradually fills from left to right with the system highlight color.
 *   @see controls_group
 */
struct cProgressBar : public cBevel
{
#ifndef _SKIP_DOC
  char cProgressBar_Members_STUB[36];
};
#endif

//============================
//=                          =
//=    object cProgressBar.  =
//=     The progress bar.    =
//=                          =
//============================

/** @fn struct cProgressBar* cProgressBar_ctor(struct cProgressBar* ptr_cpb, int width, int min_pos, int max_pos, struct Font* txtfont, color_t color, color_t back_color, int height)
 *  Constructor.
 *  @param ptr_cpb A pointer to the cProgressBar struct
 *  @param width The total width of the object. Maximal length of the progress bar itself will be width - 4 because of the frame in which it is enclosed
 *  @param min_pos The value that should be passed via SetCurrentPos to set the progress status to 0%
 *  @param max_pos The value that should be passed via SetCurrentPos to set the progress status to 100%
 *  @param txtfont The font for the progress status text output
 *  @param color Main object's color
 *  @param back_color Background color
 *  @param height The total height with frame. Should not be set below default value height = 16
 *  @return A pointer to the initialized cProgressBar object
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cProgressBar cprogr;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cProgressBar_ctor( &cprogr,
 *                          100,
 *                          0,
 *                          100,
 *                          cool_normal_font,
 *                          CLR_WHITE,
 *                          CLR_BLACK,
 *                          20 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cprogr, 10, 40 );
 *       ...
 *       cProgressBar_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see cProgressBar_dtor.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 58
#endif
struct cProgressBar* cProgressBar_ctor(struct cProgressBar*, int, int, int, struct Font*, color_t, color_t, int);

/** @fn void cProgressBar_dtor(struct cProgressBar* ptr_cpb, int memory_flag)
 *  Destructor. Deletes an object.
 *  @param ptr_cpb A pointer to the initialized cProgressBar object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was
 *  allocated for the object by malloc(), use FREE_MEMORY to free it. Use
 *  LEAVE_MEMORY if the object was static or allocated in a stack
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       ...
 *       struct cProgressBar cprogr;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cProgressBar_ctor( &cprogr,
 *                          100,
 *                          0,
 *                          100,
 *                          cool_normal_font,
 *                          CLR_WHITE,
 *                          CLR_BLACK,
 *                          20 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cprogr, 10, 40 );
 *       ...
 *       cProgressBar_dtor( &cedit, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 62
#endif
void cProgressBar_dtor(struct cProgressBar*, int);

/** @fn void cProgressBar_SetCurrentPos(struct cProgressBar* ptr_cpb, int cur_pos)
 *  Sets the current bar position and updates the object
 *  @param ptr_cpb A pointer to the initialized cProgressBar object
 *  @param cur_pos Should be between min_pos and max_pos (see cProgressBar_ctor). Progress status depends on a linear shift in position. Min_pos sets progress status to 0%, max_pos - to 100%
 *  @return None
 *  @code
 *       #include <cywin.h>
 *       {
 *       ...
 *       struct cProgressBar cprogr;
 *       struct module_t main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       // The color value may be: CLR_LTGRAY, CLR_BKGRAY, CLR_WHITE, CLR_BLACK.
 *       cProgressBar_ctor( &cprogr,
 *                          100,
 *                          0,
 *                          100,
 *                          cool_normal_font,
 *                          CLR_WHITE,
 *                          CLR_BLACK,
 *                          20 );
 *       // Puts the cBitmap object on the Cybiko screen.
 *       cWinApp_AddObj( main_module.m_process, &cprogr, 10, 40 );
 *       ...
 *       for( int i=0; i<100; i++ )
 *       {
 *          ...
 *          cProgressBar_SetCurrentPos( &cprogr, i );
 *          ...
 *       }
 *       ...
 *       cProgressBar_dtor( &cedit, LEAVE_MEMORY );\
 *       }
 *  @endcode
 *  @see cProgressBar_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos "CyWin" 61
#endif
void cProgressBar_SetCurrentPos(struct cProgressBar*, int);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_proc         cObject_proc 
#else
/**
    The Message-processing function. 
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @param ptr_message A pointer to the processed message
    @return TRUE, if the message was processed
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    struct Message* ptr_message;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    ptr_message = cWinApp_get_message( main_module.m_process, 0, 1, MSG_USER );
    // It's the same as using the cObject_proc( &cbmp, ptr_message ) function.
    // Processes the messages that the cBitmap control manages.
    cBitmap_proc( &cbmp, ptr_message );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
bool cProgressBar_proc (struct cProgressBar* ptr_cprogress_bar, struct Message* ptr_message);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Disconnect   cObject_Disconnect
#else
/**
    Disconnects a cProgressBar from it's parent object.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure.
    @return None
    @code
    #include <cywin.h>
    ...
    struct cBitmap cbmp;
    struct Bitmap bmp;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    // Creates a bitmap from the file "root.ico".
    Bitmap_ctor_Ex1( &bmp, "root.ico" );
    cBitmap_ctor( &cbmp, &bmp );
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Removes the object from the Cybiko screen. It's the same as using the cObject_Disconnect( &cbmp ) function.
    cBitmap_Disconnect( &cbmp );
    ...
    // Puts the cBitmap object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &cbmp, 10, 40 );
    ...
    // Afterwards this function disconnects the cBitmap object automatically.
    cBitmap_dtor( &cbmp, LEAVE_MEMORY );
    Bitmap_dtor( &bmp, LEAVE_MEMORY ); 
    @endcode
 */
void cProgressBar_Disconnect (struct cProgressBar* ptr_cprogress_bar);  
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Select       cObject_Select  
#else
/**
    Selects a cProgressBar object.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @return TRUE if the progress bar is selected
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Select( &button ) function.
    // Selects the button.
    cObject_Select( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY );
    @endcode
 */
bool cProgressBar_Select (struct cProgressBar* ptr_cprogress_bar);  
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_update       cObject_update 
#else
/**
    Updates a cProgressBar object. Sends a signal to the system, that the object must be redrawn. After you change the state of an object, you must call this function to redraw your object. 
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    cWinApp_clear_screen();
    // It's the same as using the cButton_update( &button ) function.
    // Redraws the button.
    cObject_update( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
void cProgressBar_update (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_GetParent    cObject_GetParent
#else
/**
    Returns a pointer to the cClip of the parent object.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @return A pointer to the cClip of the parent object
    @code 
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_GetParent( &button ) function.
    // Checks for the right parent relation.
    parent = cObject_GetParent( &button );
    if ( parent == main_module.m_process )
    {
      ...
    }
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
 */
struct cClip* cProgressBar_GetParent (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Hide         cObject_Hide
#else
/** Hides a cProgressBar object.
 * @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
 * @return None
   @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
   @endcode  
   @see cProgressBar_Show.
 */
void cProgressBar_Hide (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Show         cObject_Show
#else
/**
    Shows a cProgressBar object.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Hide( &button ) function.
    cObject_Hide( &button );
    ...
    // It's the same as using the cButton_Show( &button ) function.
    cObject_Show( &button );
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cProgressBar_Hide.
 */
void cProgressBar_Show (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Disable      cObject_Disable
#else
/**
    Disables a cProgressBar object so that it cannot be selected.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cProgressBar_Enable.
 */
void cProgressBar_Disable (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define cProgressBar_Enable       cObject_Enable
#else
/**
    Enables for selection cProgressBar object.
    @param ptr_cprogress_bar A pointer to the initialized cProgressBar structure.
    @return None
    @code
    #include <cywin.h>
    ...
    struct cButton button;
    struct module_t main_module;
    struct cClip* parent;
    ...
    init_module( &main_module );
    ...
    cButton_ctor( &button, "Button text", mrOk );
    // Puts the 'button' object on the Cybiko screen.
    cWinApp_AddObj( main_module.m_process, &button, 20, 50 );
    ...
    // It's the same as using the cButton_Disable( &button ) function.
    // Now the user can't use this object.
    cObject_Disable( &button )
    ...
    // It's the same as using the cButton_Enable( &button ) function.
    // Now the user can use this object again.
    cObject_Enable( &button )
    ...
    cButton_dtor( &button, LEAVE_MEMORY ); 
    @endcode
    @see cProgressBar_Disable.
 */
void cProgressBar_Enable (struct cProgressBar* ptr_cprogress_bar);
#endif

#ifdef _SKIP_DOC
};
#endif

#endif  // #ifndef __CONTROLS_H_INCLUDE__
