/**
 *  @file cy3d.h
 *  Hooks to CyOS 3D library.
 */
#ifndef __CY3D_H_INCLUDE__
#define __CY3D_H_INCLUDE__

/**
 *  @ingroup three_d
 *  @typedef typedef short fixed_t;
 *  @brief Values with the fixed point ( 8.8 )
 *
 *  Values with the fixed point ( 8.8 ).
 *  The 8 high bits represent the integer.
 *  The 8 low bits represent a fraction.
 *  (i.e. 0x0F80 is 15.5).
 */
typedef short fixed_t;

/**
 *  @ingroup three_d
 *  @brief A point on a plane, with the coordinates represented by fixed point (8.8) values.
 *
 *  A point on a plane, with the coordinates represented by fixed point (8.8) values.
 *  @see three_d
 */
typedef struct
{
  /**
   *  x-coordinate of the point.
   */
  fixed_t p_x;

  /**
   *  y-coordinate of the point.
   */
  fixed_t p_y;
} point_t;

/**
 *  @ingroup three_d
 *  @brief A point on a plane, with the coordinates represented by integer values.
 *
 *  A point on a plane, with the coordinates represented by the integer values.
 *  @see three_d
 */
typedef struct
{
  /**
   *  x-coordinate of the point.
   */
  short ps_x;

  /**
   *  y-coordinate of the point.
   */
  short ps_y;
} pos_t;

/**
 *  @ingroup three_d
 *  @brief a collection of sprites or textures used by the CyOS 3D library.
 *
 *  Collection of sprites or textures used by the CyOS 3D library.
 *  @see three_d
 */
typedef struct
{
  /**
   *  Number of images in the collection.
   */
  short r_num;

  /**
   *  Size of a single image in the collection.
   */
  short r_size;

  /**
   *  Data for the images.
   */
  char r_data[ 4 ];
} raster_t;

/**
 *  @ingroup three_d
 *
 *  Fills the top and bottom of the screen with the specified colors.
 *  @param buff Pointer to the video buffer.
 *  @param sky_clr Color of the top of the screen
 *  @param floor_clr Color of the bottom of the screen.
 *  Note: cy3d.dl must be loaded on the Cybiko computer.
 *  @return None
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 0
#endif
void cy3d_draw_background( char* buff, color_t sky_clr, color_t floor_clr );

/**
 *  @ingroup three_d
 *
 *  Fills the top part of the screen with the specified color.
 *  @param buff Pointer to the video buffer
 *  @param clr Color of the screen top
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   struct module_t main_module;
 *   ...
 *   init_module( &main_module );
 *   ...
 *   char* ptr_video_page =
 *     DisplayGraphics_get_page_ptr( main_module.m_gfx, 0 );
 *   ...
 *   cy3d_draw_sky( ptr_video_page, CLR_LTGRAY );
 *   ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 1
#endif
void cy3d_draw_sky( char* buff, color_t clr );

/**
 *  @ingroup three_d
 *
 *  Copies from the source video buffer to the destination video buffer
 *  performing bitwise OR.
 *  @param dst Pointer to the destination video buffer.
 *  @param src Pointer to the source video buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return None
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 2
#endif
void cy3d_draw_map_buffer( char* dst, char* src );

/**
 *  @ingroup three_d
 *
 *  Copies from the source video buffer to the destination video buffer
 *  performing bitwise XOR.
 *  @param dst Pointer to the destination video buffer.
 *  @param src Pointer to the source video buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer.
 *  @return None.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 3
#endif
void cy3d_draw_map_buffer_xor( char* dst, char* src );

/**
 *  @ingroup three_d
 *
 *  Copies the top part of the screen to the bottom part of the screen.
 *  @param buff Pointer to the video buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   struct module_t main_module;
 *   ...
 *   init_module( &main_module );
 *   ...
 *   char* ptr_video_page =
 *     DisplayGraphics_get_page_ptr( main_module.m_gfx, 0 );
 *   ...
 *   cy3d_mirror_buffer( ptr_video_page );
 *   ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 4
#endif
void cy3d_mirror_buffer( char* buff );

/**
 *  @ingroup three_d
 *
 *  Draws a column of the texture. This is useful for rendering the walls in the
 *  3D world. The corresponding element of the z-buffer will be updated.
 *  @param buff Pointer to the video buffer
 *  @param ray Index ( 0 - 159 ). tg( angle ) = ( 80 - ray ) / 100
 *  @param y y-coordinate of the column in the 3D world
 *  @param tex Pointer to the texture collection
 *  @param index Texture's index in the collection
 *  @param column Column's index in the texture ( 0 - 31 )
 *  @param shift Shifts the texture to vertical
 *  @param zbuff Pointer to the z-buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return None
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 5
#endif
void cy3d_draw_tex_column( char* buff,
                           int ray,
                           fixed_t y,
                           raster_t* tex,
                           int index,
                           int column,
                           int shift,
                           int* zbuff );

/**
 *  @ingroup three_d
 *
 *  Draws a sprite in the specified position in the 3D world.
 *  Parts of the image that are not visible will be truncated (due to the z-buffer).
 *  @param buff Pointer to the video buffer
 *  @param p Position of the sprite in the 3D world
 *  @param spr Pointer to the collection of the sprites
 *  @param index The sprite's index in the sprite collection
 *  @param zbuff Pointer to the z-buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer.
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   // Coordinates of the Cybiko sign (sprite) in the scene.
 *   point_t sign_location[ 2 ] =
 *   {
 *     { 0, 512 }, { 0, -512 }
 *   };
 *   ...
 *   void render_objects( char* ptr_gfx_buffer,
 *     point_t camera, int direction, int* z_buffer )
 *   {
 *    int index;
 *    ...
 *    // Transforms and renders column.
 *    for( index = 0; index < MAX_COLUMN; index++ )
 *    {
 *      cy3d_draw_sprite( ptr_gfx_buffer,
 *        cy3d_warp( column_location[ index ], camera, direction ),
 *          ptr_column, 0, z_buffer );
 *    }
 *    // Transforms and renders signs.
 *    for( index = 0; index < MAX_SIGN; index++ )
 *    {
 *      int effective_index = ( camera.p_y > 0 ) ? 1 - index : index ;
 *      ...
 *      cy3d_draw_sprite( ptr_gfx_buffer,
 *        cy3d_warp( sign_location[ effective_index ], camera, direction ),
 *          ptr_sign, sign_frame, z_buffer );
 *    }
 *     sign_frame++;
 *     sign_frame %= 19;
 *   }
 *   ...
 *  @endcode
 *  @see cy3d_draw_wall.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 6
#endif
void cy3d_draw_sprite( char* buff,
                       point_t p,
                       raster_t* spr,
                       int index,
                       int* zbuff );

/**
 *  @ingroup three_d
 *
 *  Draws wall in the 3D world.
 *  @param pointer To the video buffer
 *  @param p1 Starting point of the of the wall in the 3D world
 *  @param p2 Ending point of the of the wall in the 3D world
 *  @param tex Pointer to the texture collection
 *  @param index Texture's index in the collection
 *  @param shift Shifts the texture to vertical
 *  @param zbuff Pointer to the z-buffer.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return None
 *  @code
 *   #include <cywin.h>
 *   ...
 *   int iIndex, iShift, *Z_buff, iDirection;
 *   raster_t* ptr_walls_tex;
 *   char* ptr_gfx_buffer;
 *   point_t camera;
 *   ...
 *   struct Wall
 *   {
 *    point_t p1;
 *    point_t p2;
 *   };
 *   ...
 *   struct Wall* ptr_wall;
 *   ...
 *   cy3d_draw_wall( ptr_gfx_buffer,
 *      cy3d_warp( ptr_wall->p1, camera, iDirection ),
 *      cy3d_warp( ptr_wall->p2, camera, iDirection ),
 *      ptr_walls_tex,
 *      iIndex,
 *      iShift,
 *      Z_buff );
 *   ...
 *  @endcode
 *  @see cy3d_draw_sprite.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 7
#endif
void cy3d_draw_wall( char* buff,
                     point_t p1,
                     point_t p2,
                     raster_t* tex,
                     int index,
                     int shift,
                     int* zbuff );

/**
 *  @ingroup three_d
 *
 *  Loads the image collection with the specified name.
 *  If the name of the file ends with the ".spr" the size of the image will be set
 *  to 256. Otherwise it will be set to 128.
 *  Note: The collection must be linked to the application using the filer utility.
 *  @param name Name of the image collection.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Pointer to the image collection
 *  @code
 *   #include <cywin.h>
 *   ...
 *   raster_t* ptr_coll;
 *   ...
 *   ptr_coll = cy3d_load( "sign.spr" );
 *   ...
 *   free( ptr_coll );
 *   ...
 *   ptr_coll = cy3d_load( "wall.tex" );
 *   ...
 *   free( ptr_coll );
 *   ...
 *  @endcode
 *  @see cy3d_load_tex, cy3d_load_spr.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 8
#endif
raster_t* cy3d_load( char* name );

/**
 *  @ingroup three_d
 *
 *  Loads the texture collection with the specified index in the application
 *  resource list. Size of the image will be set to 128.
 *  Note: The collection must be linked to the application using filer utility.
 *  @param index Index of the collection in the application resource list.
 *  Note: cy3d.dl must be loaded on the Cybiko computer.
 *  @return Pointer to the texture collection.
 *  @code
 *   #include <cywin.h>
 *   ...
 *   raster_t* ptr_walls_tex;
 *   ...
 *   ptr_walls_tex = cy3d_load_tex( 1 );
 *   ...
 *   free( ptr_walls_tex );
 *   ...
 *  @endcode
 *  @see cy3d_load, cy3d_load_spr.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 9
#endif
raster_t* cy3d_load_tex( int index );

/**
 *  @ingroup three_d
 *
 *  Loads the sprite collection with the specified index in the application
 *  resource list. Size of the image will be set to 256.
 *  Note: The collection must be linked to the application using the filer utility.
 *  @param index Index of the collection in the application resource list.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Pointer to the sprite collection
 *  @code
 *   #include <cywin.h>
 *   ...
 *   raster_t* ptr_obj_spr;
 *   ...
 *   ptr_obj_spr = cy3d_load_spr( 5 );
 *   ...
 *   free( ptr_obj_spr );
 *   ...
 *  @endcode
 *  @see cy3d_load, cy3d_load_tex.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 10
#endif
raster_t* cy3d_load_spr( int index );

/**
 *  @ingroup three_d
 *
 *  Calculates sine of the angle.
 *  @param direction The angle's value (in degrees).
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Sine of the angle
 *  @see cy3d_cos.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 11
#endif
fixed_t cy3d_sin( int direction );


/**
 *  @ingroup three_d
 *
 *  Calculates the cosine of the angle.
 *  @param direction Value of the angle (in degrees).
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Cosine of the angle
 *  @see cy3d_sin.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 12
#endif
fixed_t cy3d_cos( int direction );

/**
 *  @ingroup three_d
 *
 *  Retrieves square root of the value.
 *  @param value Nonnegative value whose square root will be obtained
 *  Note: cy3d.dl must be loaded to the Cybiko computer.
 *  @return square root of the value.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 13
#endif
int cy3d_sqrt( long value );

/**
 *  @ingroup three_d
 *
 *  Transforms the point's position in the 3D world to its
 *  position in the camera's coordinate system
 *  (i.e. in this coordinate system, the camera is in (0, 0) and
 *  directed to the 0Y direction)
 *  @param p the point's position before transformation
 *  @param camera The camera's position
 *  @param direction The camera's direction.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return the point's position after transformation
 *  @code
 *   #include <cywin.h>
 *   ...
 *   // Coordinates of the Cybiko sign (sprite) in the scene.
 *   point_t sign_location[ 2 ] =
 *   {
 *     { 0, 512 }, { 0, -512 }
 *   };
 *   ...
 *   void render_objects( char* ptr_gfx_buffer,
 *     point_t camera, int direction, int* z_buffer )
 *   {
 *    int index;
 *    ...
 *    // Transforms and renders column.
 *    for( index = 0; index < MAX_COLUMN; index++ )
 *    {
 *      cy3d_draw_sprite( ptr_gfx_buffer,
 *        cy3d_warp( column_location[ index ], camera, direction ),
 *          ptr_column, 0, z_buffer );
 *    }
 *    // Transforms and renders signs.
 *    for( index = 0; index < MAX_SIGN; index++ )
 *    {
 *      int effective_index = ( camera.p_y > 0 ) ? 1 - index : index ;
 *      ...
 *      cy3d_draw_sprite( ptr_gfx_buffer,
 *        cy3d_warp( sign_location[ effective_index ], camera, direction ),
 *          ptr_sign, sign_frame, z_buffer );
 *    }
 *     sign_frame++;
 *     sign_frame %= 19;
 *   }
 *   ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 14
#endif
point_t cy3d_warp( point_t p, point_t camera, int direction );

/**
 *  @ingroup three_d
 *
 *  Moves a point by a fixed distance in the specified direction.
 *  @param p Starting coordinate of the point
 *  @param distance Distance the point is to move
 *  @param direction Direction of the movement.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Position of the point after it has moved
 *  @code
 *   #include <cywin.h>
 *   ...
 *   point_t new_position;
 *   ...
 *   if( DirectKeyboard_is_key_pressed( ptr_keyboard, KEY_UP ))
 *   {
 *     new_position = cy3d_move( camera, 64, direction );
 *     ...
 *     if( is_position_acceptable( new_position ))
 *     {
 *       camera = new_position;
 *     }
 *   }
 *   ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 15
#endif
point_t cy3d_move( point_t p, fixed_t distance, int direction );

/**
 *  @ingroup three_d
 *
 *  Converts the point's coordinates, from fixed point (8.8) values
 *  to integer values.
 *  @param p The point's coordinates, represented by
 *  fixed point ( 8.8 ) values.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return The pointer's coordinates, represented by integer values
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 16
#endif
pos_t cy3d_pos( point_t p );

/**
 *  @ingroup three_d
 *
 *  Converts the point's coordinates from integer values to fixed
 *  point (8.8) values.
 *  @param p The point's coordinates, represented by integer values.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return The point's coordinates, represented by
 *  fixed point (8.8) values
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 17
#endif
point_t cy3d_point( pos_t p );

/**
 *  @ingroup three_d
 *
 *  Retrieves the distance between two points
 *  @param p1 Position of the first point
 *  @param p2 Position of the second point
 *  Note: cy3d.dl must be loaded to the Cybiko computer
 *  @return the distance between two points.
 *  @see cy3d_distance2.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 18
#endif
fixed_t cy3d_distance( point_t p1, point_t p2 );

/**
 *  @ingroup three_d
 *
 *  Retrieves the square of the distance between the two points.
 *  @param p1 Position of the first point.
 *  @param p2 Position of the second point.
 *  Note: cy3d.dl must be loaded on the Cybiko computer
 *  @return Square of the distance between the two points
 *  @code
 *   #include <cywin.h>
 *   ...
 *   long lSDistance;
 *   point_t first_location;
 *   point_t second_location;
 *   ...
 *   //Retrieves the square of the distance between the two points.
 *   lSDistance = cy3d_distance2( first_location, second_location );
 *   ...
 *  @endcode
 *  @see cy3d_distance.
 */
#ifndef _SKIP_DOC
__cyos "cy3d.dl" 19
#endif
long cy3d_distance2( point_t p1, point_t p2 );

#endif
