/** 
 *  @file cyber-files.h
 *  CyOS file system.
 */

#ifndef __CYBER_FILES_H_INCLUDE__
#define __CYBER_FILES_H_INCLUDE__

/**
 *  Maximum filename length.
 *   @ingroup filesystem_overview
 */
#define MAX_NAME_LEN 64  

//============================
//=                          =
//=   object FileInput.      =
//=  Input file stream.      =
//=                          =
//============================

/** 
 *  @brief The Input file stream.
 *  @ingroup filesystem_overview
 *  Implements basic file input operations.
 *  Attempting to open or construct FileInput for a non-existing file will
 *  corrupt the file (FileInput_is_bad returns TRUE)
 *  @see filesystem_overview
 */
struct FileInput : public Input
{
#ifndef _SKIP_DOC
  char FileInput_Members_STUB[64];
};
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_ctor(ptr_file_input)  FileInput_ctor_STUB((ptr_file_input), 1)
#else
/**
 *  Creates an empty input file stream object.
 *  @param ptr_file_input A pointer to the FileInput structure
 *  @return A pointer to the initialized FileInput object.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seek( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *         ...
 *       }
 *  @endcode
 *  @see FileInput_ctor_Ex.
 */
struct FileInput* FileInput_ctor(struct FileInput* ptr_file_input);
#endif

#ifndef _SKIP_DOC
__cyos 250
struct FileInput* FileInput_ctor_STUB(struct FileInput* ptr_file_input, int Magic);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_ctor_Ex(ptr_file_input, file_name)  FileInput_ctor_Ex_STUB((ptr_file_input), 1, (file_name)) 
#else
/**
 *  Creates an input file stream object for a file with the specified name.
 *  @param ptr_file_input A pointer to the FileInput structure
 *  @param sz_file_name Name of the file to open
 *  @return A pointer to the initialized FileInput object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte;
 *       struct FileInput file_input;
 *       ...
 *       FileInput_ctor_Ex( &file_input, "game.save");
 *       ...
 *       FileInput_seek( &file_input, 10, SEEK_SET );
 *       tenth_byte = FileInput_read_byte( &file_input );
 *       ...
 *       FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FileInput_ctor.
 */
struct FileInput* FileInput_ctor_Ex(struct FileInput* ptr_file_input, char* sz_file_name);
#endif

#ifndef _SKIP_DOC
__cyos 251
struct FileInput* FileInput_ctor_Ex_STUB(struct FileInput* ptr_file_input, int Magic, char* file_name);
#endif

#ifndef _SKIP_DOC
__cyos 436
char* FileInput_class_name(struct FileInput* ptr_file_input);
#endif

/**
 *  Opens an input stream for the file with the specified name.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @param sz_file_name Name of the file to open
 *  @return TRUE if the stream was opened successfully
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seek( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 914
#endif
bool FileInput_open(struct FileInput* ptr_file_input, char* sz_file_name);

/**
 *  Seeks to the specified position, then returns the sought position.
 *  If the stream supports a seek operation, it seeks to the specified position in
 *  the specified mode. If the requested seek operation can not be done (not
 *  supported or wrong parameters), -1 will be returned.
 *  Seeking prior to the beginning of the stream sets the pointer to the stream's first byte.
 *  Seeking after the end of the stream sets the pointer to the end of stream
 *  just after the stream's last byte.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position, or -1 if an error occurred
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seek( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
#ifndef _SKIP_DOC
__cyos 1061
#endif
long FileInput_seek(struct FileInput* ptr_file_input, long pos, seek_t mode);

/**
 *  Returns the stream's current position.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return The stream's position, or -1 if that operation is not supported 
 *  for this stream
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte;
 *       struct FileInput file_input;
 *       long cur_pos;
 *       ...
 *       FileInput_ctor_Ex( &file_input, "game.save");
 *       ...
 *       FileInput_seek( &file_input, 10, SEEK_SET );
 *       tenth_byte = FileInput_read_byte( &file_input );
 *       // cur_pos will be equal to 11.
 *       cur_pos = FileInput_tell( &file_input );
 *       ...
 *       FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1236
#endif
long FileInput_tell(struct FileInput* ptr_file_input);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_dtor Input_dtor 
#else
/**
 *  Destructor.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seek( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY
 */
void FileInput_dtor(struct FileInput* ptr_file_input, int memory_flag);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_read Input_read 
#else
/**
 *  Reads a specified number of bytes from the stream. 
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @param ptr_buffer A pointer to the allocated buffer
 *  @param length The number of bytes to read
 *  @return The exact number of bytes read, or 0 if no bytes were read
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileInput file_input;
 *       struct score_t high_scores[10];
 *       ...
 *       FileInput_ctor( &file_input );
 *       if( FileInput_open( &file_input, "game.save" ) )
 *       ...
 *       Input_read( &file_input, high_scores, sizeof(high_scores) );
 *       ...
 *       Input_dtor( &file_input, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FileInput_read_byte.
 */
long FileInput_read(struct FileInput* ptr_file_input, void* ptr_buffer, long length);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_read_byte Input_read_byte 
#else
/**
 *  Reads the next byte from the stream.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return The next byte from the stream
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seek( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see FileInput_read.
 */
int FileInput_read_byte(struct FileInput* ptr_file_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_seekg Input_seekg 
#else
/**
 *  Seeks an input stream.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position, or -1 if an error occurred
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seekg( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
long FileInput_seekg(struct FileInput* ptr_file_input, long pos, seek_t mode);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_tellg Input_tellg 
#else
/**
 *  Returns the stream's position.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return The stream's position
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         long current_pos;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "game.save" ) )
 *         {
 *           FileInput_seekg( &file_input, 10, SEEK_SET );
 *           tenth_byte = FileInput_read_byte( &file_input );
 *         }
 *         //  The current_pos value will be equal to 11.
 *         current_pos = FileInput_tellg( &file_input );
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 */
long FileInput_tellg(struct FileInput* ptr_file_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_get_size Input_get_size 
#else
/**
 *  Returns the stream size (if applicable).
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return Stream size, or -1 if this operation can't be done on this stream
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         long size;
 *         long index;
 *         long sum = 0;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "values.txt" ) )
 *         {
 *           size = FileInput_get_size( &file_input);
 *           for( index = 0; index < size; index++ )
 *           {
 *             sum += (long)FileInput_read_byte( &file_input );
 *           }
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 */
long FileInput_get_size(struct FileInput* ptr_file_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_get_flags Input_get_flags 
#else
/**
 *  Returns the stream's current state.
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return The stream's current state
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         long size;
 *         long sum = 0;
 *         long index = 0;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "values.txt" ) )
 *         {
 *           while( ! ( FileInput_get_flags( &file_input ) & FLAG_EOF ) )
 *           {
 *             sum += (long)FileInput_read_byte( &file_input );
 *             index ++;
 *           }
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see Flags
 */
int FileInput_get_flags(struct FileInput* ptr_file_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_is_eof(ptr_file_input) (Input_get_flags(ptr_file_input)&FLAG_EOF) 
#else
/**
 *  Returns TRUE if the EOF flag is set (stream reached end-of-file).
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return TRUE if the EOF flag is set (stream reached end-of-file)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte;
 *         long size;
 *         long sum = 0;
 *         long index = 0;
 *         struct FileInput file_input;
 *         ...
 *         FileInput_ctor( &file_input );
 *         ...
 *         if( FileInput_open( &file_input, "values.txt" ) )
 *         {
 *           while( ! FileInput_is_eof( &file_input ) )
 *           {
 *             sum += (long)FileInput_read_byte( &file_input );
 *             index ++;
 *           }
 *         }
 *         ...
 *         FileInput_dtor( &file_input, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 */
bool FileInput_is_eof(struct FileInput* ptr_file_input);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_is_bad(ptr_file_input) (Input_get_flags(ptr_file_input)&FLAG_BAD) 
#else
/**
 *  Returns TRUE if the BAD flag is set (stream is bad). 
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return TRUE if the BAD flag is set (stream is bad)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileInput file_input;
 *       ...
 *       FileInput_ctor_Ex( &file_input , "values.txt");
 *       if ( ! FileInput_is_bad( &file_input ) )
 *       {
 *         //  Success.
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see Flags
 */
bool FileInput_is_bad (struct FileInput* ptr_file_input); 
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileInput_is_good(ptr_file_input) (!(Input_get_flags(ptr_file_input)&FLAG_BAD)) 
#else
/**
 *  Returns TRUE if the BAD flag is not set (stream is good).
 *  @param ptr_file_input A pointer to the initialized FileInput object
 *  @return TRUE if the BAD flag is not set (stream is good). 
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileInput file_input;
 *       ...
 *       FileInput_ctor_Ex( &file_input , "values.txt");
 *       if( FileInput_is_good( &file_input ) )
 *       {
 *         //  Success.
 *         ...
 *       }
 *       ...
 *  @endcode
 *  @see Flags
 */
bool FileInput_is_good (struct FileInput* ptr_file_input);
#endif

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=   object FileOutput.     =
//=  Output file stream.     =
//=                          =
//============================

/** 
 *  @brief The Output file stream.
 *  @ingroup filesystem_overview
 *  Implements basic file output operations.
 *  @see filesystem_overview
 */
struct FileOutput : public Output
{
#ifndef _SKIP_DOC
  char FileOutput_Members_STUB[64];
};
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_ctor(ptr_file_output)  FileOutput_ctor_STUB((ptr_file_output), 1) 
#else
/**
 *  Creates an empty output file stream object.
 *  @param ptr_file_output A pointer to the FileInput structure
 *  @return A pointer to the initialized FileOutput object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte = 100;
 *         struct FileOutput file_output;
 *         ...
 *         FileOutput_ctor( &file_output );
 *         ...
 *         if( FileOutput_open( &file_output, "game.save", TRUE ) )
 *         {
 *           FileOutput_seek( &file_output, 10, SEEK_SET );
 *           FileOutput_write_byte( &file_output, tenth_byte);
 *         }
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *
 */
struct FileOutput* FileOutput_ctor(struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_ctor_Ex(ptr_file_output, file_name, create)  FileOutput_ctor_Ex_STUB((ptr_file_output), 1, (file_name), (create), 0) 
#else
/**
 *  Creates an output file stream object for the file with the specified name.
 *  @param ptr_file_output A pointer to the FileInput structure
 *  @param sz_file_name The file's name
 *  @param create Specifies whether or not to create a new file if the specified file 
 *  does not exist
 *  @return A pointer to the initialized FileOutput object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" , TRUE);
 *       ...
 *       FileOutput_seek( &file_output, 10, SEEK_SET );
 *       FileOutput_write_byte( &file_output, tenth_byte);
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
struct FileOutput* FileOutput_ctor_Ex(struct FileOutput* ptr_file_output, char* sz_file_name, bool create);
#endif

#ifndef _SKIP_DOC
__cyos 174
struct FileOutput* FileOutput_ctor_STUB(struct FileOutput* ptr_file_output, int Magic);

__cyos 175
struct FileOutput* FileOutput_ctor_Ex_STUB(struct FileOutput* ptr_file_output, int Magic, char* filename, bool create, short attr);

__cyos 420
char* FileOutput_class_name(struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_open(ptr_file_output, file_name, create)  FileOutput_open_STUB((ptr_file_output), (file_name), (create), 0 )
#else
/**
 *  Opens an output stream for the file with the specified name
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param sz_file_name The name of the file
 *  @param create Specifies whether or not to create a new file if the specified file 
 *  does not exist
 *  @return TRUE if the function succeeded
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte = 100;
 *         struct FileOutput file_output;
 *         ...
 *         FileOutput_ctor( &file_output );
 *         ...
 *         if( FileOutput_open( &file_output, "game.save", TRUE ) )
 *         {
 *           FileOutput_seek( &file_output, 10, SEEK_SET );
 *           FileOutput_write_byte( &file_output, tenth_byte);
 *         }
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 * 
 */
bool FileOutput_open(struct FileOutput* ptr_file_output, char* sz_file_name, bool create);
#endif

#ifndef _SKIP_DOC
__cyos 912
bool FileOutput_open_STUB(struct FileOutput* ptr_file_output, char* file_name, bool create, short attr);
#endif

/**
 *  Seeks to the specified position, then returns the sought position. If the stream supports a seek operation, it seeks to the specified position in the specified mode. If the requested seek operation cannot be done (not supported or wrong parameters), -1 will be returned. Seeking prior to the beginning of the stream sets the pointer to the stream's first byte. Seeking after the end of the stream sets the pointer to the end of the stream; then the stream will be extended with garbage bytes to the specified size.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position, or -1 if an error occurred
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       {
 *         char tenth_byte = 100;
 *         struct FileOutput file_output;
 *         ...
 *         FileOutput_ctor( &file_output );
 *         ...
 *         if( FileOutput_open( &file_output, "game.save", TRUE ) )
 *         {
 *           FileOutput_seek( &file_output, 10, SEEK_SET );
 *           FileOutput_write_byte( &file_output, tenth_byte);
 *         }
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
#ifndef _SKIP_DOC
__cyos 1060
#endif
long FileOutput_seek(struct FileOutput* ptr_file_output, long pos, seek_t mode);

/**
 *  Returns the stream's position.
 *  @param ptr_file_output A pointer to the initialized FileOutput object.
 *  @return The stream's position, or -1 if the operation is not supported for 
 *  this stream.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       struct FileOutput file_output;
 *       long cur_pos;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save", TRUE );
 *       ...
 *       FileOutput_seek( &file_output, 10, SEEK_SET );
 *       FileOutput_write_byte( &file_output, tenth_byte);
 *       // cur_pos will be equal to 11.
 *       cur_pos = FileOutput_tell( &file_output );
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
*/
#ifndef _SKIP_DOC
__cyos 1235
#endif
long FileOutput_tell(struct FileOutput* ptr_file_output);

/**
 *  Truncates a file associated with the stream. Thread right access sensitive!
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param position The truncation position
 *  @return TRUE if the function succeeded
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save", TRUE );
 *       ...
 *       FileOutput_seek( &file_output, 10, SEEK_SET );
 *       // The file size is 11 bytes now.
 *       FileOutput_truncate( &file_output , 10 );
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1257
#endif
bool FileOutput_truncate(struct FileOutput* ptr_file_output, long position);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_dtor Output_dtor 
#else
/**
 *  Destructor.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" , TRUE);
 *       ...
 *       FileOutput_seek( &file_output, 10, SEEK_SET );
 *       FileOutput_write_byte( &file_output, tenth_byte);
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
void FileOutput_dtor(struct FileOutput* ptr_file_output, int memory_flag);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_write Output_write 
#else
/**
 *  Writes the specified number of bytes to the stream.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param ptr_buffer The buffer with the data to write
 *  @param length The number of bytes to write
 *  @return The exact number of bytes written, or 0 if no bytes were written
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct score_t high_scores[10];
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save", TRUE );
 *       ...
 *       FileOutput_write(&file_output, high_scores, sizeof( high_scores ));
 *       ....
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
long FileOutput_write(struct FileOutput* ptr_file_output, void* ptr_buffer, long length);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_write_byte Output_write_byte 
#else
/**
 *  Writes a byte to the stream.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param fbyte The byte to write
 *  @return -1 in the case of failure, otherwise the value of the byte written
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor( &file_output );
 *       ...
 *       if( FileOutput_open( &file_output, "game.save", TRUE ) )
 *       {
 *         FileOutput_seek( &file_output, 10, SEEK_SET );
 *         FileOutput_write_byte( &file_output, tenth_byte);
 *       }
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
int FileOutput_write_byte(struct FileOutput* ptr_file_output, int fbyte);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_seekp Output_seekp
#else
/**
 *  Seeks an output stream.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @param pos The seek offset
 *  @param mode The seek mode
 *  @return The new position, or -1 if an error occurred
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" );
 *       ...
 *       FileOutput_seekp( &file_output, 10, SEEK_SET );
 *       FileOutput_write_byte( &file_output, tenth_byte);
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
long FileOutput_seekp(struct FileOutput* ptr_file_output, long pos, seek_t mode);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_tellp Output_tellp
#else
/**
 *  Returns the stream's position.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return The stream position, or -1 if the operation is not supported for 
 *  this stream
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char tenth_byte = 100;
 *       long cur_pos;
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" );
 *       ...
 *       FileOutput_seekp( &file_output, 10, SEEK_SET );
 *       FileOutput_write_byte( &file_output, tenth_byte);
 *       //  cur_pos will be equal to 11.
 *       cur_pos = FileOutput_tellp( &file_output );
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
long FileOutput_tellp(struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_get_size Output_get_size 
#else
/**
 *  Returns the stream size (if applicable).
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return The stream size (if applicable)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char obtain_new_value();
 *       ...
 *       {
 *         struct FileOutput file_output;
 *         char current_value;
 *         long size;
 *         ...
 *         FileOutput_ctor_Ex( &file_output, "game.save" );
 *         ...
 *         size = Output_get_size( &file_output );
 *         for( index = 0; index < size; index++ )
 *         {    
 *           current_value = obtain_new_value();
 *           FileOutput_write_byte( &file_output, current_value );
 *         }
 *         cur_pos = FileOutput_tellp(&file_output);
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 */
long FileOutput_get_size(struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_get_flags Output_get_flags 
#else
/**
 *  Returns the stream's current state.
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return The stream's current state
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char obtain_new_value();
 *       ...
 *       {
 *         struct FileOutput file_output;
 *         char current_value;
 *         long size;
 *         ...
 *         FileOutput_ctor_Ex( &file_output, "game.save" );
 *         ...
 *         while( !( FileOutput_get_flags( &file_output ) & FLAG_EOF ) )
 *         {
 *           current_value = obtain_new_value();
 *           FileOutput_write_byte(& file_output, current_value );
 *         }
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see Flags
 */
int FileOutput_get_flags (struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_is_eof(ptr_file_output) (Output_get_flags(ptr_file_output)&FLAG_EOF) 
#else
/**
 *  Returns TRUE if the EOF flag is set (stream reached end-of-file).
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return TRUE if the EOF flag is set (stream reached end-of-file)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char obtain_new_value();
 *       ...
 *       {
 *         struct FileOutput file_output;
 *         char current_value;
 *         long size;
 *         ...
 *         FileOutput_ctor_Ex( &file_output, "game.save" );
 *         ...
 *         while( !( FileOutput_is_eof( &file_output ) ) )
 *         {
 *           current_value = obtain_new_value();
 *           FileOutput_write_byte( &file_output, current_value );
 *         }
 *         ...
 *         FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see Flags
 */
bool FileOutput_is_eof(struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_is_bad(ptr_file_output) (Output_get_flags(ptr_file_output)&FLAG_BAD)
#else
/**
 *  Returns TRUE if the BAD flag is set (stream is bad).
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return TRUE if the BAD flag is set (stream is bad)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" );
 *       if( ! FileOutput_is_bad( &file_output ) )
 *       {
 *          //  Success.
 *       }
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flags
 */
bool FileOutput_is_bad (struct FileOutput* ptr_file_output);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileOutput_is_good(ptr_file_output) (!(Output_get_flags(ptr_file_output)&FLAG_BAD))
#else
/**
 *  Returns TRUE if the BAD flag is not set (stream is good)
 *  @param ptr_file_output A pointer to the initialized FileOutput object
 *  @return TRUE if the BAD flag is not set (stream is good)
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileOutput file_output;
 *       ...
 *       FileOutput_ctor_Ex( &file_output, "game.save" );
 *       if( FileOutput_is_good( &file_output ) )
 *       {
 *         //  Success.
 *       }
 *       ...
 *       FileOutput_dtor( &file_output, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see Flags
 */
bool FileOutput_is_good(struct FileOutput* ptr_file_output);
#endif

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=      object File.        =
//=   Basic file functions.  =
//=                          =
//============================

/**
 *  @brief the object to be manipulated.
 * 
 *  Object for manipulation files.
 *  @ingroup filesystem_overview
 *  @see filesystem_overview
 */
struct File
{
#ifndef _SKIP_DOC
  void* vtable;
#endif

  /**
   *  File name.
   */
  char name[MAX_NAME_LEN];

  /**
   *  File size.
   */
  long size;

  /**
   *  Time of the last modification.
   */
  time_t modification_time;
#ifndef _SKIP_DOC
};
#endif

/**
 *  Creates an empty File object.
 *  @param ptr_file A pointer to the File structure
 *  @return A pointer to the initialized File object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file );
 *       if( File_get_info( &file, "share_data.txt" ) 
 *           && File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see File_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos 207
#endif
struct File* File_ctor(struct File* ptr_file);

/**
 *  Extended version of the File_ctor function.
 *  Creates a File object and initializes it, using the specified file's parameters.
 *  @param ptr_file A pointer to the File structure
 *  @param sz_file_name The name of the file with which to initialize the File object
 *  @return A pointer to the initialized File object.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file, "share_data.txt" );
 *       if( File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see File_ctor.
 */
#ifndef _SKIP_DOC
__cyos 208
#endif
struct File* File_ctor_Ex(struct File* ptr_file, char* sz_file_name);

#ifndef _SKIP_DOC
__cyos 1003
bool File_read_opened(struct File* ptr_file);
#endif

#ifndef _SKIP_DOC
__cyos 1295
bool File_write_opened(struct File* ptr_file);
#endif

/**
 *  Checks whether a file is open for IO functions.
 *  @param ptr_file A pointer to the initialized File object
 *  @return TRUE if the file is open for IO functions
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file );
 *       if( File_get_info( &file, "share_data.txt" ) 
 *           && File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 810
#endif
bool File_is_free(struct File* ptr_file);

#ifndef _SKIP_DOC
__cyos 1093
bool File_set_attribute(struct File* ptr_file, short attr);

__cyos 631
short File_get_attribute(struct File* ptr_file);
#endif

/**
 *  Returns the time the file was last modified.
 *  @param ptr_file A pointer to the initialized File object
 *  @return The time the file was last modified 
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       struct Time file_time;
 *       ...
 *       {
 *         File_ctor( &file );
 *         if( File_get_info( &file, "share_data.txt" ) 
 *             && File_is_free( &file ) )
 *         {
 *           Time_decode( &file_time, File_get_date( &file ) );
 *           TRACE("Date of modification is  %d.%d.%d",
 *                  file_time.month,
 *                  file_time.day,
 *                  file_time.year + 1990 );
 *         }
 *         ...
 *         File_dtor( &file, LEAVE_MEMORY );
 *       }
 *       ...
 *  @endcode
 *  @see File_set_date.
 */
#ifndef _SKIP_DOC
__cyos 1348
#endif
time_t File_get_date(struct File* ptr_file);

/**
 *  Sets date the file was last modified.
 *  @param ptr_file A pointer to the initialized File object
 *  @param date A date the file was last modified
 *  @return TRUE if the time set was successful
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file );
 *       if( File_get_info( &file, "share_data.txt" ) 
 *           && File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *         //  Updates the date of the last modifications.
 *         File_set_date( &file, time() );
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see File_get_date.
 */
#ifndef _SKIP_DOC
__cyos 1349
#endif
bool File_set_date(struct File* ptr_file, time_t date);

/**
 *  Initializes the File object's fields (name, date, length) of the File object, using the 
 *  specified file's parameters.
 *  @param ptr_file A pointer to the initialized File object
 *  @param sz_file_name The name of the file with which to initialize the File object
 *  @return FALSE if the file is not found
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file );
 *       if( File_get_info( &file, "share_data.txt" ) 
 *           && File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 680
#endif
bool File_get_info(struct File* ptr_file, char* sz_file_name);

#ifndef _SKIP_DOC
__cyos 427
char* File_class_name(struct File* ptr_file);

__cyos 572
int File_file_header_size(char* name);
#endif

/**
 *  Estimates the number of free bytes on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return Number of free bytes on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_free_user_bytes_total.
 */
#ifndef _SKIP_DOC
__cyos 614
#endif
long File_free_bytes_total(char* sz_device_name);

/**
 *  Estimates the number of free bytes for user data on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return Number of free bytes on the specified device for user data
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_free_bytes_total.
 */
#ifndef _SKIP_DOC
__cyos 616
#endif
long File_free_user_bytes_total(char* sz_device_name);

/**
 *  Returns the total number of files on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The total number of files on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_bytes_total.
 */
#ifndef _SKIP_DOC
__cyos 582
#endif
long File_files_total(char* sz_device_name);

/**
 *  Returns the total number of bytes on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The total number of bytes on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_files_total.
 */
#ifndef _SKIP_DOC
__cyos 396
#endif
long File_bytes_total(char* sz_device_name);

/**
 *  Returns the total number of blocks on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The total number of blocks on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_bad_blocks_total.
 */
#ifndef _SKIP_DOC
__cyos 386
#endif
long File_blocks_total(char* sz_device_name);

/**
 *  Returns the total number of bad blocks on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The total number of bad blocks on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 *  @see File_blocks_total.
 */
#ifndef _SKIP_DOC
__cyos 377
#endif
long File_bad_blocks_total(char* sz_device_name);

/**
 *  Returns the block size for the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The block size for the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 384
#endif
int File_block_size(char* sz_device_name);

/**
 *  Returns the number of free blocks on the specified device.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return The number of free blocks on the specified device
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 613
#endif
long File_free_blocks_total(char* sz_device_name);

/**
 *  Returns TRUE if the specified device is bootable.
 *  @param sz_device_name The device name (or "" for local flash memory)
 *  @return TRUE if the specified device is bootable
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       TRACE( "------Flash Info---------" );
 *       TRACE( "Bytes total on your device : %ld ", File_bytes_total( "" ) );
 *       TRACE( "Block size : %ld ", File_block_size( "" ) );
 *       TRACE( "Blocks total on your device : %ld ", File_blocks_total( "" ) );
 *       TRACE( "Free blocks total on your device : %ld ", File_free_blocks_total( "" ) );
 *       TRACE( "Bad blocks total on your device : %ld ", File_bad_blocks_total( "" ) );
 *       TRACE( "Free bytes total on your device : %ld ", File_free_bytes_total( "" ) );
 *       TRACE( "Free bytes for user needs on your device : %ld ", File_free_user_bytes_total( "" ) );
 *       TRACE( "File number on your device  : %ld ", File_files_total( "" ) );
 *       TRACE( "Device is %s ", File_bootable("") ? "bootable" : "not bootable" );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 389
#endif
bool File_bootable(char* sz_device_name);

/**
 *  Returns TRUE if the specified string is valid as a filename.
 *  @param sz_file_name The file's name
 *  @return TRUE if the specified string is valid as a file name
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char* input_new_file_name();
 *       ...
 *       if( File_is_valid_name( input_new_file_name() ) )
 *       {
 *         //  Creating new file.
 *         ...
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 826
#endif
bool File_is_valid_name(char* sz_file_name);

/**
 *  Renames the specified file if it is not being modified (written to or read).
 *  @param sz_old_filename Old file name
 *  @param sz_new_filename New file name
 *  @return TRUE if the file is successfully renamed, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( File_exists( "last_game.sav" ) )
 *       {
 *         File_rename( "last_game.sav", "prev_game.sav" );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1033
#endif
bool File_rename(char* sz_old_filename, char* sz_new_filename);

/**
 *  Deletes the specified file if it is not being modified (written to or read).
 *  @param sz_file_name The file's name
 *  @return TRUE if the file is successfully removed, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( File_exists( "save.dat" ) )
 *       {
 *         //  Loads unfinished game.
 *         ...
 *       }
 *       ...
 *       if( game_finished )
 *       {
 *         File_remove( "save.dat" );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1026
#endif
bool File_remove(char* sz_file_name);

/**
 *  Checks whether a file with specified name exists.
 *  @param sz_file_name The file name to check.
 *  @return TRUE if the file with the specified name exists, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( File_exists( "save.dat" ) )
 *       {
 *         //  Load unfinished game.
 *         ...
 *       }
 *       ...
 *       if( game_finished )
 *       {
 *         File_remove( "save.dat" );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 557
#endif
bool File_exists(char* sz_file_name);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define File_dtor(ptr_file, memory_flag)  if((memory_flag) == FREE_MEMORY) free(ptr_file); 
#else
/**
 *  Destructor.
 *  @param ptr_file A pointer to the initialized File object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated
 *  for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the
 *  object was static or allocated in a stack
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct File file;
 *       ...
 *       File_ctor( &file );
 *       if( File_get_info( &file, "share_data.txt" ) 
 *           && File_is_free( &file ) )
 *       {
 *         //  Performs IO operations.
 *         ...
 *       }
 *       ...
 *       File_dtor( &file, LEAVE_MEMORY ); 
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
void File_dtor(struct File* ptr_file, int memory_flag);
#endif


#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=    object FileFind.      =
//=   File finder object.    =
//=                          =
//============================

/**
 *  @brief File finder. Implements the functions needed to find a set of files.
 *
 *  Implements the functions needed to find a set of files.
 *  @ingroup filesystem_overview
 *  @see filesystem_overview
 */
struct FileFind
{
#ifndef _SKIP_DOC
  void*        vtable;
  char         mask[ MAX_NAME_LEN ];
  char         fname[ MAX_NAME_LEN ];
  int          lastfile;
  int          PADDING;
  struct File* fileinfo;
};
#endif

/**
 *  Starts a new search session using a specified search mask.
 *  @param ptr_file_find A pointer to the FileFind structure
 *  @param ptr_file_info A pointer the initialized File object to store the result
 *  @param sz_mask File search mask (symbols '*' and '?' supported)
 *  @return A pointer to the initialized FileFind object
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileFind searcher;
 *       struct File file_info;
 *       ...
 *       File_ctor( &file_info );
 *       FileFind_ctor( &searcher, &file_info, "quake.*" );
 *       ...
 *       TRACE( "Search results:" );
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       File_dtor( &file_info , LEAVE_MEMORY );
 *       FileFind_dtor( &searcher, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 241
#endif
struct FileFind* FileFind_ctor(struct FileFind* ptr_file_find, struct File* ptr_file_info, char* sz_mask);

/**
 *  Checks wheter there are any elements left in the search.
 *  @param ptr_file_find A pointer to the initialized FileFind object
 *  @return TRUE if there are any elements left in the search, otherwise FALSE
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileFind searcher;
 *       struct File file_info;
 *       ...
 *       File_ctor( &file_info );
 *       FileFind_ctor( &searcher, &file_info, "quake.*" );
 *       ...
 *       TRACE("Search results:");
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       File_dtor( &file_info , LEAVE_MEMORY );
 *       FileFind_dtor( &searcher, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 760
#endif
bool FileFind_has_more_elements(struct FileFind* ptr_file_find);

/**
 *  Initializes the new file search session.
 *  @param ptr_file_find A pointer to the initialized FileFind object
 *  @param ptr_file_info A pointer to the initialized File object to store the result
 *  @param sz_mask File search mask (symbols '*' and '?' supported).
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileFind searcher;
 *       struct File file_info;
 *       ...
 *       File_ctor( &file_info );
 *       FileFind_ctor( &searcher, &file_info, "quake.*" );
 *       ...
 *       TRACE( "Search results for quake :" );
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       FileFind_init( &searcher, &file_info, "warcraft.*" );
 *       TRACE( "Search results for warcraft :" );
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       File_dtor( &file_info , LEAVE_MEMORY );
 *       FileFind_dtor( &searcher, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 791
#endif
void FileFind_init(struct FileFind* ptr_file_find, struct File* ptr_file_info, char* sz_mask);

#ifndef _SKIP_DOC
__cyos 434
char* FileFind_class_name(struct FileFind* ptr_file_find);
#endif

/**
 *  Retrieves the next element of the file search session.
 *  @param ptr_file_find A pointer to the initialized FileFind object
 *  @return File's address, or NULL if there are no more elements
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileFind searcher;
 *       struct File file_info;
 *       ...
 *       FileFind_ctor( &searcher, &file_info, "quake.*" );
 *       ...
 *       TRACE( "Search results:" );
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       File_dtor( &file_info , LEAVE_MEMORY );
 *       FileFind_dtor( &searcher, LEAVE_MEMORY );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 901
#endif
struct File* FileFind_next_element(struct FileFind* ptr_file_find);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define FileFind_dtor(ptr_file_find, memory_flag) if((memory_flag) == FREE_MEMORY) free(ptr_file_find);
#else
/**
 *  Deletes an object.
 *  @param ptr_file_find A pointer to the initialized FileFind object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or allocated in a stack
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct FileFind searcher;
 *       struct File file_info;
 *       ...
 *       FileFind_ctor( &searcher, &file_info, "quake.*" );
 *       File_ctor( &file_info );
 *       ...
 *       TRACE( "Search results:");
 *       do
 *       {
 *         FileFind_next_element( &searcher );
 *         TRACE( "\t%s has size %ld", file_info.name, file_info.size );
 *       }while( FileFind_has_more_elements( &searcher ) );
 *       ...
 *       File_dtor( &file_info, LEAVE_MEMORY );
 *       FileFind_dtor( &searcher , LEAVE_MEMORY );
 *       ...
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
void FileFind_dtor(struct FileFind* ptr_file_find, int memory_flag);
#endif

#ifdef _SKIP_DOC
};
#endif


//  Helper structure and function.
#ifndef _SKIP_DOC

struct FileBase
{
  char FileBase_Members_STUB[72];
};
#endif



#endif  //  #ifndef __CYBER_FILES_H_INCLUDE__

