/** @file cyber-graph.h
 *   CyOS graphics.
 */
#ifndef __CYBER_GRAPH_H_INCLUDE__
#define __CYBER_GRAPH_H_INCLUDE__
/** CYBIKO screen width.
 *  @ingroup drawing_primitives_group
*/
#define SCREEN_WIDTH  160
/**  CYBIKO screen height.
 *  @ingroup drawing_primitives_group
*/
#define SCREEN_HEIGHT 100
/**  Keep as is.
 *  @ingroup bitmaps_and_bitmap_sequence_group
*/
#define BM_NORMAL  0
/**  Invert colors. <b>Obsolete</b> (no longer supported)
 *  @ingroup bitmaps_and_bitmap_sequence_group
*/
#define BM_INVERSE 1
/**  Invert x coordinate.
 *  @ingroup bitmaps_and_bitmap_sequence_group
*/
#define BM_FLIP_X  2
/**  Invert y coordinate.
 *  @ingroup bitmaps_and_bitmap_sequence_group
*/
#define BM_FLIP_Y  4

/**  Make the object white.
 *  @ingroup drawing_primitives_group
*/
#define CLR_WHITE  0
/**  Make the object light gray.
 *  @ingroup drawing_primitives_group
*/
#define CLR_LTGRAY 85
/**  Make the object dark gray.
 *  @ingroup drawing_primitives_group
*/
#define CLR_DKGRAY 171
/**  Make the object black.
 *  @ingroup drawing_primitives_group
*/
#define CLR_BLACK  255

/**  Place the image as is.
 *  @ingroup drawing_primitives_group
*/
#define DM_PUT 0
/**  Make the image opaque; do not allow background color to show through the object. See set_bckcolor().
 *  @ingroup drawing_primitives_group
*/
#define DM_OR  1
/**  XOR pixels overlay bitmap pixels.
 *  @ingroup drawing_primitives_group
*/
#define DM_XOR 2


/**
 *  @brief a structure for creating and working with simple bitmapped objects.
 *  @ingroup bitmaps_and_bitmap_sequence_group
 *  A structure for creating and working with simple bitmap object.
 *  Using this structure, you can create a bitmap object in one of four ways:
 *  <ol>
 *  <li>Create an empty bitmap object the same way Bitmap_ctor does;</li>
 *  <li>Create a Bitmap object using a resource;</li>
 *  <li>Create an empty bitmap with memory allocated for it;</li>
 *  <li>Create a Bitmap object that is an exact copy of an another bitmap.</li>
 *  </ol>
 *  Make sure to create every Bitmap object using one of the "constructor" functions, and
 *  to release each Bitmap object with the Bitmap_dtor function.
 *  <br>
 *  <br>
 *  Use the Bitmap_fill function to fill a bitmap with the color marked by the second function parameter's pointer. The second parameter also specifies
 *  the type color, via the color_t type function. This function is defined in the header, and described in the "Typedefs" section. *  a header and described in the "Typedefs" section.
 *  You can implement operations with streams using the Bitmap_load and Bitmap_storefunctions.
 *  <br>
 *  <br>
 *  Please study the examples for these functions carefully.
 *  @see bitmaps_and_bitmap_sequence_group
 */
struct Bitmap
{
#ifndef _SKIP_DOC
  char  Members_STUB_1[12];
#endif  //  #ifndef _SKIP_DOC
/**
 *  Establishes a physical bitmap width (in pixels).
*/
  int w;
/**
 *  Establishes a physical bitmap height (in pixels).
*/
  int h;

#ifndef _SKIP_DOC
  char  Members_STUB_2[12];
};
#endif  //  #ifndef _SKIP_DOC

/**
 *  The simple Bitmap constructor. Creates an empty bitmap. First you must create a Bitmap object and transfer the pointer on it into this function.
 *  @param ptr_bitmap A pointer to the Bitmap structure
 *  @return A pointer to an initialized Bitmap object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bitmap;
 *     Bitmap_ctor( &bitmap );
 *     ...
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 217
#endif
struct Bitmap* Bitmap_ctor(struct Bitmap* ptr_bitmap);

/**
 *  The first extended version of the Bitmap_ctor function. It creates an empty bitmap from a resource file.
 *  @param ptr_bitmap A pointer to the Bitmap structure
 *  @param filename The name of a resource file
 *  @return A pointer to an initialized Bitmap object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bmp;
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_ctor
 */
#ifndef _SKIP_DOC
__cyos 218
#endif
struct Bitmap* Bitmap_ctor_Ex1(struct Bitmap* ptr_bitmap, char* filename);

/**
 *  The second extended version of the Bitmap_ctor function. It creates an empty bitmap and allocates memory for it.
 *  @param ptr_bitmap A pointer to the Bitmap structure
 *  @param width The width of the bitmap object
 *  @param height The height of the bitmap object
 *  @param bpp The number of bits per pixel that you need to allocate - 2 bits for a four color display, and 1 bit for a monochrome display
 *  @return A pointer to an initialized Bitmap object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bmp;
 *     // SCREEN_WIDTH and SCREEN_HEIGHT are described in the 'Defines' section.
 *     Bitmap_ctor_Ex2( &bmp, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_ctor
 */
#ifndef _SKIP_DOC
__cyos 220
#endif
struct Bitmap* Bitmap_ctor_Ex2(struct Bitmap* ptr_bitmap, int width, int height, int bpp);

/**
 *  The third extended version of the Bitmap_ctor function. Builds the exact copy of another bitmap.
 *  @param ptr_bitmap A pointer to destination Bitmap object
 *  @param templ A pointer to template Bitmap object
 *  @return A pointer to initialized Bitmap object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap  bitmap_templ;
 *     Bitmap_ctor_Ex1( &bitmap_templ, "bmp.pic" );
 *     ...
 *     // Working with the 'ptr_bitmap_templ' bitmap.
 *     ...
 *     struct Bitmap  bitmap;
 *     Bitmap_ctor_Ex3( &bitmap, &bitmap_templ );
 *     ...
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *     Bitmap_dtor( &bitmap_templ, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_ctor
 */
#ifndef _SKIP_DOC
__cyos 219
#endif
struct Bitmap* Bitmap_ctor_Ex3(struct Bitmap* ptr_bitmap, struct Bitmap* templ);

/**
 *  Destructor. This function deletes a Bitmap object.
 *  @param ptr_bitmap A pointer to the initialized Bitmap object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If memory_flag is
 *  FREE_MEMORY when destructor frees memory, allocated for the object.
 *  FREE_MEMORY used if memory was allocated by malloc().
 *  If object was static or allocated in stack use LEAVE_MEMORY
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bitmap;
 *     // SCREEN_WIDTH and SCREEN_HEIGHT are defined in the "cyber-graph.h" file.
 *     Bitmap_ctor_Ex2( &bitmap, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );
 *     ...
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos 106
#endif
void Bitmap_dtor(struct Bitmap* ptr_bitmap, int memory_flag);

#ifndef _SKIP_DOC
__cyos 430
char* Bitmap_class_name(struct Bitmap*);
#endif

/**
 *  Fills whole bitmap by a color pointed on the color_t value. The color_t type is described in the 'Typedefs' section.
 *  @param ptr_bitmap A pointer to the initialized Bitmap object
 *  @param color The color_t value. This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bitmap;
 *     ...
 *     Bitmap_fill( &bitmap, CLR_BLACK );
 *     ...
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_clear, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 584
#endif
void Bitmap_fill(struct Bitmap* ptr_bitmap, color_t color);

/**
 *  Fills whole bitmap by white color.
 *  @param ptr_bitmap A pointer to the initialized Bitmap object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bitmap;
 *     ...
 *     Bitmap_fill( &bitmap, CLR_DKGRAY );
 *     ...
 *     // It is the same as Bitmap_fill( ptr_bitmap, CLR_WHITE );
 *     Bitmap_clear( &bitmap );
 *     ...
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_fill.
 */
#ifndef _SKIP_DOC
__cyos 440
#endif
void Bitmap_clear(struct Bitmap* ptr_bitmap);

/**
 *  Loads bitmap from an opened stream.
 *  @param ptr_bitmap A pointer to the initialized Bitmap object
 *  @param input A pointer to Input stream
 *  @return TRUE if a bitmap object was loaded successfully
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Input * ptr_input;
 *     struct Bitmap  bmp;
 *     ...
 *     Bitmap_ctor(&bmp);
 *     ...
 *     // Your make file must have the archive resource "bmp.pic" linked by filer.exe.
 *     // For example, see the Cybiko Compiler's Introduction Section
 *     ptr_input = open_resource_Ex( "bmp.pic" );
 *     if( !Bitmap_load( &bmp, ptr_input ) )
 *         return FALSE;
 *     else
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     Input_dtor( ptr_input, LEAVE_MEMORY );
 *  @endcode
 *  @see Bitmap_store.
 */
#ifndef _SKIP_DOC
__cyos 846
#endif
bool Bitmap_load(struct Bitmap* ptr_bitmap, struct Input* input);

/**
 *  Stores bitmap to an opened stream.
 *  @param ptr_bitmap A pointer to the initialized Bitmap object
 *  @param output A pointer to Output stream
 *  @return TRUE if a bitmap object was stored successfully
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Output * ptr_output;
 *     bool   fBmp_store;
 *     struct Bitmap bmp;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module ); // Application module initialization
 *     ...
 *     Bitmap_ctor_Ex2( &bmp, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );
 *     ...
 *     // Your make file must have the archive resource "bmp.pic" linked by filer.exe.
 *     // For example, see the Cybiko Compiler's Introduction Section.
 *     ptr_output = Archive_open_write_Ex( main_module.m_process->module->archive, "bmp.pic" );
 *     fBmp_store = Bitmap_store( &bmp, ptr_output );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     Output_dtor( ptr_output, FREE_MEMORY );
 *  @endcode
 *  @see Bitmap_load.
 */
#ifndef _SKIP_DOC
__cyos 1195
#endif
bool Bitmap_store(struct Bitmap* ptr_bitmap, struct Output* output);

#ifndef _SKIP_DOC
__cyos 750
int Bitmap_get_width(struct Bitmap* ptr_bitmap);
#endif

#ifndef _SKIP_DOC
__cyos 677
int Bitmap_get_height(struct Bitmap* ptr_bitmap);
#endif

#ifndef _SKIP_DOC
__cyos 714
struct rect_t* Bitmap_get_rect(struct Bitmap* ptr_bitmap, struct rect_t* rc);
#endif

#ifndef _SKIP_DOC
__cyos 713
struct rect_t* Bitmap_get_r(struct Bitmap* ptr_bitmap, struct rect_t* rc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Bitmap_get_w(ptr_bitmap)         (ptr_bitmap->w)
#else
/**
 *  Returns physical width of the Bitmap (in pixels).
 *  @param ptr_bitmap A pointer to the BitmapSequence structure
 *  @return A physical width of the Bitmap (in pixels)
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap gfx_bmp;
 *     int height_gfx, width_gfx;
 *     ...
 *     // Returns physical height of the Bitmap object
 *     height_gfx = Bitmap_get_h( &gfx_bmp );
 *     // Returns physical width of the Bitmap object
 *     width_gfx = Bitmap_get_w( &gfx_bmp );
 *     ...
 *  @endcode
 *  @see Bitmap_get_h.
 */
int Bitmap_get_w(struct Bitmap* ptr_bitmap);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Bitmap_get_h(ptr_bitmap)         (ptr_bitmap->h)
#else
/**
 *  Returns physical height of the Bitmap (in pixels).
 *  @param ptr_bitmap A pointer to the BitmapSequence structure
 *  @return A physical height of the Bitmap (in pixels)
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap gfx_bmp;
 *     int height_gfx, width_gfx;
 *     ...
 *     // Returns physical height of the Bitmap object
 *     height_gfx = Bitmap_get_h( &gfx_bmp );
 *     // Returns physical width of the Bitmap object
 *     width_gfx = Bitmap_get_w( &gfx_bmp );
 *     ...
 *  @endcode
 *  @see Bitmap_get_w.
 */
int Bitmap_get_h(struct Bitmap* ptr_bitmap);
#endif

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=     BitmapSequence.      =
//=  The array of Bitmaps.   =
//=                          =
//============================

/** @brief While the Bitmap structure is used for working with a single bitmap, the BitmapSequence is used for working with a Bitmap array.
 *  @ingroup bitmaps_and_bitmap_sequence_group
 *  While the Bitmap structure above is used for working with a single bitmapped image, the BitmapSequence is used for working with a Bitmap array.
 *  To create a BitmapSequence object, you must use the BitmapSequence_ctor function.
 *  <br>
 *  <br>
 *  Use the BitmapSequence_dtor function to release the created object by after you work with it.
 *  To receive the bitmap object from the bitmap sequence, use the BitmapSequence_get_bitmap function.
 *  <br>
 *  <br>
 *  The BitmapSequence get size function determines the number of bitmap objects in the BitmapSequence object you can receive.
 *  Bitmap structure operations with streams are designated the same way. They are implemented in the BitmapSequence_load and BitmapSequence_store functions.
 *  <br>
 *  <br>
 *  Please pay special attention to the examples for these functions.
 *  @see bitmaps_and_bitmap_sequence_group
 */
struct BitmapSequence
{
#ifndef _SKIP_DOC
  char BitmapSequence_Members_STUB[ 48 ];
};
#endif  //  #ifndef _SKIP_DOC

/**
 *  Simple BitmapSequence constructor. Creates an empty bitmap sequence.
 *  @param ptr_bitmap_sequence A pointer to the BitmapSequence structure
 *  @return A pointer to the initialized BitmapSequence object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct BitmapSequence bmps;
 *     BitmapSequence_ctor( &bmps );
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *  @endcode
 *  @see BitmapSequence_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos 192
#endif
struct BitmapSequence* BitmapSequence_ctor(struct BitmapSequence* ptr_bitmap_sequence);

/**
 *  This is the extended version of the BitmapSequence_ctor function. It allows you to use a resource for creating a BitmapSequence object.
 *  @param ptr_bitmap_sequence A pointer to the BitmapSequence structure
 *  @param file_name A name of file, where BitmapSequense's data were stored
 *  @return A pointer to the initialized BitmapSequence object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct BitmapSequence bmps;
 *     BitmapSequence_ctor_Ex( &bmps, "MyFont.pic" );
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *  @endcode
 *  @see BitmapSequence_ctor.
 */
#ifndef _SKIP_DOC
__cyos 193
#endif
struct BitmapSequence* BitmapSequence_ctor_Ex(struct BitmapSequence* ptr_bitmap_sequence, char* file_name);

/**
 *  Destructor. This function deletes the BitmapSequence object.
 *  @param ptr_bitmap_sequence A pointer to the initialized BitmapSequence object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY.
 *  If the memory was allocated for the object by malloc(),
 *  use FREE_MEMORY to free it.
 *  If the object was static or allocated in stack
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct BitmapSequence bmps;
 *     BitmapSequence_ctor_Ex( &bmps, "MyFont.pic" );
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos 96
#endif
void BitmapSequence_dtor(struct BitmapSequence* ptr_bitmap_sequence, int memory_flag);

#ifndef _SKIP_DOC
__cyos 424
char* BitmapSequence_class_name(struct BitmapSequence* ptr_bitmap_sequence);
#endif

/**
 *  Returns the number of bitmaps in a BitmapSequence object.
 *  @param ptr_bitmap_sequence A pointer to the initialized BitmapSequence object
 *  @return The number of bitmaps in the BitmapSequence object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct BitmapSequence bmps;
 *     BitmapSequence_ctor_Ex( &bmps, "MyFont.pic" );
 *     ...
 *     TRACE( "%d symbols in this font", BitmapSequence_get_size( &bmps ) );
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 727
#endif
int BitmapSequence_get_size(struct BitmapSequence* ptr_bitmap_sequence);

#ifndef _SKIP_DOC
__cyos 751
int BitmapSequence_get_x(struct BitmapSequence* ptr_bitmap_sequence, int bitmap_index);
#endif

#ifndef _SKIP_DOC
__cyos 753
int BitmapSequence_get_y(struct BitmapSequence* ptr_bitmap_sequence, int bitmap_index);
#endif

/**
 *  Returns a pointer on the Bitmap object that has the 'bitmap_index' index.
 *  @param ptr_bitmap_sequence A pointer to the BitmapSequence object
 *  @param bitmap_index The bitmap index in the BitmapSequence structure
 *  @return A pointer to the Bitmap object that has the 'bitmap_index' index
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct BitmapSequence bmps;
 *     struct Bitmap * ptr_bmp_symbol_A;
 *     BitmapSequence_ctor_Ex( &bmps , "font_1.pic" );
 *     ...
 *     ptr_bmp_symbol_A = BitmapSequence_get_bitmap( &bmps, 0 );
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 634
#endif
struct Bitmap* BitmapSequence_get_bitmap(struct BitmapSequence* ptr_bitmap_sequence, int bitmap_index);

/**
 *  Loads the bitmap sequence from an opened stream.
 *  @param ptr_bitmap_sequence A pointer to the BitmapSequence object
 *  @param input A pointer to the Input stream
 *  @return TRUE if bitmap sequence was loaded successfully
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Input * ptr_input;
 *     struct BitmapSequence bmps;
 *     ...
 *     // Your make file must have the archive resource "bmp.pic" linked by filer.exe.
 *     // For an example, see the Cybiko Compiler's Introduction Section.
 *     ptr_input = open_resource_Ex( "font_1.pic" );
 *     ...
 *     if( !BitmapSequence_load( &bmps, ptr_input ) )
 *     {
 *        ...
 *     }
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *     Input_dtor( ptr_input, FREE_MEMORY );
 *  @endcode
 *  @see BitmapSequence_store.
 */
#ifndef _SKIP_DOC
__cyos 842
#endif
bool BitmapSequence_load(struct BitmapSequence* ptr_bitmap_sequence, struct Input* input);


/**
 *  Stores a bitmap sequence to an opened stream.
 *  @param ptr_bitmap_sequence A pointer to the BitmapSequence object
 *  @param output A pointer to the Output stream
 *  @return TRUE if the bitmap sequence was stored successfully
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Output * ptr_output;
 *     bool fBmp_store;
 *     struct BitmapSequence bmps;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module ); // Application module initialization
 *     ...
 *     Bitmap_ctor_Ex2( &bmp, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );
 *     ...
 *     // Your make file must have the archive resource "bmp.pic" linked by filer.exe.
 *     // For an example, see the Cybiko Compiler's Introduction Section.
 *     ptr_output = Archive_open_write_Ex( main_module.m_process->module->archive, "font_1.pic" );
 *     fBmp_store = BitmapSequence_store( &bmps, ptr_output );
 *     ...
 *     BitmapSequence_dtor( &bmps, LEAVE_MEMORY );
 *     Output_dtor( ptr_output, FREE_MEMORY );
 *  @endcode
 *  @see BitmapSequence_load.
 */
#ifndef _SKIP_DOC
__cyos 1194
#endif
bool BitmapSequence_store(struct BitmapSequence* ptr_bitmap_sequence, struct Output* output);

#ifdef _SKIP_DOC
};
#endif


//============================
//=                          =
//=      object Font.        =
//=   A character font.      =
//=                          =
//============================

/** @brief provides functions for manipulating the appearance of text characters.
 *  @ingroup fonts_and_text_group
 *  To use a Font object, construct it with the Font_ctor or Font_ctor_Ex functions.
 *  Use Font_dtor to deconstruct a Font object after use. Font object operations include
 *  receiving information about the character width and height, string length or length of
 *  a predefined number of characters in a string; as well as receiving the Bitmap representation
 *  of any character in a string.
 *  <br>
 *  <br>
 *  You can fix the current font by calling the Font_set_fixed function.
 *  You can set the font spacing value by calling the Font_set_spacing function.
 *  You can also use the extended version of the Font_ctor function, which calls
 *  Font_ctor_Ex and enables it to create a font object using a resource.
 *  <br>
 *  <br>
 *  Since all characters in a text string are elements of a BitmapSequence object,
 *  you can manipulate a text string as if it were a BitmapSequence object.
 *  @see fonts_and_text_group
 */
struct Font : public BitmapSequence
{
#ifndef _SKIP_DOC
  char Font_Members_STUB[ 40 ];
};
#endif  //  #ifndef _SKIP_DOC

/** Handle for the 'cool bold' font.
*/
#ifndef _SKIP_DOC
__cyos 468
#endif
struct Font* cool_bold_font;

/** Handle for the 'cool normal' font.
*/
#ifndef _SKIP_DOC
__cyos 469
#endif
struct Font* cool_normal_font;

/** Handle for the 'mini bold' font.
*/
#ifndef _SKIP_DOC
__cyos 891
#endif
struct Font* mini_bold_font;

/** Handle for the 'mini normal' font.
*/
#ifndef _SKIP_DOC
__cyos 892
#endif
struct Font* mini_normal_font;

/**
 *  A simple Font constructor. Creates an empty Font object.
 *  @param ptr_font A pointer to the Font structure
 *  @return A pointer to the initialized Font object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor( &font );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos 210
#endif
struct Font* Font_ctor(struct Font* ptr_font);

/**
 *  The extended version of the Font_ctor function. It creates a Font object from a file.
 *  @param ptr_font A pointer to the Font structure
 *  @param file_name The name of a file that contains a font
 *  @param fixed sets TRUE for fonts with a fixed width
 *  @param spacing The font spacing value in pixels
 *  @return A pointer to the initialized Font object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 * @see Font_ctor.
 */
#ifndef _SKIP_DOC
__cyos 211
#endif
struct Font* Font_ctor_Ex(struct Font* ptr_font, char* file_name, bool fixed, int spacing);

/**
 *  Destructor. Deletes a Font object.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY.
 *  If the memory was allocated for the object by malloc(),
 *  use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object
 *  was static or allocated in a stackcan
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor( &font );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos 102
#endif
void Font_dtor(struct Font* ptr_font, int memory_flag);

#ifndef _SKIP_DOC
__cyos 428
char* Font_class_name(struct Font* ptr_font);
#endif

/**
 *  Returns the font name. This is equal to the full font file name.
 *  @param ptr_font A pointer to the initialized Font object
 *  @return The font name
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor( &font );
 *     ...
 *     char*  ptr_font_name;
 *     ptr_font_name = Font_get_name( &font );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_set_name.
 */
#ifndef _SKIP_DOC
__cyos 691
#endif
char* Font_get_name(struct Font* ptr_font);

/**
 *  Sets the font name manually.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param filename The file name
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor( &font );
 *     ...
 *     Font_set_name( &font, "MyFont.fnt" );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_get_name.
 */
#ifndef _SKIP_DOC
__cyos 1135
#endif
void Font_set_name(struct Font* ptr_font, char* filename);

/**
 *  Returns the character's width in pixels.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param chr A character
 *  @return The character's width in pixels
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     int char_width;
 *     Font_ctor( &font );
 *     ...
 *     char_width = Font_get_char_width( &font, 'c' );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_get_char_height, Font_string_width.
 */
#ifndef _SKIP_DOC
__cyos 651
#endif
int Font_get_char_width(struct Font* ptr_font, int chr);

/**
 *  Returns the string's width in pixels.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param str A string
 *  @return The string's width in pixels
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     int    str_width;
 *     char*  str = "any string";
 *     ...
 *     Font_ctor( &font );
 *     ...
 *     str_width = Font_string_width( &font, str );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_get_char_width, Font_string_width_Ex.
 */
#ifndef _SKIP_DOC
__cyos 736
#endif
int Font_string_width(struct Font* ptr_font, char* str);

/**
 *  The extended version of the Font_string_width function. This function returns the width in pixels of the first 'flen' characters of a string.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param str A string
 *  @param len The number of characters needed
 *  @return The string's width in pixels
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     int    str_width;
 *     char*  str = "any string";
 *     ...
 *     Font_ctor( &font );
 *     ...
 *     str_width = Font_string_width_Ex( &font, str, 3 );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_string_width, Font_get_char_width.
 */
#ifndef _SKIP_DOC
__cyos 737
#endif
int Font_string_width_Ex(struct Font* ptr_font, char* str, int len);

/**
 *  Returns the character's height in pixels.
 *  @param ptr_font A pointer to the initialized Font object
 *  @return The character's height in pixels
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     int char_height;
 *     ...
 *     Font_ctor( &font );
 *     ...
 *     char_height = Font_get_char_height( &font );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_get_char_width.
 */
#ifndef _SKIP_DOC
__cyos 649
#endif
int Font_get_char_height(struct Font* ptr_font);

/**
 *  Determines if the font is fixed.
 *  @param ptr_font A pointer to the initialized Font object
 *  @return TRUE if the font is fixed
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     if ( !Font_is_fixed( &font ) )
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_set_fixed.
 */
#ifndef _SKIP_DOC
__cyos 809
#endif
bool Font_is_fixed(struct Font* ptr_font);

/**
 *  Sets or doesn't set a fixed font.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param fixed TRUE if it's need to set a fixed font
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     Font_set_fixed( &font, FALSE );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_is_fixed.
 */
#ifndef _SKIP_DOC
__cyos 1123
#endif
void Font_set_fixed(struct Font* ptr_font, bool fixed);

/**
 *  Returns the font spacing value.
 *  @param ptr_font A pointer to the initialized Font object
 *  @return The font spacing value
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     int    font_space;
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     font_space = Font_get_spacing( &font );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_set_spacing.
 */
#ifndef _SKIP_DOC
__cyos 730
#endif
int Font_get_spacing(struct Font* ptr_font);

/**
 *  Sets the font spacing value.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param spacing A font's spacing value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font font;
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     if ( Font_get_spacing( &font ) < 3 )
 *       Font_set_spacing( &font, 5 );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *  @endcode
 *  @see Font_get_spacing.
 */
 #ifndef _SKIP_DOC
__cyos 1155
#endif
void Font_set_spacing(struct Font* ptr_font, int spacing);

/**
 *  Loads the font from an opened stream.
 *  @param ptr_font A pointer to the initialized Font object
 *  @param input A pointer to an opened stream
 *  @return TRUE if the font was loaded successfully
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Input * ptr_input;
 *     struct Font  font;
 *     ...
 *     // Your make file must have the archive resource "bmp.pic" linked by filer.exe.
 *     // For example, see the Cybiko Compiler's Introduction Section.
 *     ptr_input = open_resource_Ex( "font_1.pic" );
 *     ...
 *     if( !Font_load( &font, ptr_input ) )
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *     Input_dtor( ptr_input, FREE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 844
#endif
bool Font_load(struct Font* ptr_font, struct Input* input);

/**
 *  Returns the bitmap object of the 'chr' character. (Bitmap representation of the 'chr' character).
 *  @param ptr_font A pointer to the initialized Font object
 *  @param chr A character
 *  @return A pointer to a Bitmap object that represents the 'chr' character
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font   font;
 *     struct Bitmap * bmp;
 *     ...
 *     Font_ctor_Ex( &font, "MyFont.fnt", TRUE, 5 );
 *     ...
 *     bmp = Font_bmp_by_char( &font, 'c' );
 *     ...
 *     Font_dtor( &font, LEAVE_MEMORY );
 *     Bitmap_dtor( bmp, FREE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 387
#endif
struct Bitmap* Font_bmp_by_char(struct Font* ptr_font, int chr);

/**
 *  Splits string #str to substrings that fit specified #width. This function doesn't modify the
 *  source string, but calculates start and length of each substring instead.
 *  <p>The first call to this function returns an address of the source string and calculates
 *  first substring's length. Subsequent calls return remaining substrings' addresses and lengths, in order.
 *  <p>The string may contain explicit line breaks ('\\n').
 *  @param ptr_font A pointer to a font which metrics will be used to measure the string width
 *  @param str A string to be splitted; this parameter is modified by the function to handle subsequent calls
 *  @param width Width, in pixels, to fit the source text in
 *  @param len Returns length of the current substring, in characters
 *  @return A pointer to the current substring, or NULL if no text is left
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     char *str, *ptr;
 *     int  len;
 *     ...
 *     str = "Any   string";
 *     while( ptr = Font_split_string( &str, 6, &len ))
 *     {
 *       draw_text( ptr, 0, y*12, len );
 *       y++;
 *     }
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1169
#endif
char* Font_split_string(struct Font* ptr_font, char** str, int width, int* len);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Font_get_size  BitmapSequence_get_size
#else
/**
 *  Returns the number of bitmaps in a font object.
 *  @param ptr_font A pointer to the initialized  Font object
 *  @return The number of bitmaps in a font object
 */
int Font_get_size (struct Font* ptr_font);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Font_get_x BitmapSequence_get_x
#else
/**
 *  Returns the X origin of the Nth bitmap in a font object.
 *  @param ptr_font A pointer to the initialized  Font object
 *  @param bitmap_index The bitmap's index
 *  @return The X origin of the Nth bitmap in a font object
 *  @see Font_get_y
 */
int Font_get_x(struct Font* ptr_font, int bitmap_index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Font_get_y BitmapSequence_get_y
#else
/**
 *  Returns the Y origin of the Nth bitmap in a font object.
 *  @param ptr_font A pointer to the initialized  Font object
 *  @param bitmap_index The bitmap's index
 *  @return The Y origin of the Nth bitmap in a font object
 *  @see Font_get_x
 */
int Font_get_y(struct Font* ptr_font, int bitmap_index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Font_get_bitmap BitmapSequence_get_bitmap
#else
/**
 *  Returns a pointer to the bitmap with a specified index.
 *  @param ptr_font A pointer to the initialized  Font object
 *  @param bitmap_index The bitmap's index
 *  @return A pointer to the bitmap with the specified index
 */
struct Bitmap * Font_get_bitmap(struct Font* ptr_font, int bitmap_index);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Font_store BitmapSequence_store
#else
/**
 *  Stores a font object to an opened stream.
 *  @param ptr_font A pointer to the initialized  Font object
 *  @param output A pointer to the Output stream
 *  @return TRUE if the font sequence was stored successfully
 */
bool Font_store (struct Font* ptr_font, struct Output* output);
#endif

#ifdef _SKIP_DOC
};
#endif


//============================
//=                          =
//=     object TGraph.       =
//=  System graphics object. =
//=                          =
//============================

/** @brief the implementation of raster and 2D-graphics. It is designated for work with primitive graphics such as rectangles, lines and pixels.
 *  @ingroup drawing_primitives_group
 * The TGraph structure is implemented by raster and 2D-graphics.
 * It is designated for working with graphic primitives like rectangles,
 * lines and pixels. It functions similarly to many other SDK structures,
 * in that you must use a TGraph constructor to create the TGraph object,
 * and release it after use with a TGraph deconstructor.
 *  <br>
 *  <br>
 * The structure diagram below shows that TGraph is a parent structure for
 * other structures such as Graphics and DisplayGraphics. This means that raster
 * and 2D-graphics operations for Graphics and DisplayGraphics structures are
 * implemented in their parent structure.
 *  <br>
 *  <br>
 * To use all TGraph functions you must obtain the Display Graphics Context from
 * the init_module function. It has only one parameter, called the main application
 * module parameter, and it must be obtained by the module_t structure.
 *  <br>
 *  <br>
 * See examples for related functions.
 *  @see drawing_primitives_group
 */
struct TGraph
{
#ifndef _SKIP_DOC
  char TGraph_Members_STUB_1[12];
  char* _buf;
  char TGraph_Members_STUB_2[16];
};
#endif  //  #ifndef _SKIP_DOC

// for the "outer world", most of the following functions look just like
// those of CyOS, although they're actually implemented by the rtl; note
// that there may exist custom rtls with different import sets;
// application using such rtl must #define DISABLE_STANDARD_RTL right
// before #include <cybiko.h>
#ifndef DISABLE_STANDARD_RTL

/**
 *  Draws a rectangular frame.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param x The rectangle's left coordinate
 *  @param y The rectangle's top coordinate
 *  @param w The rectangle's width
 *  @param h The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to Cybiko graphics context
 *      TGraph_draw_rect( main_module.m_gfx, 10, 10, 100, 100 );
 *     ...
 *     // To send the current graphics page to Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_draw_rect_Ex.
 */
#ifndef _SKIP_DOC
import 9
#endif
void TGraph_draw_rect(struct TGraph* ptr_graph, int x, int y, int w, int h);

/**
 *  The extended version of the TGraph_draw_rect function. Draws a rectangular frame using the info contained in a rect_t object.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param rc A pointer to a tect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct rect_t rc;
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     rc.top = rc.left = 10;
 *     rc.width = rc.height = 100;
 *     ...
 *     // main_module.m_gfx is a pointer to Cybiko graphics context
 *      TGraph_draw_rect_Ex( main_module.m_gfx, &rc );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_draw_rect.
 */
#ifndef _SKIP_DOC
import 10
#endif
void TGraph_draw_rect_Ex(struct TGraph* ptr_graph, struct rect_t* rc);

#ifndef _SKIP_DOC
__cyos 112
void TGraph_dtor(struct TGraph* ptr_graph, int);
#endif

/**
 *  Fills a rectangle at the specified coordinates with the current color.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param x The rectangle's left coordinate
 *  @param y The rectangle's top coordinate
 *  @param w The rectangle's width
 *  @param h The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *      TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_fill_rect_Ex.
 */
#ifndef _SKIP_DOC
import 11
#endif
void TGraph_fill_rect(struct TGraph* ptr_graph, int x, int y, int w, int h);

/**
 *  The extended version of the TGraph_fill_rect function. Fills a rectangle at the specified coordinates with the current color.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param rc A pointer to a tect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct rect_t rc;
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     rc.top = rc.left = 10;
 *     rc.width = rc.height = 100;
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *      TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_fill_rect_Ex( main_module.m_gfx, &rc );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_fill_rect.
 */
#ifndef _SKIP_DOC
import 12
#endif
void TGraph_fill_rect_Ex(struct TGraph* ptr_graph, struct rect_t* rc);

/**
 *  Returns a TGraph object's current color.
 *  @param ptr_graph A pointer to a TGraph object
 *  @return The TGraph object's current color
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *        TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_set_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
import 13
#endif
color_t TGraph_get_color(struct TGraph* ptr_graph);

/**
 *  Returns the current draw mode of a TGraph object.
 *  @param ptr_graph A pointer to a TGraph object
 *  @return The current draw mode of a TGraph object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     ...
 *     // The second parameter in the TGraph_set_draw_mode function is the draw mode value.
 *     // This value must be one of the following: DM_XOR, DM_OR, or DM_PUT.
 *     if ( TGraph_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *        TGraph_set_draw_mode( main_module.m_gfx, DM_PUT );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_set_draw_mode, DM_XOR, DM_OR, DM_PUT.
 */
#ifndef _SKIP_DOC
import 14
#endif
drawmode_t TGraph_get_draw_mode(struct TGraph* ptr_graph);

/**
 *  Sets the current draw mode of a TGraph object.
 *  @param ptr_graph A pointer to a TGraph structure
 *  @param mode A drawmode_t object
 *  @return None
 *  @code
 *   #include "cybiko.h"
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     ...
 *     // The second parameter in the TGraph_set_draw_mode function is the draw mode value.
 *     // This value must be one of the following: DM_XOR, DM_OR, or DM_PUT.
 *     if ( TGraph_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *        TGraph_set_draw_mode( main_module.m_gfx, DM_PUT );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_get_draw_mode, DM_XOR, DM_OR, DM_PUT.
 */
#ifndef _SKIP_DOC
import 15
#endif
void TGraph_set_draw_mode(struct TGraph* ptr_graph, drawmode_t mode);

//  For future use.
#ifndef _SKIP_DOC
import 16
void TGraph_set_line_style(struct TGraph* ptr_graph, short mask);
#endif

#endif  //----------------------------- DISABLE_STANDARD_RTL ----------------------------

/**
 *  Draws a horizontal line from point ( x , y ) to point ( dx , y ).
 *  @param ptr_graph A pointer to the graphics context
 *  @param x x-coordinate of the horizontal line's starting point
 *  @param y y-coordinate of the horizontal line's starting point
 *  @param dx x-coordinate of the horizontal line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *        TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_draw_hline( main_module.m_gfx, 10, 10, 20);
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_draw_line, TGraph_draw_vline.
 */
#ifndef _SKIP_DOC
__cyos 528
#endif
void TGraph_draw_hline(struct TGraph* ptr_graph, int x, int y, int dx);


/**
 *  Draws vertical line from point ( x , y ) to point ( x , dy ).
 *  @param ptr_graph A pointer to the graphic context
 *  @param x x-coordinate of the vertical line's starting point
 *  @param y y-coordinate of the vertical line's starting point
 *  @param dy y-coordinate of the vertical line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one from following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *        TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_draw_vline( main_module.m_gfx, 10, 10, 20);
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_draw_line, TGraph_draw_hline.
 */
#ifndef _SKIP_DOC
__cyos 534
#endif
void TGraph_draw_vline(struct TGraph* ptr_graph, int x, int y, int dy);

/**
 *  Draws a solid line from point ( x1 , y1 ) to point ( x2 , y2 ).
 *  @param ptr_graph A pointer to the graphic context
 *  @param x1 x-coordinate of the line's starting point
 *  @param y1 y-coordinate of the line's starting point
 *  @param x2 x-coordinate of the line's finishing point
 *  @param y2 y-coordinate of the line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY or CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *        TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_draw_line( main_module.m_gfx, 10, 10, 20, 60 );
 *     ...
 *     // To send the current graphics page to the TGraph_draw_rect_Ex Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_draw_hline, TGraph_draw_vline.
 */
#ifndef _SKIP_DOC
__cyos 530
#endif
void TGraph_draw_line(struct TGraph* ptr_graph, int x1, int y1, int x2, int y2);

/**
 *  Fills the screen with the color 'fc'.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param fc The color value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_fill_screen function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     TGraph_fill_screen( main_module.m_gfx, CLR_BLACK );
 *     ...
 *     // To send the current graphisc page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_put_background, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 587
#endif
void TGraph_fill_screen(struct TGraph* ptr_graph, color_t fc);

/**
 *  Returns the clip region stored in the 'rc' object.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param rc A pointer to a rect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct rect_t clip_region;
 *       init_module( &main_module );
 *       ...
 *       // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *       TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *       if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *            ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *             TGraph_set_clip( main_module.m_gfx, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
 *       ...
 *  @endcode
 *  @see TGraph_set_clip.
 */
#ifndef _SKIP_DOC
__cyos 655
#endif
void TGraph_get_clip(struct TGraph* ptr_graph, struct rect_t* rc);

/**
 *  Returns the color of the pixel at coordinate ( x , y ).
 *  @param ptr_graph A pointer to a TGraph object
 *  @param x x-coordinate of the required pixel
 *  @param y y-coordinate of the required pixel
 *  @return The color of the pixel at coordinate ( x , y )
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( TGraph_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          TGraph_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE )
 *     ...
 *  @endcode
 *  @see TGraph_set_pixel, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 709
#endif
color_t TGraph_get_pixel(struct TGraph* ptr_graph, int x, int y);

/**
 *  Sets an image pointed by #ptr_background as a background image.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param ptr_background A pointer to a background image
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     char*  ptr_display_buffer;
 *     init_module( &main_module );
 *     ...
 *     // Allocating memory for buffer
 *     ptr_display_buffer = (char* ) malloc( TGraph_get_bytes_total( main_module.m_gfx ) );
 *     ...
 *     // Make some graphics operation ( draw a picture, for instance)
 *     ...
 *     // Save the screen
 *     memcpy( ptr_display_buffer, DisplayGraphics_get_page_ptr(main_module.m_gfx, 0), TGraph_get_bytes_total(main_module.m_gfx));
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen. Works very fast.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen again.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     free( ptr_display_buffer );
 *  @endcode
 *  @see TGraph_fill_screen.
 */
#ifndef _SKIP_DOC
__cyos 966
#endif
void TGraph_put_background(struct TGraph* ptr_graph, char* ptr_background);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define TGraph_get_buf_addr(ptr_graph)  ((ptr_graph)->_buf)
#else
/**
 *  Returns a pointer to an image of a TGraph object's destination.
 *  @param ptr_graph A pointer to a TGraph object
 *  @return A pointer to an image of a TGraph object's destination
 */
char* TGraph_get_buf_addr(struct TGraph* ptr_graph);
#endif

/**
 *  Returns the screen buffer size, in bytes.
 *  @param ptr_graph A pointer to a TGraph object
 *  @return The screen bufzfer size in bytes
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     char*  ptr_display_buffer;
 *     init_module( &main_module );
 *     ...
 *     // Allocating memory for buffer
 *     ptr_display_buffer = (char* ) malloc( TGraph_get_bytes_total( main_module.m_gfx ) );
 *     ...
 *     // Make some graphics operation ( draw a picture, for instance)
 *     ...
 *     // Save the screen
 *     memcpy( ptr_display_buffer, DisplayGraphics_get_page_ptr(main_module.m_gfx, 0), TGraph_get_bytes_total(main_module.m_gfx));
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen. Works very fast.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen again.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     free( ptr_display_buffer );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 645
#endif
int TGraph_get_bytes_total(struct TGraph* ptr_graph);

/**
 *  Scrolls the rectangle defined by ( x, y, width, height ) parameters. The scroll values are dx and dy.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param left x-coordinate of the rectangle to be scrolled
 *  @param top y-coordinate of the rectangle to be srolled
 *  @param width A width of the rectangle to be scrolled
 *  @param height A height of the rectangle to be scrolled
 *  @param dx A horizontal shift value
 *  @param dy A vertical shift value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t  rectangle;
 *     init_module(&main_module);
 *     rectangle.x = 10;
 *     rectangle.y = 10;
 *     rectangle.h = 20;
 *     rectangle.w = 40;
 *     ...
 *     TGraph_fill_rect_Ex( main_module.m_gfx, &rectangle );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     // Make some graphics operation
 *     ...
 *     TGraph_scroll( main_module.m_gfx, &rectangle, 10, 10 );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1057
#endif
void TGraph_scroll(struct TGraph* ptr_graph, int left, int top, int width, int height, int dx, int dy);

/**
 *  Sets the transparent color as the color defined in the 'color' parameter.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param color A color value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bmp;
 *     ...
 *     init_module( &main_module );
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     ...
 *     // Set draw mode to mode OR.
 *     TGraph_set_draw_mode( main_module.m_gfx, DM_OR );
 *     // Set transparent color to CLR_BLACK
 *     TGraph_set_bkcolor( main_module.m_gfx, CLR_BLACK );
 *     // Draw all pixels of the bitmap except pixels with CLR_BLACK color.
 *     Graphics_draw_bitmap( main_module.m_gfx, &bmp, 30, 40, BM_NORMAL );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see TGraph_set_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 1098
#endif
void TGraph_set_bkcolor(struct TGraph* ptr_graph, color_t color);

/**
 *  Sets the clipping region.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param x x-coordinate of the clipping region
 *  @param y y-coordinate of the clipping region
 *  @param width A width of the clipping region
 *  @param height A height of the clipping region
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *          ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *           TGraph_set_clip( main_module.m_gfx, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
 *     ...
 *  @endcode
 *  @see TGraph_set_clip_Ex.
 */
#ifndef _SKIP_DOC
__cyos 1104
#endif
void TGraph_set_clip(struct TGraph* ptr_graph, int x, int y, int width, int height);

/**
 *  Sets the clipping region. This is an extended version of the TGraph_set_clip function.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param rc A pointer to a rect_t object. By this structure we can define the region coordinates
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     struct rect_t new_clip_reg;
 *     init_module( &main_module );
 *     ...
 *     new_clip_reg.x = 10;
 *     new_clip_reg.y = 10;
 *     new_clip_reg.h = 100;
 *     new_clip_reg.w = 100;
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *          ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *           TGraph_set_clip( main_module.m_gfx, &new_clip_reg );
 *     ...
 *  @endcode
 *  @see TGraph_set_clip.
 */
#ifndef _SKIP_DOC
__cyos 1103
#endif
void TGraph_set_clip_Ex(struct TGraph* ptr_graph, struct rect_t* rc);

/**
 *  Sets the foreground color.
 *  @param ptr_graph A pointer to a TGraph object
 *  @param color A pointer to a color_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *        TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_draw_line( main_module.m_gfx, 10, 10, 20, 60 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see TGraph_set_bkcolor, TGraph_get_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, or CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 1108
#endif
void TGraph_set_color(struct TGraph* ptr_graph, color_t color);

/**
 *  Sets the color for the pixel at the coordinates ( x, y ).
 *  @param ptr_graph A pointer to a TGraph object
 *  @param x x-coordinate of the required pixel
 *  @param y y-coordinate of the required pixel
 *  @param color A pointer to a color_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( TGraph_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          TGraph_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE )
 *     ...
 *  @endcode
 *  @see TGraph_get_pixel, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
#ifndef _SKIP_DOC
__cyos 1142
#endif
void TGraph_set_pixel(struct TGraph* ptr_graph, int x, int y, color_t color);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=     object Graphics.     =
//= Base object for drawing. =
//=                          =
//============================

/** @brief This structure allows draw operations to be applied to Bitmap objects.
 *  @ingroup drawing_primitives_group
 *  The Graphics structure uses bitmaps to maintain many operations. The DisplayGraphics structure allows you to draw operations directly on the screen.
 *  You can draw graphics primitives,as well as text with the current font.
 *  <br>
 *  <br>
 *  Because TGraph is its parent structure, the Graphics structure's functions are defined in TGraph structure.
 *  For example, manipulations of graphics primitives are implemented in TGraph structure.
 *  <br>
 *  <br>
 *  To use some Graphics functions, you must obtain the Display Graphics Context from the init_module function.
 *  This context has only one parameter (the main application module parameter) and it must be obtained from the module_t structure.
 *  <br>
 *  <br>
 *  See examples for related functions.
 *  @see drawing_primitives_group
 */
struct Graphics : public TGraph
{
#ifndef _SKIP_DOC
  char Graphics_Members_STUB[12];
};
#endif  //  #ifndef _SKIP_DOC

/**
 *  Simple Graphics constructor. Creates an empty Graphics object.
 *  @param ptr_gfx A pointer to the Graphics structure
 *  @return A pointer to the initialized Graphics object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Graphics gfx;
 *     Graphics_ctor( &gfx );
 *     ...
 *     Graphics_dtor( &gfx, LEAVE_MEMORY );
 *  @endcode
 *  @see Graphics_ctor_Ex.
 */
#ifndef _SKIP_DOC
__cyos 243
#endif
struct Graphics* Graphics_ctor(struct Graphics* ptr_gfx);

/**
 *  The extended version of the Graphics_ctor function. Creates a Graphics object that is associated with a Bitmap object.
 *  @param ptr_gfx A pointer to a Graphics object
 *  @param bitmap A pointer to a Bitmap object
 *  @return A pointer to the initialized Graphics object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap  bitmap;
 *     struct Graphics gfx;
 *     ...
 *     Bitmap_ctor_Ex1( &bitmap, "screen.pic" );
 *     Graphics_ctor_Ex( &gfx, &bitmap );
 *     ...
 *     Graphics_dtor( &gfx, LEAVE_MEMORY );
 *     Bitmap_dtor( &bitmap, LEAVE_MEMORY );
 *  @endcode
 *  @see Graphics_ctor.
 */
#ifndef _SKIP_DOC
__cyos 244
#endif
struct Graphics* Graphics_ctor_Ex(struct Graphics* ptr_gfx, struct Bitmap* bitmap);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_dtor TGraph_dtor
#else
/**
 *  Deletes a Graphics object.
 *  @param ptr_gfx A pointer to the Graphics structure
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY.
 *  If the memory was allocated for the object by malloc(),
 *  use FREE_MEMORY to free it. Use LEAVE_MEMORY If the
 *  object was static or allocated in a stack
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Graphics gfx;
 *     Graphics_ctor( &gfx );
 *     ...
 *     Graphics_dtor( &gfx, LEAVE_MEMORY );
 *  @endcode
 *  @see FREE_MEMORY, LEAVE_MEMORY.
 */
void Graphics_dtor (struct Graphics* ptr_gfx, int memory_flag);
#endif
#ifndef _SKIP_DOC
__cyos 435
char* Graphics_class_name(struct Graphics* ptr_gfx);
#endif

/**
 *  Allows the bitmap 'bmp' to be drawn in a specified position (left, top) in 'fm' mode.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param bmp A pointer to the initialized Bitmap object
 *  @param left x-coordinate of the point where bitmap will be placed
 *  @param top y-coordinate of the point where bitmap will be placed
 *  @param fm Drawing mode
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bitmap;
 *     ...
 *     init_module( &main_module );
 *     Bitmap_ctor_Ex1( &bitmap, "screen.pic" );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_draw_bitmap( main_module.m_gfx, &bitmap, 30, 40, BM_NORMAL );
 *     ...
 *  @endcode
 *  @see BM_NORMAL, BM_INVERSE, BM_FLIP_X, BM_FLIP_Y.
 */
#ifndef _SKIP_DOC
__cyos 526
#endif
void Graphics_draw_bitmap(struct Graphics* ptr_gfx, struct Bitmap* bmp, int left, int top, short fm);

/**
 *  Allows the character 'fc' to be drawn in the current font in specified position (x, y).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param x x-coordinate of the point where the char will be places
 *  @param y y-coordinate of the point where the char will be places
 *  @param fc A char to be drawn
 *  @return The position of the next pixel after the 'fc' character
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int pos;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     pos = Graphics_draw_char( main_module.m_gfx, 10, 10, 'c' );
 *     ...
 *  @endcode
 *  @see Graphics_draw_text.
 */
#ifndef _SKIP_DOC
__cyos 527
#endif
int Graphics_draw_char(struct Graphics* ptr_gfx, int x, int y, char fc);

/**
 *  Allows 'text' to be drawn in the current font, in the specified position (left , top).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param text A pointer to the string to be drawn
 *  @param left x-coordinate of the string to be drawn
 *  @param top y-coordinate of the string to be drawn
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     Graphics_draw_text( main_module.m_gfx, "Any text", 10, 10 );
 *     ...
 *  @endcode
 *  @see Graphics_draw_text_Ex, Graphics_draw_char.
 */
#ifndef _SKIP_DOC
__cyos 532
#endif
void Graphics_draw_text(struct Graphics* ptr_gfx, char* text, int left, int top);

/**
 *  Duplicates the Graphics_draw_text function. If the length of the string exceeds the number of 'flen' pixels, the string will be truncated.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param str A pointer to the string to be drawn
 *  @param left x-coordinate of the place where the string to be drawn
 *  @param top y-coordinate of the place where the string to be drawn
 *  @param len A maximum number of chars to draw
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     Graphics_draw_text_Ex( main_module.m_gfx, "Any Text", 10, 10, 20 );
 *     ...
 *  @endcode
 *  @see Graphics_draw_text.
 */
#ifndef _SKIP_DOC
__cyos 533
#endif
void Graphics_draw_text_Ex(struct Graphics* ptr_gfx, char* str, int left, int top, int len);

/**
 *  Returns the current Bitmap context for drawing
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return A pointer to the current Graphics page's Bitmap object
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap * bitmap;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     bitmap = Graphics_get_bitmap( main_module.m_gfx );
 *     ...
 *  @endcode
 *  @see Graphics_set_bitmap.
 */
#ifndef _SKIP_DOC
__cyos 635
#endif
struct Bitmap* Graphics_get_bitmap(struct Graphics* ptr_gfx);

/**
 *  Returns the current font's height.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return The current font's height
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int font_height;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     font_height = Graphics_get_char_height( main_module.m_gfx );
 *     ...
 *  @endcode
 *  @see Graphics_get_char_width, Graphics_get_font.
 */
#ifndef _SKIP_DOC
__cyos 650
#endif
int Graphics_get_char_height(struct Graphics* ptr_gfx);

/**
 *  Returns the current font's character width.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param chr A character
 *  @return The current font's character width
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int char_width;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     char_width = Graphics_get_char_width( main_module.m_gfx, 'g' );
 *     ...
 *  @endcode
 *  @see Graphics_get_char_height, Graphics_get_font.
 */
#ifndef _SKIP_DOC
__cyos 652
#endif
int Graphics_get_char_width(struct Graphics* ptr_gfx, char chr);

/**
 *  Returns the current font.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return A pointer to the current font
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Font * ptr_font;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *     ptr_font = Graphics_get_font( main_module.m_gfx );
 *     ...
 *  @endcode
 *  @see Graphics_set_font.
 */
#ifndef _SKIP_DOC
__cyos 673
#endif
struct Font* Graphics_get_font(struct Graphics* ptr_gfx);

/**
 *  Returns the width (in pixels) of a string 'str' for the current font.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param str A pointer to the string
 *  @return The width (in pixels) of a string for the current font
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int str_plen;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *     str_plen = Graphics_string_width( main_module.m_gfx, "Any string" );
 *     ...
 *  @endcode
 *  @see Graphics_string_width_Ex.
 */
#ifndef _SKIP_DOC
__cyos 738
#endif
int Graphics_string_width(struct Graphics* ptr_gfx, char* str);

/**
 *  The extended version of the Graphics_string_width function. Returns the width (in pixels) of the first 'len' symbols of a string 'str'.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param str A pointer to the string
 *  @param len A maximum possible length of the drawn string
 *  @return The width (in pixels) of the first 'len' symbols of a string
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int str_plen;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *     str_plen = Graphics_string_width_Ex( main_module.m_gfx, "Any string", 20 );
 *     ...
 *  @endcode
 *  @see Graphics_string_width.
 */
#ifndef _SKIP_DOC
__cyos 739
#endif
int Graphics_string_width_Ex(struct Graphics* ptr_gfx, char* str, int len);

/**
 *  Sets a bitmap 'bmp' as the destination for drawings.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param bmp A pointer to the destination bitmap
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct Bitmap bitmap;
 *     struct Graphics vgfx;
 *     ...
 *     Graphics_ctor( &vgfx );
 *     // SCREEN_WIDTH and SCREEN_HEIGHT are described in the 'Defines' section.
 *     Bitmap_ctor_Ex2( &bitmap, SCREEN_WIDTH, SCREEN_HEIGHT, 2 );
 *     ...
 *     // Now all drawing operations in the 'vgfx' graphics context will be performed on the 'bitmap' object.
 *     Graphics_set_bitmap( &vgfx, &bitmap );
 *     ...
 *  @endcode
 *  @see Graphics_get_bitmap.
 */
#ifndef _SKIP_DOC
__cyos 1097
#endif
void Graphics_set_bitmap(struct Graphics* ptr_gfx, struct Bitmap* bmp);

/**
 *  Sets the current font.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param font A pointer to the initialized  Font object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int str_plen;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *  @endcode
 *  @see Graphics_get_font.
 */
#ifndef _SKIP_DOC
__cyos 1126
#endif
void Graphics_set_font(struct Graphics* ptr_gfx, struct Font* font);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_scroll            TGraph_scroll
#else
/**
 *  Scrolls a rectangle area that is defined by delta x, delta y in 'ptr_rectangle'.
 *  @param ptr_gfx A pointer to the Graphics structure
 *  @param ptr_rectangle A pointer to the rectangle to be scrolled
 *  @param dx A horizontal scroll shift value
 *  @param dy A vertical scroll shift value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t rectangle;
 *     init_module(&main_module);
 *     rectangle.x = 10;
 *     rectangle.y = 10;
 *     rectangle.h = 20;
 *     rectangle.w = 40;
 *     ...
 *     TGraph_fill_rect_Ex( main_module.m_gfx, &rectangle );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     // Make some graphic operation
 *     ...
 *     Graphics_scroll( main_module.m_gfx, &rectangle, 10, 10 );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *  @endcode
 */
void Graphics_scroll(struct Graphics* ptr_gfx, struct rect_t* ptr_rectangle, int dx, int dy) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_draw_hline        TGraph_draw_hline
#else
/**
 *  Draws a horizontal line from point (x , y) to point (xx , y).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param x x-coordinate of the horizontal line's starting point
 *  @param y y-coordinate of the horizontal line's starting point
 *  @param xx x-coordinate of the horizontal line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_draw_hline( main_module.m_gfx, 10, 10, 20, 60 );
 *     ...
 *  @endcode
 *  @see Graphics_draw_line, Graphics_draw_vline.
 */
void Graphics_draw_hline(struct Graphics* ptr_gfx, int x, int y, int xx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_draw_vline        TGraph_draw_vline
#else
/**
 *  Draws a vertical line from point (x , y) to point (x , yy).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param x x-coordinate of the vertical line's starting point
 *  @param y y-coordinate of the vertical line's starting point
 *  @param yy y-coordinate of the vertical line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_draw_vline( main_module.m_gfx, 10, 10, 20);
 *     ...
 *  @endcode
 *  @see Graphics_draw_line, Graphics_draw_hline.
 */
void Graphics_draw_vline (struct Graphics* ptr_gfx, int x, int y, int yy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_draw_line         TGraph_draw_line
#else
/**
 *  Draws a solid line from point ( x , y ) to point ( xx , yy ).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param x x-coordinate of the line's starting point
 *  @param y y-coordinate of the line's starting point
 *  @param xx x-coordinate of the line's finishing point
 *  @param yy y-coordinate of the line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_draw_line( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see Graphics_draw_vline, Graphics_draw_hline.
 */
void Graphics_draw_line (struct Graphics* ptr_gfx, int x, int y, int xx, int yy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_draw_rect         TGraph_draw_rect
#else
/**
 *  Draws a rectangular frame in the current color.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fx The rectangle's left coordinate
 *  @param fy The rectangle's top coordinate
 *  @param fw The rectangle's width
 *  @param fh The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_draw_rect( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see Graphics_draw_rect_Ex.
 */
void Graphics_draw_rect (struct Graphics* ptr_gfx, int fx, int fy, int fw, int fh);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_draw_rect_Ex      TGraph_draw_rect_Ex
#else
/**
 *  Duplicates the Graphics_draw_rect function.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param ptr_rectangle A pointer to a tect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t rectangle={0,0,100,100};
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_draw_rect_Ex( main_module.m_gfx, &rect_t);
 *     ...
 *  @endcode
 *  @see Graphics_draw_rect.
 */
void Graphics_draw_rect_Ex (struct Graphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_fill_rect         TGraph_fill_rect
#else
/**
 *  Fills a rectangle with the current color. The rectangle is defined by fx, fy, fw, and fh parameters.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fx The rectangle's left coordinate
 *  @param fy The rectangle's top coordinate
 *  @param fw The rectangle's width
 *  @param fh The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_fill_rect( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see Graphics_fill_rect_Ex.
 */
void Graphics_fill_rect (struct Graphics* ptr_gfx, int fx ,int fy, int fw, int fh);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_fill_rect_Ex      TGraph_fill_rect_Ex
#else
/**
 *  Duplicates the Graphics_fill_rect function.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param ptr_rectangle A pointer to a tect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t rectangle={0,0,100,100};
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     Graphics_fill_rect_Ex( main_module.m_gfx, &rect_t);
 *     ...
 * @endcode
 * @see Graphics_fill_rect.
 */
void Graphics_fill_rect_Ex (struct Graphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_fill_screen       TGraph_fill_screen
#else
/**
 *  Fills the screen with the 'fc' color.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fc A value of the color
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     Graphics_fill_screen( main_module.m_gfx, CLR_LTGRAY);
 *     ...
 *  @endcode
 *  @see CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void Graphics_fill_screen (struct Graphics* ptr_gfx, color_t fc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_bytes_total TGraph_get_bytes_total
#else
/**
 *  Returns the total number of bytes in the screen buffer.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return The total number of bytes in the screen buffer
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     char* ptr_display_buffer;
 *     ...
 *     init_module( &main_module );
 *     ptr_display_buffer = (char* ) malloc( Graphics_get_bytes_total( main_module.m_gfx ) );
 *     ...
 *  @endcode
 */
int Graphics_get_bytes_total (struct Graphics* ptr_gfx) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_clip          TGraph_set_clip
#else
/**
 *  Sets a clip region, defined by the coordinates stored in the fx, fy, fw, and fh parameters.
 *  @param ptr_gfx A pointer to a Graphics object
 *  @param fx x-coordinate of the upper-left corner of the area to be clipped
 *  @param fy y-coordinate of the upper-left corner of the area to be clipped
 *  @param fw Width of the area to be clipped
 *  @param fh Height of the area to be clipped
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *          ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *           Graphics_set_clip( main_module.m_gfx, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
 *     ...
 *  @endcode
 *  @see Graphics_set_clip_Ex, Graphics_get_clip.
 */
void Graphics_set_clip (struct Graphics* ptr_gfx, int fx, int fy, int fw, int fh) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_clip          TGraph_get_clip
#else
/**
 *  Returns a clip region. It is defined by coordinates that are stored in a rect_t object.
 *  @param ptr_gfx A pointer to a Graphics object
 *  @param ptr_rectangle A pointer to a rect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *          ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *           Graphics_set_clip( main_module.m_gfx, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
 *     ...
 *  @endcode
 *  @see Graphics_set_clip.
 */
void Graphics_get_clip (struct Graphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_clip_Ex       TGraph_set_clip_Ex
#else
/**
 *  Duplicates the Graphics_set_clip function. The region coordinates are stored in a rect_t object.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param ptr_rectangle A pointer to a rect_t object. By this structure we can define the region coordinates
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     struct rect_t new_clip_reg;
 *     init_module( &main_module );
 *     ...
 *     new_clip_reg.x = 10;
 *     new_clip_reg.y = 10;
 *     new_clip_reg.h = 100;
 *     new_clip_reg.w = 100;
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *          ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *           TGraph_set_clip( main_module.m_gfx, &new_clip_reg );
 *     ...
 *  @endcode
 *  @see Graphics_set_clip.
 */
void Graphics_set_clip_Ex (struct Graphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_color         TGraph_get_color
#else
/**
 *  Returns the current foreground color.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return The current foreground color
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( Graphics_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *     Graphics_set_color( main_module.m_gfx, CLR_LTGRAY);
 *     ...
 *  @endcode
 *  @see Graphics_set_color.
 */
color_t Graphics_get_color (struct Graphics* ptr_gfx) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_color         TGraph_set_color
#else
/**
 *  Sets the current foreground color.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fc A color value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( Graphics_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *     Graphics_set_color( main_module.m_gfx, CLR_LTGRAY);
 *     ...
 *  @endcode
 *  @see Graphics_get_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void Graphics_set_color (struct Graphics* ptr_gfx, color_t fc) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_draw_mode     TGraph_get_draw_mode
#else
/**
 *  Returns the current draw mode.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return The current draw mode
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( Graphics_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *     Graphics_set_draw_mode( main_module.m_gfx, DM_PUT);
 *     ...
 *  @endcode
 *  @see Graphics_set_draw_mode, DM_PUT, DM_OR, DM_XOR.
 */
drawmode_t Graphics_get_draw_mode (struct Graphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_draw_mode     TGraph_set_draw_mode
#else
/**
 *  Sets the current draw mode.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param dm A draw mode
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( Graphics_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *     Graphics_set_draw_mode( main_module.m_gfx, DM_PUT);
 *     ...
 *  @endcode
 *  @see Graphics_get_draw_mode, DM_PUT, DM_OR, DM_XOR.
 */
void Graphics_set_draw_mode (struct Graphics* ptr_gfx, drawmode_t dm);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_put_background    TGraph_put_background
#else
/**
 *  Treats #fp as a pointer to a background image and sends the image to its destination.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fp A pointer to the background image
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Graphics gfx_fp;
 *     ...
 *     Graphics_put_background( main_module.m_gfx,
 *                              Graphics_get_buf_addr( &gfx_fp )) ;
 *     ...
 *  @endcode
 */
void Graphics_put_background (struct Graphics* ptr_gfx, char* fp) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_pixel         TGraph_get_pixel
#else
/**
 *  Returns the color of the pixel with coordinate ( fx , fy ).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fx An x-coordinate of the required pixel
 *  @param fy An y-coordinate of the required pixel
 *  @return The color of the pixel with coordinate ( fx , fy )
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     if ( Graphics_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          Graphics_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE );
 *     ...
 *  @endcode
 *  @see Graphics_set_pixel.
 */
color_t Graphics_get_pixel (struct Graphics* ptr_gfx, int fx, int fy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_pixel         TGraph_set_pixel
#else
/**
 *  The color, defined by the 'fc' parameter. Sets the color of the pixel at coordinates ( fx , fy ).
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @param fx x-coordinate of the required pixel
 *  @param fy y-coordinate of the required pixel
 *  @param fc A color value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     if ( Graphics_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          Graphics_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE );
 *     ...
 *  @endcode
 *  @see Graphics_get_pixel, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void Graphics_set_pixel (struct Graphics* ptr_gfx, int fx, int fy, color_t fc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_get_buf_addr(ptr_gfx)  ((ptr_gfx)->_buf)
#else
/**
 *  Returns a pointer to the Graphics object's associated image.
 *  @param ptr_gfx A pointer to the initialized Graphics object
 *  @return A pointer to the Graphics object's associated image.
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Graphics gfx_fp;
 *     ...
 *     Graphics_put_background( main_module.m_gfx,
 *       Graphics_get_buf_addr( &gfx_fp )) ;
 *     ...
 *  @endcode
 */
char* Graphics_get_buf_addr (struct Graphics* ptr_gfx);
#endif


#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Graphics_set_bkcolor       TGraph_set_bkcolor
#else
/**
 *  Sets the current background color by calling TGraph_set_bkcolor.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param fc A color_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bmp;
 *     ...
 *     init_module( &main_module );
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     ...
 *     // Set draw mode to mode OR.
 *     Graphics_set_draw_mode( main_module.m_gfx, DM_OR );
 *     // Set transparent color to CLR_BLACK
 *     Graphics_set_bkcolor( main_module.m_gfx, CLR_BLACK );
 *     // Draw all pixels of the bitmap except pixels with CLR_BLACK color.
 *     Graphics_draw_bitmap( main_module.m_gfx, &bmp, 30, 40, BM_NORMAL );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see TGraph_set_bkcolor, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void Graphics_set_bkcolor (struct Graphics* ptr_gfx, color_t fc);
#endif

#ifdef _SKIP_DOC
};
#endif


#ifndef DISABLE_STANDARD_RTL

// for the "outer world", most of the following functions look just like
// those of CyOS, although they're actually implemented by the rtl; note
// that there may exist custom rtls with different import sets;
// application using such rtl must #define DISABLE_STANDARD_RTL right
// before #include <cybiko.h>

/**
 *  @ingroup bitmaps_and_bitmap_sequence_group
 *  Draws a bitmap in the specified position.
 *  @param lib_index is the index of a *.pic file in the "filer.exe" file list (the first element has a 0 index)
 *  @param frame is the index of a bitmap frame in a *.pic file (the first frame has a 0 index)
 *  @param x is the x-coordinate of a bitmap
 *  @param y is the y-coordinate of a bitmap
 *  @param mode must be one of the following: BM_NORMAL, BM_INVERSE, BM_FLIP_X, BM_FLIP_Y
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     // This code is from makefile for test.app
 *     // test.app : tmp/bytecode.bin
 *     //            filer.exe a test.app res/font.pic tmp/bytecode.bin $(SDK)/lib/main.e
 *     draw_lib( 0, 0, 10, 10, BM_NORMAL );
 *     // The first bitmap from the "font.pic" bitmaps sequence would be draw.
 *     ...
 *  @endcode
 *  @see BM_NORMAL, BM_INVERSE, BM_FLIP_X, BM_FLIP_Y.
 */
#ifndef _SKIP_DOC
import 18
#endif
void draw_lib(int lib_index,int frame,int x,int y,short mode);

#endif // DISABLE_STANDARD_RTL

//============================
//=                          =
//= object DisplayGraphics.  =
//= Drawing on LCD display.  =
//=                          =
//============================

/**
 *  @ingroup drawing_primitives_group
 *  @brief This structure is designited to execute any draw operation on the cybiko display.
 *  This structure is designited to execute any draw operation on the Cybiko computer's display.
 *  If you want to make draw operations on a bitmap object, it's best to use the Graphics structure.
 *  <br>
 *  <br>
 *  The Cybiko computer features two displays; one of them virtual. We refer to these displays as
 *  'graphics pages.' The draw operations are performed on the virtual graphics page, and then
 *  shown as the Cybiko screen's graphics page.
 *  @see drawing_primitives_group
 */
struct DisplayGraphics : public Graphics
{
#ifndef _SKIP_DOC
  char DisplayGraphics_Members_STUB[12];
};
#endif

#ifndef _SKIP_DOC
__cyos 425
char* DisplayGraphics_class_name(struct DisplayGraphics*);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_bitmap        Graphics_draw_bitmap
#else
/**
 *  Draws a bitmap 'bmp' in a specified position  (left, top) in 'fm' mode.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param bmp A pointer to the initialized Bitmap object
 *  @param left x-coordinate of the left side of the bitmap to be drawn
 *  @param top y-coordinate of the top side of the bitmap to be drawn
 *  @param fm A drawing mode
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bitmap;
 *     ...
 *     init_module( &main_module );
 *     Bitmap_ctor_Ex1( &bitmap, "screen.pic" );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     DisplayGraphics_draw_bitmap( main_module.m_gfx, &bitmap, 30, 40, BM_NORMAL );
 *     ...
 *  @endcode
 */
void DisplayGraphics_draw_bitmap (struct DisplayGraphics* ptr_gfx, struct Bitmap* bmp, int left, int top, int fm);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_char          Graphics_draw_char
#else
/**
 *  Draws a character 'fc' with the current font in a specified position (fx, fy).
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param fx x-coordinate of the character to be drawn
 *  @param fy y-coordinate of the character to be drawn
 *  @param fc A character to be drawn
 *  @return The position of the next pixel after the 'fc' character
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font);
 *     pos = Graphics_draw_char( main_module.m_gfx, 10, 10, 'c' );
 *     ...
 *  @endcode
 */
int DisplayGraphics_draw_char (struct DisplayGraphics* ptr_gfx, int fx, int fy, char_t fc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_hline         TGraph_draw_hline
#else
/**
 *  Draws a horizontal line from point (x , y) to point (xx , y).
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param x x-coordinate horizontal line's starting point
 *  @param y y-coordinate horizontal line's starting point
 *  @param xx x-coordinate horizontal line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_hline( main_module.m_gfx, 10, 10, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_line, DisplayGraphics_draw_vline.
 */
void DisplayGraphics_draw_hline (struct DisplayGraphics* ptr_gfx, int x, int y, int xx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_vline         TGraph_draw_vline
#else
/**
 *  Draws a vertical line from point (x, y) to point (x , yy).
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param x x-coordinate of the vertical line's starting point
 *  @param y y-coordinate of the vertical line's starting point
 *  @param yy y-coordinate of the vertical line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_vline( main_module.m_gfx, 10, 10, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_line, DisplayGraphics_draw_hline.
 */
void DisplayGraphics_draw_vline (struct DisplayGraphics* ptr_gfx, int x, int y, int yy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_line          TGraph_draw_line
#else
/**
 *  Draws a solid line from point (x,y) to point (xx,yy).
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param x x-coordinate of the line's starting point
 *  @param y y-coordinate of the line's starting point
 *  @param xx x-coordinate of the line's finishing point
 *  @param yy y-coordinate of the line's finishing point
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_line( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_vline, DisplayGraphics_draw_hline.
 */
void DisplayGraphics_draw_line (struct DisplayGraphics* ptr_gfx, int x, int y, int xx, int yy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_rect          TGraph_draw_rect
#else
/**
 *  Draws a rectangular frame in the current color.
 *  It is defined by the coordinates stored in the fx, fy, fw and fh parameters.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param fx The rectangle's left coordinate
 *  @param fy The rectangle's top coordinate
 *  @param fw The rectangle's width
 *  @param fh The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_rect( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_rect_Ex.
 */
void DisplayGraphics_draw_rect (struct DisplayGraphics* ptr_gfx, int fx, int fy, int fw, int fh);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_rect_Ex       TGraph_draw_rect_Ex
#else
/**
 *  Duplicates the DisplayGraphics_draw_rect function
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param ptr_rectangle A pointer to a rect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t rectangle={0,0,100,100};
 *     ...
 *     init_module( &main_module );
 *     Graph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_rect_Ex( main_module.m_gfx, &rect_t);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_rect.
 */
void DisplayGraphics_draw_rect_Ex (struct DisplayGraphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_text          Graphics_draw_text
#else
/**
 *  Draws text in a specified position (left, top) with the current font.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param text A pointer to the string to be drawn
 *  @param left x-coordinate of the text to be drawn
 *  @param top y-coordinate of the text to be drawn
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_text( main_module.m_gfx, "Any text", 10, 10 );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_text_Ex.
 */
void DisplayGraphics_draw_text (struct DisplayGraphics* ptr_gfx, char* text, int left, int top);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_draw_text_Ex       Graphics_draw_text_Ex
#else
/**
 *  Duplicates the DisplayGraphics_draw_text function. If the length of the string
 *  exceeds the number of 'flen' pixels, the string will be truncated.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param text A pointer to the string to be drawn
 *  @param left x-coordinate of the text to be drawn
 *  @param top y-coordinate of the text to be drawn
 *  @param flen Maximum length (in pixels) of the string to be drawn
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_draw_text_Ex( main_module.m_gfx, "Any text", 10, 10, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_draw_text.
 */
void DisplayGraphics_draw_text_Ex (struct DisplayGraphics* ptr_gfx, char* text, int left, int top, int flen) ;
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_fill_rect          TGraph_fill_rect
#else
/**
 *  Fills a rectangle with the current color. The rectangle's area is defined by the fx, fy, fw and fh parameters.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param fx The rectangle's left coordinate
 *  @param fy The rectangle's top coordinate
 *  @param fw The rectangle's width
 *  @param fh The rectangle's height
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int pos;
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_fill_rect( main_module.m_gfx, 10, 10, 20, 20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_fill_rect_Ex.
 */
void DisplayGraphics_fill_rect (struct DisplayGraphics* ptr_gfx, int fx ,int fy, int fw, int fh);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_fill_rect_Ex       TGraph_fill_rect_Ex
#else
/**
 *  Duplicates the Graphics_fill_rect function.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param ptr_rectangle A pointer to a tect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t rectangle={0,0,100,100};
 *     ...
 *     init_module( &main_module );
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     DisplayGraphics_fill_rect_Ex( main_module.m_gfx, &rect_t);
 *     ...
 *  @endcode
 *  @see Graphics_fill_rect.
 */
void DisplayGraphics_fill_rect_Ex (struct DisplayGraphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_fill_screen        TGraph_fill_screen
#else
/**
 *  Fills the screen with the 'fc' color.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param fc A color's value
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     DisplayGraphics_fill_screen( main_module.m_gfx, CLR_LTGRAY);
 *     ...
 *  @endcode
 *  @see CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void DisplayGraphics_fill_screen (struct DisplayGraphics* ptr_gfx, color_t fc);
#endif


/**
 *  Switches background and foreground pages and shows the background screen on the LCD.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int i;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     DisplayGraphics_set_page( main_module.m_gfx, 0 );
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_WHITE );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     TGraph_set_color( main_module.m_gfx, CLR_BLACK );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     for(i = 0; i < 20 ; i++)
 *     {
 *       DisplayGraphics_flip( main_module.m_gfx );
 *       cWinApp_pause( main_module.m_gfx , 250 );   // 250 milliseconds delay.
 *     }
 *     //  Black and white rectangles will change places with each other.
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 599
#endif
void DisplayGraphics_flip(struct DisplayGraphics* ptr_gfx);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_char_height    Graphics_get_char_height
#else
/**
 *  Returns the current font's character's height.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return Current font's character height
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     DisplayGraphics_set_font( main_module.m_gfx, cool_normal_font );
 *     font_height = DisplayGraphics_get_char_height( main_module.m_gfx );
 *     ...
 *  @endcode
 */
int DisplayGraphics_get_char_height (struct DisplayGraphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_char_width     Graphics_get_char_width
#else
/**
 *  Returns the current font's character width.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return Current font's character width
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     DisplayGraphics_set_font( main_module.m_gfx, cool_normal_font );
 *     font_height = DisplayGraphics_get_char_width( main_module.m_gfx );
 *     ...
 *  @endcode
 */
int DisplayGraphics_get_char_width (struct DisplayGraphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_clip           TGraph_get_clip
#else
/**
 *  Returns a clip region.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param ptr_rectangle A pointer to a rect_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct rect_t clip_region;
 *     ...
 *     DisplayGraphics_get_clip( main_module.m_gfx, &clip_region)
 *     ...
 *  @endcode
 */
void DisplayGraphics_get_clip (struct DisplayGraphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_color          TGraph_get_color
#else
/**
 *  Returns the current foreground color.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return Current foreground color
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( DisplayGraphics_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *       DisplayGraphics_set_color( main_module.m_gfx, CLR_LTGRAY);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_set_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
color_t DisplayGraphics_get_color (struct DisplayGraphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_draw_mode      TGraph_get_draw_mode
#else
/**
 *  Returns the current draw mode.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return Current draw mode
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     if ( DisplayGraphics_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *       DisplayGraphics_set_draw_mode( main_module.m_gfx, DM_PUT);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_set_draw_mode, DM_PUT, DM_OR, DM_XOR.
 */
drawmode_t DisplayGraphics_get_draw_mode (struct DisplayGraphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_font           Graphics_get_font
#else
/**
 *  Returns the current font.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return A pointer to the current font
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Font *curFont;
 *     ...
 *     init_module( &main_module );
 *     curFont=DisplayGraphics_get_font( main_module.m_gfx);
 *     ...
 *  @endcode
 */
struct Font* DisplayGraphics_get_font (struct DisplayGraphics* ptr_gfx);
#endif

/**
 *  Returns the current graphics page number (0 or 1).
 *  @param ptr_gfx A pointer to a Graphics object
 *  @return Current graphics page number (0 or 1)
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     if ( !DisplayGraphics_get_page( main_module.m_gfx ) )
 *       DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_flip, DisplayGraphics_set_page.
 */
#ifndef _SKIP_DOC
__cyos 702
#endif
int DisplayGraphics_get_page(struct DisplayGraphics* ptr_gfx);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_bytes_total    TGraph_get_bytes_total
#else
/**
 *  Returns the number of bytes amount in the screen buffer.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @return The number of bytes in the screen buffer
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     char* ptr_display_buffer;
 *     ...
 *     init_module( &main_module );
 *     ptr_display_buffer = (char* ) malloc( DisplayGraphics_get_bytes_total( main_module.m_gfx ) );
 *     ...
 *  @endcode
 */
int DisplayGraphics_get_bytes_total (struct DisplayGraphics* ptr_gfx);
#endif


/**
 *  Returns the pointer to the screen buffer for the 'fp' graphics page. Use with caution!!!
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param page A number of the graphics page
 *  @return A pointer to the graphics page's screen buffer
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     char*  ptr_display_buffer;
 *     init_module( &main_module );
 *     ...
 *     // Allocating memory for buffer
 *     ptr_display_buffer = (char* ) malloc( TGraph_get_bytes_total( main_module.m_gfx ) );
 *     ...
 *     // Make some graphics operations ( draw a picture, for instance)
 *     ...
 *     // Save the screen
 *     memcpy( ptr_display_buffer, DisplayGraphics_get_page_ptr(main_module.m_gfx, 0), TGraph_get_bytes_total(main_module.m_gfx));
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen. Works very fast.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     // Make another graphics operation
 *     ...
 *     // Restore the screen again.
 *     TGraph_put_background( main_module.m_gfx, ptr_display_buffer );
 *     ...
 *     free( ptr_display_buffer );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 703
#endif
char* DisplayGraphics_get_page_ptr(struct DisplayGraphics* ptr_gfx, int page);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_pixel          TGraph_get_pixel
#else
/**
 *  Returns the color of the pixel with coordinate (fx , fy).
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param fx x-coordinate of the required pixel
 *  @param fy y-coordinate of the required pixel
 *  @return The color of the pixel with coordinate (fx , fy)
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     if ( DisplayGraphics_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          DisplayGraphics_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE );
 *       ...
 *  @endcode
 *  @see DisplayGraphics_set_pixel.
 */
color_t DisplayGraphics_get_pixel (struct DisplayGraphics* ptr_gfx, int fx, int fy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_string_width       Graphics_string_width
#else
/**
 *  Returns the width (in pixels) of a string in the current font.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param string A pointer to the string
 *  @return The width (in pixels) of a string in the current font
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int str_plen;
 *     DisplayGraphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *     str_plen = DisplayGraphics_string_width( main_module.m_gfx, "Any string" );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_string_width_Ex.
 */
int DisplayGraphics_string_width (struct DisplayGraphics* ptr_gfx, char *string);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_string_width_Ex    Graphics_string_width_Ex
#else
/**
 *  Returns the width (in pixels) of the first 'flen' symbols in a string.
 *  @param ptr_gfx A pointer to the initialized DisplayGraphics object
 *  @param string A pointer to the string
 *  @param flen A number of the symbols
 *  @return The width (in pixels) of the first 'flen' symbols in a string
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     int str_plen;
 *     DisplayGraphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *     str_plen = DisplayGraphics_string_width_Ex( main_module.m_gfx, "Any string" ,20);
 *     ...
 *  @endcode
 *  @see DisplayGraphics_string_width.
 */
int DisplayGraphics_string_width_Ex (struct DisplayGraphics* ptr_gfx, char *string, int flen);
#endif


/**
 *  Copies a rectangular area from one graphics page to another. A rectangular area is defined by x,y,h and w parameters.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param from_page The source page to copy
 *  @param to_page The destination page for the copied image
 *  @param x x-coordinate of the rectangular area to be copied
 *  @param y y-coordinate of the rectangular area to be copied
 *  @param w A width of the page to be copied
 *  @param h A height of the page to be copied
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx A pointer to the Cybiko graphics context
 *     DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_WHITE );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show_Ex( main_module.m_gfx , 1 );
 *     ...
 *     DisplayGraphics_set_page( main_module.m_gfx, 0 );
 *     DisplayGraphics_page_copy( main_module.m_gfx, 1, 0, 5, 5, 30, 30 );
 *     // Now the white rectangle is on both Cybiko graphics pages.
 *     ...
 *  @endcode
 *  @see DisplayGraphics_page_copy_Ex.
 */
#ifndef _SKIP_DOC
__cyos 935
#endif
void DisplayGraphics_page_copy(struct DisplayGraphics* ptr_gfx,int from_page,int to_page,int x,int y,int w,int h);

/**
 *  Duplicates the DisplayGraphics_page_copy function. The region coordinates are defined in the rect_t object.
 *  @param ptr_gfx A pointer to a DisplayGraphicsGraphics object
 *  @param from_page The source page to copy
 *  @param to_page The destination page for the copied image
 *  @param rc A pointer to the rectangle to be copied
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t white_rect;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     white_rect.x = 5;
 *     white_rect.y = 5;
 *     white_rect.w = 30;
 *     white_rect.h = 30;
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_WHITE );
 *     TGraph_fill_rect_Ex( main_module.m_gfx, &white_rect );
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show_Ex( main_module.m_gfx , 1 );
 *     ...
 *     DisplayGraphics_set_page( main_module.m_gfx, 0 );
 *     DisplayGraphics_page_copy_Ex( main_module.m_gfx, 1, 0 , &white_rect );
 *     // Now the white rectangle is on both Cybiko graphics pages.
 *  @endcode
 *  @see DisplayGraphics_page_copy.
 */
#ifndef _SKIP_DOC
__cyos 934
#endif
void DisplayGraphics_page_copy_Ex(struct DisplayGraphics* ptr_gfx,int from_page,int to_page,struct rect_t* rc);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_scroll             TGraph_scroll
#else
/**
 *  Scrolls a rectangular area by dx and dy.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param ptr_rectangle A pointer to the rect_t type structure to be scrolled
 *  @param dx A number of pixels in the x-direction
 *  @param dy A number of pixels in the y-direction
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     int scroll_shift;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     DisplayGraphics_scroll( main_module.m_gfx, 0, 0, 160, 78, scroll_shift, 0 );
 *     ...
 *  @endcode
 */
void DisplayGraphics_scroll (struct DisplayGraphics* ptr_gfx, struct rect_t* ptr_rectangle, int dx, int dy);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_bitmap         Graphics_set_bitmap
#else
/**
 *  Sets a bitmap 'bmp' as the destination for drawings.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param bmp A pointer to the Bitmap object
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bitmap;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     DisplayGraphics_set_bitmap( main_module.m_gfx, &bitmap );
 *     ...
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *  @endcode
 */
void DisplayGraphics_set_bitmap (struct DisplayGraphics* ptr_gfx, struct Bitmap* bmp);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_put_background     TGraph_put_background
#else
/**
 *  Sets an image pointed by #ptr_background as a background image.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param ptr_background A pointer to a background image
 *  @return None
 *  @code
 *   #include <cywin.h>
 *     ...
 *     struct module_t main_module;
 *     struct Graphics gfx;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     DisplayGraphics_put_background( main_module.m_gfx,
 *              Graphics_get_buf_addr( &gfx ));
 *     ...
 *  @endcode
 */
void DisplayGraphics_put_background (struct DisplayGraphics* ptr_gfx, char* ptr_background);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_get_buf_addr(ptr_gfx)  ((ptr_gfx)->_buf)
#else
/**
 *  Returns a pointer to an image of the DisplayGraphics object destination.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @return A pointer to an image of the DisplayGraphics object destination
 *  @see DisplayGraphics_page_copy_Ex.
 */
char* DisplayGraphics_get_buf_addr (struct DisplayGraphics* ptr_gfx);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_bkcolor        TGraph_set_bkcolor
#else
/**
 *  Sets the current background color.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param fc A color_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct Bitmap bmp;
 *     ...
 *     init_module( &main_module );
 *     // Creates a bitmap from the file "root.ico".
 *     Bitmap_ctor_Ex1( &bmp, "root.ico" );
 *     ...
 *     // Set draw mode to mode OR.
 *     TGraph_set_draw_mode( main_module.m_gfx, DM_OR );
 *     // Set transparent color to CLR_BLACK
 *     TGraph_set_bkcolor( main_module.m_gfx, CLR_BLACK );
 *     // Draw all pixels of the bitmap except pixels with CLR_BLACK color.
 *     Graphics_draw_bitmap( main_module.m_gfx, &bmp, 30, 40, BM_NORMAL );
 *     ...
 *     Bitmap_dtor( &bmp, LEAVE_MEMORY );
 *     ...
 *  @endcode
 *  @see CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void DisplayGraphics_set_bkcolor (struct DisplayGraphics* ptr_gfx, color_t fc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_clip           TGraph_set_clip ///<
#else
/**
 *  Sets a clip region, defined by coordinates that are stored in fx, fy, fw and fh parameters.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param fx x-coordinate of upper-left corner of the area to be clipped
 *  @param fy y-coordinate of upper-left corner of the area to be clipped
 *  @param fw Width of the area to be clipped
 *  @param fh Height of the area to be clipped
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *        ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *         TGraph_set_clip( main_module.m_gfx, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_set_clip_Ex.
 */
void DisplayGraphics_set_clip (struct DisplayGraphics* ptr_gfx, int fx, int fy, int fw, int fh);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_clip_Ex        TGraph_set_clip_Ex
#else
/**
 *  Duplicates the Graphics_set_clip function. The region coordinates are stored in the rect_t object.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param ptr_rectangle A pointer to a rect_t object. By this structure we can define the region coordinates
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     struct rect_t new_clip_reg;
 *     init_module( &main_module );
 *     ...
 *     new_clip_reg.x = 10;
 *     new_clip_reg.y = 10;
 *     new_clip_reg.h = 100;
 *     new_clip_reg.w = 100;
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( (clip_region.x != 0) || ( clip_region.y != 0) ||
 *        ( clip_region.w != SCREEN_WIDTH) ||( clip_region.h != SCREEN_HEIGHT)) )
 *        TGraph_set_clip( main_module.m_gfx, &new_clip_reg );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_set_clip.
 */
void DisplayGraphics_set_clip_Ex (struct DisplayGraphics* ptr_gfx, struct rect_t* ptr_rectangle);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_color          TGraph_set_color
#else
/**
 *  Sets the current foreground color.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param fc A pointer to a color_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     if ( TGraph_get_color( main_module.m_gfx ) != CLR_LTGRAY )
 *          TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     TGraph_draw_line( main_module.m_gfx, 10, 10, 20, 60 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see DisplayGraphics_get_color, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void DisplayGraphics_set_color (struct DisplayGraphics* ptr_gfx, color_t fc);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_draw_mode      TGraph_set_draw_mode
#else
/**
 *  Sets the current drawing mode.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param dm A drawmode_t object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_LTGRAY );
 *     ...
 *     // The second parameter in the TGraph_set_draw_mode function is the draw mode value.
 *     // This value must be one of the following: DM_XOR, DM_OR, DM_PUT.
 *     if ( TGraph_get_draw_mode( main_module.m_gfx ) != DM_PUT )
 *          TGraph_set_draw_mode( main_module.m_gfx, DM_PUT );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     ...
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *  @endcode
 *  @see DisplayGraphics_get_draw_mode, DM_XOR, DM_OR, DM_PUT.
 */
void DisplayGraphics_set_draw_mode (struct DisplayGraphics* ptr_gfx, drawmode_t dm);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_font           Graphics_set_font
#else
/**
 *  Sets the font object as a current font.
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param font A pointer to the Font structure
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int str_plen;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     Graphics_set_font( main_module.m_gfx, cool_normal_font );
 *     ...
 *  @endcode
 */
void DisplayGraphics_set_font (struct DisplayGraphics* ptr_gfx, struct Font* font);
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define DisplayGraphics_set_pixel          TGraph_set_pixel
#else
/**
 *  Sets the color value 'fc' for pixel (fx, fy).
 *  @param ptr_gfx A pointer to a TGraph object
 *  @param fx x-coordinate of the pixel to be drawn
 *  @param fy y-coordinate of the pixel to be drawn
 *  @param fc Color of the pixel to be drawn
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     struct rect_t clip_region;
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     TGraph_get_clip ( main_module.m_gfx, &clip_region );
 *     if ( TGraph_get_pixel( main_module.m_gfx, 10, 30 ) == CLR_BLACK )
 *          TGraph_set_pixel( main_module.m_gfx, 10, 30, CLR_WHITE )
 *     ...
 *  @endcode
 *  @see DisplayGraphics_get_pixel, CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
void DisplayGraphics_set_pixel (struct DisplayGraphics* ptr_gfx, int fx,int fy, color_t fc);
#endif

/**
 *  Sets the page 'page' as a target for drawing, and in default sends it to the LCD.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param page Can be 0 or 1
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     struct module_t main_module;
 *     init_module( &main_module );
 *     ...
 *     if ( !DisplayGraphics_get_page( main_module.m_gfx ) )
 *        DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     ...
 *  @endcode
 *  @see DisplayGraphics_get_page.
 */
#ifndef _SKIP_DOC
__cyos 1137
#endif
void DisplayGraphics_set_page(struct DisplayGraphics* ptr_gfx, int page);

/**
 *  Sends the current graphics page to the LCD.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int i;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     DisplayGraphics_set_page( main_module.m_gfx, 0 );
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is the color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_WHITE );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     TGraph_set_color( main_module.m_gfx, CLR_BLACK );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     for(i=0; i<20; i++)
 *     {
 *       DisplayGraphics_flip( main_module.m_gfx );
 *       cWinApp_pause( main_module.m_gfx, 250);   // 250 milliseconds delay.
 *     }
 *     //  Black and white rectangles will change places with each other.
 *  @endcode
 *  @see DisplayGraphics_set_page, DisplayGraphics_show_Ex.
 */
#ifndef _SKIP_DOC
__cyos 1163
#endif
void DisplayGraphics_show(struct DisplayGraphics* ptr_gfx);

/**
 *  The extended version of the DisplayGraphics_show function. Sends a specified graphics page to the LCD. The graphics page defines are in the 'pg' parameter.
 *  @param ptr_gfx A pointer to a DisplayGraphics object
 *  @param page Graphics page (0 or 1)
 *  @return None
 *  @code
 *   #include <cybiko.h>
 *     ...
 *     int i;
 *     struct module_t main_module;
 *     ...
 *     init_module( &main_module );
 *     ...
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     DisplayGraphics_set_page( main_module.m_gfx, 0 );
 *     // main_module.m_gfx is a pointer to the Cybiko graphics context.
 *     // The second parameter in the TGraph_set_color function is color value.
 *     // This value must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 *     TGraph_set_color( main_module.m_gfx, CLR_WHITE );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     // To send the current graphics page to the Cybiko display you should use this function
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     DisplayGraphics_set_page( main_module.m_gfx, 1 );
 *     TGraph_set_color( main_module.m_gfx, CLR_BLACK );
 *     TGraph_fill_rect( main_module.m_gfx, 5, 5, 30, 30 );
 *     DisplayGraphics_show( main_module.m_gfx );
 *     ...
 *     for(i=0; i<20; i++)
 *     {
 *       DisplayGraphics_show_Ex( main_module.m_gfx, i%2 );
 *       cWinApp_pause(main_module.m_process , 250);   // 250 milliseconds delay.
 *     }
 *     //  Black and white rectangles will change places with each other.
 *  @endcode
 *  @see DisplayGraphics_set_page, DisplayGraphics_show.
 */
#ifndef _SKIP_DOC
__cyos 1164
#endif
void DisplayGraphics_show_Ex(struct DisplayGraphics* ptr_gfx, int page);

#ifdef _SKIP_DOC
};
#endif

#endif //  #ifndef __CYBER_GRAPH_H_INCLUDE__
