/**  
 *   @file cyber-lib.h
 *   Bitset object (describes a set of bits with simple bit operations).
 */
#ifndef __CYBER_LIB_H_INCLUDE__
#define __CYBER_LIB_H_INCLUDE__

/** 
 *  @ingroup math_group
 *  @brief Works with a set of bits using simple bit operations.
 *
 *  Works with a set of bits using simple bit operations. 
 *  <br>
 *  <br>
 *  You must call the bitset_ctor() function before use, and the bitset_dtor() function 
 *  after use.
 *  @see math_group
 */
struct bitset
{
#ifndef _SKIP_DOC
  short size;
  short PADDING;
  char* val;
};
#endif

/**
 *  Creates a set of bits.
 *  @param ptr_bitset A pointer to a bitset object
 *  @param size The number of bits in a bitset object
 *  @return A pointer to the initialized bitset object
 *  @code
 *       #include <cybiko.h>
 *       #define SIGN_BIT    31
 *       ...
 *       struct bitset float_value;
 *       bitset_ctor( &float_value, 32 );
 *       ...
 *       //  Takes abs of a value.
 *       bitset_excl( &float_value, SIGN_BIT );
 *       ...
 *       bitset_dtor( &float_value, LEAVE_MEMORY );
 *  @endcode
 *  @see bitset_dtor.
 */
#ifndef _SKIP_DOC
__cyos 233
#endif
struct bitset* bitset_ctor(struct bitset* ptr_bitset, short size);

/**
 *  Sets all bits to zero.
 *  @param ptr_bitset A pointer to the initialized bitset object
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct bitset float_value;
 *       bitset_ctor( &float_value, 32 );
 *       ...
 *       //  Sets a value to zero.
 *       bitset_clear( &float_value );
 *       ...
 *       bitset_dtor( &float_value, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 441
#endif
void bitset_clear(struct bitset* ptr_bitset);

/**
 *  Excludes a bit from the set at a specified position.
 *  @param ptr_bitset A pointer to the initialized bitset object
 *  @param fbit The position of the bit to exclude
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define SIGN_BIT    31
 *       ...
 *       struct bitset float_value;
 *       bitset_ctor( &float_value, 32 );
 *       ...
 *       //  Takes abs of a value.
 *       bitset_excl( &float_value, SIGN_BIT );
 *       ...
 *       bitset_dtor( &float_value, LEAVE_MEMORY );
 *  @endcode
 *  @see bitset_in, bitset_incl.
 */
#ifndef _SKIP_DOC
__cyos 553
#endif
void bitset_excl(struct bitset* ptr_bitset, short fbit);

/**
 *  Determines whether the bit at a specified position was excluded.
 *  @param ptr_bitset A pointer to the initialized bitset object.
 *  @param fbit The position of the bit to test
 *  @return TRUE if the bit at the specified position was not excluded
 *  @code
 *       #include <cybiko.h>
 *       #define SIGN_BIT    31
 *       ...
 *       {
 *         struct bitset float_value;
 *         bitset_ctor( &float_value, 32 );
 *         ...
 *         //  Inverses a value.
 *         if( bitset_in( &float_value, SIGN_BIT ) ) 
 *         {
 *           bitset_excl( &float_value, SIGN_BIT );
 *         } 
 *         else 
 *         {
 *           bitset_incl( &float_value, SIGN_BIT );
 *         }
 *         ...
 *         bitset_dtor( &float_value, LEAVE_MEMORY );
 *       }
 *  @endcode
 *  @see bitset_excl, bitset_incl.
 */
#ifndef _SKIP_DOC
__cyos 767
#endif
bool bitset_in(struct bitset* ptr_bitset, short fbit);

/**
 *  Adds the bit at a specified position to the set.
 *  @param ptr_bitset A pointer to the initialized bitset object
 *  @param fbit The position of the bit to test
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define SIGN_BIT    31
 *       ...
 *       {
 *         struct bitset float_value;
 *         bitset_ctor( &float_value, 32 );
 *         ...
 *         //  Inverses a value.
 *         if( bitset_in( &float_value, SIGN_BIT ) )
 *         {
 *           bitset_excl( &float_value, SIGN_BIT );
 *         }
 *         else
 *         {
 *           bitset_incl( &float_value, SIGN_BIT );
 *         }
 *         ...
 *         bitset_dtor( &float_value, LEAVE_MEMORY );
 *       }
 *  @endcode
 *  @see bitset_excl, bitset_in.
 */
#ifndef _SKIP_DOC
__cyos 768
#endif
void bitset_incl(struct bitset* ptr_bitset, short fbit);

/**
 *  Adds two bitsets, the same as logical OR.
 *  The result will be stored in the first bitset.
 *  @param ptr_bitset_1 A pointer to the first bitset
 *  @param ptr_bitset_2 A pointer to the second bitset
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct bitset btset1;
 *       struct bitset btset2;
 *       bitset_ctor( &btset1, 8 );
 *       bitset_ctor( &btset2, 8 );
 *       ...
 *       bitset_clear( &btset1 );
 *       bitset_clear( &btset2 );
 *       
 *       bitset_incl( &btset1 , 0 );
 *       bitset_incl( &btset1 , 1 );
 *       bitset_incl( &btset2 , 0 );
 *       bitset_incl( &btset2 , 2 );
 *       ...
 *       //  Performs the logical OR operation.
 *       //  Before call:
 *       //      btset1   00000011  
 *       //      btset2   00000101
 *       //  After call:
 *       //      btset1   00000111  
 *       //      btset2   00000101
 *       bitset_add( &btset1, &btset2 );
 *       ...
 *       bitset_dtor( &btset1, LEAVE_MEMORY );
 *       bitset_dtor( &btset2, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 366
#endif
void bitset_add(struct bitset* ptr_bitset_1, struct bitset* ptr_bitset_2);

/**
 *  Multiplies two bitsets, the same as logical AND.
 *  The result will be stored in the first bitset.
 *  @param ptr_bitset_1 A pointer to the first bitset
 *  @param ptr_bitset_2 A pointer to the second bitset
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct bitset btset1;
 *       struct bitset btset2;
 *       bitset_ctor( &btset1, 8 );
 *       bitset_ctor( &btset2, 8 );
 *       ...
 *       bitset_clear( &btset1 );
 *       bitset_clear( &btset2 );
 *       
 *       bitset_incl( &btset1 , 0 );
 *       bitset_incl( &btset1 , 1 );
 *       bitset_incl( &btset2 , 0 );
 *       bitset_incl( &btset2 , 2 );
 *       ...
 *       //  Performs the logical AND operation.
 *       //  Before call:
 *       //      btset1   00000011  
 *       //      btset2   00000101
 *       //  After call:
 *       //      btset1   00000001  
 *       //      btset2   00000101
 *       bitset_mul( &btset1, &btset2 );
 *       ...
 *       bitset_dtor( &btset1, LEAVE_MEMORY );
 *       bitset_dtor( &btset2, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 898
#endif
void bitset_mul(struct bitset* ptr_bitset_1, struct bitset* ptr_bitset_2);

/**
 *  Subtracts two bitsets, the same as logical XOR.
 *  The result will be stored in the first bitset.
 *  @param ptr_bitset_1 A pointer to the first bitset.
 *  @param ptr_bitset_2 A pointer to the second bitset.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct bitset btset1;
 *       struct bitset btset2;
 *       bitset_ctor( &btset1, 8 );
 *       bitset_ctor( &btset2, 8 );
 *       ...
 *       bitset_clear( &btset1 );
 *       bitset_clear( &btset2 );
 *       
 *       bitset_incl( &btset1 , 0 );
 *       bitset_incl( &btset1 , 1 );
 *       bitset_incl( &btset2 , 0 );
 *       bitset_incl( &btset2 , 2 );
 *       ...
 *       //  Performs the logical XOR operation.
 *       //  Before call:
 *       //      btset1   00000011  
 *       //      btset2   00000101
 *       //  After call:
 *       //      btset1   00000110  
 *       //      btset2   00000101
 *       bitset_sub( &btset1, &btset2 );
 *       ...
 *       bitset_dtor( &btset1, LEAVE_MEMORY );
 *       bitset_dtor( &btset2, LEAVE_MEMORY );
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1219
#endif
void bitset_sub(struct bitset* ptr_bitset_1, struct bitset* ptr_bitset_2);

/**
 *  The bitset destructor. Deletes a bitset object.
 *  @param ptr_bitset A pointer to the initialized bitset object.
 *  @param memory_flag Can be FREE_MEMORY or LEAVE_MEMORY. If the memory was allocated for the object by 
 *  malloc(), use FREE_MEMORY to free it. Use LEAVE_MEMORY If the object was static or 
 *  allocated in stack 
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define SIGN_BIT    31
 *       ...
 *       struct bitset float_value;
 *       bitset_ctor( &float_value, 32 );
 *       ...
 *       //  Takes abs of a value.
 *       bitset_excl( &float_value, SIGN_BIT );
 *       ...
 *       bitset_dtor( &float_value, LEAVE_MEMORY );
 *  @endcode
 *  @see bitset_ctor, FREE_MEMORY, LEAVE_MEMORY.
 */
#ifndef _SKIP_DOC
__cyos 114
#endif
void bitset_dtor(struct bitset* ptr_bitset, int memory_flag);

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __CYBER_LIB_H_INCLUDE__

