/** 
 *  @file cyber-message.h
 *  The interface for message communications.
 */
#ifndef __CYBER_MESSAGE_H_INCLUDE__
#define __CYBER_MESSAGE_H_INCLUDE__
                     
/**
 *  Hot key for section Main Menu. 
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION1       0x100 

/**
 *  Hot key for section Chat.
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION2       0x101 

/**
 *  Hot key for section People. 
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION3       0x102 

/**
 *  Hot key for section Applications. 
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION4       0x103 

/**
 *  Hot key for section Games. 
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION5       0x104

/** 
 *  Hot key for section Labyrinth. 
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION6       0x105 

/**
 *  Hot key for section CyLandia.
 *  @ingroup keyboard_overview
 */
#define KEY_SECTION7       0x106 

/**
 *  Escape/Backspace ( upper left corner ).
 *  @ingroup keyboard_overview
 */
#define KEY_ESC            0x107 

/**
 *  Left ( joystick, arrow key ).
 *  @ingroup keyboard_overview
 */
#define KEY_LEFT           0x108 

/**
 *  Up ( joystick, arrow key ).
 *  @ingroup keyboard_overview
 */
#define KEY_UP             0x109 

/**
 *  Right ( joystick, arrow key ). 
 *  @ingroup keyboard_overview
 */
#define KEY_RIGHT          0x10A 

/**
 *  Down ( joystick, arrow key ).
 *  @ingroup keyboard_overview
 */
#define KEY_DOWN           0x10B 

/**
 *  Insert/Yes, 1-st big key.
 *  @ingroup keyboard_overview
 */
#define KEY_INS            0x10C 

/**
 *  Delete/No, 2-nd big key.
 *  @ingroup keyboard_overview
 */
#define KEY_DEL            0x10D 

/**
 *  Tab/Switch, 3-rd big key.
 *  @ingroup keyboard_overview
 */
#define KEY_TAB            0x10E 

/**
 *  Select, 4-th big key.
 *  @ingroup keyboard_overview
 */
#define KEY_SELECT         0x10F 

/**
 *  Enter, 5-th big key.
 *  @ingroup keyboard_overview
 */
#define KEY_ENTER          0x110 

/**
 *  Backspace, move left and delete.
 *  @ingroup keyboard_overview
 */
#define KEY_BACKSPACE      0x111 

/**
 *  Help key, in the upper-left corner of the case.
 *  @ingroup keyboard_overview
 */
#define KEY_HELP           0x112 

/**
 *  Shift ( "upper case" ) key.
 *  @ingroup keyboard_overview
 */
#define KEY_SHIFT          0x113 

/**
 *  Ctrl ( "alternative function" ) key.
 *  @ingroup keyboard_overview
 */
#define KEY_CONTROL        0x114 

/**
 *  CY sign (Fn+C).
 *  @ingroup keyboard_overview
 */
#define KEY_CY             133  

/**
 *  Space key.
 *  @ingroup keyboard_overview
 */
#define KEY_SPACE          0x20 

/**
 *  0 key.
 *  @ingroup keyboard_overview
 */
#define KEY_0              0x30

/**
 *  1 key.
 *  @ingroup keyboard_overview
 */
#define KEY_1              0x31 

/**
 *  2 key.
 *  @ingroup keyboard_overview
 */
#define KEY_2              0x32 

/**
 *  3 key.
 *  @ingroup keyboard_overview
 */
#define KEY_3              0x33 

/**
 *  4 key.
 *  @ingroup keyboard_overview
 */
#define KEY_4              0x34 

/**
 *  5 key.
 *  @ingroup keyboard_overview
 */
#define KEY_5              0x35 

/** 
 *  6 key.
 *  @ingroup keyboard_overview
 */
#define KEY_6              0x36 

/**
 *  7 key.
 *  @ingroup keyboard_overview
 */
#define KEY_7              0x37 

/**
 *  8 key.
 *  @ingroup keyboard_overview
 */
#define KEY_8              0x38 

/**
 *  9 key.
 *  @ingroup keyboard_overview
 */
#define KEY_9              0x39 

/**
 *  ' key.
 *  @ingroup keyboard_overview
 */
#define KEY_QUOTE          0x27 

/**
 *  , key.
 *  @ingroup keyboard_overview
 */
#define KEY_COMMA          0x2C 

/**
 *  "-" key.
 *  @ingroup keyboard_overview
 */
#define KEY_MINUS          0x2D 

/**
 *  "." key
 *  @ingroup keyboard_overview
 */
#define KEY_PERIOD         0x2E 

/**
 *  / key.
 *  @ingroup keyboard_overview
*/
#define KEY_SLASH          0x2F 

/**
 *  ; key.
 *  @ingroup keyboard_overview
*/
#define KEY_SEMICOLON      0x3B 

/**
 *  \ key.
 *  @ingroup keyboard_overview
*/
#define KEY_BACKSLASH      0x5C 

/**
 *  = key.
 *  @ingroup keyboard_overview
*/
#define KEY_EQUAL          0x3D 

/**
 *  [ key.
 *  @ingroup keyboard_overview
*/
#define KEY_OPEN_SBRACKET  0x5B 

/**
 *  ] key.
 *  @ingroup keyboard_overview
*/
#define KEY_CLOSE_SBRACKET 0x5D 

/**
 *  ` key.
 *  @ingroup keyboard_overview
*/
#define KEY_BACKQUOTE      0x60 

/**
 *  A key.
 *  @ingroup keyboard_overview
*/
#define KEY_A              0x61 

/**
 *  B key.
 *  @ingroup keyboard_overview
*/
#define KEY_B              0x62 

/**
 *  C key.
 *  @ingroup keyboard_overview
*/
#define KEY_C              0x63 

/**
 *  D key.
 *  @ingroup keyboard_overview
*/
#define KEY_D              0x64 

/**
 *  E key.
 *  @ingroup keyboard_overview
*/
#define KEY_E              0x65 

/**
 *  F key.
 *  @ingroup keyboard_overview
*/
#define KEY_F              0x66 

/**
 *  G key.
 *  @ingroup keyboard_overview
*/
#define KEY_G              0x67 

/**
 *  H key.
 *  @ingroup keyboard_overview
*/
#define KEY_H              0x68 

/**
 *  I key.
 *  @ingroup keyboard_overview
*/
#define KEY_I              0x69 

/**
 *  J key.
 *  @ingroup keyboard_overview
*/
#define KEY_J              0x6A 

/**
 *  K key.
 *  @ingroup keyboard_overview
*/
#define KEY_K              0x6B

/**
 *  L key.
 *  @ingroup keyboard_overview
*/
#define KEY_L              0x6C 

/**
 *  M key.
 *  @ingroup keyboard_overview
*/
#define KEY_M              0x6D 

/**
 *  N key.
 *  @ingroup keyboard_overview
*/
#define KEY_N              0x6E 

/**
 *  O key.
 *  @ingroup keyboard_overview
*/
#define KEY_O              0x6F 

/**
 *  P key.
 *  @ingroup keyboard_overview
*/
#define KEY_P              0x70 

/**
 *  Q key.
 *  @ingroup keyboard_overview
*/
#define KEY_Q              0x71 

/**
 *  R key.
 *  @ingroup keyboard_overview
*/
#define KEY_R              0x72 

/**
 *  S key.
 *  @ingroup keyboard_overview
*/
#define KEY_S              0x73 

/**
 *  T key.
 *  @ingroup keyboard_overview
*/
#define KEY_T              0x74 

/**
 *  U key.
 *  @ingroup keyboard_overview
*/
#define KEY_U              0x75 

/**
 *  V key.
 *  @ingroup keyboard_overview
*/
#define KEY_V              0x76 

/**
 *  W key.
 *  @ingroup keyboard_overview
*/
#define KEY_W              0x77 

/**
 *  X key.
 *  @ingroup keyboard_overview
*/
#define KEY_X              0x78

/**
 *  Y key.
 *  @ingroup keyboard_overview
*/
#define KEY_Y              0x79 

/**
 *  Z key.
 *  @ingroup keyboard_overview
*/
#define KEY_Z              0x7A 

/**
 *  Message_deliver() flag, which deletes the message after it was delivered
 *  @ingroup messaging_group
*/
#define MSG_AUTODELETE   0x80000000

/**
 *  Message_deliver() maximum timeout
 *  @ingroup messaging_group
*/
#define MSG_MAX_TIMEOUT  0x0FFFFFFF 

#ifndef _SKIP_DOC
#define MSG_TIMEOUT_MASK MSG_MAX_TIMEOUT 

#define MSG_NONE         0      
#endif

/**
 *  Minimum ID of a keyboard message.
 *  @ingroup messaging_group
*/
#define MSG_KEYMIN       1      

/**
 *  Sends the message to the focused program when the user presses a key
 *  @ingroup messaging_group
*/
#define MSG_KEYDOWN      1   

/**
 *  Sends the message to the focused program when the user releases a key
 *  @ingroup messaging_group
*/
#define MSG_KEYUP        2      

/**
 *  Sends the message to the focused program when the user releases a key
 *  @ingroup messaging_group
*/
#define MSG_CHARTYPED    3

/**
 *  Maximum ID of a keyboard message.
 *  @ingroup http://www.flight-toys.com/boomerangs.htm
*/
#define MSG_KEYMAX       0xF   

/**
 *  Sent when the pre-set timer counts down to 0
 *  @ingroup messaging_group
*/
#define MSG_TIMER        0x20   

/**
 *  Batteries are low, warn the user ! 
 *  @ingroup messaging_group
*/
#define MSG_POWERDOWN    0x30   

/**
 *  Sends this message when the process should finish
 *  @ingroup messaging_group
*/
#define MSG_QUIT         0x31   

/**
 *  About repaint
 *  @ingroup messaging_group
*/
#define MSG_PAINT        0x32   

/**
 *  Lists changes in the file system
 *  @ingroup messaging_group
*/
#define MSG_FILES        0x33

/**
 *  The Process gets the message when it loses focus
 *  @ingroup messaging_group
*/
#define MSG_LOSTFOCUS    0x40   

/**
 *  The process gets the message once it gets the focus
 *  @ingroup messaging_group
*/
#define MSG_GOTFOCUS     0x41   

/**
 *  Sends the message to the Process you want to launch
 *  @ingroup messaging_group
*/
#define MSG_LAUNCH       0x42   

/**
 *  Receiver sends the message when a device appears or disappears on the channel
 *  @ingroup messaging_group
*/
#define MSG_DEVICE       0x43   

/**
 *  Message is sent by the Receiver to ping the listener process
 *  @ingroup messaging_group
*/
#define MSG_PING         0x44  

#ifndef _SKIP_DOC
#define MSG_RUN          0x45  
#define MSG_MELODY       0x46   
/**
 *  Packed Message for retransmitting.
 *  @ingroup messaging_group
*/
#define MSG_PACKED       0x47   

/**
 *  Device is sleeping.
 *  @ingroup messaging_group
*/
#define MSG_DEVICE_SLEEP 0x49
#endif
 
/**
 *  Process must be terminated
 *  @ingroup messaging_group
*/
#define MSG_SHUTUP       0x101A 

#ifndef _SKIP_DOC
#define MSG_SPLASH_TIMER 0x04D2
#define MSG_SPLASH_STOP  0x04D3
#endif

/**
 *  Start of the user_defined message ID
 *  @ingroup messaging_group
*/
#define MSG_USER         0x2000

/** Message was created and possibly posted
 *  @ingroup messaging_group
*/
#define DL_NONE          0

/**
 *  Message is in queue to deliver; please, don't delete the message 
 *  while in this state.
 *  @ingroup messaging_group
*/
#define DL_INQUEUE       1 

/**
 *  Message is being delivered
 *  @ingroup messaging_group
*/
#define DL_WAIT          2 

/**
 *  Message delivery aborted by internal error or user request
 *  @ingroup messaging_group
*/
#define DL_ABORT         3 

/**
 *  Message delivery finished by timeout
 *  @ingroup messaging_group
*/
#define DL_TIMEOUT       4 

/**
 *  Message was delivered successfully
 *  @ingroup messaging_group
*/
#define DL_SUCCESS       5 

/**
 *  State is undefined.
 *  @ingroup filesystem_overview
*/
#define FS_NONE          0 

/**
 *  File was created.
 *  @ingroup filesystem_overview
*/
#define FS_CREATED       1
 
/**
 *  File was removed.
 *  @ingroup filesystem_overview
*/
#define FS_REMOVED       2

/**
 *  File was changed.
 *  @ingroup filesystem_overview
*/
#define FC_CHANGED       3 

/**
 *  File was renamed.
 *  @ingroup filesystem_overview
*/
#define FC_RENAMED       4

/**
 *  Disk was formatted.
 *  @ingroup filesystem_overview
*/
#define FC_FORMATTED     5 

/**
 *  Enumeration of masks for local broadcast messages
 *  @ingroup messaging_group
 */
enum qflag_t
{
  /**
   *  Queue will get only messages for QF_NONE
   */
  QF_NONE = 0,    

  /**
   *  Queue will get only messages for unit changes
   */
  QF_DEVICES    = 1,    

  /**
   *  Queue will get only messages for file system changes
   */
  QF_FILES    = 2,      

  /**
   *  First value for application masks
   */
  QF_APPMASK  = 0x1000,       

  /**
   *  Mask for all MASKED queues
   */
  QF_ALL  = 0xffff        
};

/**
 *  Keyboard modificators masks.
 *  @ingroup messaging_group
 */
enum keymask_t
{
    /**
     *  Shift was pressed.
     */
    KEYMASK_SHIFT      = 0x0001,

    /**
     *  Control was pressed.
     */
    KEYMASK_CTRL       = 0x0002,

    /**
     *  It is autorepeat-generated message (not the first key down).
     */
    KEYMASK_AUTOREPEAT = 0x0004
};

//============================
//=                          =
//=     object Message.      =
//=  Communication object.   =
//=                          =
//============================

/**
 *  @brief Structure for interprocess communications; can be sent either locally or remotely.
 *  @ingroup messaging_group
 *  Structure for interprocess communications; can be sent 
 *  either locally or remotely.
 *  @see messaging_group
 */
struct Message
{
#ifndef _SKIP_DOC

  /** 
   *  Pointer to the Buffer object.
   */
  struct Buffer* buffer;              

  /** 
   *  The name of the current process.
   */
  char procname[12];        

#endif
  /**
   *  Pointer to the next Message in the queue.
   */
  struct Message* next; 

  /**
   *  The name of the target process.
   */
  char* dst_name;            

  /** 
   *  Cy ID of the sender device.
   */
  cyid_t cyid_from;           

  /**
   *  Cy ID of the target device.
   */
  cyid_t cyid_to;             

  /** 
   *  TRUE if the Message has been deleted from the queue.
   */
  bool deleted;             

#ifndef _SKIP_DOC
  bool PADDING;
#endif

  /** 
   *  ID of the Message.
   */
  short msgid;               

  /**
   *  Additional parameters for the Message.
   */
  long param[2];            

#ifndef _SKIP_DOC
  void* session;
};
#endif

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Message_get_key_param(ptr_message) \
        ((struct KeyParam*)((ptr_message)->param))
#else
/**
 *  Returns a pointer to the message's KeyParam structure.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return A pointer to the KeyParam structure of the message
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t  main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = cWinApp_get_message( main_module.m_process, 
 *                                          0, 
 *                                          1, 
 *                                          MSG_USER );
 *       if( ptr_message->msgid == MSG_KEYDOWN )
 *       {
 *         if( Message_get_key_param( ptr_message )->scancode == MSG_ESC )
 *         {
 *           //  Processes key 'Esc'.
 *         }
 *       }
 *       ...
 *  @endcode
 */
struct KeyParam* Message_get_key_param (struct Message* ptr_message);
#endif

/**
 *  Returns a new Message from the system stores.
 *  @param sizeof_message The size of the Message structure
 *  @return A pointer to the Message object
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       //  Sends the message to the 'data_reciever' process which 
 *       //  is on the same computer.
 *       Message_post( ptr_message, "data_reciever", get_own_id() );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 270
#endif
struct Message* Message_new(size_t sizeof_message);

#ifndef _REPLACE_DEFINE_BY_FUNCTION
#define Message_get_sender_id(ptr_message) ((ptr_message)->cyid_from)  
#else
/**
 *  Returns the 32-bit device ID (CyID) of the Message's sender.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return The 32-bit device ID (CyID) of the Message's sender
 *  @code
 *  @endcode
 */
cyid_t Message_get_sender_id(struct Message* ptr_message);
#endif

/**
 *  Determines if the message has no specific address.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return TRUE if the message has no specific address
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TASK_REQUEST     MSG_USER + 1
 *       ...
 *       struct module_t  main_module;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       struct Message* ptr_message = cWinApp_get_message( main_module.m_process, 
 *                                                          0, 
 *                                                          1, 
 *                                                          MSG_TASK_REQUEST );
 *       if( Message_is_broadcast( ptr_message ) )
 *       {
 *         //  Declines task.
 *       }
 *       else
 *       {
 *         //  Accepts task.
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 804
#endif
bool Message_is_broadcast(struct Message* ptr_message);

/**
 *  Posts a message to a specified target.
 *  @param ptr_message A pointer to the initialized Message object.
 *  @param sz_process_name The target process' name
 *  @param cyid The target computer's Cy ID
 *  @return TRUE if a local Message was put into the Process' Queue
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       //  Sends the message to the 'data_reciever' process which 
 *       //  is on the same computer.
 *       Message_post( ptr_message, "data_reciever", get_own_id() );
 *       ...
 *  @endcode
 *  @see Message_post_all.
 */
#ifndef _SKIP_DOC
__cyos 943
#endif
bool Message_post(struct Message* ptr_message, char* sz_process_name, cyid_t cyid);

/**
 *  Posts a message to all Queues.
 *  @param ptr_message A pointer to the initialized Message object.
 *  @param mask One of the following: QF_NONE, QF_DEVICES, QF_FILES, QF_APPMASK, QF_ALL.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       //  Sends the message to all. This message will receive all 
 *       //  processes on the Cybiko computer.
 *       Message_post_all( ptr_message, QF_ALL );
 *       ...
 *  @endcode
 *  @see Message_post.
 */
#ifndef _SKIP_DOC
__cyos 945
#endif
void Message_post_all(struct Message* ptr_message, short mask);

/**
 *  Delivers a message to a specified target, with delivery result notification.
 *  @param ptr_message A pointer to the initialized Message object
 *  @param sz_process_name The target process' name
 *  @param cyid The target device's Cy ID
 *  @param timeout A special time to wait. If the Message is not delivered to 
 *  the target during the timeout, then the delivery state becomes DL_TIMEOUT
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Flag delivery_flag;
 *       char partner_name[9];
 *       cyid_t partner_id;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       partner_id = select_game_partner( main_module.m_process,
 *                                         "Billiard",
 *                                         SGP_CYBIKO | SGP_HOT_SEAT,
 *                                         partner_name );
 *       if( partner_id )
 *       {
 *         ...
 *         struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *         ptr_message->msgid = MSG_TEST;
 *         //  Sends message to the "Billiard" application on the remote computer.
 *         Message_deliver( ptr_message, "Billiard", partner_id, 60*1000 );
 *       }
 *       Flag_ctor(&delivery_flag, "MessageMutex", TRUE);
 *       if( Message_wait_delivery( ptr_message, &delivery_flag ) == DL_SUCCESS )
 *       {
 *         TRACE( "Message was delivered successfully" );
 *       }
 *       Flag_dtor(&delivery_flag, LEAVE_MEMORY);
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 501
#endif
void* Message_deliver(struct Message* ptr_message, char* sz_process_name, cyid_t cyid, clock_t timeout);

/**
 *  Checks the delivery state.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return Delivery state
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       char partner_name[9];
 *       cyid_t partner_id;
 *       delivery_t delivery_result;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       partner_id = select_game_partner( main_module.m_process,
 *                                         "Billiard",
 *                                         SGP_CYBIKO | SGP_HOT_SEAT,
 *                                         partner_name );
 *       if( partner_id )
 *       {
 *         ...
 *         struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *         ptr_message->msgid = MSG_TEST;
 *         //  Sends message to "Billiard" application on the remote computer.
 *         Message_deliver( ptr_message, "Billiard", partner_id, 60*1000 );
 *       }
 *
 *       while(1)
 *       {
 *         cWinApp_pause( module.m_process, 250 );
 *         delivery_result = Message_check_delivery( ptr_test_message );
 *         if( delivery_result == DL_SUCCESS )
 *         {
 *           TRACE( "Message was delivered successfully" );
 *           break;
 *         }
 *         else if( ( delivery_result == DL_ABORT ) 
 *                  || ( delivery_result == DL_TIMEOUT ) )
 *         {
 *           TRACE( "Message was not delivered successfully" );
 *           break;
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see DL_INQUEUE, DL_WAIT, DL_ABORT, DL_TIMEOUT, DL_SUCCESS.
 */
#ifndef _SKIP_DOC
__cyos 413
#endif
delivery_t Message_check_delivery(struct Message* ptr_message);

/**
 *  Waits until Message delivery is completed or flag is set.
 *  @param ptr_message A pointer to the initialized Message object
 *  @param ptr_flag A pointer to the synchronizing object, or 0
 *  @return Message delivery state
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Flag delivery_flag;
 *       char partner_name[9];
 *       cyid_t partner_id;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       partner_id = select_game_partner( main_module.m_process,
 *                                         "Billiard",
 *                                         SGP_CYBIKO | SGP_HOT_SEAT,
 *                                         partner_name );
 *       if( partner_id )
 *       {
 *         ...
 *         struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *         ptr_message->msgid = MSG_TEST;
 *         //  Sends message to the "Billiard" application on the remote computer.
 *         Message_deliver( ptr_message, "Billiard", partner_id, 60*1000 );
 *       }
 *       Flag_ctor(&delivery_flag, "MessageMutex", TRUE);
 *       if( Message_wait_delivery( ptr_message, &delivery_flag ) == DL_SUCCESS )
 *       {
 *         TRACE( "Message was delivered successfully" );
 *       }
 *       Flag_dtor(&delivery_flag, LEAVE_MEMORY);
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1277
#endif
delivery_t Message_wait_delivery(struct Message* ptr_message, struct Flag* ptr_flag);

/**
 *  Tests whether the Message has an attached buffer.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return TRUE if the message has a buffer
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_CHAT_STRING     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       char sz_string_text[32];
 *       ...
 *       init_module( &main_module );
 *       ...
 *       struct Message* ptr_message = cWinApp_get_message( main_module.m_process, 
 *                                                          0, 
 *                                                          1, 
 *                                                          MSG_CHAT_STRING );
 *       if( ptr_message->msgid == MSG_CHAT_STRING )
 *       {
 *         if( Message_has_buffer( ptr_message ) )
 *         {
 *            Buffer_load( Message_get_buffer( ptr_message ), 
 *                         sz_string_text , 
 *                         0, 
 *                         0 );
 *            TRACE( "Message from the opponent: %s", sz_string_text );
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see Message_get_buffer.
 */
#ifndef _SKIP_DOC
__cyos 757
#endif
bool Message_has_buffer(struct Message* ptr_message);

/**
 *  Gets a pointer to the attached buffer. Note, the Buffer created by the system when 
 *  that receives the message will recycle automatically along with the Message.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return A pointer to the Buffer object
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_CHAT_STRING     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       char sz_string_text[32];
 *       ...
 *       init_module( &main_module );
 *       ...
 *       struct Message* ptr_message = cWinApp_get_message( main_module.m_process, 
 *                                                          0, 
 *                                                          1, 
 *                                                          MSG_CHAT_STRING );
 *       if( ptr_message->msgid == MSG_CHAT_STRING )
 *       {
 *         if( Message_has_buffer( ptr_message ) )
 *         {
 *            Buffer_load( Message_get_buffer( ptr_message ), 
 *                         sz_string_text, 
 *                         0, 
 *                         0 );
 *            TRACE( "Message from the opponent: %s", sz_string_text );
 *         }
 *       }
 *       ...
 *  @endcode
 *  @see Message_has_buffer.
 */
#ifndef _SKIP_DOC
__cyos 640
#endif
struct Buffer* Message_get_buffer(struct Message* ptr_message);

/**
 *  Attaches a buffer to the message. Note: the Buffer recycles automatically along 
 *  with the Message.
 *  @param ptr_message A pointer to the initialized Message object
 *  @param ptr_buffer A pointer to the initialized Buffer object to attach
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_CHAT_STRING     MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Buffer message_buffer;
 *       char partner_name[9];
 *       cyid_t partner_id;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       partner_id = select_game_partner( main_module.m_process,
 *                                         "Billiard",
 *                                         SGP_CYBIKO | SGP_HOT_SEAT,
 *                                         partner_name );
 *       if( partner_id )
 *       {
 *         ...
 *         struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *         ptr_message->msgid = MSG_CHAT_STRING;
 *         Buffer_ctor( &message_buffer, 20, 10 );
 *         Buffer_store_string( &message_buffer, "Hey, dude!", 0 );
 *         Message_attach_buffer( ptr_message, &message_buffer );
 *         Message_deliver( ptr_message, "Billiard", partner_id, 60*1000 );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 373
#endif
void Message_attach_buffer(struct Message* ptr_message, struct Buffer* ptr_buffer);

/**
 *  Makes the destination Message an exact copy of the source Message.
 *  @param ptr_message A pointer to the destination Message object
 *  @param ptr_templ A pointer to the source Message object
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       struct Message* ptr_message_repeat = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       ...
 *       Message_copy( ptr_message_repeat, ptr_message );
 *       // Sends the message to the 'data_reciever' process which 
 *       // is on the same device.
 *       Message_post( ptr_message, "data_reciever", get_own_id() );
 *       //  Repeats message.
 *       Message_post( ptr_message_repeat, "data_reciever", get_own_id() );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 470
#endif
void Message_copy(struct Message* ptr_message, struct Message* ptr_templ);

/**
 *  Deletes a message.
 *  @param ptr_message A pointer to the initialized Message object
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_TEST     MSG_USER + 1
 *       ...
 *       struct Message* ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_TEST;
 *       //  Sends the message to the 'data_reciever' process which 
 *       //  is on the same computer.
 *       Message_post( ptr_message, "data_reciever", get_own_id() );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 265
#endif
void Message_delete(struct Message* ptr_message);

#ifdef _SKIP_DOC
};
#endif

//============================
//=                          =
//=     object Queue.        =
//=  Messages queue object.  =
//=                          =
//============================

/**
 *   @brief Messages queue; messages are stored in the queue.
 *   @ingroup multithreads_overview
 *   Queues can peek at and/or remove messages from a specified range.
 *   They have names and are targets for messages; in particular,
 *   all processes have queues and thus can get messages.
 *   @see multithreads_overview
 */
struct Queue
{
#ifndef _SKIP_DOC
  char Queue_Members_STUB[56]; 
};
#endif

/**
 *  Peeks at or gets a message from the queue. 
 *  If remove is TRUE, the message will be removed. It does not wait for a message 
 *  to be available; it returns 0 immediately if there are no messages in 
 *  the specified range. Don't use this function unless really need to.
 *  @param ptr_queue A pointer to the initialized Queue object
 *  @param remove If TRUE, the message will be removed from the message queue
 *  @param min Specifies the lowest message value to be retrieved, as an integer
 *  @param max Specifies the highest message value to be retrieved, as an integer
 *  @return A pointer to the Message if one is in the queue, 
 *  otherwise 0
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       //  Removes all keyboard messages from the queue.
 *       while( ptr_message = Queue_peek_message( main_module.m_process, 
 *                                                TRUE, 
 *                                                MSG_KEYDOWN, 
 *                                                MSG_KEYDOWN ) )
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see Queue_get_message.
 */
#ifndef _SKIP_DOC
__cyos 940
#endif
struct Message* Queue_peek_message(struct Queue* ptr_queue, bool remove, int min, int max);

/**
 *  Returns the message from the specified interval. The calling thread waits 
 *  until the first message appears in the queue. Be careful not to overload, the queue.
 *  Delete the message when you are finished with it.
 *  @param ptr_queue A pointer to the initialized Queue object
 *  @param timeout Timeout value(in milliseconds)
 *  @param min deprecated
 *  @param max deprecated
 *  @return A pointer to the Message if one is in the queue, 
 *  or 0 if a timeout occurs
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = Queue_get_message( main_module.m_process, 
 *                                        0, 
 *                                        1, 
 *                                        MSG_USER );
 *       if( ptr_message->msgid == MSG_KEYDOWN )
 *       {
 *         if( Message_get_key_param( ptr_message )->scancode == MSG_ESC )
 *         {
 *           //  Processes key 'Esc'.
 *         }
 *       }
 *       ...
 *       Message_delete( ptr_message );
 *       ...
 *  @endcode
 *  @see Queue_peek_message.
 */
#ifndef _SKIP_DOC
__cyos 687
#endif
struct Message* Queue_get_message(struct Queue* ptr_queue, long timeout, int min, int max);

/**
 *  Puts a Message in the Queue.
 *  @param ptr_queue A pointer to the initialized Queue object
 *  @param ptr_message A pointer to the specified message to put
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_MAKE_TASK  MSG_USER + 1
 *       ...
 *       struct module_t main_module;
 *       struct Message* ptr_message;
 *       ...
 *       init_module( &main_module );
 *       ...
 *       ptr_message = Queue_get_message( main_module.m_process, 
 *                                        0, 
 *                                        1, 
 *                                        MSG_MAKE_TASK );
 *       if( ptr_message->msgid == MSG_MAKE_TASK )
 *       {
 *         ...
 *         //  Performs some periodical calculation.
 *         ...
 *         sleep( 250 );
 *         Queue_put_message( main_module.m_process, ptr_message );
 *       }
 *       else
 *       {
 *         Message_delete( ptr_message );
 *       }
 *       ...
 *  @endcode
 *  @see Queue_get_message , Queue_peek_message.
 */
#ifndef _SKIP_DOC
__cyos 975
#endif
void Queue_put_message(struct Queue* ptr_queue, struct Message* ptr_message);

#ifndef _SKIP_DOC
__cyos 692
char* Queue_get_name(struct Queue* ptr_queue);
#endif

#ifdef _SKIP_DOC
};
#endif

/**
 *  @brief The parameter holder for keyboard messages.
 *  @ingroup messaging_group
 *  The parameter holder for keyboard messages.
 *  @see messaging_group
 */
struct KeyParam
{
  /**
   *  Keyboard virtual keys code. 
   */
  int scancode;                        

  /**
   *  Keyboard modifier mask. 
   */
  int mask;                          

  /**
   *  Character read from the keyboard.
   */  
  char ch;                           
};

#ifndef _SKIP_DOC
/** 
 *  @ingroup filesystem_overview
 *  File event.
 *  @see FS_NONE, FS_CREATED, FS_REMOVED, FC_CHANGED, FC_RENAMED, FC_FORMATTED.
 */
struct file_event_t
{
  /** 
   *  Event type. Must be one of the following: FS_NONE , FS_CREATED ,
   *  FS_REMOVED , FC_CHANGED , FC_RENAMED or FC_FORMATTED .
   */
  int kind;

  /**
   *  File device ID.
   */
  char device;
};
#endif

//  Helper functions.
#ifndef _SKIP_DOC
__cyos 1076
#endif
/**
 *  @fn bool send_msg(char* app_name,int msgid,long d0,long d1,void* data,size_t size);
 *  @ingroup messaging_group
 *  Sends a message to another application in the same Cybiko computer.
 *  @param app_name The name of the application to which the message is addressed
 *  @param msgid Message identifier
 *  @param d0 The message's 1st additional parameter (default value - 0)
 *  @param d1 The message's 2nd additional parameter of message (default value - 0)
 *  @param data Pointer to message-related data (default value - 0)
 *  @param size Size of related data (default value - 0)
 *  @return 1, if message was sent successfully, 0 otherwise
*/
bool send_msg(char* app_name,int msgid,long d0,long d1,void* data,size_t size);

#ifndef _SKIP_DOC
__cyos 1075
#endif
/**
 *  @fn bool send_copy_msg(char* app_name,struct Message* msg);
 *  @ingroup messaging_group
 *  Sends a pre-written message to another application in the same Cybiko computer.
 *  @param app_name The name of the application to which the message is addressed
 *  @param msg Pointer to the pre-written message to be sent in Message structure
 *  @return 1, if message was sent successfully, 0 otherwise
*/
bool send_copy_msg(char* app_name,struct Message* msg);

#ifndef _SKIP_DOC
__cyos 1081
#endif
/**
 *  @fn bool send_remote_msg(cyid_t cyid,char* app_name,int msgid,long d0,long d1,void* data,size_t size);
 *  @ingroup messaging_group
 *  Sends message to another application in the same or another Cybiko computer.
 *  @param cy_id CyID of the Cybiko computer to which the message is addressed
 *  @param app_name The name of the application to which the message is addressed
 *  @param msgid Message identifier
 *  @param d0 The message's 1st additional parameter (default value - 0)
 *  @param d1 The message's 2nd additional parameter (default value - 0)
 *  @param data Pointer to message-related data (default value - 0)
 *  @param size Size of related data (default value - 0)
 *  @return 1, if message was sent successfully, 0 otherwise
*/
bool send_remote_msg(cyid_t cyid,char* app_name,int msgid,long d0,long d1,void* data,size_t size);

#ifndef _SKIP_DOC
__cyos 1080
#endif
/**
 *  @fn bool send_remote_copy_msg(cyid_t cyid,char* app_name,struct Message* msg);
 *  @ingroup messaging_group
 *  Sends pre-written message to another application in the same or other Cybiko computer
 *  @param cyid CyID of the Cybiko computer to which the message is addressed
 *  @param app_name The name of the application to which the message is addressed
 *  @param msg Pointer to the pre-written message to be sent in Message structure
 *  @return 1, if message was sent successfully, 0 otherwise
*/
bool send_remote_copy_msg(cyid_t cyid,char* app_name,struct Message* msg);

#ifndef _SKIP_DOC
__cyos 1078
#endif
/**
 *  @fn bool send_msg_launch(char* app_name);
 *  @ingroup messaging_group
 *  Launches the 'app_name' application.
 *  @param app_name Name of the application to be launched
 *  @return 1, if message was sent successfully, 0 otherwise
*/
bool send_msg_launch(char* app_name);

#endif  //  #ifndef __CYBER_MESSAGE_H_INCLUDE__

