/** 
 *  @file cyber-rf.h
 *  The application interface for Radio Frequency Communications (CyDP version).
 */

#ifndef __CYBER_RF_H_INCLUDED__
#define __CYBER_RF_H_INCLUDED__

#ifndef _SKIP_DOC
#define RF_MAILGATE_ID 0xFFFFFF01    

#define RF_APPCHANNELS        30     
#define RF_APPCHANNEL_GROUPS  10     
#endif

/**
 *  Default group for unknown applications.
 * @ingroup low_level_group
 */
#define RF_GROUP_DEFAULT       0        

/**
 *  Chat channel group.
 * @ingroup low_level_group
*/
#define RF_GROUP_CHAT          1        

/**
 *  CyLandia channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_CYLAND        2        

/**
 *  Labyrinth channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_LABYRINTH     3       

/**
 *  Server channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_SERVER        4        

/**
 *  Files channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_FILES         5        

/**
 *  Games channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_GAMES         6        

/**
 *  Applications channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_APPS          7       

/**
 *  Reserved channel group.
 * @ingroup low_level_group
 */
#define RF_GROUP_OTHER         8        

/**
 *  System channel group (for debugging purposes).
 * @ingroup low_level_group
 */
#define RF_GROUP_SYSTEM        9        

#ifndef _SKIP_DOC
__cyos 710
bool get_power(void);

__cyos 1145
void set_power(bool increased);
#endif

/**
 *  @ingroup low_level_group
 *  Returns the communications state.
 *  @return TRUE if communications are enabled.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if ( !get_communications() )
 *             set_communications( TRUE );
 *       ...
 *  @endcode
 *  @see set_communications.
 */
#ifndef _SKIP_DOC
__cyos 660
#endif
bool get_communications(void);

/**
 *  @ingroup low_level_group
 *  Enables or disables communications through RF. Disabling RF communications 
 *  turns off all communications devices (e.g. AVR, PLL and RF2915), so power 
 *  consumption is decreased. Also, it disables periodically sending pings.
 *  @param enable specifies whether communications should be enabled. 
 *  A TRUE value enables communications; a FALSE value disables them.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( !get_communications() )
 *            set_communications( TRUE );
 *       ...
 *  @endcode
 *  @see get_communications.
 */
#ifndef _SKIP_DOC
__cyos 1109
#endif
void set_communications(bool enable);

/**
 *  @ingroup low_level_group
 *  Returns the current application's channel.
 *  @return the current application channel.
 *  @code
 *       #include <cybiko.h>
 *       #define PRIVATE_CHAT_CHANNEL    0x1     
 *       ...
 *       char old_channel;
 *       ...
 *       //  Sets new application channel.
 *       old_channel = get_appchannel();
 *       set_appchannel( get_groupchannel( RF_GROUP_CHAT, 
 *                                         PRIVATE_CHAT_CHANNEL ) ); 
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ...
 *  @endcode
 *  @see set_appchannel.
 */
#ifndef _SKIP_DOC
__cyos 629
#endif
char get_appchannel(void);

/**
 *  @ingroup low_level_group
 *  Returns the current application channel of the specified device.
 *  @param device_id the ID of the device, to determine its RF channel.
 *  @return current application channel of the specified device.
 *  @code
 *       #include<cybiko.h>
 *       ...
 *       char old_channel;
 *       cyid_t partner_id;
 *       ...
 *       //  Obtains partner_id.
 *       ...
 *       //  Sets application channel equal to your partner channel.
 *       old_channel = get_appchannel();
 *       set_appchannel( get_appchannel_of( partner_id ) );
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ... 
 *  @endcode
 *  @see set_appchannel.
 */
#ifndef _SKIP_DOC
__cyos 630
#endif
char get_appchannel_of(cyid_t device_id);

/**
 *  @ingroup low_level_group
 *  Sets current application channel. The channel #0 is the Base channel.
 *  @param rf_channel RF channel to set.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       #define PRIVATE_CHAT_CHANNEL    0x1     
 *       ...
 *       char old_channel;
 *       ...
 *       //  Sets new application channel.
 *       old_channel = get_appchannel();
 *       set_appchannel( get_groupchannel( RF_GROUP_CHAT, 
 *                                         PRIVATE_CHAT_CHANNEL ) ); 
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ...
 *  @endcode
 *  @see get_appchannel_of.
 */
#ifndef _SKIP_DOC
__cyos 1091
#endif
void set_appchannel(char rf_channel);

/**
 *  @ingroup low_level_group
 *  Selects the application channel with the last traffic.
 *  @param group_id ID of the group to which the application belongs.
 *  @return ID of the channel with the last traffic.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char old_channel;
 *       ...
 *       // Sets application channel with the least traffic.
 *       old_channel = get_appchannel();
 *       set_best_appchannel( RF_GROUP_CHAT );
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ... 
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1096
#endif
char set_best_appchannel(char group_id);

#ifndef _SKIP_DOC
__cyos 716
int get_rfchannel(void);

__cyos 1150
void set_rfchannel(int no);

__cyos 718
bool get_server_mode(void);

__cyos 1152
void set_server_mode(bool enable);
#endif

/**
 *  @ingroup low_level_group
 *  Returns the application channel, by channel group index.
 *  @param group_id ID of the group to which the application belongs.
 *  @param channel index of the application in the group. 
 *  @return the application channel, by channel group index.
 *  @code
 *       #include <cybiko.h>
 *       #define PRIVATE_CHAT_CHANNEL    0x1     
 *       ...
 *       char old_channel;
 *       ...
 *       //  Sets new application channel.
 *       old_channel = get_appchannel();
 *       set_appchannel( get_groupchannel( RF_GROUP_CHAT, 
 *                                         PRIVATE_CHAT_CHANNEL ) ); 
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ...
 *  @endcode
 *  @see set_groupchannel.
 */
#ifndef _SKIP_DOC
__cyos 675
#endif
char get_groupchannel(char group_id, char channel);

/**
 *  @ingroup low_level_group
 *  Selects the application channel by channel group index.
 *  @param group_id ID of the group to which the application belongs.
 *  @param channel index of the application in the group. 
 *  @return returns the application channel.
 *  @code
 *       #include <cybiko.h>
 *       #define PRIVATE_CHAT_CHANNEL    0x1     
 *       ...
 *       char old_channel;
 *       ...
 *       //  Sets new application channel.
 *       old_channel = get_appchannel();
 *       set_groupchannel( RF_GROUP_CHAT, PRIVATE_CHAT_CHANNEL ); 
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ...
 *  @endcode
 *  @see get_groupchannel.
 */
#ifndef _SKIP_DOC
__cyos 1127
#endif
char set_groupchannel(char group_id, char channel);

/**
 *  @ingroup low_level_group
 *  Returns the number of channels in a specified group.
 *  @param group ID of the group.
 *  @return the number of channels in the group.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char max_chat_channel;
 *       chat current_channel;
 *       int people_in_chat = 0;
 *       ...
 *       //  Counts all people on chat channels.
 *       max_chat_channel = get_group_size( RF_GROUP_CHAT );
 *       for( current_channel = 0; current_channel < max_chat_channel; current_channel ++ )
 *       {
 *         people_in_chat += get_people_on_channel( get_groupchannel( RF_GROUP_CHAT, 
 *                                                                    current_channel ) );
 *       }  
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1315
#endif
char get_group_size(char group);

/**
 *  @ingroup low_level_group
 *  Returns the device count on the channel.
 *  @param channel the index of the RF channel. 
 *  @return the device count on the channel.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char max_chat_channel;
 *       char current_channel;
 *       int people_in_chat = 0;
 *       ...
 *       //  Counts all people on chat channels.
 *       max_chat_channel = get_group_size( RF_GROUP_CHAT );
 *       for( current_channel = 0; current_channel < max_chat_channel; current_channel ++ )
 *       {
 *         people_in_chat += get_people_on_channel( get_groupchannel( RF_GROUP_CHAT, 
 *                                                                    current_channel ) );
 *       }  
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 705
#endif
int get_people_on_channel(char channel);

/**
 *  @ingroup low_level_group
 *  Returns the CyID of the next device after prev_device (the CyIDs of 
 *  people in the vicinity). To retrieve the first device in the list, set 
 *  prev_device to zero.  
 *  @param prev_device CyID of the device in the previous CyID list of 
 *  people in the vicinity. 
 *  @return the CyID of a device.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char sz_str_ID[10];
 *       cyid_t current_device_id = 0;
 *       ...
 *       TRACE( "Devices around:" );
 *       while( current_device_id = get_people_around( current_device_id ) )
 *       {
 *         cyid2str( sz_str_ID, current_device_id );
 *         TRACE( "%s", sz_str_ID );
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 704
#endif
cyid_t get_people_around(cyid_t prev_device);

/**
 *  @ingroup low_level_group
 *  Selects the base channel, waits the timeout, and resets the work channel.
 *  @param timeout the amount of time to wait.
 *  @return None.
 *  @code
 *       #include <cybiko.h>
 *       #define PRIVATE_CHAT_CHANNEL    0x1
 *       ...
 *       char old_channel;
 *       ...
 *       //  Sets new application channel.
 *       old_channel = get_appchannel();
 *       set_groupchannel( RF_GROUP_CHAT, PRIVATE_CHAT_CHANNEL );
 *       ...
 *       update_people_around( 250 );
 *       //  Retrives people arround.
 *       ...
 *       //  Restores application channel.
 *       set_appchannel( old_channel );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1268
#endif
void update_people_around(clock_t timeout);

/**
 *  @ingroup low_level_group
 *  Returns TRUE if RF communications are paused.
 *  @return TRUE if RF communications are paused.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool rf_paused;
 *       ...
 *       //  Disables communications.
 *       rf_paused = is_rf_paused();
 *       rf_pause();
 *       ...
 *       //  Restores communications.
 *       if( ! rf_paused )
 *       {
 *         rf_resume();
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1342
#endif
bool is_rf_paused(void);

/**
 *  @ingroup low_level_group
 *  Pauses RF communications. Another way to stop RF.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool rf_paused;
 *       ...
 *       //  Disables communications.
 *       rf_paused = is_rf_paused();
 *       rf_pause();
 *       ...
 *       //  Restores communications.
 *       if( ! rf_paused )
 *       {
 *         rf_resume();
 *       }
 *       ...
 *  @endcode
 *  @see rf_resume.
 */
#ifndef _SKIP_DOC
__cyos 1344
#endif
void rf_pause(void);

/**
 *  @ingroup low_level_group
 *  Resumes RF communications after a pause.
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       bool rf_paused;
 *       ...
 *       //  Disables communications.
 *       rf_paused = is_rf_paused();
 *       rf_pause();
 *       ...
 *       //  Restores communications.
 *       if( ! rf_paused )
 *       {
 *         rf_resume();
 *       }
 *       ...
 *  @endcode
 *  @see rf_pause.
 */
#ifndef _SKIP_DOC
__cyos 1345
#endif
void rf_resume(void);

#ifndef _SKIP_DOC
__cyos 707
char* get_ping_listener(void);

__cyos 1141
void set_ping_listener(char* app_name);

__cyos 1039
void reset_ping_listener(char* app_name);

__cyos 1341
bool is_gate_visible(void);

__cyos 1340
long get_gate_cyid(void);
#endif

/**
 *  @ingroup low_level_group
 *  Returns the clock_t value of the last sent or received frame except sent 
 *  pings (last RF activity clock).
 *  @return the clock_t value of the last sent or received frame.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       int value;
 *       ...
 *       //  Initializes a pseudorandom sequence.
 *       srand( last_rf_clock() );
 *       ...
 *       value = random( 10 );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1343
#endif
clock_t last_rf_clock(void);

/**
 *  @ingroup low_level_group
 *  Returns the time the last frame was sent or received, except sent pings.
 *  @return the time the last frame was sent or received.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       struct Time last_message_time;
 *       ...
 *       Time_decode( &last_message_time, last_rf_time() );
 *       TRACE("Last message was sent or received at hour: %d, minutes: %d, seconds: %d",
 *             last_message_time.hour,
 *             last_message_time.minute,
 *             last_message_time.second );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 832
#endif
time_t last_rf_time(void);

#ifndef _SKIP_DOC
__cyos 1269
void update_rf_time(void);
#endif

/**
 *  @ingroup convertion_overview
 *  Converts CyID from cyid_t to text.
 *  @param sz_cy_id buffer for the CyID in text form.
 *  @param cy_id CyID to convert.
 *  @return None 
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       char sz_str_ID[10];
 *       cyid_t current_device_id = 0;
 *       ...
 *       TRACE( "Devices around:" );
 *       while( current_device_id = get_people_around( current_device_id ) )
 *       {
 *         cyid2str( sz_str_ID, current_device_id );
 *         TRACE( "%s", sz_str_ID );
 *       }
 *       ...
 *  @endcode
 *  @see str2cyid.
 */
#ifndef _SKIP_DOC
__cyos 484
#endif
void cyid2str(char* sz_cy_id, cyid_t cy_id);

/**
 *  @ingroup convertion_overview
 *  Converts CyID from text form to cyid_t.
 *  @param sz_cy_id CyID in text form.
 *  @return CyID in cyid_t form.
 *  @code
 *       #include <cybiko.h>
 *       #define MSG_CLIENT_PING    MSG_USER + 1
 *       ...
 *       struct Message* ptr_message;
 *       char sz_game_server_ID[10];
 *       ...
 *       //  Obtains sz_game_server_ID.
 *       ...
 *       ptr_message = Message_new( sizeof( struct Message ) );
 *       ptr_message->msgid = MSG_CLIENT_PING;
 *       Message_post(ptr_message, "game_server", str2cyid( sz_partner_ID ) );
 *       ...
 *  @endcode
 *  @see cyid2str.
 */
#ifndef _SKIP_DOC
__cyos 1205
#endif
cyid_t str2cyid(char* sz_cy_id);

/**
 *  @ingroup low_level_group
 *  Sets a special critical mode to disable network or internal task interrupts, 
 *  such as invite.
 *  @param set if TRUE, enables critical mode; any other answer disables it
 *  @return previous state.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( ! get_critical_mode() )
 *       {
 *         set_critical_mode( TRUE ); 
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "servers" 10
#endif
bool set_critical_mode(bool set);

/**
 *  @ingroup low_level_group
 *  Returns the critical mode state.
 *  @return the critical mode state.
 *  @code
 *       #include <cybiko.h>
 *       ...
 *       if( ! get_critical_mode() )
 *       {
 *         set_critical_mode( TRUE ); 
 *       }
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos "servers" 9 
#endif
bool get_critical_mode(void);


#endif  //  #ifndef __CYBER_RF_H_INCLUDED__
