/** 
 *  @file cyber-types.h
 *  Main CyOS types.
 */

#ifndef __CYBER_TYPES_H_INCLUDED__
#define __CYBER_TYPES_H_INCLUDED__

#ifndef _SKIP_DOC

#define TRUE  1
#define FALSE 0
#define NULL  ((char*)0)

typedef char bool;
#endif  //  #ifndef _SKIP_DOC

/**
 *  @typedef typedef int seek_t;
 *  Defines the initial position in the stream and must be one of the following: 
 *  SEEK_SET, SEEK_CUR or SEEK_END.
 *  @ingroup filesystem_overview
 *  @see SEEK_SET, SEEK_CUR, SEEK_END.
 */
typedef int seek_t;

/** 
 *  @typedef typedef long clock_t;
 *  Stores time values: used by the clock() function.
 *  @ingroup time_overview
 *  @see clock.
 */
typedef long clock_t;

/**
 *  @typedef typedef clock_t time_t; 
 *  Represents time values in the time() function and Time object.
 *  @ingroup time_overview
 *  @see time.
 */
typedef clock_t time_t;

/**
 *  @typedef typedef int size_t;
 *  Result of the sizeof operator.
 *  @ingroup misc_overview
 */
typedef int size_t;

/**
 *  @typedef typedef int priority_t;
 *  The thread priority. 
 *  Must be one of the following: PR_IDLE, PR_NORMAL or PR_RUNTIME.
 *  @ingroup multithreads_overview
 *  @see PR_IDLE, PR_NORMAL, PR_RUNTIME.
 */
typedef int priority_t;

/**
 *  @typedef typedef int click_kind_t;
 *  Keyboard click sound. 
 *  Must be one of the following: CK_NO, CK_BEEP_SHORT or CK_BEEP_SHORT.
 *  @ingroup keyboard_overview
 *  @see CK_NO, CK_BEEP_SHORT, CK_BEEP_SHORT.
 */
typedef int click_kind_t;

/** 
 *  @typedef typedef long cyid_t;
 *  Device's Cy ID
 *  @ingroup messaging_group
 */
typedef long cyid_t;

/**
 *  @typedef typedef int delivery_t;
 *  Message delivery state. 
 *  Must be one of the following: DL_NONE, DL_INQUEUE, DL_WAIT, DL_ABORT, 
 *  DL_TIMEOUT or DL_SUCCESS.
 *  @ingroup messaging_group
 *  @see DL_NONE, DL_INQUEUE, DL_WAIT, DL_ABORT, DL_TIMEOUT, DL_SUCCESS.
 */
typedef int delivery_t;

/**
 *  @typedef typedef int color_t;
 *  Display color.
 *  Must be one of the following: CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY or CLR_BLACK.
 *  @ingroup drawing_primitives_group
 *  @see CLR_WHITE, CLR_LTGRAY, CLR_DKGRAY, CLR_BLACK.
 */
typedef int color_t;

/**
 *  @typedef typedef int drawmode_t;
 *  Drawing mode.
 *  Must be one of the following: DM_PUT, DM_OR or DM_XOR.
 *  @ingroup drawing_primitives_group
 *  @see DM_PUT, DM_OR, DM_XOR.
 */
typedef int drawmode_t;

/**
 *  @brief A rectangle.
 *  @ingroup drawing_primitives_group
 *  A rectangle.
 *  @see drawing_primitives_group
 */
struct rect_t
{
  /**
   *  Rectangle's x-coordinate.
   */
  short x;

  /**
   *  Rectangle's y-coordinate.
   */
  short y;

  /**
   *  Rectangle's width.
   */
  short w;

  /**
   *  Rectangle's height.
   */
  short h;

#ifndef _SKIP_DOC
};
#endif

/**
 *  Initializes a rectangle.
 *  @param ptr_rect A pointer to the rect_t structure
 *  @param x The rectangle's x-coordinate
 *  @param y The rectangle's y-coordinate
 *  @param width The rectangle's width
 *  @param height The rectangle's height
 *  @return None
 *  @code
 *       #include <cybiko.h>
 *       #define VEHICLE_WIDTH          60
 *       #define VEHICLE_HEIGHT         20
 *       ...
 *       int x1;
 *       int x2;
 *       int y1;
 *       int y2;
 *       struct rect_t vehicle_1_boundary;
 *       struct rect_t vehicle_2_boundary;
 *       struct rect_t intersection;
 *       ...
 *       rect_set( &vehicle_1_boundary, x1, y1, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       rect_set( &vehicle_2_boundary, x2, y2, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       //  Checks collision of the vehicles.
 *       if( rect_and( &intersection, &vehicle_1_boundary, &vehicle_2_boundary ) )
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 1089
#endif
void rect_set(struct rect_t* ptr_rect, int x, int y, int width, int height);

/**
 *  Intersects two rectangles. 
 *  Sets the value ptr_rectangle into an intersection of two given rectangles.
 *  @param ptr_rectangle The intersection result
 *  @param ptr_rectangle_1 The first rectangle to intersect
 *  @param ptr_rectangle_2 The second rectangle to intersect
 *  @return TRUE if a valid rectangle intersection was obtained, or FALSE if no intersection 
 *  was found
 *  @code
 *       #include <cybiko.h>
 *       #define VEHICLE_WIDTH          60
 *       #define VEHICLE_HEIGHT         20
 *       ...
 *       int x1;
 *       int x2;
 *       int y1;
 *       int y2;
 *       struct rect_t vehicle_1_boundary;
 *       struct rect_t vehicle_2_boundary;
 *       struct rect_t intersection;
 *       ...
 *       rect_set( &vehicle_1_boundary, x1, y1, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       rect_set( &vehicle_2_boundary, x2, y2, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       //  Checks collision of the vehicles.
 *       if( rect_and( &intersection, &vehicle_1_boundary, &vehicle_2_boundary ) )
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 368
#endif
bool rect_and(struct rect_t* ptr_rectangle, struct rect_t* ptr_rectangle_1, struct rect_t* ptr_rectangle_2);

/**
 *  Merges two rectangles. 
 *  Sets the value ptr_rectangle into a union of two given rectangles.
 *  @param ptr_rectangle The merged result
 *  @param ptr_rectangle_1 The first rectangle to join
 *  @param ptr_rectangle_2 The second rectangle to join
 *  @return TRUE if a valid rectangle was obtained, or FALSE otherwise
 *  @code
 *       #include <cybiko.h>
 *       #define VEHICLE_WIDTH          60
 *       #define VEHICLE_HEIGHT         20
 *       ...
 *       int x1;
 *       int x2;
 *       int y1;
 *       int y2;
 *       struct rect_t vehicle_1_boundary;
 *       struct rect_t vehicle_2_boundary;
 *       struct rect_t common_boundary;
 *       ...
 *       rect_set( &vehicle_1_boundary, x1, y1, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       rect_set( &vehicle_2_boundary, x2, y2, VEHICLE_WIDTH, VEHICLE_HEIGHT );
 *       rect_or( &common_boundary, &vehicle_1_boundary, &vehicle_2_boundary );
 *       ...
 *  @endcode
 */
#ifndef _SKIP_DOC
__cyos 922
#endif
bool rect_or(struct rect_t* ptr_rectangle, struct rect_t* ptr_rectangle_1, struct rect_t* ptr_rectangle_2);

#ifdef _SKIP_DOC
};
#endif

#endif  //  #ifndef __CYBER_TYPES_H_INCLUDED__


